/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.accumulo.tserver;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThrows;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.accumulo.core.client.SampleNotPresentException;
import org.apache.accumulo.core.client.impl.BaseIteratorEnvironment;
import org.apache.accumulo.core.client.sample.RowSampler;
import org.apache.accumulo.core.client.sample.Sampler;
import org.apache.accumulo.core.client.sample.SamplerConfiguration;
import org.apache.accumulo.core.conf.ConfigurationCopy;
import org.apache.accumulo.core.conf.DefaultConfiguration;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.data.ArrayByteSequence;
import org.apache.accumulo.core.data.ByteSequence;
import org.apache.accumulo.core.data.Key;
import org.apache.accumulo.core.data.Mutation;
import org.apache.accumulo.core.data.Range;
import org.apache.accumulo.core.data.Value;
import org.apache.accumulo.core.iterators.IterationInterruptedException;
import org.apache.accumulo.core.iterators.SortedKeyValueIterator;
import org.apache.accumulo.core.iterators.system.ColumnFamilySkippingIterator;
import org.apache.accumulo.core.sample.impl.SamplerConfigurationImpl;
import org.apache.accumulo.core.sample.impl.SamplerFactory;
import org.apache.accumulo.core.util.LocalityGroupUtil;
import org.apache.accumulo.core.util.LocalityGroupUtil.LocalityGroupConfigurationError;
import org.apache.accumulo.server.client.HdfsZooInstance;
import org.apache.accumulo.server.conf.ZooConfiguration;
import org.apache.accumulo.tserver.InMemoryMap.MemoryIterator;
import org.apache.hadoop.io.Text;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import com.google.common.collect.ImmutableMap;

public class InMemoryMapTest {

  private static class SampleIE extends BaseIteratorEnvironment {

    private final SamplerConfiguration sampleConfig;

    public SampleIE() {
      this.sampleConfig = null;
    }

    public SampleIE(SamplerConfigurationImpl sampleConfig) {
      this.sampleConfig = sampleConfig.toSamplerConfiguration();
    }

    @Override
    public boolean isSamplingEnabled() {
      return sampleConfig != null;
    }

    @Override
    public SamplerConfiguration getSamplerConfiguration() {
      return sampleConfig;
    }
  }

  @BeforeClass
  public static void setUp() throws Exception {
    // suppress log messages having to do with not having an instance
    Logger.getLogger(ZooConfiguration.class).setLevel(Level.OFF);
    Logger.getLogger(HdfsZooInstance.class).setLevel(Level.OFF);
  }

  @Rule
  public TemporaryFolder tempFolder =
      new TemporaryFolder(new File(System.getProperty("user.dir") + "/target"));

  public void mutate(InMemoryMap imm, String row, String column, long ts) {
    Mutation m = new Mutation(new Text(row));
    String[] sa = column.split(":");
    m.putDelete(new Text(sa[0]), new Text(sa[1]), ts);

    imm.mutate(Collections.singletonList(m));
  }

  public void mutate(InMemoryMap imm, String row, String column, long ts, String value) {
    Mutation m = new Mutation(new Text(row));
    String[] sa = column.split(":");
    m.put(new Text(sa[0]), new Text(sa[1]), ts, new Value(value.getBytes()));

    imm.mutate(Collections.singletonList(m));
  }

  static Key newKey(String row, String column, long ts) {
    String[] sa = column.split(":");
    Key k = new Key(new Text(row), new Text(sa[0]), new Text(sa[1]), ts);
    return k;
  }

  static void testAndCallNext(SortedKeyValueIterator<Key,Value> dc, String row, String column,
      int ts, String val) throws IOException {
    assertTrue(dc.hasTop());
    assertEquals(newKey(row, column, ts), dc.getTopKey());
    assertEquals(new Value(val.getBytes()), dc.getTopValue());
    dc.next();

  }

  static void assertEqualsNoNext(SortedKeyValueIterator<Key,Value> dc, String row, String column,
      int ts, String val) throws IOException {
    assertTrue(dc.hasTop());
    assertEquals(newKey(row, column, ts), dc.getTopKey());
    assertEquals(new Value(val.getBytes()), dc.getTopValue());

  }

  static Set<ByteSequence> newCFSet(String... cfs) {
    HashSet<ByteSequence> cfSet = new HashSet<>();
    for (String cf : cfs) {
      cfSet.add(new ArrayByteSequence(cf));
    }
    return cfSet;
  }

  static Set<Text> toTextSet(String... cfs) {
    HashSet<Text> cfSet = new HashSet<>();
    for (String cf : cfs) {
      cfSet.add(new Text(cf));
    }
    return cfSet;
  }

  static ConfigurationCopy newConfig(String memDumpDir) {
    ConfigurationCopy config = new ConfigurationCopy(DefaultConfiguration.getInstance());
    config.set(Property.TSERV_NATIVEMAP_ENABLED, "" + false);
    config.set(Property.TSERV_MEMDUMP_DIR, memDumpDir);
    return config;
  }

  static InMemoryMap newInMemoryMap(boolean useNative, String memDumpDir)
      throws LocalityGroupConfigurationError {
    ConfigurationCopy config = new ConfigurationCopy(DefaultConfiguration.getInstance());
    config.set(Property.TSERV_NATIVEMAP_ENABLED, "" + useNative);
    config.set(Property.TSERV_MEMDUMP_DIR, memDumpDir);
    return new InMemoryMap(config, "--TEST--");
  }

  @Test
  public void test2() throws Exception {
    InMemoryMap imm = newInMemoryMap(false, tempFolder.newFolder().getAbsolutePath());

    MemoryIterator ski1 = imm.skvIterator(null);
    mutate(imm, "r1", "foo:cq1", 3, "bar1");
    MemoryIterator ski2 = imm.skvIterator(null);

    ski1.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);
    assertFalse(ski1.hasTop());

    ski2.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);
    assertTrue(ski2.hasTop());
    testAndCallNext(ski2, "r1", "foo:cq1", 3, "bar1");
    assertFalse(ski2.hasTop());

  }

  @Test
  public void test3() throws Exception {
    InMemoryMap imm = newInMemoryMap(false, tempFolder.newFolder().getAbsolutePath());

    mutate(imm, "r1", "foo:cq1", 3, "bar1");
    mutate(imm, "r1", "foo:cq1", 3, "bar2");
    MemoryIterator ski1 = imm.skvIterator(null);
    mutate(imm, "r1", "foo:cq1", 3, "bar3");

    mutate(imm, "r3", "foo:cq1", 3, "bar9");
    mutate(imm, "r3", "foo:cq1", 3, "bara");

    MemoryIterator ski2 = imm.skvIterator(null);

    ski1.seek(new Range(new Text("r1")), LocalityGroupUtil.EMPTY_CF_SET, false);
    testAndCallNext(ski1, "r1", "foo:cq1", 3, "bar2");
    testAndCallNext(ski1, "r1", "foo:cq1", 3, "bar1");
    assertFalse(ski1.hasTop());

    ski2.seek(new Range(new Text("r3")), LocalityGroupUtil.EMPTY_CF_SET, false);
    testAndCallNext(ski2, "r3", "foo:cq1", 3, "bara");
    testAndCallNext(ski2, "r3", "foo:cq1", 3, "bar9");
    assertFalse(ski1.hasTop());

  }

  @Test
  public void test4() throws Exception {
    InMemoryMap imm = newInMemoryMap(false, tempFolder.newFolder().getAbsolutePath());

    mutate(imm, "r1", "foo:cq1", 3, "bar1");
    mutate(imm, "r1", "foo:cq1", 3, "bar2");
    MemoryIterator ski1 = imm.skvIterator(null);
    mutate(imm, "r1", "foo:cq1", 3, "bar3");

    imm.delete(0);

    ski1.seek(new Range(new Text("r1")), LocalityGroupUtil.EMPTY_CF_SET, false);
    testAndCallNext(ski1, "r1", "foo:cq1", 3, "bar2");
    testAndCallNext(ski1, "r1", "foo:cq1", 3, "bar1");
    assertFalse(ski1.hasTop());

    ski1.seek(new Range(new Text("r1")), LocalityGroupUtil.EMPTY_CF_SET, false);
    testAndCallNext(ski1, "r1", "foo:cq1", 3, "bar2");
    testAndCallNext(ski1, "r1", "foo:cq1", 3, "bar1");
    assertFalse(ski1.hasTop());

    ski1.seek(new Range(new Text("r2")), LocalityGroupUtil.EMPTY_CF_SET, false);
    assertFalse(ski1.hasTop());

    ski1.seek(new Range(newKey("r1", "foo:cq1", 3), null), LocalityGroupUtil.EMPTY_CF_SET, false);
    testAndCallNext(ski1, "r1", "foo:cq1", 3, "bar2");
    testAndCallNext(ski1, "r1", "foo:cq1", 3, "bar1");
    assertFalse(ski1.hasTop());

    ski1.close();
  }

  @Test
  public void testDecodeValueModification() throws Exception {
    // This test case is the fix for ACCUMULO-4483
    InMemoryMap imm = newInMemoryMap(false, tempFolder.newFolder().getAbsolutePath());

    mutate(imm, "r1", "foo:cq1", 3, "");
    MemoryIterator ski1 = imm.skvIterator(null);

    imm.delete(0);

    ski1.seek(new Range(new Text("r1")), LocalityGroupUtil.EMPTY_CF_SET, false);
    assertEqualsNoNext(ski1, "r1", "foo:cq1", 3, "");
    ski1.seek(new Range(new Text("r1")), LocalityGroupUtil.EMPTY_CF_SET, false);
    testAndCallNext(ski1, "r1", "foo:cq1", 3, "");
    assertFalse(ski1.hasTop());

    ski1.close();
  }

  @Test
  public void test5() throws Exception {
    InMemoryMap imm = newInMemoryMap(false, tempFolder.newFolder().getAbsolutePath());

    mutate(imm, "r1", "foo:cq1", 3, "bar1");
    mutate(imm, "r1", "foo:cq1", 3, "bar2");
    mutate(imm, "r1", "foo:cq1", 3, "bar3");

    MemoryIterator ski1 = imm.skvIterator(null);
    ski1.seek(new Range(new Text("r1")), LocalityGroupUtil.EMPTY_CF_SET, false);
    testAndCallNext(ski1, "r1", "foo:cq1", 3, "bar3");

    imm.delete(0);

    testAndCallNext(ski1, "r1", "foo:cq1", 3, "bar2");
    testAndCallNext(ski1, "r1", "foo:cq1", 3, "bar1");
    assertFalse(ski1.hasTop());

    ski1.close();

    imm = newInMemoryMap(false, tempFolder.newFolder().getAbsolutePath());

    mutate(imm, "r1", "foo:cq1", 3, "bar1");
    mutate(imm, "r1", "foo:cq2", 3, "bar2");
    mutate(imm, "r1", "foo:cq3", 3, "bar3");

    ski1 = imm.skvIterator(null);
    ski1.seek(new Range(new Text("r1")), LocalityGroupUtil.EMPTY_CF_SET, false);
    testAndCallNext(ski1, "r1", "foo:cq1", 3, "bar1");

    imm.delete(0);

    testAndCallNext(ski1, "r1", "foo:cq2", 3, "bar2");
    testAndCallNext(ski1, "r1", "foo:cq3", 3, "bar3");
    assertFalse(ski1.hasTop());

    ski1.close();
  }

  @Test
  public void test6() throws Exception {
    InMemoryMap imm = newInMemoryMap(false, tempFolder.newFolder().getAbsolutePath());

    mutate(imm, "r1", "foo:cq1", 3, "bar1");
    mutate(imm, "r1", "foo:cq2", 3, "bar2");
    mutate(imm, "r1", "foo:cq3", 3, "bar3");
    mutate(imm, "r1", "foo:cq4", 3, "bar4");

    MemoryIterator ski1 = imm.skvIterator(null);

    mutate(imm, "r1", "foo:cq5", 3, "bar5");

    SortedKeyValueIterator<Key,Value> dc = ski1.deepCopy(new SampleIE());

    ski1.seek(new Range(newKey("r1", "foo:cq1", 3), null), LocalityGroupUtil.EMPTY_CF_SET, false);
    testAndCallNext(ski1, "r1", "foo:cq1", 3, "bar1");

    dc.seek(new Range(newKey("r1", "foo:cq2", 3), null), LocalityGroupUtil.EMPTY_CF_SET, false);
    testAndCallNext(dc, "r1", "foo:cq2", 3, "bar2");

    imm.delete(0);

    testAndCallNext(ski1, "r1", "foo:cq2", 3, "bar2");
    testAndCallNext(dc, "r1", "foo:cq3", 3, "bar3");
    testAndCallNext(ski1, "r1", "foo:cq3", 3, "bar3");
    testAndCallNext(dc, "r1", "foo:cq4", 3, "bar4");
    testAndCallNext(ski1, "r1", "foo:cq4", 3, "bar4");
    assertFalse(ski1.hasTop());
    assertFalse(dc.hasTop());

    ski1.seek(new Range(newKey("r1", "foo:cq3", 3), null), LocalityGroupUtil.EMPTY_CF_SET, false);

    dc.seek(new Range(newKey("r1", "foo:cq4", 3), null), LocalityGroupUtil.EMPTY_CF_SET, false);
    testAndCallNext(dc, "r1", "foo:cq4", 3, "bar4");
    assertFalse(dc.hasTop());

    testAndCallNext(ski1, "r1", "foo:cq3", 3, "bar3");
    testAndCallNext(ski1, "r1", "foo:cq4", 3, "bar4");
    assertFalse(ski1.hasTop());
    assertFalse(dc.hasTop());

    ski1.close();
  }

  private void deepCopyAndDelete(int interleaving, boolean interrupt) throws Exception {
    // interleaving == 0 intentionally omitted, this runs the test w/o deleting in mem map

    InMemoryMap imm = newInMemoryMap(false, tempFolder.newFolder().getAbsolutePath());

    mutate(imm, "r1", "foo:cq1", 3, "bar1");
    mutate(imm, "r1", "foo:cq2", 3, "bar2");

    MemoryIterator ski1 = imm.skvIterator(null);

    AtomicBoolean iflag = new AtomicBoolean(false);
    ski1.setInterruptFlag(iflag);

    if (interleaving == 1) {
      imm.delete(0);
      if (interrupt)
        iflag.set(true);
    }

    SortedKeyValueIterator<Key,Value> dc = ski1.deepCopy(new SampleIE());

    if (interleaving == 2) {
      imm.delete(0);
      if (interrupt)
        iflag.set(true);
    }

    dc.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);
    ski1.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);

    if (interleaving == 3) {
      imm.delete(0);
      if (interrupt)
        iflag.set(true);
    }

    testAndCallNext(dc, "r1", "foo:cq1", 3, "bar1");
    testAndCallNext(ski1, "r1", "foo:cq1", 3, "bar1");
    dc.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);

    if (interleaving == 4) {
      imm.delete(0);
      if (interrupt)
        iflag.set(true);
    }

    testAndCallNext(ski1, "r1", "foo:cq2", 3, "bar2");
    testAndCallNext(dc, "r1", "foo:cq1", 3, "bar1");
    testAndCallNext(dc, "r1", "foo:cq2", 3, "bar2");
    assertFalse(dc.hasTop());
    assertFalse(ski1.hasTop());

    if (interrupt)
      dc.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);
  }

  @Test
  public void testDeepCopyAndDelete() throws Exception {
    for (int i = 0; i <= 4; i++)
      deepCopyAndDelete(i, false);

    for (int i = 1; i <= 4; i++)
      try {
        deepCopyAndDelete(i, true);
        fail("i = " + i);
      } catch (IterationInterruptedException iie) {}
  }

  @Test
  public void testBug1() throws Exception {
    InMemoryMap imm = newInMemoryMap(false, tempFolder.newFolder().getAbsolutePath());

    for (int i = 0; i < 20; i++) {
      mutate(imm, "r1", "foo:cq" + i, 3, "bar" + i);
    }

    for (int i = 0; i < 20; i++) {
      mutate(imm, "r2", "foo:cq" + i, 3, "bar" + i);
    }

    MemoryIterator ski1 = imm.skvIterator(null);
    ColumnFamilySkippingIterator cfsi = new ColumnFamilySkippingIterator(ski1);

    imm.delete(0);

    ArrayList<ByteSequence> columns = new ArrayList<>();
    columns.add(new ArrayByteSequence("bar"));

    // this seek resulted in an infinite loop before a bug was fixed
    cfsi.seek(new Range("r1"), columns, true);

    assertFalse(cfsi.hasTop());

    ski1.close();
  }

  @Test
  public void testSeekBackWards() throws Exception {
    InMemoryMap imm = newInMemoryMap(false, tempFolder.newFolder().getAbsolutePath());

    mutate(imm, "r1", "foo:cq1", 3, "bar1");
    mutate(imm, "r1", "foo:cq2", 3, "bar2");
    mutate(imm, "r1", "foo:cq3", 3, "bar3");
    mutate(imm, "r1", "foo:cq4", 3, "bar4");

    MemoryIterator skvi1 = imm.skvIterator(null);

    skvi1.seek(new Range(newKey("r1", "foo:cq3", 3), null), LocalityGroupUtil.EMPTY_CF_SET, false);
    testAndCallNext(skvi1, "r1", "foo:cq3", 3, "bar3");

    skvi1.seek(new Range(newKey("r1", "foo:cq1", 3), null), LocalityGroupUtil.EMPTY_CF_SET, false);
    testAndCallNext(skvi1, "r1", "foo:cq1", 3, "bar1");

  }

  @Test
  public void testDuplicateKey() throws Exception {
    InMemoryMap imm = newInMemoryMap(false, tempFolder.newFolder().getAbsolutePath());

    Mutation m = new Mutation(new Text("r1"));
    m.put(new Text("foo"), new Text("cq"), 3, new Value("v1".getBytes()));
    m.put(new Text("foo"), new Text("cq"), 3, new Value("v2".getBytes()));
    imm.mutate(Collections.singletonList(m));

    MemoryIterator skvi1 = imm.skvIterator(null);
    skvi1.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);
    testAndCallNext(skvi1, "r1", "foo:cq", 3, "v2");
    testAndCallNext(skvi1, "r1", "foo:cq", 3, "v1");
  }

  private static final Logger log = Logger.getLogger(InMemoryMapTest.class);

  static long sum(long[] counts) {
    long result = 0;
    for (long count : counts)
      result += count;
    return result;
  }

  // - hard to get this timing test to run well on apache build machines
  @Test
  @Ignore
  public void parallelWriteSpeed() throws Exception {
    List<Double> timings = new ArrayList<>();
    for (int threads : new int[] {1, 2, 16, /* 64, 256 */}) {
      final long now = System.currentTimeMillis();
      final long counts[] = new long[threads];
      final InMemoryMap imm = newInMemoryMap(false, tempFolder.newFolder().getAbsolutePath());
      ExecutorService e = Executors.newFixedThreadPool(threads);
      for (int j = 0; j < threads; j++) {
        final int threadId = j;
        e.execute(new Runnable() {
          @Override
          public void run() {
            while (System.currentTimeMillis() - now < 1000) {
              for (int k = 0; k < 1000; k++) {
                Mutation m = new Mutation("row");
                m.put("cf", "cq", new Value("v".getBytes()));
                List<Mutation> mutations = Collections.singletonList(m);
                imm.mutate(mutations);
                counts[threadId]++;
              }
            }
          }
        });
      }
      e.shutdown();
      e.awaitTermination(10, TimeUnit.SECONDS);
      imm.delete(10000);
      double mutationsPerSecond = sum(counts) / ((System.currentTimeMillis() - now) / 1000.);
      timings.add(mutationsPerSecond);
      log.info(
          String.format("%.1f mutations per second with %d threads", mutationsPerSecond, threads));
    }
    // verify that more threads doesn't go a lot faster, or a lot slower than one thread
    for (Double timing : timings) {
      double ratioFirst = timings.get(0) / timing;
      assertTrue(ratioFirst < 3);
      assertTrue(ratioFirst > 0.3);
    }
  }

  @Test
  public void testLocalityGroups() throws Exception {
    ConfigurationCopy config = newConfig(tempFolder.newFolder().getAbsolutePath());
    config.set(Property.TABLE_LOCALITY_GROUP_PREFIX + "lg1",
        LocalityGroupUtil.encodeColumnFamilies(toTextSet("cf1", "cf2")));
    config.set(Property.TABLE_LOCALITY_GROUP_PREFIX + "lg2",
        LocalityGroupUtil.encodeColumnFamilies(toTextSet("cf3", "cf4")));
    config.set(Property.TABLE_LOCALITY_GROUPS.getKey(), "lg1,lg2");

    InMemoryMap imm = new InMemoryMap(config, "--TEST--");

    Mutation m1 = new Mutation("r1");
    m1.put("cf1", "x", 2, "1");
    m1.put("cf1", "y", 2, "2");
    m1.put("cf3", "z", 2, "3");
    m1.put("foo", "b", 2, "9");

    Mutation m2 = new Mutation("r2");
    m2.put("cf2", "x", 3, "5");

    Mutation m3 = new Mutation("r3");
    m3.put("foo", "b", 4, "6");

    Mutation m4 = new Mutation("r4");
    m4.put("foo", "b", 5, "7");
    m4.put("cf4", "z", 5, "8");

    Mutation m5 = new Mutation("r5");
    m5.put("cf3", "z", 6, "A");
    m5.put("cf4", "z", 6, "B");

    imm.mutate(Arrays.asList(m1, m2, m3, m4, m5));

    MemoryIterator iter1 = imm.skvIterator(null);

    seekLocalityGroups(iter1);
    SortedKeyValueIterator<Key,Value> dc1 = iter1.deepCopy(new SampleIE());
    seekLocalityGroups(dc1);

    assertTrue(imm.getNumEntries() == 10);
    assertTrue(imm.estimatedSizeInBytes() > 0);

    imm.delete(0);

    seekLocalityGroups(iter1);
    seekLocalityGroups(dc1);
    // TODO uncomment following when ACCUMULO-1628 is fixed
    // seekLocalityGroups(iter1.deepCopy(null));
  }

  @Test
  public void testSample() throws Exception {

    SamplerConfigurationImpl sampleConfig = new SamplerConfigurationImpl(RowSampler.class.getName(),
        ImmutableMap.of("hasher", "murmur3_32", "modulus", "7"));
    Sampler sampler = SamplerFactory.newSampler(sampleConfig, DefaultConfiguration.getInstance());

    ConfigurationCopy config1 = newConfig(tempFolder.newFolder().getAbsolutePath());
    for (Entry<String,String> entry : sampleConfig.toTablePropertiesMap().entrySet()) {
      config1.set(entry.getKey(), entry.getValue());
    }

    ConfigurationCopy config2 = newConfig(tempFolder.newFolder().getAbsolutePath());
    config2.set(Property.TABLE_LOCALITY_GROUP_PREFIX + "lg1",
        LocalityGroupUtil.encodeColumnFamilies(toTextSet("cf2")));
    config2.set(Property.TABLE_LOCALITY_GROUPS.getKey(), "lg1");
    for (Entry<String,String> entry : sampleConfig.toTablePropertiesMap().entrySet()) {
      config2.set(entry.getKey(), entry.getValue());
    }

    for (ConfigurationCopy config : Arrays.asList(config1, config2)) {

      InMemoryMap imm = new InMemoryMap(config, "--TEST--");

      TreeMap<Key,Value> expectedSample = new TreeMap<>();
      TreeMap<Key,Value> expectedAll = new TreeMap<>();
      TreeMap<Key,Value> expectedNone = new TreeMap<>();

      MemoryIterator iter0 = imm.skvIterator(sampleConfig);

      for (int r = 0; r < 100; r++) {
        String row = String.format("r%06d", r);
        mutate(imm, row, "cf1:cq1", 5, "v" + (2 * r), sampler, expectedSample, expectedAll);
        mutate(imm, row, "cf2:cq2", 5, "v" + ((2 * r) + 1), sampler, expectedSample, expectedAll);
      }

      assertTrue(expectedSample.size() > 0);

      MemoryIterator iter1 = imm.skvIterator(sampleConfig);
      MemoryIterator iter2 = imm.skvIterator(null);
      SortedKeyValueIterator<Key,Value> iter0dc1 = iter0.deepCopy(new SampleIE());
      SortedKeyValueIterator<Key,Value> iter0dc2 = iter0.deepCopy(new SampleIE(sampleConfig));
      SortedKeyValueIterator<Key,Value> iter1dc1 = iter1.deepCopy(new SampleIE());
      SortedKeyValueIterator<Key,Value> iter1dc2 = iter1.deepCopy(new SampleIE(sampleConfig));
      SortedKeyValueIterator<Key,Value> iter2dc1 = iter2.deepCopy(new SampleIE());
      SortedKeyValueIterator<Key,Value> iter2dc2 = iter2.deepCopy(new SampleIE(sampleConfig));

      assertEquals(expectedNone, readAll(iter0));
      assertEquals(expectedNone, readAll(iter0dc1));
      assertEquals(expectedNone, readAll(iter0dc2));
      assertEquals(expectedSample, readAll(iter1));
      assertEquals(expectedAll, readAll(iter2));
      assertEquals(expectedAll, readAll(iter1dc1));
      assertEquals(expectedAll, readAll(iter2dc1));
      assertEquals(expectedSample, readAll(iter1dc2));
      assertEquals(expectedSample, readAll(iter2dc2));

      imm.delete(0);

      assertEquals(expectedNone, readAll(iter0));
      assertEquals(expectedNone, readAll(iter0dc1));
      assertEquals(expectedNone, readAll(iter0dc2));
      assertEquals(expectedSample, readAll(iter1));
      assertEquals(expectedAll, readAll(iter2));
      assertEquals(expectedAll, readAll(iter1dc1));
      assertEquals(expectedAll, readAll(iter2dc1));
      assertEquals(expectedSample, readAll(iter1dc2));
      assertEquals(expectedSample, readAll(iter2dc2));

      SortedKeyValueIterator<Key,Value> iter0dc3 = iter0.deepCopy(new SampleIE());
      SortedKeyValueIterator<Key,Value> iter0dc4 = iter0.deepCopy(new SampleIE(sampleConfig));
      SortedKeyValueIterator<Key,Value> iter1dc3 = iter1.deepCopy(new SampleIE());
      SortedKeyValueIterator<Key,Value> iter1dc4 = iter1.deepCopy(new SampleIE(sampleConfig));
      SortedKeyValueIterator<Key,Value> iter2dc3 = iter2.deepCopy(new SampleIE());
      SortedKeyValueIterator<Key,Value> iter2dc4 = iter2.deepCopy(new SampleIE(sampleConfig));

      assertEquals(expectedNone, readAll(iter0dc3));
      assertEquals(expectedNone, readAll(iter0dc4));
      assertEquals(expectedAll, readAll(iter1dc3));
      assertEquals(expectedAll, readAll(iter2dc3));
      assertEquals(expectedSample, readAll(iter1dc4));
      assertEquals(expectedSample, readAll(iter2dc4));

      iter1.close();
      iter2.close();
    }
  }

  @Test
  public void testInterruptingSample() throws Exception {
    runInterruptSampleTest(false, false, false);
    runInterruptSampleTest(false, true, false);
    runInterruptSampleTest(true, false, false);
    runInterruptSampleTest(true, true, false);
    runInterruptSampleTest(true, true, true);
  }

  private void runInterruptSampleTest(boolean deepCopy, boolean delete, boolean dcAfterDelete)
      throws Exception {
    SamplerConfigurationImpl sampleConfig1 = new SamplerConfigurationImpl(
        RowSampler.class.getName(), ImmutableMap.of("hasher", "murmur3_32", "modulus", "2"));
    Sampler sampler = SamplerFactory.newSampler(sampleConfig1, DefaultConfiguration.getInstance());

    ConfigurationCopy config1 = newConfig(tempFolder.newFolder().getAbsolutePath());
    for (Entry<String,String> entry : sampleConfig1.toTablePropertiesMap().entrySet()) {
      config1.set(entry.getKey(), entry.getValue());
    }

    InMemoryMap imm = new InMemoryMap(config1, "--TEST--");

    TreeMap<Key,Value> expectedSample = new TreeMap<>();
    TreeMap<Key,Value> expectedAll = new TreeMap<>();

    for (int r = 0; r < 1000; r++) {
      String row = String.format("r%06d", r);
      mutate(imm, row, "cf1:cq1", 5, "v" + (2 * r), sampler, expectedSample, expectedAll);
      mutate(imm, row, "cf2:cq2", 5, "v" + ((2 * r) + 1), sampler, expectedSample, expectedAll);
    }

    assertTrue(expectedSample.size() > 0);

    MemoryIterator miter = imm.skvIterator(sampleConfig1);
    AtomicBoolean iFlag = new AtomicBoolean(false);
    miter.setInterruptFlag(iFlag);
    SortedKeyValueIterator<Key,Value> iter = miter;

    if (delete && !dcAfterDelete) {
      imm.delete(0);
    }

    if (deepCopy) {
      iter = iter.deepCopy(new SampleIE(sampleConfig1));
    }

    if (delete && dcAfterDelete) {
      imm.delete(0);
    }

    assertEquals(expectedSample, readAll(iter));
    iFlag.set(true);
    try {
      readAll(iter);
      fail();
    } catch (IterationInterruptedException iie) {}

    miter.close();
  }

  private void mutate(InMemoryMap imm, String row, String cols, int ts, String val, Sampler sampler,
      TreeMap<Key,Value> expectedSample, TreeMap<Key,Value> expectedAll) {
    mutate(imm, row, cols, ts, val);
    Key k1 = newKey(row, cols, ts);
    if (sampler.accept(k1)) {
      expectedSample.put(k1, new Value(val.getBytes()));
    }
    expectedAll.put(k1, new Value(val.getBytes()));
  }

  @Test(expected = SampleNotPresentException.class)
  public void testDifferentSampleConfig() throws Exception {
    SamplerConfigurationImpl sampleConfig = new SamplerConfigurationImpl(RowSampler.class.getName(),
        ImmutableMap.of("hasher", "murmur3_32", "modulus", "7"));

    ConfigurationCopy config1 = newConfig(tempFolder.newFolder().getAbsolutePath());
    for (Entry<String,String> entry : sampleConfig.toTablePropertiesMap().entrySet()) {
      config1.set(entry.getKey(), entry.getValue());
    }

    InMemoryMap imm = new InMemoryMap(config1, "--TEST--");

    mutate(imm, "r", "cf:cq", 5, "b");

    SamplerConfigurationImpl sampleConfig2 = new SamplerConfigurationImpl(
        RowSampler.class.getName(), ImmutableMap.of("hasher", "murmur3_32", "modulus", "9"));
    MemoryIterator iter = imm.skvIterator(sampleConfig2);
    iter.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);
  }

  @Test(expected = SampleNotPresentException.class)
  public void testNoSampleConfig() throws Exception {
    InMemoryMap imm = newInMemoryMap(false, tempFolder.newFolder().getAbsolutePath());

    mutate(imm, "r", "cf:cq", 5, "b");

    SamplerConfigurationImpl sampleConfig2 = new SamplerConfigurationImpl(
        RowSampler.class.getName(), ImmutableMap.of("hasher", "murmur3_32", "modulus", "9"));
    MemoryIterator iter = imm.skvIterator(sampleConfig2);
    iter.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);
  }

  @Test
  public void testEmptyNoSampleConfig() throws Exception {
    InMemoryMap imm = newInMemoryMap(false, tempFolder.newFolder().getAbsolutePath());

    SamplerConfigurationImpl sampleConfig2 = new SamplerConfigurationImpl(
        RowSampler.class.getName(), ImmutableMap.of("hasher", "murmur3_32", "modulus", "9"));

    // when in mem map is empty should be able to get sample iterator with any sample config
    MemoryIterator iter = imm.skvIterator(sampleConfig2);
    iter.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);
    assertFalse(iter.hasTop());
  }

  @Test
  public void testDeferredSamplerCreation() throws Exception {
    SamplerConfigurationImpl sampleConfig1 = new SamplerConfigurationImpl(
        RowSampler.class.getName(), ImmutableMap.of("hasher", "murmur3_32", "modulus", "9"));

    ConfigurationCopy config1 = newConfig(tempFolder.newFolder().getAbsolutePath());
    for (Entry<String,String> entry : sampleConfig1.toTablePropertiesMap().entrySet()) {
      config1.set(entry.getKey(), entry.getValue());
    }

    InMemoryMap imm = new InMemoryMap(config1, "--TEST--");

    // change sampler config after creating in mem map.
    SamplerConfigurationImpl sampleConfig2 = new SamplerConfigurationImpl(
        RowSampler.class.getName(), ImmutableMap.of("hasher", "murmur3_32", "modulus", "7"));
    for (Entry<String,String> entry : sampleConfig2.toTablePropertiesMap().entrySet()) {
      config1.set(entry.getKey(), entry.getValue());
    }

    TreeMap<Key,Value> expectedSample = new TreeMap<>();
    TreeMap<Key,Value> expectedAll = new TreeMap<>();
    Sampler sampler = SamplerFactory.newSampler(sampleConfig2, config1);

    for (int i = 0; i < 100; i++) {
      mutate(imm, "r" + i, "cf:cq", 5, "v" + i, sampler, expectedSample, expectedAll);
    }

    MemoryIterator iter = imm.skvIterator(sampleConfig2);
    iter.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);
    assertEquals(expectedSample, readAll(iter));

    SortedKeyValueIterator<Key,Value> dc = iter.deepCopy(new SampleIE(sampleConfig2));
    dc.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);
    assertEquals(expectedSample, readAll(dc));

    iter = imm.skvIterator(null);
    iter.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);
    assertEquals(expectedAll, readAll(iter));

    iter = imm.skvIterator(sampleConfig1);
    final MemoryIterator finalIter = iter;
    assertThrows(SampleNotPresentException.class,
        () -> finalIter.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false));
  }

  private TreeMap<Key,Value> readAll(SortedKeyValueIterator<Key,Value> iter) throws IOException {
    iter.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);

    TreeMap<Key,Value> actual = new TreeMap<>();
    while (iter.hasTop()) {
      actual.put(iter.getTopKey(), iter.getTopValue());
      iter.next();
    }
    return actual;
  }

  private void seekLocalityGroups(SortedKeyValueIterator<Key,Value> iter1) throws IOException {
    iter1.seek(new Range(), newCFSet("cf1"), true);
    testAndCallNext(iter1, "r1", "cf1:x", 2, "1");
    testAndCallNext(iter1, "r1", "cf1:y", 2, "2");
    testAndCallNext(iter1, "r2", "cf2:x", 3, "5");
    assertFalse(iter1.hasTop());

    iter1.seek(new Range("r2", "r4"), newCFSet("cf1"), true);
    testAndCallNext(iter1, "r2", "cf2:x", 3, "5");
    assertFalse(iter1.hasTop());

    iter1.seek(new Range(), newCFSet("cf3"), true);
    testAndCallNext(iter1, "r1", "cf3:z", 2, "3");
    testAndCallNext(iter1, "r4", "cf4:z", 5, "8");
    testAndCallNext(iter1, "r5", "cf3:z", 6, "A");
    testAndCallNext(iter1, "r5", "cf4:z", 6, "B");
    assertFalse(iter1.hasTop());

    iter1.seek(new Range(), newCFSet("foo"), true);
    testAndCallNext(iter1, "r1", "foo:b", 2, "9");
    testAndCallNext(iter1, "r3", "foo:b", 4, "6");
    testAndCallNext(iter1, "r4", "foo:b", 5, "7");
    assertFalse(iter1.hasTop());

    iter1.seek(new Range(), newCFSet("cf1", "cf3"), true);
    testAndCallNext(iter1, "r1", "cf1:x", 2, "1");
    testAndCallNext(iter1, "r1", "cf1:y", 2, "2");
    testAndCallNext(iter1, "r1", "cf3:z", 2, "3");
    testAndCallNext(iter1, "r2", "cf2:x", 3, "5");
    testAndCallNext(iter1, "r4", "cf4:z", 5, "8");
    testAndCallNext(iter1, "r5", "cf3:z", 6, "A");
    testAndCallNext(iter1, "r5", "cf4:z", 6, "B");
    assertFalse(iter1.hasTop());

    iter1.seek(new Range("r2", "r4"), newCFSet("cf1", "cf3"), true);
    testAndCallNext(iter1, "r2", "cf2:x", 3, "5");
    testAndCallNext(iter1, "r4", "cf4:z", 5, "8");
    assertFalse(iter1.hasTop());

    iter1.seek(new Range(), newCFSet("cf1", "cf3", "foo"), true);
    assertAll(iter1);

    iter1.seek(new Range("r1", "r2"), newCFSet("cf1", "cf3", "foo"), true);
    testAndCallNext(iter1, "r1", "cf1:x", 2, "1");
    testAndCallNext(iter1, "r1", "cf1:y", 2, "2");
    testAndCallNext(iter1, "r1", "cf3:z", 2, "3");
    testAndCallNext(iter1, "r1", "foo:b", 2, "9");
    testAndCallNext(iter1, "r2", "cf2:x", 3, "5");
    assertFalse(iter1.hasTop());

    iter1.seek(new Range(), LocalityGroupUtil.EMPTY_CF_SET, false);
    assertAll(iter1);

    iter1.seek(new Range(), newCFSet("cf1"), false);
    assertAll(iter1);

    iter1.seek(new Range(), newCFSet("cf1", "cf2"), false);
    testAndCallNext(iter1, "r1", "cf3:z", 2, "3");
    testAndCallNext(iter1, "r1", "foo:b", 2, "9");
    testAndCallNext(iter1, "r3", "foo:b", 4, "6");
    testAndCallNext(iter1, "r4", "cf4:z", 5, "8");
    testAndCallNext(iter1, "r4", "foo:b", 5, "7");
    testAndCallNext(iter1, "r5", "cf3:z", 6, "A");
    testAndCallNext(iter1, "r5", "cf4:z", 6, "B");
    assertFalse(iter1.hasTop());

    iter1.seek(new Range("r2"), newCFSet("cf1", "cf3", "foo"), true);
    testAndCallNext(iter1, "r2", "cf2:x", 3, "5");
    assertFalse(iter1.hasTop());
  }

  private void assertAll(SortedKeyValueIterator<Key,Value> iter1) throws IOException {
    testAndCallNext(iter1, "r1", "cf1:x", 2, "1");
    testAndCallNext(iter1, "r1", "cf1:y", 2, "2");
    testAndCallNext(iter1, "r1", "cf3:z", 2, "3");
    testAndCallNext(iter1, "r1", "foo:b", 2, "9");
    testAndCallNext(iter1, "r2", "cf2:x", 3, "5");
    testAndCallNext(iter1, "r3", "foo:b", 4, "6");
    testAndCallNext(iter1, "r4", "cf4:z", 5, "8");
    testAndCallNext(iter1, "r4", "foo:b", 5, "7");
    testAndCallNext(iter1, "r5", "cf3:z", 6, "A");
    testAndCallNext(iter1, "r5", "cf4:z", 6, "B");
    assertFalse(iter1.hasTop());
  }
}
