/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.server.conf.util;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import org.apache.accumulo.core.conf.DeprecatedPropertyUtil;
import org.apache.accumulo.core.fate.zookeeper.ZooReader;
import org.apache.accumulo.core.fate.zookeeper.ZooReaderWriter;
import org.apache.accumulo.core.fate.zookeeper.ZooUtil;
import org.apache.accumulo.core.util.DurationFormat;
import org.apache.accumulo.core.util.Pair;
import org.apache.accumulo.core.util.Retry;
import org.apache.accumulo.server.conf.codec.VersionedPropCodec;
import org.apache.accumulo.server.conf.codec.VersionedProperties;
import org.apache.accumulo.server.conf.store.PropStoreKey;
import org.apache.accumulo.server.conf.store.SystemPropKey;
import org.apache.accumulo.server.conf.store.impl.PropStoreWatcher;
import org.apache.accumulo.server.conf.store.impl.ZooPropStore;
import org.apache.accumulo.server.conf.util.TransformToken;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.Stat;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigTransformer {
    private static final Logger log = LoggerFactory.getLogger(ConfigTransformer.class);
    private final ZooReaderWriter zrw;
    private final VersionedPropCodec codec;
    private final PropStoreWatcher propStoreWatcher;
    private final Retry retry;

    public ConfigTransformer(ZooReaderWriter zrw, VersionedPropCodec codec, PropStoreWatcher propStoreWatcher) {
        this.zrw = zrw;
        this.codec = codec;
        this.propStoreWatcher = propStoreWatcher;
        this.retry = Retry.builder().maxRetries(15L).retryAfter(250L, TimeUnit.MILLISECONDS).incrementBy(500L, TimeUnit.MILLISECONDS).maxWait(5L, TimeUnit.SECONDS).backOffFactor(1.75).logInterval(3L, TimeUnit.MINUTES).createRetry();
    }

    public ConfigTransformer(ZooReaderWriter zrw, VersionedPropCodec codec, PropStoreWatcher propStoreWatcher, Retry retry) {
        this.zrw = zrw;
        this.codec = codec;
        this.propStoreWatcher = propStoreWatcher;
        this.retry = retry;
    }

    public VersionedProperties transform(PropStoreKey<?> propStoreKey, String legacyPath, boolean deleteLegacyNode) {
        VersionedProperties exists = this.checkNeedsTransform(propStoreKey);
        if (exists != null) {
            return exists;
        }
        TransformToken token = TransformToken.createToken(legacyPath, this.zrw);
        return this.transform(propStoreKey, token, legacyPath, deleteLegacyNode);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    VersionedProperties transform(PropStoreKey<?> propStoreKey, TransformToken token, String legacyPath, boolean deleteLegacyNode) {
        log.trace("checking for legacy property upgrade transform for {}", propStoreKey);
        Instant start = Instant.now();
        try {
            VersionedProperties results = this.checkNeedsTransform(propStoreKey);
            if (results != null) {
                VersionedProperties versionedProperties = results;
                return versionedProperties;
            }
            while (!token.haveTokenOwnership()) {
                block36: {
                    this.retry.useRetry();
                    this.retry.waitForNextAttempt(log, "transform property at " + propStoreKey.getPath());
                    log.trace("own the token - look for existing encoded node at: {}", (Object)propStoreKey.getPath());
                    results = ZooPropStore.readFromZk(propStoreKey, this.propStoreWatcher, (ZooReader)this.zrw);
                    if (results == null) break block36;
                    log.trace("Found existing node with properties after getting token at {}. skipping legacy prop conversion - version: {}, timestamp: {}", new Object[]{propStoreKey, results.getDataVersion(), results.getTimestamp()});
                    VersionedProperties versionedProperties = results;
                    return versionedProperties;
                }
                try {
                    token.getTokenOwnership();
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                    throw new IllegalStateException("Failed to hold transform token for " + String.valueOf(propStoreKey), ex);
                }
                catch (IllegalStateException ex) {
                    throw new IllegalStateException("Failed to hold transform token for " + String.valueOf(propStoreKey), ex);
                }
            }
            Set<LegacyPropNode> upgradeNodes = this.readLegacyProps(legacyPath);
            if (upgradeNodes.size() == 0) {
                log.trace("No existing legacy props {}, skipping conversion, writing default prop node", propStoreKey);
                VersionedProperties ex = this.writeNode(propStoreKey, Map.of());
                return ex;
            }
            results = this.writeConverted(propStoreKey, upgradeNodes = this.convertDeprecatedProps(propStoreKey, upgradeNodes));
            if (results == null) {
                throw new IllegalStateException("Could not create properties for " + String.valueOf(propStoreKey));
            }
            if (!token.validateToken()) {
                throw new IllegalStateException("legacy conversion failed. Lost transform token for " + String.valueOf(propStoreKey));
            }
            Pair<Integer, Integer> deleteCounts = this.deleteLegacyProps(upgradeNodes);
            log.info("property transform for {} took {} ms, delete count: {}, error count: {}", new Object[]{propStoreKey, new DurationFormat(Duration.between(start, Instant.now()).toMillis(), ""), deleteCounts.getFirst(), deleteCounts.getSecond()});
            VersionedProperties versionedProperties = results;
            return versionedProperties;
        }
        catch (Exception ex) {
            log.info("Issue on upgrading legacy properties for: " + String.valueOf(propStoreKey), (Throwable)ex);
        }
        finally {
            token.releaseToken();
            if (deleteLegacyNode) {
                log.trace("Delete legacy property base node: {}", (Object)legacyPath);
                try {
                    this.zrw.delete(legacyPath);
                }
                catch (KeeperException.NotEmptyException ex) {
                    log.info("Delete for legacy prop node {} - not empty", (Object)legacyPath);
                }
                catch (InterruptedException | KeeperException ex) {
                    Thread.currentThread().interrupt();
                }
            }
        }
        return null;
    }

    private VersionedProperties checkNeedsTransform(PropStoreKey<?> propStoreKey) {
        try {
            VersionedProperties results = ZooPropStore.readFromZk(propStoreKey, this.propStoreWatcher, (ZooReader)this.zrw);
            if (results != null) {
                log.trace("Found existing node with properties at {}. skipping legacy prop conversion - version: {}, timestamp: {}", new Object[]{propStoreKey, results.getDataVersion(), results.getTimestamp()});
                return results;
            }
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("Interrupted during zookeeper read", ex);
        }
        catch (IOException | KeeperException ex) {
            log.trace("node for {} not found for upgrade", propStoreKey);
        }
        return null;
    }

    private Set<LegacyPropNode> convertDeprecatedProps(PropStoreKey<?> propStoreKey, Set<LegacyPropNode> upgradeNodes) {
        if (!(propStoreKey instanceof SystemPropKey)) {
            return upgradeNodes;
        }
        TreeSet<LegacyPropNode> renamedNodes = new TreeSet<LegacyPropNode>();
        for (LegacyPropNode original : upgradeNodes) {
            String finalName = DeprecatedPropertyUtil.getReplacementName((String)original.getPropName(), (log, replacement) -> log.info("Automatically renaming deprecated property '{}' with its replacement '{}' in ZooKeeper configuration upgrade.", (Object)original, replacement));
            LegacyPropNode renamed = new LegacyPropNode(original.getPath(), finalName, original.getData(), original.getNodeVersion());
            renamedNodes.add(renamed);
        }
        return renamedNodes;
    }

    private @NonNull Set<LegacyPropNode> readLegacyProps(String basePath) {
        TreeSet<LegacyPropNode> legacyProps = new TreeSet<LegacyPropNode>();
        String tokenName = "/transform_token".substring(1);
        try {
            List childNames = this.zrw.getChildren(basePath);
            for (String propName : childNames) {
                log.trace("processing ZooKeeper child node: {} at path: {}", (Object)propName, (Object)basePath);
                if (tokenName.equals(propName)) continue;
                log.trace("Adding: {} to list for legacy conversion", (Object)propName);
                String path = basePath + "/" + propName;
                Stat stat = new Stat();
                byte[] bytes = this.zrw.getData(path, stat);
                try {
                    LegacyPropNode node = stat.getDataLength() > 0 ? new LegacyPropNode(path, propName, new String(bytes, StandardCharsets.UTF_8), stat.getVersion()) : new LegacyPropNode(path, propName, "", stat.getVersion());
                    legacyProps.add(node);
                }
                catch (IllegalStateException ex) {
                    log.warn("Skipping invalid property at path " + path, (Throwable)ex);
                }
            }
        }
        catch (KeeperException ex) {
            throw new IllegalStateException("Failed to read legacy props due to ZooKeeper error", ex);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("Failed to read legacy props due to interrupt read from ZooKeeper", ex);
        }
        return legacyProps;
    }

    private Pair<Integer, Integer> deleteLegacyProps(Set<LegacyPropNode> nodes) {
        int deleteCount = 0;
        int errorCount = 0;
        for (LegacyPropNode n : nodes) {
            try {
                log.trace("Delete legacy prop at path: {}, data version: {}", (Object)n.getPath(), (Object)n.getNodeVersion());
                ++deleteCount;
                this.zrw.deleteStrict(n.getPath(), n.getNodeVersion());
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                throw new IllegalStateException("interrupt received during upgrade node clean-up", ex);
            }
            catch (KeeperException ex) {
                ++errorCount;
                log.info("Failed to delete node during upgrade clean-up", (Throwable)ex);
            }
        }
        return new Pair((Object)deleteCount, (Object)errorCount);
    }

    private @Nullable VersionedProperties writeConverted(PropStoreKey<?> propStoreKey, Set<LegacyPropNode> nodes) {
        VersionedProperties vProps;
        HashMap<String, String> props = new HashMap<String, String>();
        nodes.forEach(node -> props.put(node.getPropName(), node.getData()));
        try {
            vProps = this.writeNode(propStoreKey, props);
        }
        catch (InterruptedException | KeeperException ex) {
            if (ex instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            throw new IllegalStateException("failed to create node for " + String.valueOf(propStoreKey) + " on conversion", ex);
        }
        if (!this.validateWrite(propStoreKey, vProps)) {
            log.trace("Failed property conversion validation for: {}", propStoreKey);
            return null;
        }
        return vProps;
    }

    private VersionedProperties writeNode(PropStoreKey<?> propStoreKey, Map<String, String> props) throws InterruptedException, KeeperException {
        try {
            String path = propStoreKey.getPath();
            log.trace("Writing converted properties to ZooKeeper path: {} for key: {}", (Object)path, propStoreKey);
            Stat currStat = this.zrw.getStatus(path);
            if (currStat == null || currStat.getDataLength() == 0) {
                VersionedProperties vProps = new VersionedProperties(props);
                this.zrw.putPrivatePersistentData(path, this.codec.toBytes(vProps), ZooUtil.NodeExistsPolicy.OVERWRITE);
            }
            return ZooPropStore.readFromZk(propStoreKey, this.propStoreWatcher, (ZooReader)this.zrw);
        }
        catch (IOException ex) {
            throw new IllegalStateException("failed to create node for " + String.valueOf(propStoreKey) + " on conversion", ex);
        }
    }

    private boolean validateWrite(PropStoreKey<?> propStoreKey, VersionedProperties vProps) {
        try {
            Stat stat = this.zrw.getStatus(propStoreKey.getPath(), (Watcher)this.propStoreWatcher);
            if (stat == null) {
                throw new IllegalStateException("failed to get stat to validate created node for " + String.valueOf(propStoreKey));
            }
            log.debug("Property conversion validation - version received: {}, version expected: {}", (Object)stat.getVersion(), (Object)vProps.getDataVersion());
            return (long)stat.getVersion() == vProps.getDataVersion();
        }
        catch (KeeperException ex) {
            throw new IllegalStateException("failed to validate created node for " + String.valueOf(propStoreKey), ex);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("failed to validate created node for " + String.valueOf(propStoreKey), ex);
        }
    }

    private static class LegacyPropNode
    implements Comparable<LegacyPropNode> {
        private final String path;
        private final String propName;
        private final String data;
        private final int nodeVersion;

        public LegacyPropNode(@NonNull String path, String propName, String data, int nodeVersion) {
            this.path = path;
            this.propName = propName;
            this.data = data;
            this.nodeVersion = nodeVersion;
        }

        public String getPath() {
            return this.path;
        }

        public String getPropName() {
            return this.propName;
        }

        public String getData() {
            return this.data;
        }

        public int getNodeVersion() {
            return this.nodeVersion;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            LegacyPropNode that = (LegacyPropNode)o;
            return this.path.equals(that.path);
        }

        public int hashCode() {
            return Objects.hash(this.path);
        }

        @Override
        public int compareTo(LegacyPropNode other) {
            return this.path.compareTo(other.path);
        }
    }
}

