/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.core.fate.zookeeper;

import java.io.IOException;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.apache.accumulo.core.fate.zookeeper.ZooUtil;
import org.apache.accumulo.core.singletons.SingletonManager;
import org.apache.accumulo.core.singletons.SingletonService;
import org.apache.accumulo.core.util.AddressUtil;
import org.apache.accumulo.core.util.UtilWaitThread;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZooSession {
    private static final Logger log = LoggerFactory.getLogger(ZooSession.class);
    private static Map<String, ZooSessionInfo> sessions = new HashMap<String, ZooSessionInfo>();
    private static final SecureRandom random = new SecureRandom();

    private static String sessionKey(String keepers, int timeout, String scheme, byte[] auth) {
        return keepers + ":" + timeout + ":" + (scheme == null ? "" : scheme) + ":" + (auth == null ? "" : new String(auth, StandardCharsets.UTF_8));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static ZooKeeper connect(String host, int timeout, String scheme, byte[] auth, Watcher watcher) {
        int TIME_BETWEEN_CONNECT_CHECKS_MS = 100;
        int connectTimeWait = Math.min(10000, timeout);
        boolean tryAgain = true;
        long sleepTime = 100L;
        ZooKeeper zooKeeper = null;
        long startTime = System.nanoTime();
        while (tryAgain) {
            try {
                zooKeeper = new ZooKeeper(host, timeout, watcher);
                for (int i = 0; i < connectTimeWait / 100 && tryAgain; ++i) {
                    if (zooKeeper.getState().equals((Object)ZooKeeper.States.CONNECTED)) {
                        if (auth != null) {
                            ZooUtil.auth(zooKeeper, scheme, auth);
                        }
                        tryAgain = false;
                        continue;
                    }
                    UtilWaitThread.sleep(100L);
                }
            }
            catch (IOException e) {
                if (e instanceof UnknownHostException) {
                    int ttl = AddressUtil.getAddressCacheNegativeTtl((UnknownHostException)e);
                    sleepTime = Math.max(sleepTime, (long)(ttl + 1) * 1000L);
                }
                log.warn("Connection to zooKeeper failed, will try again in " + String.format("%.2f secs", (double)sleepTime / 1000.0), (Throwable)e);
            }
            finally {
                if (tryAgain && zooKeeper != null) {
                    try {
                        zooKeeper.close();
                        zooKeeper = null;
                    }
                    catch (InterruptedException e) {
                        log.warn("interrupted", (Throwable)e);
                    }
                }
            }
            long stopTime = System.nanoTime();
            long duration = TimeUnit.NANOSECONDS.toMillis(stopTime - startTime);
            if (duration > 2L * (long)timeout) {
                throw new RuntimeException("Failed to connect to zookeeper (" + host + ") within 2x zookeeper timeout period " + timeout);
            }
            if (!tryAgain) continue;
            if (2L * (long)timeout < duration + sleepTime + (long)connectTimeWait) {
                sleepTime = 2L * (long)timeout - duration - (long)connectTimeWait;
            }
            if (sleepTime < 0L) {
                connectTimeWait = (int)((long)connectTimeWait - sleepTime);
                sleepTime = 0L;
            }
            UtilWaitThread.sleep(sleepTime);
            if (sleepTime >= 10000L) continue;
            sleepTime += (long)((double)sleepTime * random.nextDouble());
        }
        return zooKeeper;
    }

    public static ZooKeeper getAuthenticatedSession(String zooKeepers, int timeout, String scheme, byte[] auth) {
        return ZooSession.getSession(zooKeepers, timeout, scheme, auth);
    }

    public static ZooKeeper getAnonymousSession(String zooKeepers, int timeout) {
        return ZooSession.getSession(zooKeepers, timeout, null, null);
    }

    private static synchronized ZooKeeper getSession(String zooKeepers, int timeout, String scheme, byte[] auth) {
        if (sessions == null) {
            throw new ZooSessionShutdownException("The Accumulo singleton that that tracks zookeeper session is disabled.  This is likely caused by all AccumuloClients being closed or garbage collected.");
        }
        String sessionKey = ZooSession.sessionKey(zooKeepers, timeout, scheme, auth);
        String readOnlySessionKey = ZooSession.sessionKey(zooKeepers, timeout, null, null);
        ZooSessionInfo zsi = sessions.get(sessionKey);
        if (zsi != null && zsi.zooKeeper.getState() == ZooKeeper.States.CLOSED) {
            log.debug("Removing closed ZooKeeper session to {}", (Object)zooKeepers);
            if (auth != null && sessions.get(readOnlySessionKey) == zsi) {
                sessions.remove(readOnlySessionKey);
            }
            zsi = null;
            sessions.remove(sessionKey);
        }
        if (zsi == null) {
            ZooWatcher watcher = new ZooWatcher();
            log.debug("Connecting to {} with timeout {} with auth", (Object)zooKeepers, (Object)timeout);
            zsi = new ZooSessionInfo(ZooSession.connect(zooKeepers, timeout, scheme, auth, watcher));
            sessions.put(sessionKey, zsi);
            if (auth != null && !sessions.containsKey(readOnlySessionKey)) {
                sessions.put(readOnlySessionKey, zsi);
            }
        }
        return zsi.zooKeeper;
    }

    private static synchronized boolean isEnabled() {
        return sessions != null;
    }

    private static synchronized void enable() {
        if (sessions != null) {
            return;
        }
        sessions = new HashMap<String, ZooSessionInfo>();
    }

    private static synchronized void disable() {
        if (sessions == null) {
            return;
        }
        for (ZooSessionInfo zsi : sessions.values()) {
            try {
                zsi.zooKeeper.close();
            }
            catch (Exception e) {
                log.debug("Error closing zookeeper during shutdown", (Throwable)e);
            }
        }
        sessions = null;
    }

    static {
        SingletonManager.register(new SingletonService(){

            @Override
            public boolean isEnabled() {
                return ZooSession.isEnabled();
            }

            @Override
            public void enable() {
                ZooSession.enable();
            }

            @Override
            public void disable() {
                ZooSession.disable();
            }
        });
    }

    public static class ZooSessionShutdownException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        public ZooSessionShutdownException(String msg) {
            super(msg);
        }
    }

    private static class ZooSessionInfo {
        ZooKeeper zooKeeper;

        public ZooSessionInfo(ZooKeeper zooKeeper) {
            this.zooKeeper = zooKeeper;
        }
    }

    private static class ZooWatcher
    implements Watcher {
        private ZooWatcher() {
        }

        public void process(WatchedEvent event) {
            if (event.getState() == Watcher.Event.KeeperState.Expired) {
                log.debug("Session expired; {}", (Object)event);
            }
        }
    }
}

