/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.tools.offlineImageViewer;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.hdfs.tools.offlineImageViewer.DelimitedImageVisitor;
import org.apache.hadoop.hdfs.tools.offlineImageViewer.FileDistributionVisitor;
import org.apache.hadoop.hdfs.tools.offlineImageViewer.ImageLoader;
import org.apache.hadoop.hdfs.tools.offlineImageViewer.ImageVisitor;
import org.apache.hadoop.hdfs.tools.offlineImageViewer.IndentedImageVisitor;
import org.apache.hadoop.hdfs.tools.offlineImageViewer.LsImageVisitor;
import org.apache.hadoop.hdfs.tools.offlineImageViewer.NameDistributionVisitor;
import org.apache.hadoop.hdfs.tools.offlineImageViewer.TextWriterImageVisitor;
import org.apache.hadoop.hdfs.tools.offlineImageViewer.XmlImageVisitor;
import org.apache.hadoop.io.IOUtils;

@InterfaceAudience.Private
public class OfflineImageViewer {
    public static final Log LOG = LogFactory.getLog(OfflineImageViewer.class);
    private static final String usage = "Usage: bin/hadoop oiv [OPTIONS] -i INPUTFILE -o OUTPUTFILE\nOffline Image Viewer\nView a Hadoop fsimage INPUTFILE using the specified PROCESSOR,\nsaving the results in OUTPUTFILE.\n\nThe oiv utility will attempt to parse correctly formed image files\nand will abort fail with mal-formed image files.\n\nThe tool works offline and does not require a running cluster in\norder to process an image file.\n\nThe following image processors are available:\n  * Ls: The default image processor generates an lsr-style listing\n    of the files in the namespace, with the same fields in the same\n    order.  Note that in order to correctly determine file sizes,\n    this formatter cannot skip blocks and will override the\n    -skipBlocks option.\n  * Indented: This processor enumerates over all of the elements in\n    the fsimage file, using levels of indentation to delineate\n    sections within the file.\n  * Delimited: Generate a text file with all of the elements common\n    to both inodes and inodes-under-construction, separated by a\n    delimiter. The default delimiter is \u0001, though this may be\n    changed via the -delimiter argument. This processor also overrides\n    the -skipBlocks option for the same reason as the Ls processor\n  * XML: This processor creates an XML document with all elements of\n    the fsimage enumerated, suitable for further analysis by XML\n    tools.\n  * FileDistribution: This processor analyzes the file size\n    distribution in the image.\n    -maxSize specifies the range [0, maxSize] of file sizes to be\n     analyzed (128GB by default).\n    -step defines the granularity of the distribution. (2MB by default)\n  * NameDistribution: This processor analyzes the file names\n    in the image and prints total number of file names and how frequently    file names are reused.\n\nRequired command line arguments:\n-i,--inputFile <arg>   FSImage file to process.\n-o,--outputFile <arg>  Name of output file. If the specified\n                       file exists, it will be overwritten.\n\nOptional command line arguments:\n-p,--processor <arg>   Select which type of processor to apply\n                       against image file. (Ls|XML|Delimited|Indented|FileDistribution).\n-h,--help              Display usage information and exit\n-printToScreen         For processors that write to a file, also\n                       output to screen. On large image files this\n                       will dramatically increase processing time.\n-skipBlocks            Skip inodes' blocks information. May\n                       significantly decrease output.\n                       (default = false).\n-delimiter <arg>       Delimiting string to use with Delimited processor\n";
    private final boolean skipBlocks;
    private final String inputFile;
    private final ImageVisitor processor;

    public OfflineImageViewer(String inputFile, ImageVisitor processor, boolean skipBlocks) {
        this.inputFile = inputFile;
        this.processor = processor;
        this.skipBlocks = skipBlocks;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void go() throws IOException {
        DataInputStream in = null;
        PositionTrackingInputStream tracker = null;
        ImageLoader fsip = null;
        boolean done = false;
        try {
            tracker = new PositionTrackingInputStream(new BufferedInputStream(new FileInputStream(new File(this.inputFile))));
            in = new DataInputStream(tracker);
            int imageVersionFile = this.findImageVersion(in);
            fsip = ImageLoader.LoaderFactory.getLoader(imageVersionFile);
            if (fsip == null) {
                throw new IOException("No image processor to read version " + imageVersionFile + " is available.");
            }
            fsip.loadImage(in, this.processor, this.skipBlocks);
            done = true;
            if (!done) {
                LOG.error("image loading failed at offset " + tracker.getPos());
            }
        }
        catch (Throwable throwable) {
            if (!done) {
                LOG.error("image loading failed at offset " + tracker.getPos());
            }
            IOUtils.cleanup(LOG, in, tracker);
            throw throwable;
        }
        IOUtils.cleanup(LOG, in, tracker);
    }

    private int findImageVersion(DataInputStream in) throws IOException {
        in.mark(42);
        int version = in.readInt();
        in.reset();
        return version;
    }

    public static Options buildOptions() {
        Options options = new Options();
        OptionBuilder.isRequired();
        OptionBuilder.hasArgs();
        OptionBuilder.withLongOpt("outputFile");
        options.addOption(OptionBuilder.create("o"));
        OptionBuilder.isRequired();
        OptionBuilder.hasArgs();
        OptionBuilder.withLongOpt("inputFile");
        options.addOption(OptionBuilder.create("i"));
        options.addOption("p", "processor", true, "");
        options.addOption("h", "help", false, "");
        options.addOption("skipBlocks", false, "");
        options.addOption("printToScreen", false, "");
        options.addOption("delimiter", true, "");
        return options;
    }

    public static void main(String[] args) throws IOException {
        TextWriterImageVisitor v;
        CommandLine cmd;
        Options options = OfflineImageViewer.buildOptions();
        if (args.length == 0) {
            OfflineImageViewer.printUsage();
            return;
        }
        PosixParser parser = new PosixParser();
        try {
            cmd = parser.parse(options, args);
        }
        catch (ParseException e) {
            System.out.println("Error parsing command-line options: ");
            OfflineImageViewer.printUsage();
            return;
        }
        if (cmd.hasOption("h")) {
            OfflineImageViewer.printUsage();
            return;
        }
        boolean skipBlocks = cmd.hasOption("skipBlocks");
        boolean printToScreen = cmd.hasOption("printToScreen");
        String inputFile = cmd.getOptionValue("i");
        String processor = cmd.getOptionValue("p", "Ls");
        String outputFile = cmd.getOptionValue("o");
        String delimiter = cmd.getOptionValue("delimiter");
        if (delimiter != null && !processor.equals("Delimited")) {
            System.out.println("Can only specify -delimiter with Delimited processor");
            OfflineImageViewer.printUsage();
            return;
        }
        if (processor.equals("Indented")) {
            v = new IndentedImageVisitor(outputFile, printToScreen);
        } else if (processor.equals("XML")) {
            v = new XmlImageVisitor(outputFile, printToScreen);
        } else if (processor.equals("Delimited")) {
            v = delimiter == null ? new DelimitedImageVisitor(outputFile, printToScreen) : new DelimitedImageVisitor(outputFile, printToScreen, delimiter);
            skipBlocks = false;
        } else if (processor.equals("FileDistribution")) {
            long maxSize = Long.parseLong(cmd.getOptionValue("maxSize", "0"));
            int step = Integer.parseInt(cmd.getOptionValue("step", "0"));
            v = new FileDistributionVisitor(outputFile, maxSize, step);
        } else if (processor.equals("NameDistribution")) {
            v = new NameDistributionVisitor(outputFile, printToScreen);
        } else {
            v = new LsImageVisitor(outputFile, printToScreen);
            skipBlocks = false;
        }
        try {
            OfflineImageViewer d = new OfflineImageViewer(inputFile, v, skipBlocks);
            d.go();
        }
        catch (EOFException e) {
            System.err.println("Input file ended unexpectedly.  Exiting");
        }
        catch (IOException e) {
            System.err.println("Encountered exception.  Exiting: " + e.getMessage());
        }
    }

    private static void printUsage() {
        System.out.println(usage);
    }

    private static class PositionTrackingInputStream
    extends FilterInputStream {
        private long curPos = 0L;
        private long markPos = -1L;

        public PositionTrackingInputStream(InputStream is) {
            super(is);
        }

        @Override
        public int read() throws IOException {
            int ret = super.read();
            if (ret != -1) {
                ++this.curPos;
            }
            return ret;
        }

        @Override
        public int read(byte[] data) throws IOException {
            int ret = super.read(data);
            if (ret > 0) {
                this.curPos += (long)ret;
            }
            return ret;
        }

        @Override
        public int read(byte[] data, int offset, int length) throws IOException {
            int ret = super.read(data, offset, length);
            if (ret > 0) {
                this.curPos += (long)ret;
            }
            return ret;
        }

        @Override
        public void mark(int limit) {
            super.mark(limit);
            this.markPos = this.curPos;
        }

        @Override
        public void reset() throws IOException {
            if (this.markPos == -1L) {
                throw new IOException("Not marked!");
            }
            super.reset();
            this.curPos = this.markPos;
            this.markPos = -1L;
        }

        public long getPos() {
            return this.curPos;
        }

        @Override
        public long skip(long amt) throws IOException {
            long ret = super.skip(amt);
            this.curPos += ret;
            return ret;
        }
    }
}

