/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

/**
 * Create custom styled checkbox with appropriate bindings and HTML template.
 * You can specify <code><label></code> text with <code>label</code> property, or pass I18n translation key
 * to <code>labelTranslate</code> property. Checkbox specific attributes can be bound in the same way as
 * for Ember.Checkbox <code>checked</code>, <code>disabled</code>.
 * Additional class names can be added with <code>labelClassNames</code> and <code>checkboxClassNames</code>
 * properties.
 *
 * @example
 * Usage:
 * {{view App.CheckboxView checkedBinding="someBinding" label="someLabel"}}
 * {{view App.CheckboxView checkedBinding="someBinding" labelTranslate="i18n.translate"}}
 * {{view App.CheckboxView checkedBinding="someBinding"
 *        labelClassNames="class-1 class-2"
 *        checkboxClassNames="class-3 class-4"}}
 *
 * <!-- will append additional markup after <code><label></code> tag
 * {{#view App.CheckboxView}}
 *   <span>Additional content</span>
 * {{/view}}
 *
 * <!-- custom template -->
 * {{#view App.CheckboxView isCustomTemplate="true"}}
 *   <!-- any markup here -->
 *   <label {{bindAttr for="view.checkboxId"}}>Left aligned label</label>
 *   <!-- minimal template to display checkbox -->
 *   {{view view.checkboxView}}
 *   {{view view.labelView}}
 * {{/view}}
 * @type {Ember.View}
 */
App.CheckboxView = Em.View.extend({
  layoutName: require('templates/common/checkbox'),
  concatenatedProperties: ['labelClassNames', 'checkboxClassNames'],
  classNames: ['checkbox'],

  /**
   * Checkbox id generated by Ember. Used for <code>for</code> label attribute
   * @private
   * @type {String}
   */
  checkboxId: '',
  /**
   * When <code>true</code> custom html markup will be used.
   * @type {Boolean}
   */
  isCustomTemplate: false,

  /**
   * Stores checkbox's <code>checked</code> attribute value
   * @type {Boolean}
   */
  checked: false,

  /**
   * Stores checkbox's <code>disabled</code> attribute value
   * @type {Boolean}
   */
  disabled: false,

  /**
   * Label's text
   * @type {String}
   */
  label: '',

  /**
   * Label's i18n code
   * @type {[type]}
   */
  labelTranslate: null,

  /**
   * Checkbox's <code>name</code> attribute value
   * @type {String}
   */
  name: '',

  /**
   * Default class for label element
   * @type {String[]}
   */
  labelClassNames: ['checkbox-label'],

  /**
   * Label's text to display. <code>label</code> property value will be used if no
   * <code>labelTranslate</code> passed
   * @return {String}
   */
  labelText: function() {
    return this.get('labelTranslate') ?
      Em.I18n.t(this.get('labelTranslate')) :
      this.get('label');
  }.property('label', 'labelTranslate'),

  /**
   * Ember checkbox view
   * @type {Em.View}
   */
  checkboxView: Ember.Checkbox.extend({
    attributeBindings: ['name'],
    checkedBinding: 'parentView.checked',
    disabledBinding: 'parentView.disabled',
    nameBinding: 'parentView.name',
    classNameBindings: 'parentView.checkboxClassNames',

    didInsertElement: function() {
      this.set('parentView.checkboxId', this.get('elementId'));
      this._super();
    },
    change: function() {
      if (typeof this.get('parentView.change') === 'function') {
        this.get('parentView.change').apply(this.get('parentView'), arguments);
      }
    }
  }),

  /**
   * Label view
   * @type {[type]}
   */
  labelView: Em.View.extend({
    labelTextBinding: 'parentView.labelText',
    classNameBindings: 'parentView.labelClassNames',
    tagName: 'label',
    attributeBindings: ['for', 'parentView.checked:checked','data-bs-toggle','title','data-bs-content'],
    'data-bs-toggle':'popover',
    'title':'',
    'data-bs-content':'',
    'data-qaBinding': 'parentView.labelIdentifier',
    template: Em.Handlebars.compile('{{view.labelText}}'),
    for: Em.computed.alias('parentView.checkboxId')
  })
});
