/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.utils.collections;

import java.lang.invoke.MethodHandles;
import java.lang.reflect.Array;
import java.util.Comparator;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.function.Consumer;
import org.apache.activemq.artemis.utils.collections.LinkedList;
import org.apache.activemq.artemis.utils.collections.LinkedListIterator;
import org.apache.activemq.artemis.utils.collections.NodeStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LinkedListImpl<E>
implements LinkedList<E> {
    private static final Logger logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final int INITIAL_ITERATOR_ARRAY_SIZE = 10;
    private final Node<E> head = new NodeHolder<Object>(null);
    private final Comparator<E> comparator;
    private Node<E> tail = null;
    private int size;
    private volatile Iterator[] iters = this.createIteratorArray(10);
    private int numIters;
    private int nextIndex;
    private NodeStore<E> nodeStore;
    private volatile Node<E> lastAdd;

    public LinkedListImpl() {
        this(null, null);
    }

    public LinkedListImpl(Comparator<E> comparator) {
        this(comparator, null);
    }

    public LinkedListImpl(Comparator<E> comparator, NodeStore<E> supplier) {
        this.comparator = comparator;
        this.nodeStore = supplier;
    }

    @Override
    public void clearID() {
        if (this.nodeStore != null) {
            this.nodeStore.clear();
        }
        this.nodeStore = null;
    }

    @Override
    public void setNodeStore(NodeStore<E> supplier) {
        this.nodeStore = supplier;
        try (Iterator iterator = (Iterator)this.iterator();){
            while (iterator.hasNext()) {
                Object value = iterator.next();
                Node position = iterator.last;
                this.putID(value, position);
            }
        }
    }

    private void putID(E element, Node<E> node) {
        this.nodeStore.storeNode(element, node);
    }

    @Override
    public void addHead(E e) {
        Node<E> node = Node.with(e);
        node.next = this.head.next;
        node.prev = this.head;
        this.head.next = node;
        if (this.size == 0) {
            this.tail = node;
        } else {
            node.next.prev = node;
        }
        this.itemAdded(node, e);
        ++this.size;
    }

    @Override
    public E peek() {
        Node current = this.head.next;
        if (current == null) {
            return null;
        }
        return current.val();
    }

    @Override
    public E get(int position) {
        Node current = this.head.next;
        for (int i = 0; i < position && current != null; ++i) {
            current = current.next;
        }
        if (current == null) {
            throw new IndexOutOfBoundsException(position + " > " + this.size());
        }
        return current.val();
    }

    @Override
    public synchronized E removeWithID(String listID, long id) {
        assert (this.nodeStore != null);
        Node<E> node = this.nodeStore.getNode(listID, id);
        if (node == null) {
            return null;
        }
        this.removeAfter(node.prev);
        return node.val();
    }

    @Override
    public void forEach(Consumer<E> consumer) {
        try (LinkedListIterator<E> iter = this.iterator();){
            while (iter.hasNext()) {
                Object nextValue = iter.next();
                consumer.accept(nextValue);
            }
        }
    }

    private void itemAdded(Node<E> node, E item) {
        assert (node.val() == item);
        this.lastAdd = node;
        if (logger.isTraceEnabled()) {
            logger.trace("Setting lastAdd as {}, e={}", this.lastAdd, this.lastAdd.val());
        }
        if (this.nodeStore != null) {
            this.putID(item, node);
        }
    }

    private void itemRemoved(Node<E> node) {
        this.lastAdd = null;
        if (this.nodeStore != null) {
            this.nodeStore.removeNode(node.val(), node);
        }
    }

    @Override
    public void addTail(E e) {
        if (this.size == 0) {
            this.addHead(e);
        } else {
            Node<E> node = Node.with(e);
            node.prev = this.tail;
            this.tail.next = node;
            this.tail = node;
            this.itemAdded(node, e);
            ++this.size;
        }
    }

    public void addSorted(E e) {
        Node<E> localLastAdd = this.lastAdd;
        logger.trace("**** addSorted element {}", e);
        if (this.comparator == null) {
            throw new NullPointerException("comparator=null");
        }
        if (this.size != 0) {
            if (this.comparator.compare(this.head.next.val(), e) < 0) {
                if (logger.isTraceEnabled()) {
                    logger.trace("addHead as e={} and head={}", e, this.head.next.val());
                }
                this.addHead(e);
                return;
            }
            if (this.comparator.compare(this.tail.val(), e) >= 0) {
                logger.trace("addTail as e={} and tail={}", e, this.tail.val());
                this.addTail(e);
                return;
            }
            if (localLastAdd != null) {
                int compareLastAdd;
                if (logger.isDebugEnabled()) {
                    logger.debug("localLastAdd Value = {}, we are adding {}", localLastAdd.val(), e);
                }
                if ((compareLastAdd = this.comparator.compare(localLastAdd.val(), e)) > 0 && this.scanRight(localLastAdd, e)) {
                    return;
                }
                if (compareLastAdd < 0 && this.scanLeft(localLastAdd, e)) {
                    return;
                }
            }
            if (this.addSortedScan(e)) {
                return;
            }
            throw new IllegalStateException("Cannot find a suitable place for your element, There's a mismatch in the comparator or there was concurrent adccess on the queue");
        }
        logger.trace("adding head as there are no elements {}", e);
        this.addHead(e);
    }

    protected boolean scanRight(Node<E> position, E e) {
        Node fetching = position.next;
        while (fetching != null) {
            if (this.comparator.compare(fetching.val(), e) < 0) {
                this.addAfter(position, e);
                return true;
            }
            position = fetching;
            fetching = fetching.next;
        }
        return false;
    }

    protected boolean scanLeft(Node<E> position, E e) {
        Node fetching = position.prev;
        while (fetching != null) {
            if (this.comparator.compare(fetching.val(), e) > 0) {
                this.addAfter(fetching, e);
                return true;
            }
            fetching = fetching.prev;
        }
        return false;
    }

    protected boolean addSortedScan(E e) {
        logger.trace("addSortedScan {}...", e);
        Node fetching = this.head.next;
        while (fetching.next != null) {
            int compareNext = this.comparator.compare(fetching.next.val(), e);
            if (compareNext <= 0) {
                this.addAfter(fetching, e);
                logger.trace("... addSortedScan done, returning true");
                return true;
            }
            fetching = fetching.next;
        }
        logger.trace("... addSortedScan done, could not find a spot, returning false");
        return false;
    }

    private void addAfter(Node<E> node, E e) {
        Node<E> newNode = Node.with(e);
        Node nextNode = node.next;
        node.next = newNode;
        newNode.prev = node;
        newNode.next = nextNode;
        nextNode.prev = newNode;
        this.itemAdded(newNode, e);
        ++this.size;
    }

    @Override
    public E poll() {
        Node ret = this.head.next;
        if (ret != null) {
            this.removeAfter(this.head);
            return ret.val();
        }
        return null;
    }

    @Override
    public void clear() {
        while (this.poll() != null) {
        }
    }

    @Override
    public int size() {
        return this.size;
    }

    @Override
    public LinkedListIterator<E> iterator() {
        if (logger.isTraceEnabled()) {
            logger.trace("Creating new iterator at", (Throwable)new Exception("trace location"));
        }
        return new Iterator();
    }

    public String toString() {
        StringBuilder str = new StringBuilder("LinkedListImpl [ ");
        Node<E> node = this.head;
        while (node != null) {
            str.append(node.toString());
            if (node.next != null) {
                str.append(", ");
            }
            node = node.next;
        }
        return str.toString();
    }

    public int numIters() {
        return this.numIters;
    }

    private Iterator[] createIteratorArray(int size) {
        return (Iterator[])Array.newInstance(Iterator.class, size);
    }

    private void removeAfter(Node<E> node) {
        Node toRemove = node.next;
        node.next = toRemove.next;
        if (toRemove.next != null) {
            toRemove.next.prev = node;
        }
        this.itemRemoved(toRemove);
        if (toRemove == this.tail) {
            this.tail = node;
        }
        --this.size;
        if (toRemove.iterCount != 0) {
            this.nudgeIterators(toRemove);
        }
        toRemove.prev = null;
        toRemove.next = null;
    }

    private synchronized void nudgeIterators(Node<E> node) {
        for (int i = 0; i < this.numIters; ++i) {
            Iterator iter = this.iters[i];
            if (iter == null) continue;
            iter.nudged(node);
        }
    }

    private synchronized void addIter(Iterator iter) {
        if (this.numIters == this.iters.length) {
            this.resize(2 * this.numIters);
        }
        this.iters[this.nextIndex++] = iter;
        ++this.numIters;
    }

    private synchronized void resize(int newSize) {
        Iterator[] newIters = this.createIteratorArray(newSize);
        System.arraycopy(this.iters, 0, newIters, 0, this.numIters);
        this.iters = newIters;
    }

    private synchronized void removeIter(Iterator iter) {
        if (logger.isTraceEnabled()) {
            logger.trace("Removing iterator at", (Throwable)new Exception("trace location"));
        }
        for (int i = 0; i < this.numIters; ++i) {
            if (iter != this.iters[i]) continue;
            this.iters[i] = null;
            if (i != this.numIters - 1) {
                System.arraycopy(this.iters, i + 1, this.iters, i, this.numIters - i - 1);
            }
            --this.numIters;
            if (this.numIters >= 10 && this.numIters == this.iters.length / 2) {
                this.resize(this.numIters);
            }
            --this.nextIndex;
            if (this.nextIndex < this.iters.length) {
                this.iters[this.nextIndex] = null;
            }
            return;
        }
        throw new IllegalStateException("Cannot find iter to remove");
    }

    private static final class NodeHolder<E>
    extends Node<E> {
        private final E val;

        private NodeHolder(E e) {
            this.val = e;
        }

        @Override
        protected E val() {
            return this.val;
        }
    }

    public static class Node<E> {
        private Node<E> next;
        private Node<E> prev;
        private int iterCount;

        private static <E> Node<E> with(E o) {
            Objects.requireNonNull(o, "Only HEAD nodes are allowed to hold null values");
            if (o instanceof Node) {
                Node node = (Node)o;
                if (node.prev == null && node.next == null) {
                    node.iterCount = 0;
                    return node;
                }
            }
            return new NodeHolder<E>(o);
        }

        protected E val() {
            return (E)this;
        }

        protected final Node<E> next() {
            return this.next;
        }

        protected final Node<E> prev() {
            return this.prev;
        }

        public String toString() {
            return this.val() == this ? "Intrusive Node" : "Node, value = " + this.val();
        }
    }

    public class Iterator
    implements LinkedListIterator<E> {
        Node<E> last;
        Node<E> current;
        boolean repeat;

        Iterator() {
            this.current = LinkedListImpl.this.head.next;
            if (this.current != null) {
                ++this.current.iterCount;
            }
            LinkedListImpl.this.addIter(this);
        }

        @Override
        public void repeat() {
            this.repeat = true;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public boolean hasNext() {
            LinkedListImpl linkedListImpl = LinkedListImpl.this;
            synchronized (linkedListImpl) {
                Node e = this.getNode();
                if (e != null && (e != this.last || this.repeat)) {
                    return true;
                }
                return this.canAdvance();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public E next() {
            LinkedListImpl linkedListImpl = LinkedListImpl.this;
            synchronized (linkedListImpl) {
                Node e = this.getNode();
                if (this.repeat) {
                    this.repeat = false;
                    if (e != null) {
                        return e.val();
                    }
                    if (this.canAdvance()) {
                        this.advance();
                        e = this.getNode();
                        return e.val();
                    }
                    throw new NoSuchElementException();
                }
                if (e == null || e == this.last) {
                    if (this.canAdvance()) {
                        this.advance();
                        e = this.getNode();
                    } else {
                        throw new NoSuchElementException();
                    }
                }
                this.last = e;
                this.repeat = false;
                return e.val();
            }
        }

        @Override
        public void remove() {
            this.removeLastElement();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public E removeLastElement() {
            LinkedListImpl linkedListImpl = LinkedListImpl.this;
            synchronized (linkedListImpl) {
                if (this.last == null) {
                    throw new NoSuchElementException();
                }
                if (this.current == null) {
                    return null;
                }
                Object returningElement = this.current.val();
                Node prev = this.current.prev;
                if (prev != null) {
                    LinkedListImpl.this.removeAfter(prev);
                    this.last = null;
                }
                return returningElement;
            }
        }

        @Override
        public void close() {
            LinkedListImpl.this.removeIter(this);
        }

        public void nudged(Node<E> node) {
            if (this.current == node) {
                if (this.canAdvance()) {
                    this.advance();
                } else if (this.current.prev != LinkedListImpl.this.head) {
                    --this.current.iterCount;
                    this.current = this.current.prev;
                    ++this.current.iterCount;
                    if (this.last == node) {
                        this.last = this.current;
                    }
                } else {
                    this.current = null;
                }
            }
        }

        private Node<E> getNode() {
            if (this.current == null) {
                this.current = LinkedListImpl.this.head.next;
                if (this.current != null) {
                    ++this.current.iterCount;
                }
            }
            if (this.current != null) {
                return this.current;
            }
            return null;
        }

        private boolean canAdvance() {
            if (this.current == null) {
                this.current = LinkedListImpl.this.head.next;
                if (this.current != null) {
                    ++this.current.iterCount;
                }
            }
            return this.current != null && this.current.next != null;
        }

        private void advance() {
            if (this.current == null || this.current.next == null) {
                throw new NoSuchElementException();
            }
            --this.current.iterCount;
            this.current = this.current.next;
            ++this.current.iterCount;
        }
    }
}

