/*
 * Decompiled with CFR 0.152.
 */
package org.apache.empire.db.validation;

import java.sql.Connection;
import java.util.List;
import org.apache.empire.data.DataType;
import org.apache.empire.db.DBColumn;
import org.apache.empire.db.DBDatabase;
import org.apache.empire.db.DBMaterializedView;
import org.apache.empire.db.DBRelation;
import org.apache.empire.db.DBRowSet;
import org.apache.empire.db.DBTable;
import org.apache.empire.db.DBTableColumn;
import org.apache.empire.db.DBView;
import org.apache.empire.db.validation.DBModelErrorHandler;
import org.apache.empire.db.validation.DBModelParser;
import org.apache.empire.exceptions.ObjectNotValidException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DBModelChecker {
    private static final Logger log = LoggerFactory.getLogger(DBModelChecker.class);
    protected final DBModelParser modelParser;
    protected DBDatabase remoteDb = null;

    public DBModelChecker(DBModelParser modelParser) {
        this.modelParser = modelParser;
    }

    public DBDatabase getRemoteDatabase() {
        return this.remoteDb;
    }

    public void checkModel(DBDatabase db, Connection conn, DBModelErrorHandler handler) {
        this.modelParser.parseModel(conn);
        this.remoteDb = this.modelParser.getDatabase();
        this.checkRemoteAgainst(db, handler);
    }

    public void checkRemoteAgainst(DBDatabase db, DBModelErrorHandler handler) {
        if (this.remoteDb == null) {
            throw new ObjectNotValidException(this);
        }
        for (DBTable table : db.getTables()) {
            this.checkTable(table, handler);
        }
        for (DBView view : db.getViews()) {
            this.checkView(view, handler);
        }
    }

    protected void checkTable(DBTable table, DBModelErrorHandler handler) {
        DBTable remoteTable = this.remoteDb.getTable(table.getName());
        if (remoteTable == null) {
            DBView remoteView = this.remoteDb.getView(table.getName());
            if (remoteView != null) {
                handler.objectTypeMismatch(remoteView, table.getName(), DBTable.class);
            } else {
                handler.itemNotFound(table);
            }
            return;
        }
        this.checkPrimaryKey(table, remoteTable, handler);
        this.checkForeignKeys(table, remoteTable, handler);
        for (DBColumn column : table.getColumns()) {
            DBColumn remoteColumn = remoteTable.getColumn(column.getName());
            if (remoteColumn == null) {
                handler.itemNotFound(column);
                continue;
            }
            this.checkColumn(column, remoteColumn, handler);
        }
    }

    protected void checkView(DBView view, DBModelErrorHandler handler) {
        DBRowSet remoteView = this.remoteDb.getView(view.getName());
        if (remoteView == null) {
            DBTable remoteTable = this.remoteDb.getTable(view.getName());
            if (remoteTable == null || !(view instanceof DBMaterializedView)) {
                if (remoteTable != null) {
                    handler.objectTypeMismatch(remoteTable, view.getName(), DBView.class);
                } else {
                    handler.itemNotFound(view);
                }
                return;
            }
            remoteView = remoteTable;
        }
        for (DBColumn column : view.getColumns()) {
            DBColumn remoteColumn = remoteView.getColumn(column.getName());
            if (remoteColumn == null) {
                handler.itemNotFound(column);
                continue;
            }
            this.checkColumnType(column, remoteColumn, handler);
        }
    }

    protected void checkPrimaryKey(DBTable table, DBTable remoteTable, DBModelErrorHandler handler) {
        if (table.getPrimaryKey() == null) {
            return;
        }
        if (remoteTable.getPrimaryKey() == null) {
            handler.itemNotFound(table.getPrimaryKey());
            return;
        }
        DBColumn[] pk = table.getPrimaryKey().getColumns();
        DBColumn[] remotePk = remoteTable.getPrimaryKey().getColumns();
        int pkColIdx = -1;
        boolean hasOrderMismatch = false;
        block0: for (DBColumn pkCol : pk) {
            ++pkColIdx;
            int remColIdx = -1;
            for (DBColumn remotePkCol : remotePk) {
                ++remColIdx;
                if (!pkCol.getName().equalsIgnoreCase(remotePkCol.getName())) continue;
                hasOrderMismatch |= pkColIdx != remColIdx;
                continue block0;
            }
            handler.primaryKeyColumnMissing(table.getPrimaryKey(), pkCol);
        }
        if (hasOrderMismatch || remotePk.length > pk.length) {
            handler.primaryKeyMismatch(table.getPrimaryKey(), remotePk);
        }
    }

    protected void checkForeignKeys(DBTable table, DBTable remoteTable, DBModelErrorHandler handler) {
        if (table.getForeignKeyRelations().isEmpty()) {
            return;
        }
        List<DBRelation> relations = table.getForeignKeyRelations();
        List<DBRelation> remoteRelations = remoteTable.getForeignKeyRelations();
        block0: for (DBRelation relation : relations) {
            block1: for (DBRelation.DBReference reference : relation.getReferences()) {
                if (reference.getTargetColumn().getRowSet() instanceof DBTable) {
                    DBTable targetTable = (DBTable)reference.getTargetColumn().getRowSet();
                    DBTableColumn targetColumn = reference.getTargetColumn();
                    if (!targetTable.getPrimaryKey().contains(targetColumn)) {
                        log.info("The column {} of foreign key {} is not a primary key of table {}", new Object[]{targetColumn.getName(), relation.getName(), targetTable.getName()});
                        continue;
                    }
                }
                for (DBRelation remoteRelation : remoteRelations) {
                    for (DBRelation.DBReference remoteReference : remoteRelation.getReferences()) {
                        if (reference.getSourceColumn().toString().equalsIgnoreCase(remoteReference.getSourceColumn().toString()) && reference.getTargetColumn().toString().equalsIgnoreCase(remoteReference.getTargetColumn().toString())) continue block1;
                    }
                }
                handler.itemNotFound(relation);
                continue block0;
            }
        }
    }

    protected void checkColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        switch (column.getDataType()) {
            case UNKNOWN: {
                this.checkUnknownColumn(column, remoteColumn, handler);
                break;
            }
            case INTEGER: {
                this.checkIntegerColumn(column, remoteColumn, handler);
                break;
            }
            case AUTOINC: {
                this.checkAutoIncColumn(column, remoteColumn, handler);
                break;
            }
            case VARCHAR: {
                this.checkTextColumn(column, remoteColumn, handler);
                break;
            }
            case DATE: 
            case TIME: 
            case DATETIME: 
            case TIMESTAMP: {
                this.checkDateColumn(column, remoteColumn, handler);
                break;
            }
            case CHAR: {
                this.checkCharColumn(column, remoteColumn, handler);
                break;
            }
            case FLOAT: {
                this.checkFloatColumn(column, remoteColumn, handler);
                break;
            }
            case DECIMAL: {
                this.checkDecimalColumn(column, remoteColumn, handler);
                break;
            }
            case BOOL: {
                this.checkBoolColumn(column, remoteColumn, handler);
                break;
            }
            case CLOB: {
                this.checkClobColumn(column, remoteColumn, handler);
                break;
            }
            case BLOB: {
                this.checkBlobColumn(column, remoteColumn, handler);
                break;
            }
            case UNIQUEID: {
                this.checkUniqueIdColumn(column, remoteColumn, handler);
                break;
            }
            default: {
                throw new RuntimeException("Invalid DataType " + (Object)((Object)column.getDataType()));
            }
        }
    }

    private void checkGenericColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkColumnType(column, remoteColumn, handler);
        this.checkColumnNullable(column, remoteColumn, handler);
        this.checkColumnSize(column, remoteColumn, handler);
    }

    protected void checkColumnType(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        if (column.getDataType() != remoteColumn.getDataType()) {
            handler.columnTypeMismatch(column, remoteColumn.getDataType());
        }
    }

    protected void checkColumnNullable(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        if (column.isRequired() && !remoteColumn.isRequired()) {
            handler.columnNullableMismatch(column, remoteColumn.isRequired());
        }
    }

    protected void checkColumnSize(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        int colSize = (int)column.getSize();
        if (colSize == 0) {
            return;
        }
        if (column.getDataType() == DataType.AUTOINC) {
            return;
        }
        if (colSize != (int)remoteColumn.getSize()) {
            handler.columnSizeMismatch(column, (int)remoteColumn.getSize(), 0);
        }
    }

    protected void checkUnknownColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkGenericColumn(column, remoteColumn, handler);
    }

    protected void checkIntegerColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkGenericColumn(column, remoteColumn, handler);
    }

    protected void checkAutoIncColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkColumnSize(column, remoteColumn, handler);
        this.checkColumnNullable(column, remoteColumn, handler);
    }

    protected void checkTextColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkGenericColumn(column, remoteColumn, handler);
    }

    protected void checkDateColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkColumnNullable(column, remoteColumn, handler);
        if (!remoteColumn.getDataType().isDate()) {
            handler.columnTypeMismatch(column, remoteColumn.getDataType());
        }
    }

    protected void checkCharColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkGenericColumn(column, remoteColumn, handler);
    }

    protected void checkFloatColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkGenericColumn(column, remoteColumn, handler);
    }

    protected void checkDecimalColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkGenericColumn(column, remoteColumn, handler);
        if (column instanceof DBTableColumn && remoteColumn instanceof DBTableColumn) {
            DBTableColumn tableColumn = (DBTableColumn)column;
            DBTableColumn tableRemoteColumn = (DBTableColumn)remoteColumn;
            if (tableColumn.getDecimalScale() != tableRemoteColumn.getDecimalScale()) {
                handler.columnSizeMismatch(column, (int)remoteColumn.getSize(), tableRemoteColumn.getDecimalScale());
            }
        }
    }

    protected void checkBoolColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkColumnType(column, remoteColumn, handler);
        this.checkColumnNullable(column, remoteColumn, handler);
    }

    protected void checkBlobColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkColumnType(column, remoteColumn, handler);
        this.checkColumnNullable(column, remoteColumn, handler);
    }

    protected void checkClobColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkColumnType(column, remoteColumn, handler);
        this.checkColumnNullable(column, remoteColumn, handler);
    }

    protected void checkUniqueIdColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkGenericColumn(column, remoteColumn, handler);
    }
}

