/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.fineract.portfolio.loanproduct.productmix.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.fineract.infrastructure.core.api.JsonCommand;
import org.apache.fineract.infrastructure.core.data.CommandProcessingResult;
import org.apache.fineract.infrastructure.core.data.CommandProcessingResultBuilder;
import org.apache.fineract.infrastructure.core.exception.ErrorHandler;
import org.apache.fineract.infrastructure.security.service.PlatformSecurityContext;
import org.apache.fineract.portfolio.loanproduct.domain.LoanProduct;
import org.apache.fineract.portfolio.loanproduct.domain.LoanProductRepository;
import org.apache.fineract.portfolio.loanproduct.exception.LoanProductNotFoundException;
import org.apache.fineract.portfolio.loanproduct.productmix.domain.ProductMix;
import org.apache.fineract.portfolio.loanproduct.productmix.domain.ProductMixRepository;
import org.apache.fineract.portfolio.loanproduct.productmix.exception.ProductMixNotFoundException;
import org.apache.fineract.portfolio.loanproduct.productmix.serialization.ProductMixDataValidator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.NonTransientDataAccessException;
import org.springframework.orm.jpa.JpaSystemException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

@Service
public class ProductMixWritePlatformServiceJpaRepositoryImpl implements ProductMixWritePlatformService {

    private static final Logger LOG = LoggerFactory.getLogger(ProductMixWritePlatformServiceJpaRepositoryImpl.class);
    private final PlatformSecurityContext context;
    private final ProductMixDataValidator fromApiJsonDeserializer;
    private final ProductMixRepository productMixRepository;
    private final LoanProductRepository productRepository;

    @Autowired
    public ProductMixWritePlatformServiceJpaRepositoryImpl(final PlatformSecurityContext context,
            final ProductMixDataValidator fromApiJsonDeserializer, final ProductMixRepository productMixRepository,
            final LoanProductRepository productRepository) {
        this.context = context;
        this.fromApiJsonDeserializer = fromApiJsonDeserializer;
        this.productMixRepository = productMixRepository;
        this.productRepository = productRepository;
    }

    @Transactional
    @Override
    public CommandProcessingResult createProductMix(final Long productId, final JsonCommand command) {

        try {

            this.context.authenticatedUser();

            this.fromApiJsonDeserializer.validateForCreate(command.json());

            final Set<String> restrictedIds = new HashSet<>(Arrays.asList(command.arrayValueOfParameterNamed("restrictedProducts")));

            // remove the existed restriction if it is not exists in
            // restrictedIds.
            final List<Long> removedRestrictions = updateRestrictionsForProduct(productId, restrictedIds);
            final Map<Long, LoanProduct> restrictedProductsAsMap = getRestrictedProducts(restrictedIds);
            final List<ProductMix> productMixes = new ArrayList<>();

            createNewProductMix(restrictedProductsAsMap, productId, productMixes);

            this.productMixRepository.saveAll(productMixes);

            final Map<String, Object> changes = new LinkedHashMap<>();
            changes.put("restrictedProductsForMix", restrictedProductsAsMap.keySet());
            changes.put("removedProductsForMix", removedRestrictions);
            return new CommandProcessingResultBuilder().withProductId(productId).with(changes).withCommandId(command.commandId()).build();
        } catch (final JpaSystemException | DataIntegrityViolationException dve) {

            handleDataIntegrityIssues(dve);
            return CommandProcessingResult.empty();
        }
    }

    private List<Long> updateRestrictionsForProduct(final Long productId, final Set<String> restrictedIds) {

        final List<Long> removedRestrictions = new ArrayList<>();
        final List<ProductMix> mixesToRemove = new ArrayList<>();

        final List<ProductMix> existedProductMixes = this.productMixRepository.findRestrictedProducts(productId);
        for (final ProductMix productMix : existedProductMixes) {
            if (!restrictedIds.contains(productMix.getProductId().toString())) {
                mixesToRemove.add(productMix);
                removedRestrictions.add(productMix.getId());
            }
        }
        if (!CollectionUtils.isEmpty(mixesToRemove)) {
            this.productMixRepository.deleteAll(mixesToRemove);
        }
        return removedRestrictions;
    }

    private void createNewProductMix(final Map<Long, LoanProduct> restrictedProductsAsMap, final Long productId,
            final List<ProductMix> productMixes) {

        final LoanProduct productMixInstance = findByProductIdIfProvided(productId);
        for (final LoanProduct restrictedProduct : restrictedProductsAsMap.values()) {
            final ProductMix productMix = ProductMix.createNew(productMixInstance, restrictedProduct);
            productMixes.add(productMix);
        }
    }

    @Override
    public CommandProcessingResult updateProductMix(final Long productId, final JsonCommand command) {

        try {
            this.context.authenticatedUser();
            this.fromApiJsonDeserializer.validateForUpdate(command.json());
            final Map<String, Object> changes = new LinkedHashMap<>();

            final List<ProductMix> existedProductMixes = new ArrayList<>(this.productMixRepository.findByProductId(productId));
            if (CollectionUtils.isEmpty(existedProductMixes)) {
                throw new ProductMixNotFoundException(productId);
            }
            final Set<String> restrictedIds = new HashSet<>(Arrays.asList(command.arrayValueOfParameterNamed("restrictedProducts")));

            // updating with empty array means deleting the existed records.
            if (restrictedIds.isEmpty()) {
                final List<Long> removedRestrictedProductIds = this.productMixRepository.findRestrictedProductIdsByProductId(productId);
                this.productMixRepository.deleteAll(existedProductMixes);
                changes.put("removedProductsForMix", removedRestrictedProductIds);
                return new CommandProcessingResultBuilder().with(changes).withProductId(productId).withCommandId(command.commandId())
                        .build();
            }

            /*
             * if restrictedProducts array is not empty delete the duplicate ids which are already exists and update
             * existedProductMixes
             */
            final List<ProductMix> productMixesToRemove = updateRestrictedIds(restrictedIds, existedProductMixes);
            final Map<Long, LoanProduct> restrictedProductsAsMap = getRestrictedProducts(restrictedIds);
            createNewProductMix(restrictedProductsAsMap, productId, existedProductMixes);

            this.productMixRepository.saveAll(existedProductMixes);
            changes.put("restrictedProductsForMix", getProductIdsFromCollection(existedProductMixes));

            if (!CollectionUtils.isEmpty(productMixesToRemove)) {
                this.productMixRepository.deleteAll(productMixesToRemove);
                changes.put("removedProductsForMix", getProductIdsFromCollection(productMixesToRemove));
            }
            return new CommandProcessingResultBuilder().with(changes).withProductId(productId).withCommandId(command.commandId()).build();
        } catch (final JpaSystemException | DataIntegrityViolationException dve) {

            handleDataIntegrityIssues(dve);
            return CommandProcessingResult.empty();
        }
    }

    private LoanProduct findByProductIdIfProvided(final Long productId) {
        return this.productRepository.findById(productId).orElseThrow(() -> new LoanProductNotFoundException(productId));
    }

    private Map<Long, LoanProduct> getRestrictedProducts(final Set<String> restrictedIds) {

        final Map<Long, LoanProduct> restrictedProducts = new HashMap<>();

        for (final String restrictedId : restrictedIds) {
            final Long restrictedIdAsLong = Long.valueOf(restrictedId);
            final LoanProduct restrictedProduct = findByProductIdIfProvided(Long.valueOf(restrictedId));
            restrictedProducts.put(restrictedIdAsLong, restrictedProduct);
        }
        return restrictedProducts;
    }

    private void handleDataIntegrityIssues(final NonTransientDataAccessException dve) {
        LOG.error("Error occurred.", dve);
        throw ErrorHandler.getMappable(dve, "error.msg.product.loan.unknown.data.integrity.issue",
                "Unknown data integrity issue with resource.");
    }

    private List<ProductMix> updateRestrictedIds(final Set<String> restrictedIds, final List<ProductMix> existedProductMixes) {

        final List<ProductMix> productMixesToRemove = new ArrayList<>();
        for (final ProductMix productMix : existedProductMixes) {
            final String currentMixId = productMix.getRestrictedProductId().toString();
            if (restrictedIds.contains(currentMixId)) {
                restrictedIds.remove(currentMixId);
            } else {
                productMixesToRemove.add(productMix);
            }
        }
        existedProductMixes.removeAll(productMixesToRemove);
        return productMixesToRemove;
    }

    @Override
    public CommandProcessingResult deleteProductMix(final Long productId) {
        try {
            this.context.authenticatedUser();
            final Map<String, Object> changes = new LinkedHashMap<>();

            final List<ProductMix> existedProductMixes = this.productMixRepository.findByProductId(productId);
            if (CollectionUtils.isEmpty(existedProductMixes)) {
                throw new ProductMixNotFoundException(productId);
            }
            this.productMixRepository.deleteAll(existedProductMixes);
            changes.put("removedProductsForMix", getProductIdsFromCollection(existedProductMixes));
            return new CommandProcessingResultBuilder().with(changes).withProductId(productId).build();
        } catch (final JpaSystemException | DataIntegrityViolationException dve) {
            handleDataIntegrityIssues(dve);
            return CommandProcessingResult.empty();
        }
    }

    private List<Long> getProductIdsFromCollection(final List<ProductMix> collection) {
        final List<Long> productIds = new ArrayList<>();
        for (final ProductMix productMix : collection) {
            productIds.add(productMix.getRestrictedProductId());
        }
        return productIds;
    }

}
