/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.model.util;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.helix.HelixDefinedState;
import org.apache.helix.model.StateModelDefinition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StateModelDefinitionValidator {
    private static final Logger _logger = LoggerFactory.getLogger(StateModelDefinitionValidator.class);
    private final StateModelDefinition _stateModelDef;
    private final List<String> _statePriorityList;
    private final List<String> _transitionPriorityList;
    private final Set<String> _stateSet;

    private StateModelDefinitionValidator(StateModelDefinition stateModelDef) {
        this._stateModelDef = stateModelDef;
        this._statePriorityList = stateModelDef.getStatesPriorityList();
        this._transitionPriorityList = stateModelDef.getStateTransitionPriorityList();
        this._stateSet = Sets.newHashSet(this._statePriorityList);
    }

    public boolean isStateModelDefinitionValid() {
        if (this._stateModelDef.getId() == null || this._stateModelDef.getId().isEmpty()) {
            _logger.error("State model does not have a name");
            return false;
        }
        if (this._stateModelDef.getInitialState() == null || this._stateModelDef.getInitialState().isEmpty()) {
            _logger.error("State model does not contain init state, statemodel:" + this._stateModelDef.getId());
            return false;
        }
        if (this._statePriorityList == null || this._statePriorityList.isEmpty()) {
            _logger.error("CurrentState does not contain StatesPriorityList, state model : " + this._stateModelDef.getId());
            return false;
        }
        if (!this._stateSet.contains(this._stateModelDef.getInitialState())) {
            _logger.error("Defined states does not include the initial state, state model: " + this._stateModelDef.getId());
            return false;
        }
        if (!this._stateSet.contains(HelixDefinedState.DROPPED.toString())) {
            _logger.error("Defined states does not include the DROPPED state, state model: " + this._stateModelDef.getId());
            return false;
        }
        return this.areStateCountsValid() && this.areNextStatesValid() && this.isTransitionPriorityListValid() && this.arePathsValid();
    }

    private boolean areStateCountsValid() {
        for (String state : this._statePriorityList) {
            String count = this._stateModelDef.getNumInstancesPerState(state);
            if (count == null) {
                _logger.error("State " + state + " needs an upper bound constraint, state model: " + this._stateModelDef.getId());
                return false;
            }
            try {
                Integer.parseInt(count);
            }
            catch (NumberFormatException e) {
                if (count.equals("N") || count.equals("R")) continue;
                _logger.error("State " + state + " has invalid count " + count + ", state model: " + this._stateModelDef.getId());
                return false;
            }
        }
        return true;
    }

    private boolean isTransitionPriorityListValid() {
        if (this._transitionPriorityList != null) {
            for (String transition : this._transitionPriorityList) {
                int index = transition.indexOf(45);
                int lastIndex = transition.indexOf(45);
                if (index <= 0 || index >= transition.length() - 1 || index != lastIndex) {
                    _logger.error("Transition " + transition + " is not of the form SRC-DEST, state model: " + this._stateModelDef.getId());
                    return false;
                }
                String from = transition.substring(0, index);
                String to = transition.substring(index + 1);
                if (!this._stateSet.contains(from)) {
                    _logger.error("State " + from + " in " + transition + " is not a defined state, state model" + this._stateModelDef.getId());
                    return false;
                }
                if (!this._stateSet.contains(to)) {
                    _logger.error("State " + to + " in " + transition + " is not a defined state, state model: " + this._stateModelDef.getId());
                    return false;
                }
                if (to.equals(this._stateModelDef.getNextStateForTransition(from, to))) continue;
                _logger.error("Transition " + transition + " must have " + to + " as the next state");
                return false;
            }
        }
        return true;
    }

    private boolean areNextStatesValid() {
        for (String state : this._statePriorityList) {
            if (!state.equals(HelixDefinedState.DROPPED.toString()) && this._stateModelDef.getNextStateForTransition(state, HelixDefinedState.DROPPED.toString()) == null) {
                _logger.error("State " + state + " cannot reach the DROPPED state, state model: " + this._stateModelDef.getId());
                return false;
            }
            if (!state.equals(this._stateModelDef.getInitialState()) && !state.equals(HelixDefinedState.ERROR.toString()) && this._stateModelDef.getNextStateForTransition(this._stateModelDef.getInitialState(), state) == null) {
                _logger.error("Initial state " + this._stateModelDef.getInitialState() + " should be able to reach all states, state model: " + this._stateModelDef.getId());
                return false;
            }
            for (String destState : this._statePriorityList) {
                if (state.equals(destState)) continue;
                String intermediate = this._stateModelDef.getNextStateForTransition(state, destState);
                if (intermediate != null && !this._stateSet.contains(intermediate)) {
                    _logger.error("Intermediate state " + intermediate + " for transition " + state + "-" + destState + " is not a valid state, state model: " + this._stateModelDef.getId());
                    return false;
                }
                if (intermediate == null || !intermediate.equals(state)) continue;
                _logger.error("Intermediate state " + intermediate + " for transition " + state + "-" + destState + " should never be the from state, state model: " + this._stateModelDef.getId());
                return false;
            }
        }
        return true;
    }

    private boolean arePathsValid() {
        HashMap alreadyChecked = Maps.newHashMap();
        for (String state : this._statePriorityList) {
            alreadyChecked.put(state, new HashSet());
        }
        for (String from : this._statePriorityList) {
            for (String to : this._statePriorityList) {
                if (from.equals(to)) continue;
                HashSet used = Sets.newHashSet((Object[])new String[]{from});
                String next = this._stateModelDef.getNextStateForTransition(from, to);
                if (next == null) {
                    if (!from.equals(this._stateModelDef.getInitialState()) || to.equals(HelixDefinedState.ERROR.toString())) continue;
                    _logger.error("Initial state " + from + " cannot reach " + to + ", state model: " + this._stateModelDef.getId());
                    return false;
                }
                while (!to.equals(next) && !((Set)alreadyChecked.get(next)).contains(to)) {
                    if (used.contains(next)) {
                        _logger.error("Path from " + from + " to " + to + " contains an infinite loop, state model: " + this._stateModelDef.getId());
                        return false;
                    }
                    ((Set)alreadyChecked.get(next)).add(to);
                    used.add(next);
                    if ((next = this._stateModelDef.getNextStateForTransition(next, to)) != null) continue;
                    _logger.error("Path from " + from + " to " + to + " is incomplete, state model: " + this._stateModelDef.getId());
                    return false;
                }
                ((Set)alreadyChecked.get(from)).add(to);
            }
        }
        return true;
    }

    public static boolean isStateModelDefinitionValid(StateModelDefinition stateModelDef) {
        return new StateModelDefinitionValidator(stateModelDef).isStateModelDefinitionValid();
    }
}

