/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.participant;

import com.google.common.collect.ImmutableList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import org.apache.helix.HelixDataAccessor;
import org.apache.helix.HelixException;
import org.apache.helix.HelixManager;
import org.apache.helix.InstanceType;
import org.apache.helix.NotificationContext;
import org.apache.helix.PropertyKey;
import org.apache.helix.messaging.handling.BatchMessageHandler;
import org.apache.helix.messaging.handling.BatchMessageWrapper;
import org.apache.helix.messaging.handling.HelixStateTransitionCancellationHandler;
import org.apache.helix.messaging.handling.HelixStateTransitionHandler;
import org.apache.helix.messaging.handling.MessageHandler;
import org.apache.helix.messaging.handling.TaskExecutor;
import org.apache.helix.model.CurrentState;
import org.apache.helix.model.Message;
import org.apache.helix.model.StateModelDefinition;
import org.apache.helix.participant.StateMachineEngine;
import org.apache.helix.participant.statemachine.StateModel;
import org.apache.helix.participant.statemachine.StateModelFactory;
import org.apache.helix.participant.statemachine.StateModelParser;
import org.apache.helix.task.TaskPartitionState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HelixStateMachineEngine
implements StateMachineEngine {
    private static Logger logger = LoggerFactory.getLogger(HelixStateMachineEngine.class);
    private final Map<String, Map<String, StateModelFactory<? extends StateModel>>> _stateModelFactoryMap;
    private final StateModelParser _stateModelParser = new StateModelParser();
    private final HelixManager _manager;
    private final ConcurrentHashMap<String, StateModelDefinition> _stateModelDefs;

    public HelixStateMachineEngine(HelixManager manager) {
        this._manager = manager;
        this._stateModelFactoryMap = new ConcurrentHashMap<String, Map<String, StateModelFactory<? extends StateModel>>>();
        this._stateModelDefs = new ConcurrentHashMap();
    }

    @Override
    public StateModelFactory<? extends StateModel> getStateModelFactory(String stateModelName) {
        return this.getStateModelFactory(stateModelName, "DEFAULT");
    }

    @Override
    public StateModelFactory<? extends StateModel> getStateModelFactory(String stateModelName, String factoryName) {
        if (!this._stateModelFactoryMap.containsKey(stateModelName)) {
            return null;
        }
        return this._stateModelFactoryMap.get(stateModelName).get(factoryName);
    }

    @Override
    public boolean registerStateModelFactory(String stateModelDef, StateModelFactory<? extends StateModel> factory) {
        return this.registerStateModelFactory(stateModelDef, factory, "DEFAULT");
    }

    @Override
    public boolean registerStateModelFactory(String stateModelName, StateModelFactory<? extends StateModel> factory, String factoryName) {
        if (stateModelName == null || factory == null || factoryName == null) {
            throw new HelixException("stateModelDef|stateModelFactory|factoryName cannot be null");
        }
        logger.info("Register state model factory for state model " + stateModelName + " using factory name " + factoryName + " with " + factory);
        if (!this._stateModelFactoryMap.containsKey(stateModelName)) {
            this._stateModelFactoryMap.put(stateModelName, new ConcurrentHashMap());
        }
        if (this._stateModelFactoryMap.get(stateModelName).containsKey(factoryName)) {
            logger.warn("stateModelFactory for " + stateModelName + " using factoryName " + factoryName + " has already been registered.");
            return false;
        }
        this._stateModelFactoryMap.get(stateModelName).put(factoryName, factory);
        this.sendNopMessage();
        return true;
    }

    private void sendNopMessage() {
        if (this._manager.isConnected()) {
            try {
                Message nopMsg = new Message(Message.MessageType.NO_OP, UUID.randomUUID().toString());
                nopMsg.setSrcName(this._manager.getInstanceName());
                HelixDataAccessor accessor = this._manager.getHelixDataAccessor();
                PropertyKey.Builder keyBuilder = accessor.keyBuilder();
                if (this._manager.getInstanceType() == InstanceType.CONTROLLER || this._manager.getInstanceType() == InstanceType.CONTROLLER_PARTICIPANT) {
                    nopMsg.setTgtName(InstanceType.CONTROLLER.name());
                    accessor.setProperty(keyBuilder.controllerMessage(nopMsg.getId()), nopMsg);
                }
                if (this._manager.getInstanceType() == InstanceType.PARTICIPANT || this._manager.getInstanceType() == InstanceType.CONTROLLER_PARTICIPANT) {
                    nopMsg.setTgtName(this._manager.getInstanceName());
                    accessor.setProperty(keyBuilder.message(nopMsg.getTgtName(), nopMsg.getId()), nopMsg);
                }
                logger.info("Send NO_OP message to " + nopMsg.getTgtName() + ", msgId: " + nopMsg.getId());
            }
            catch (Exception e) {
                logger.error(e.toString());
            }
        }
    }

    @Override
    public void reset() {
        logger.info("Resetting HelixStateMachineEngine");
        this.loopStateModelFactories(stateModel -> {
            stateModel.reset();
            String initialState = this._stateModelParser.getInitialState(stateModel.getClass());
            stateModel.updateState(initialState);
        });
        logger.info("Successfully reset HelixStateMachineEngine");
    }

    @Override
    public void sync() {
        logger.info("Syncing HelixStateMachineEngine");
        this.loopStateModelFactories(StateModel::syncState);
        logger.info("Successfully synced HelixStateMachineEngine");
    }

    private void loopStateModelFactories(Consumer<StateModel> consumer) {
        for (Map<String, StateModelFactory<? extends StateModel>> ftyMap : this._stateModelFactoryMap.values()) {
            for (StateModelFactory<? extends StateModel> stateModelFactory : ftyMap.values()) {
                for (String resourceName : stateModelFactory.getResourceSet()) {
                    for (String partitionKey : stateModelFactory.getPartitionSet(resourceName)) {
                        logger.info("Operating on {}::{}", (Object)resourceName, (Object)partitionKey);
                        StateModel stateModel = stateModelFactory.getStateModel(resourceName, partitionKey);
                        if (stateModel != null) {
                            consumer.accept(stateModel);
                            continue;
                        }
                        logger.warn("Failed operation due to StateModel being null! Resource: {}, Partition: {}", (Object)resourceName, (Object)partitionKey);
                    }
                }
            }
        }
    }

    @Override
    public MessageHandler createHandler(Message message, NotificationContext context) {
        TaskExecutor executor;
        StateModelFactory<? extends StateModel> stateModelFactory;
        String type = message.getMsgType();
        if (!type.equals(Message.MessageType.STATE_TRANSITION.name()) && !type.equals(Message.MessageType.STATE_TRANSITION_CANCELLATION.name())) {
            throw new HelixException("Expect state-transition message type, but was " + message.getMsgType() + ", msgId: " + message.getMsgId());
        }
        String partitionKey = message.getPartitionName();
        String stateModelName = message.getStateModelDef();
        String resourceName = message.getResourceName();
        String sessionId = message.getTgtSessionId();
        int bucketSize = message.getBucketSize();
        if (stateModelName == null) {
            logger.error("Fail to create msg-handler because message does not contain stateModelDef. msgId: " + message.getId());
            return null;
        }
        String factoryName = message.getStateModelFactoryName();
        if (factoryName == null) {
            factoryName = "DEFAULT";
        }
        if ((stateModelFactory = this.getStateModelFactory(stateModelName, factoryName)) == null) {
            logger.warn("Fail to create msg-handler because cannot find stateModelFactory for model: " + stateModelName + " using factoryName: " + factoryName + " for resource: " + resourceName);
            return null;
        }
        if (!this._stateModelDefs.containsKey(stateModelName)) {
            PropertyKey.Builder keyBuilder;
            HelixDataAccessor accessor = this._manager.getHelixDataAccessor();
            StateModelDefinition stateModelDef = (StateModelDefinition)accessor.getProperty((keyBuilder = accessor.keyBuilder()).stateModelDef(stateModelName));
            if (stateModelDef == null) {
                throw new HelixException("fail to create msg-handler because stateModelDef for " + stateModelName + " does NOT exist");
            }
            this._stateModelDefs.put(stateModelName, stateModelDef);
        }
        if (!message.getBatchMessageMode()) {
            String initState = this._stateModelDefs.get(message.getStateModelDef()).getInitialState();
            StateModel stateModel = stateModelFactory.getStateModel(resourceName, partitionKey);
            if (stateModel == null) {
                stateModel = stateModelFactory.createAndAddStateModel(resourceName, partitionKey);
                if (stateModelName.equals("Task") && message.getToState().equals(TaskPartitionState.DROPPED.name())) {
                    stateModel.updateState(message.getFromState());
                } else {
                    stateModel.updateState(initState);
                }
            }
            if (message.getMsgType().equals(Message.MessageType.STATE_TRANSITION_CANCELLATION.name())) {
                return new HelixStateTransitionCancellationHandler(stateModel, message, context);
            }
            CurrentState currentStateDelta = new CurrentState(resourceName);
            currentStateDelta.setSessionId(sessionId);
            currentStateDelta.setStateModelDefRef(stateModelName);
            currentStateDelta.setStateModelFactoryName(factoryName);
            currentStateDelta.setBucketSize(bucketSize);
            currentStateDelta.setState(partitionKey, stateModel.getCurrentState() == null ? initState : stateModel.getCurrentState());
            return new HelixStateTransitionHandler(stateModelFactory, stateModel, message, context, currentStateDelta);
        }
        BatchMessageWrapper wrapper = stateModelFactory.getBatchMessageWrapper(resourceName);
        if (wrapper == null) {
            wrapper = stateModelFactory.createAndAddBatchMessageWrapper(resourceName);
        }
        if ((executor = (TaskExecutor)context.get(NotificationContext.MapKey.TASK_EXECUTOR.toString())) == null) {
            logger.error("fail to get executor-service for batch message: " + message.getId() + ". msgType: " + message.getMsgType() + ", resource: " + message.getResourceName());
            return null;
        }
        return new BatchMessageHandler(message, context, this, wrapper, executor);
    }

    @Override
    public List<String> getMessageTypes() {
        return ImmutableList.of((Object)Message.MessageType.STATE_TRANSITION.name(), (Object)Message.MessageType.STATE_TRANSITION_CANCELLATION.name());
    }

    @Override
    public boolean removeStateModelFactory(String stateModelDef, StateModelFactory<? extends StateModel> factory) {
        throw new UnsupportedOperationException("Remove not yet supported");
    }

    @Override
    public boolean removeStateModelFactory(String stateModelDef, StateModelFactory<? extends StateModel> factory, String factoryName) {
        throw new UnsupportedOperationException("Remove not yet supported");
    }
}

