/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hop.pipeline.transforms.getserversequence;

import org.apache.hop.core.Const;
import org.apache.hop.core.util.Utils;
import org.apache.hop.core.variables.IVariables;
import org.apache.hop.i18n.BaseMessages;
import org.apache.hop.pipeline.PipelineMeta;
import org.apache.hop.pipeline.transform.BaseTransformMeta;
import org.apache.hop.pipeline.transform.ITransformDialog;
import org.apache.hop.server.HopServer;
import org.apache.hop.ui.core.dialog.BaseDialog;
import org.apache.hop.ui.core.widget.MetaSelectionLine;
import org.apache.hop.ui.core.widget.TextVar;
import org.apache.hop.ui.pipeline.transform.BaseTransformDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

public class GetServerSequenceDialog extends BaseTransformDialog implements ITransformDialog {
  private static final Class<?> PKG = GetServerSequenceMeta.class; // For Translator

  private Text wValuename;

  private MetaSelectionLine<HopServer> wHopServer;

  private TextVar wSeqname;

  private TextVar wIncrement;

  private final GetServerSequenceMeta input;

  public GetServerSequenceDialog(
      Shell parent, IVariables variables, Object in, PipelineMeta pipelineMeta, String sname) {
    super(parent, variables, (BaseTransformMeta) in, pipelineMeta, sname);
    input = (GetServerSequenceMeta) in;
  }

  @Override
  public String open() {
    Shell parent = getParent();

    shell = new Shell(parent, SWT.DIALOG_TRIM | SWT.RESIZE | SWT.MAX | SWT.MIN);
    props.setLook(shell);
    setShellImage(shell, input);

    ModifyListener lsMod = e -> input.setChanged();
    changed = input.hasChanged();

    FormLayout formLayout = new FormLayout();
    formLayout.marginWidth = Const.FORM_MARGIN;
    formLayout.marginHeight = Const.FORM_MARGIN;

    shell.setLayout(formLayout);
    shell.setText(BaseMessages.getString(PKG, "GetSequenceDialog.Shell.Title"));

    int middle = props.getMiddlePct();
    int margin = props.getMargin();

    // TransformName line
    wlTransformName = new Label(shell, SWT.RIGHT);
    wlTransformName.setText(BaseMessages.getString(PKG, "GetSequenceDialog.TransformName.Label"));
    props.setLook(wlTransformName);
    fdlTransformName = new FormData();
    fdlTransformName.left = new FormAttachment(0, 0);
    fdlTransformName.right = new FormAttachment(middle, -margin);
    fdlTransformName.top = new FormAttachment(0, margin);
    wlTransformName.setLayoutData(fdlTransformName);
    wTransformName = new Text(shell, SWT.SINGLE | SWT.LEFT | SWT.BORDER);
    wTransformName.setText(transformName);
    props.setLook(wTransformName);
    wTransformName.addModifyListener(lsMod);
    fdTransformName = new FormData();
    fdTransformName.left = new FormAttachment(middle, 0);
    fdTransformName.top = new FormAttachment(0, margin);
    fdTransformName.right = new FormAttachment(100, 0);
    wTransformName.setLayoutData(fdTransformName);

    // Valuename line
    Label wlValuename = new Label(shell, SWT.RIGHT);
    wlValuename.setText(BaseMessages.getString(PKG, "GetSequenceDialog.Valuename.Label"));
    props.setLook(wlValuename);
    FormData fdlValuename = new FormData();
    fdlValuename.left = new FormAttachment(0, 0);
    fdlValuename.top = new FormAttachment(wTransformName, margin);
    fdlValuename.right = new FormAttachment(middle, -margin);
    wlValuename.setLayoutData(fdlValuename);
    wValuename = new Text(shell, SWT.SINGLE | SWT.LEFT | SWT.BORDER);
    wValuename.setText("");
    props.setLook(wValuename);
    wValuename.addModifyListener(lsMod);
    FormData fdValuename = new FormData();
    fdValuename.left = new FormAttachment(middle, 0);
    fdValuename.top = new FormAttachment(wTransformName, margin);
    fdValuename.right = new FormAttachment(100, 0);
    wValuename.setLayoutData(fdValuename);

    // Hop server line
    wHopServer =
        new MetaSelectionLine<>(
            variables,
            pipelineMeta.getMetadataProvider(),
            HopServer.class,
            shell,
            SWT.BORDER,
            BaseMessages.getString(PKG, "GetSequenceDialog.HopServer.Label"),
            BaseMessages.getString(PKG, "GetSequenceDialog.HopServer.Tooltip"),
            false);
    FormData fdHopServer = new FormData();
    fdHopServer.left = new FormAttachment(0, 0);
    fdHopServer.top = new FormAttachment(wValuename, margin);
    fdHopServer.right = new FormAttachment(100, 0);
    wHopServer.setLayoutData(fdHopServer);
    wHopServer.addToConnectionLine(shell, wValuename, null, lsMod);
    props.setLook(wHopServer);

    // Sequence name line
    Label wlSeqname = new Label(shell, SWT.RIGHT);
    wlSeqname.setText(BaseMessages.getString(PKG, "GetSequenceDialog.Seqname.Label"));
    props.setLook(wlSeqname);
    FormData fdlSeqname = new FormData();
    fdlSeqname.left = new FormAttachment(0, 0);
    fdlSeqname.right = new FormAttachment(middle, -margin);
    fdlSeqname.top = new FormAttachment(wHopServer, margin);
    wlSeqname.setLayoutData(fdlSeqname);

    wSeqname = new TextVar(variables, shell, SWT.SINGLE | SWT.LEFT | SWT.BORDER);
    wSeqname.setText("");
    props.setLook(wSeqname);
    wSeqname.addModifyListener(lsMod);
    FormData fdSeqname = new FormData();
    fdSeqname.left = new FormAttachment(middle, 0);
    fdSeqname.top = new FormAttachment(wHopServer, margin);
    fdSeqname.right = new FormAttachment(100, 0);
    wSeqname.setLayoutData(fdSeqname);

    // Increment line
    Label wlIncrement = new Label(shell, SWT.RIGHT);
    wlIncrement.setText(BaseMessages.getString(PKG, "GetSequenceDialog.Increment.Label"));
    props.setLook(wlIncrement);
    FormData fdlIncrement = new FormData();
    fdlIncrement.left = new FormAttachment(0, 0);
    fdlIncrement.right = new FormAttachment(middle, -margin);
    fdlIncrement.top = new FormAttachment(wSeqname, margin);
    wlIncrement.setLayoutData(fdlIncrement);

    wIncrement = new TextVar(variables, shell, SWT.SINGLE | SWT.LEFT | SWT.BORDER);
    wIncrement.setText("");
    props.setLook(wIncrement);
    wIncrement.addModifyListener(lsMod);
    FormData fdIncrement = new FormData();
    fdIncrement.left = new FormAttachment(middle, 0);
    fdIncrement.top = new FormAttachment(wSeqname, margin);
    fdIncrement.right = new FormAttachment(100, 0);
    wIncrement.setLayoutData(fdIncrement);

    // THE BUTTONS
    wOk = new Button(shell, SWT.PUSH);
    wOk.setText(BaseMessages.getString(PKG, "System.Button.OK"));
    wCancel = new Button(shell, SWT.PUSH);
    wCancel.setText(BaseMessages.getString(PKG, "System.Button.Cancel"));

    setButtonPositions(new Button[] {wOk, wCancel}, margin, wIncrement);

    // Add listeners
    wOk.addListener(SWT.Selection, e -> ok());
    wCancel.addListener(SWT.Selection, e -> cancel());

    getData();
    input.setChanged(changed);

    BaseDialog.defaultShellHandling(shell, c -> ok(), c -> cancel());

    return transformName;
  }

  /** Copy information from the meta-data input to the dialog fields. */
  public void getData() {
    logDebug(BaseMessages.getString(PKG, "GetSequenceDialog.Log.GettingKeyInfo"));

    wValuename.setText(Const.NVL(input.getValuename(), ""));
    wHopServer.setText(Const.NVL(input.getHopServerName(), ""));
    wSeqname.setText(Const.NVL(input.getSequenceName(), ""));
    wIncrement.setText(Const.NVL(input.getIncrement(), ""));

    wTransformName.selectAll();
    wTransformName.setFocus();
  }

  private void cancel() {
    transformName = null;
    input.setChanged(changed);
    dispose();
  }

  private void ok() {
    if (Utils.isEmpty(wTransformName.getText())) {
      return;
    }

    transformName = wTransformName.getText(); // return value

    input.setHopServerName(wHopServer.getText());
    input.setValuename(wValuename.getText());
    input.setSequenceName(wSeqname.getText());
    input.setIncrement(wIncrement.getText());

    dispose();
  }
}
