/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hop.databases.kingbasees;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import org.apache.hop.core.database.DatabaseMeta;
import org.apache.hop.core.row.value.ValueMetaBigNumber;
import org.apache.hop.core.row.value.ValueMetaBoolean;
import org.apache.hop.core.row.value.ValueMetaDate;
import org.apache.hop.core.row.value.ValueMetaInteger;
import org.apache.hop.core.row.value.ValueMetaInternetAddress;
import org.apache.hop.core.row.value.ValueMetaNumber;
import org.apache.hop.core.row.value.ValueMetaString;
import org.apache.hop.core.row.value.ValueMetaTimestamp;
import org.junit.Before;
import org.junit.Test;

public class KingbaseESDatabaseMetaTest {

  private KingbaseESDatabaseMeta nativeMeta;

  @Before
  public void setupBefore() {
    nativeMeta = new KingbaseESDatabaseMeta();
    nativeMeta.setAccessType(DatabaseMeta.TYPE_ACCESS_NATIVE);
  }

  @Test
  public void testSettings() {
    assertArrayEquals(new int[] {DatabaseMeta.TYPE_ACCESS_NATIVE}, nativeMeta.getAccessTypeList());
    assertEquals(54321, nativeMeta.getDefaultDatabasePort());
    assertEquals(0, nativeMeta.getNotFoundTK(true));
    assertEquals(0, nativeMeta.getNotFoundTK(false));
    assertEquals("com.kingbase.Driver", nativeMeta.getDriverClass());
    assertEquals("jdbc:kingbase://FOO:BAR/WIBBLE", nativeMeta.getURL("FOO", "BAR", "WIBBLE"));
    assertEquals(
        "jdbc:kingbase://FOO:/WIBBLE",
        nativeMeta.getURL("FOO", "", "WIBBLE")); // Pretty sure this is a bug (colon after foo)

    assertArrayEquals(
        new String[] {
          // http://www.postgresql.org/docs/8.1/static/sql-keywords-appendix.html
          // added also non-reserved key words because there is progress from the Postgre developers
          // to add them
          "A",
          "ABORT",
          "ABS",
          "ABSOLUTE",
          "ACCESS",
          "ACTION",
          "ADA",
          "ADD",
          "ADMIN",
          "AFTER",
          "AGGREGATE",
          "ALIAS",
          "ALL",
          "ALLOCATE",
          "ALSO",
          "ALTER",
          "ALWAYS",
          "ANALYSE",
          "ANALYZE",
          "AND",
          "ANY",
          "ARE",
          "ARRAY",
          "AS",
          "ASC",
          "ASENSITIVE",
          "ASSERTION",
          "ASSIGNMENT",
          "ASYMMETRIC",
          "AT",
          "ATOMIC",
          "ATTRIBUTE",
          "ATTRIBUTES",
          "AUTHORIZATION",
          "AVG",
          "BACKWARD",
          "BEFORE",
          "BEGIN",
          "BERNOULLI",
          "BETWEEN",
          "BIGINT",
          "BINARY",
          "BIT",
          "BITVAR",
          "BIT_LENGTH",
          "BLOB",
          "BOOLEAN",
          "BOTH",
          "BREADTH",
          "BY",
          "C",
          "CACHE",
          "CALL",
          "CALLED",
          "CARDINALITY",
          "CASCADE",
          "CASCADED",
          "CASE",
          "CAST",
          "CATALOG",
          "CATALOG_NAME",
          "CEIL",
          "CEILING",
          "CHAIN",
          "CHAR",
          "CHARACTER",
          "CHARACTERISTICS",
          "CHARACTERS",
          "CHARACTER_LENGTH",
          "CHARACTER_SET_CATALOG",
          "CHARACTER_SET_NAME",
          "CHARACTER_SET_SCHEMA",
          "CHAR_LENGTH",
          "CHECK",
          "CHECKED",
          "CHECKPOINT",
          "CLASS",
          "CLASS_ORIGIN",
          "CLOB",
          "CLOSE",
          "CLUSTER",
          "COALESCE",
          "COBOL",
          "COLLATE",
          "COLLATION",
          "COLLATION_CATALOG",
          "COLLATION_NAME",
          "COLLATION_SCHEMA",
          "COLLECT",
          "COLUMN",
          "COLUMN_NAME",
          "COMMAND_FUNCTION",
          "COMMAND_FUNCTION_CODE",
          "COMMENT",
          "COMMIT",
          "COMMITTED",
          "COMPLETION",
          "CONDITION",
          "CONDITION_NUMBER",
          "CONNECT",
          "CONNECTION",
          "CONNECTION_NAME",
          "CONSTRAINT",
          "CONSTRAINTS",
          "CONSTRAINT_CATALOG",
          "CONSTRAINT_NAME",
          "CONSTRAINT_SCHEMA",
          "CONSTRUCTOR",
          "CONTAINS",
          "CONTINUE",
          "CONVERSION",
          "CONVERT",
          "COPY",
          "CORR",
          "CORRESPONDING",
          "COUNT",
          "COVAR_POP",
          "COVAR_SAMP",
          "CREATE",
          "CREATEDB",
          "CREATEROLE",
          "CREATEUSER",
          "CROSS",
          "CSV",
          "CUBE",
          "CUME_DIST",
          "CURRENT",
          "CURRENT_DATE",
          "CURRENT_DEFAULT_TRANSFORM_GROUP",
          "CURRENT_PATH",
          "CURRENT_ROLE",
          "CURRENT_TIME",
          "CURRENT_TIMESTAMP",
          "CURRENT_TRANSFORM_GROUP_FOR_TYPE",
          "CURRENT_USER",
          "CURSOR",
          "CURSOR_NAME",
          "CYCLE",
          "DATA",
          "DATABASE",
          "DATE",
          "DATETIME_INTERVAL_CODE",
          "DATETIME_INTERVAL_PRECISION",
          "DAY",
          "DEALLOCATE",
          "DEC",
          "DECIMAL",
          "DECLARE",
          "DEFAULT",
          "DEFAULTS",
          "DEFERRABLE",
          "DEFERRED",
          "DEFINED",
          "DEFINER",
          "DEGREE",
          "DELETE",
          "DELIMITER",
          "DELIMITERS",
          "DENSE_RANK",
          "DEPTH",
          "DEREF",
          "DERIVED",
          "DESC",
          "DESCRIBE",
          "DESCRIPTOR",
          "DESTROY",
          "DESTRUCTOR",
          "DETERMINISTIC",
          "DIAGNOSTICS",
          "DICTIONARY",
          "DISABLE",
          "DISCONNECT",
          "DISPATCH",
          "DISTINCT",
          "DO",
          "DOMAIN",
          "DOUBLE",
          "DROP",
          "DYNAMIC",
          "DYNAMIC_FUNCTION",
          "DYNAMIC_FUNCTION_CODE",
          "EACH",
          "ELEMENT",
          "ELSE",
          "ENABLE",
          "ENCODING",
          "ENCRYPTED",
          "END",
          "END-EXEC",
          "EQUALS",
          "ESCAPE",
          "EVERY",
          "EXCEPT",
          "EXCEPTION",
          "EXCLUDE",
          "EXCLUDING",
          "EXCLUSIVE",
          "EXEC",
          "EXECUTE",
          "EXISTING",
          "EXISTS",
          "EXP",
          "EXPLAIN",
          "EXTERNAL",
          "EXTRACT",
          "FALSE",
          "FETCH",
          "FILTER",
          "FINAL",
          "FIRST",
          "FLOAT",
          "FLOOR",
          "FOLLOWING",
          "FOR",
          "FORCE",
          "FOREIGN",
          "FORTRAN",
          "FORWARD",
          "FOUND",
          "FREE",
          "FREEZE",
          "FROM",
          "FULL",
          "FUNCTION",
          "FUSION",
          "G",
          "GENERAL",
          "GENERATED",
          "GET",
          "GLOBAL",
          "GO",
          "GOTO",
          "GRANT",
          "GRANTED",
          "GREATEST",
          "GROUP",
          "GROUPING",
          "HANDLER",
          "HAVING",
          "HEADER",
          "HIERARCHY",
          "HOLD",
          "HOST",
          "HOUR",
          "IDENTITY",
          "IGNORE",
          "ILIKE",
          "IMMEDIATE",
          "IMMUTABLE",
          "IMPLEMENTATION",
          "IMPLICIT",
          "IN",
          "INCLUDING",
          "INCREMENT",
          "INDEX",
          "INDICATOR",
          "INFIX",
          "INHERIT",
          "INHERITS",
          "INITIALIZE",
          "INITIALLY",
          "INNER",
          "INOUT",
          "INPUT",
          "INSENSITIVE",
          "INSERT",
          "INSTANCE",
          "INSTANTIABLE",
          "INSTEAD",
          "INT",
          "INTEGER",
          "INTERSECT",
          "INTERSECTION",
          "INTERVAL",
          "INTO",
          "INVOKER",
          "IS",
          "ISNULL",
          "ISOLATION",
          "ITERATE",
          "JOIN",
          "K",
          "KEY",
          "KEY_MEMBER",
          "KEY_TYPE",
          "LANCOMPILER",
          "LANGUAGE",
          "LARGE",
          "LAST",
          "LATERAL",
          "LEADING",
          "LEAST",
          "LEFT",
          "LENGTH",
          "LESS",
          "LEVEL",
          "LIKE",
          "LIMIT",
          "LISTEN",
          "LN",
          "LOAD",
          "LOCAL",
          "LOCALTIME",
          "LOCALTIMESTAMP",
          "LOCATION",
          "LOCATOR",
          "LOCK",
          "LOGIN",
          "LOWER",
          "M",
          "MAP",
          "MATCH",
          "MATCHED",
          "MAX",
          "MAXVALUE",
          "MEMBER",
          "MERGE",
          "MESSAGE_LENGTH",
          "MESSAGE_OCTET_LENGTH",
          "MESSAGE_TEXT",
          "METHOD",
          "MIN",
          "MINUTE",
          "MINVALUE",
          "MOD",
          "MODE",
          "MODIFIES",
          "MODIFY",
          "MODULE",
          "MONTH",
          "MORE",
          "MOVE",
          "MULTISET",
          "MUMPS",
          "NAME",
          "NAMES",
          "NATIONAL",
          "NATURAL",
          "NCHAR",
          "NCLOB",
          "NESTING",
          "NEW",
          "NEXT",
          "NO",
          "NOCREATEDB",
          "NOCREATEROLE",
          "NOCREATEUSER",
          "NOINHERIT",
          "NOLOGIN",
          "NONE",
          "NORMALIZE",
          "NORMALIZED",
          "NOSUPERUSER",
          "NOT",
          "NOTHING",
          "NOTIFY",
          "NOTNULL",
          "NOWAIT",
          "NULL",
          "NULLABLE",
          "NULLIF",
          "NULLS",
          "NUMBER",
          "NUMERIC",
          "OBJECT",
          "OCTETS",
          "OCTET_LENGTH",
          "OF",
          "OFF",
          "OFFSET",
          "OIDS",
          "OLD",
          "ON",
          "ONLY",
          "OPEN",
          "OPERATION",
          "OPERATOR",
          "OPTION",
          "OPTIONS",
          "OR",
          "ORDER",
          "ORDERING",
          "ORDINALITY",
          "OTHERS",
          "OUT",
          "OUTER",
          "OUTPUT",
          "OVER",
          "OVERLAPS",
          "OVERLAY",
          "OVERRIDING",
          "OWNER",
          "PAD",
          "PARAMETER",
          "PARAMETERS",
          "PARAMETER_MODE",
          "PARAMETER_NAME",
          "PARAMETER_ORDINAL_POSITION",
          "PARAMETER_SPECIFIC_CATALOG",
          "PARAMETER_SPECIFIC_NAME",
          "PARAMETER_SPECIFIC_SCHEMA",
          "PARTIAL",
          "PARTITION",
          "PASCAL",
          "PASSWORD",
          "PATH",
          "PERCENTILE_CONT",
          "PERCENTILE_DISC",
          "PERCENT_RANK",
          "PLACING",
          "PLI",
          "POSITION",
          "POSTFIX",
          "POWER",
          "PRECEDING",
          "PRECISION",
          "PREFIX",
          "PREORDER",
          "PREPARE",
          "PREPARED",
          "PRESERVE",
          "PRIMARY",
          "PRIOR",
          "PRIVILEGES",
          "PROCEDURAL",
          "PROCEDURE",
          "PUBLIC",
          "QUOTE",
          "RANGE",
          "RANK",
          "READ",
          "READS",
          "REAL",
          "RECHECK",
          "RECURSIVE",
          "REF",
          "REFERENCES",
          "REFERENCING",
          "REGR_AVGX",
          "REGR_AVGY",
          "REGR_COUNT",
          "REGR_INTERCEPT",
          "REGR_R2",
          "REGR_SLOPE",
          "REGR_SXX",
          "REGR_SXY",
          "REGR_SYY",
          "REINDEX",
          "RELATIVE",
          "RELEASE",
          "RENAME",
          "REPEATABLE",
          "REPLACE",
          "RESET",
          "RESTART",
          "RESTRICT",
          "RESULT",
          "RETURN",
          "RETURNED_CARDINALITY",
          "RETURNED_LENGTH",
          "RETURNED_OCTET_LENGTH",
          "RETURNED_SQLSTATE",
          "RETURNS",
          "REVOKE",
          "RIGHT",
          "ROLE",
          "ROLLBACK",
          "ROLLUP",
          "ROUTINE",
          "ROUTINE_CATALOG",
          "ROUTINE_NAME",
          "ROUTINE_SCHEMA",
          "ROW",
          "ROWS",
          "ROW_COUNT",
          "ROW_NUMBER",
          "RULE",
          "SAVEPOINT",
          "SCALE",
          "SCHEMA",
          "SCHEMA_NAME",
          "SCOPE",
          "SCOPE_CATALOG",
          "SCOPE_NAME",
          "SCOPE_SCHEMA",
          "SCROLL",
          "SEARCH",
          "SECOND",
          "SECTION",
          "SECURITY",
          "SELECT",
          "SELF",
          "SENSITIVE",
          "SEQUENCE",
          "SERIALIZABLE",
          "SERVER_NAME",
          "SESSION",
          "SESSION_USER",
          "SET",
          "SETOF",
          "SETS",
          "SHARE",
          "SHOW",
          "SIMILAR",
          "SIMPLE",
          "SIZE",
          "SMALLINT",
          "SOME",
          "SOURCE",
          "SPACE",
          "SPECIFIC",
          "SPECIFICTYPE",
          "SPECIFIC_NAME",
          "SQL",
          "SQLCODE",
          "SQLERROR",
          "SQLEXCEPTION",
          "SQLSTATE",
          "SQLWARNING",
          "SQRT",
          "STABLE",
          "START",
          "STATE",
          "STATEMENT",
          "STATIC",
          "STATISTICS",
          "STDDEV_POP",
          "STDDEV_SAMP",
          "STDIN",
          "STDOUT",
          "STORAGE",
          "STRICT",
          "STRUCTURE",
          "STYLE",
          "SUBCLASS_ORIGIN",
          "SUBLIST",
          "SUBMULTISET",
          "SUBSTRING",
          "SUM",
          "SUPERUSER",
          "SYMMETRIC",
          "SYSID",
          "SYSTEM",
          "SYSTEM_USER",
          "TABLE",
          "TABLESAMPLE",
          "TABLESPACE",
          "TABLE_NAME",
          "TEMP",
          "TEMPLATE",
          "TEMPORARY",
          "TERMINATE",
          "THAN",
          "THEN",
          "TIES",
          "TIME",
          "TIMESTAMP",
          "TIMEZONE_HOUR",
          "TIMEZONE_MINUTE",
          "TO",
          "TOAST",
          "TOP_LEVEL_COUNT",
          "TRAILING",
          "TRANSACTION",
          "TRANSACTIONS_COMMITTED",
          "TRANSACTIONS_ROLLED_BACK",
          "TRANSACTION_ACTIVE",
          "TRANSFORM",
          "TRANSFORMS",
          "TRANSLATE",
          "TRANSLATION",
          "TREAT",
          "TRIGGER",
          "TRIGGER_CATALOG",
          "TRIGGER_NAME",
          "TRIGGER_SCHEMA",
          "TRIM",
          "TRUE",
          "TRUNCATE",
          "TRUSTED",
          "TYPE",
          "UESCAPE",
          "UNBOUNDED",
          "UNCOMMITTED",
          "UNDER",
          "UNENCRYPTED",
          "UNION",
          "UNIQUE",
          "UNKNOWN",
          "UNLISTEN",
          "UNNAMED",
          "UNNEST",
          "UNTIL",
          "UPDATE",
          "UPPER",
          "USAGE",
          "USER",
          "USER_DEFINED_TYPE_CATALOG",
          "USER_DEFINED_TYPE_CODE",
          "USER_DEFINED_TYPE_NAME",
          "USER_DEFINED_TYPE_SCHEMA",
          "USING",
          "VACUUM",
          "VALID",
          "VALIDATOR",
          "VALUE",
          "VALUES",
          "VARCHAR",
          "VARIABLE",
          "VARYING",
          "VAR_POP",
          "VAR_SAMP",
          "VERBOSE",
          "VIEW",
          "VOLATILE",
          "WHEN",
          "WHENEVER",
          "WHERE",
          "WIDTH_BUCKET",
          "WINDOW",
          "WITH",
          "WITHIN",
          "WITHOUT",
          "WORK",
          "WRITE",
          "YEAR",
          "ZONE"
        },
        nativeMeta.getReservedWords());

    assertEquals("&", nativeMeta.getExtraOptionSeparator());
    assertEquals("?", nativeMeta.getExtraOptionIndicator());
    assertFalse(nativeMeta.isSupportsAutoInc());
    assertTrue(nativeMeta.isFetchSizeSupported());
    assertFalse(nativeMeta.isSupportsBitmapIndex());
    assertFalse(nativeMeta.isSupportsSynonyms());
    assertTrue(nativeMeta.isSupportsSequences());
    assertEquals(" limit 5", nativeMeta.getLimitClause(5));
    assertFalse(nativeMeta.isDefaultingToUppercase());
    assertFalse(nativeMeta.isSupportsTimeStampToDateConversion());
  }

  @Test
  public void testSqlStatements() {
    assertEquals("SELECT currval('FOO')", nativeMeta.getSqlCurrentSequenceValue("FOO"));
    assertEquals("SELECT nextval('FOO')", nativeMeta.getSqlNextSequenceValue("FOO"));
    assertEquals(
        "SELECT relname AS sequence_name FROM sys_class WHERE relname = 'foo'",
        nativeMeta.getSqlSequenceExists("FoO"));
    assertEquals(
        "SELECT relname AS sequence_name FROM sys_class", nativeMeta.getSqlListOfSequences());
    assertEquals(
        "ALTER TABLE FOO ADD COLUMN BAR VARCHAR(15)",
        nativeMeta.getAddColumnStatement(
            "FOO", new ValueMetaString("BAR", 15, 0), "", false, "", false));
    String lineSep = System.getProperty("line.separator");
    assertEquals(
        "ALTER TABLE FOO DROP COLUMN BAR" + lineSep,
        nativeMeta.getDropColumnStatement(
            "FOO", new ValueMetaString("BAR", 15, 0), "", false, "", false));
    assertEquals(
        "ALTER TABLE FOO DROP COLUMN BAR"
            + lineSep
            + ";"
            + lineSep
            + "ALTER TABLE FOO ADD COLUMN BAR VARCHAR(15)",
        nativeMeta.getModifyColumnStatement(
            "FOO", new ValueMetaString("BAR", 15, 0), "", false, "", false));
    assertEquals(
        "insert into FOO(FOOKEY, FOOVERSION) values (0, 1)",
        nativeMeta.getSqlInsertAutoIncUnknownDimensionRow("FOO", "FOOKEY", "FOOVERSION"));

    nativeMeta.setUsername("FOoUsEr");
    assertEquals(
        "select proname from sys_proc, sys_user where sys_user.usesysid = sys_proc.proowner and upper(sys_user.usename) = 'FOOUSER'",
        nativeMeta.getSqlListOfProcedures());
    assertEquals(
        "LOCK TABLE FOO , BAR IN ACCESS EXCLUSIVE MODE;" + lineSep,
        nativeMeta.getSqlLockTables(new String[] {"FOO", "BAR"}));
    assertNull(nativeMeta.getSqlUnlockTables(new String[] {}));
  }

  @Test
  public void testGetFieldDefinition() {
    assertEquals(
        "FOO TIMESTAMP",
        nativeMeta.getFieldDefinition(new ValueMetaDate("FOO"), "", "", false, true, false));
    assertEquals(
        "TIMESTAMP",
        nativeMeta.getFieldDefinition(new ValueMetaTimestamp("FOO"), "", "", false, false, false));
    // Simple hack to prevent duplication of code. Checking the case of supported boolean type
    // both supported and unsupported. Should return BOOLEAN if supported, or CHAR(1) if not.
    String[] typeCk = new String[] {"CHAR(1)", "BOOLEAN", "CHAR(1)"};
    int i = (nativeMeta.isSupportsBooleanDataType() ? 1 : 0);
    assertEquals(
        typeCk[i],
        nativeMeta.getFieldDefinition(new ValueMetaBoolean("FOO"), "", "", false, false, false));

    assertEquals(
        "BIGSERIAL",
        nativeMeta.getFieldDefinition(
            new ValueMetaNumber("FOO", 10, 0), "FOO", "", false, false, false));
    assertEquals(
        "BIGSERIAL",
        nativeMeta.getFieldDefinition(
            new ValueMetaInteger("FOO", 8, 0), "", "FOO", false, false, false));

    assertEquals(
        "NUMERIC(10, 5)",
        nativeMeta.getFieldDefinition(
            new ValueMetaBigNumber("FOO", 10, 5), "", "", false, false, false));

    assertEquals(
        "NUMERIC(19, 0)",
        nativeMeta.getFieldDefinition(
            new ValueMetaInteger("FOO", 19, 0), "", "", false, false, false));

    assertEquals(
        "BIGINT",
        nativeMeta.getFieldDefinition(
            new ValueMetaNumber("FOO", 10, 0), "", "", false, false, false));
    assertEquals(
        "INTEGER",
        nativeMeta.getFieldDefinition(
            new ValueMetaInteger("FOO", 9, 0), "", "", false, false, false));
    assertEquals(
        "SMALLINT",
        nativeMeta.getFieldDefinition(
            new ValueMetaInteger("FOO", 4, 0), "", "", false, false, false));
    assertEquals(
        "INTEGER",
        nativeMeta.getFieldDefinition(
            new ValueMetaInteger("FOO", 5, 0), "", "", false, false, false));

    assertEquals(
        "DOUBLE PRECISION",
        nativeMeta.getFieldDefinition(
            new ValueMetaBigNumber("FOO", -7, -3), "", "", false, false, false));

    assertEquals(
        "TEXT",
        nativeMeta.getFieldDefinition(
            new ValueMetaString("FOO", DatabaseMeta.CLOB_LENGTH + 1, 0),
            "",
            "",
            false,
            false,
            false));
    assertEquals(
        "TEXT",
        nativeMeta.getFieldDefinition(
            new ValueMetaString("FOO", 0, 0), "", "", false, false, false));
    assertEquals(
        "TEXT",
        nativeMeta.getFieldDefinition(new ValueMetaString("FOO"), "", "", false, false, false));
    assertEquals(
        "VARCHAR(15)",
        nativeMeta.getFieldDefinition(
            new ValueMetaString("FOO", 15, 0), "", "", false, false, false));

    assertEquals(
        " UNKNOWN",
        nativeMeta.getFieldDefinition(
            new ValueMetaInternetAddress("FOO"), "", "", false, false, false));
    assertEquals(
        " UNKNOWN" + System.getProperty("line.separator"),
        nativeMeta.getFieldDefinition(
            new ValueMetaInternetAddress("FOO"), "", "", false, false, true));
  }
}
