/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.flogger.GoogleLogger;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.grpc.ClientCall;
import io.grpc.Context;
import io.grpc.stub.StreamObserver;
import java.time.Clock;
import java.time.Duration;
import java.util.Iterator;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import javax.annotation.Nonnull;
import javax.annotation.concurrent.GuardedBy;

final class Watchdog
implements Runnable {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    private final ConcurrentHashMap.KeySetView<WatchdogStream, Boolean> openStreams = ConcurrentHashMap.newKeySet();
    private final Duration scheduleInterval;
    private final Clock clock = Clock.systemUTC();
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("gcs-background-watchdog-pool-%d").setDaemon(true).build());

    public static Watchdog create(Duration scheduleInterval) {
        Watchdog watchdog = new Watchdog(scheduleInterval);
        watchdog.start();
        return watchdog;
    }

    private Watchdog(Duration scheduleInterval) {
        this.scheduleInterval = scheduleInterval;
    }

    private void start() {
        this.executor.scheduleAtFixedRate(this, 0L, this.scheduleInterval.toMillis(), TimeUnit.MILLISECONDS);
    }

    public <T> Iterator<T> watch(Context.CancellableContext requestContext, Iterator<T> responseIterator, @Nonnull Duration idleTimeout) {
        Preconditions.checkNotNull(responseIterator, (Object)"responseIterator can't be null");
        Preconditions.checkNotNull((Object)idleTimeout, (Object)"idleTimeout can't be null");
        if (idleTimeout.isZero()) {
            return responseIterator;
        }
        ServerStreamingRPCWatchdogStream<T> stream = new ServerStreamingRPCWatchdogStream<T>(requestContext, responseIterator, idleTimeout);
        this.openStreams.add(stream);
        return stream;
    }

    public <R, T> StreamObserver<R> watch(ClientCall<R, T> clientCall, StreamObserver<R> streamObserver, @Nonnull Duration idleTimeout) {
        Preconditions.checkNotNull(streamObserver, (Object)"streamObserver can't be null");
        Preconditions.checkNotNull((Object)idleTimeout, (Object)"idleTimeout can't be null");
        if (idleTimeout.isZero()) {
            return streamObserver;
        }
        ClientStreamingRPCWatchdogStream<R, T> stream = new ClientStreamingRPCWatchdogStream<R, T>(clientCall, streamObserver, idleTimeout);
        this.openStreams.add(stream);
        return stream;
    }

    @Override
    public void run() {
        try {
            this.runUnsafe();
        }
        catch (RuntimeException e) {
            ((GoogleLogger.Api)((GoogleLogger.Api)logger.atWarning()).withCause((Throwable)e)).log("Caught RuntimeException in periodic Watchdog run, continuing.");
        }
    }

    @VisibleForTesting
    ConcurrentHashMap.KeySetView<WatchdogStream, Boolean> getOpenStreams() {
        return this.openStreams;
    }

    private void runUnsafe() {
        this.openStreams.removeIf(WatchdogStream::cancelIfStale);
    }

    public void shutdown() {
        this.executor.shutdown();
    }

    class ClientStreamingRPCWatchdogStream<R, T>
    implements StreamObserver<R>,
    WatchdogStream {
        private final Object lock = new Object();
        private final Duration waitTimeout;
        private final StreamObserver<R> innerStreamObserver;
        private final ClientCall<R, T> clientCall;
        @GuardedBy(value="lock")
        private long lastActivityAt = Watchdog.access$000(Watchdog.this).millis();
        @GuardedBy(value="lock")
        private State state = State.IDLE;

        public ClientStreamingRPCWatchdogStream(ClientCall<R, T> clientCall, StreamObserver<R> innerStreamObserver, Duration waitTimeout) {
            this.clientCall = clientCall;
            this.innerStreamObserver = innerStreamObserver;
            this.waitTimeout = waitTimeout;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public boolean cancelIfStale() {
            if (this.innerStreamObserver == null || this.clientCall == null) {
                return false;
            }
            Throwable throwable = null;
            Object object = this.lock;
            synchronized (object) {
                long waitTime = Watchdog.this.clock.millis() - this.lastActivityAt;
                if (this.state == State.WAITING && !this.waitTimeout.isZero() && waitTime >= this.waitTimeout.toMillis()) {
                    throwable = new TimeoutException("Canceled due to timeout waiting for next response");
                }
            }
            if (throwable != null) {
                this.clientCall.cancel(throwable.getMessage(), throwable);
                return true;
            }
            return false;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void onNext(R value) {
            Object object = this.lock;
            synchronized (object) {
                this.lastActivityAt = Watchdog.this.clock.millis();
                this.state = State.WAITING;
            }
            this.innerStreamObserver.onNext(value);
            object = this.lock;
            synchronized (object) {
                this.state = State.IDLE;
            }
        }

        public void onError(Throwable t) {
            Watchdog.this.openStreams.remove(this);
            this.innerStreamObserver.onError(t);
        }

        public void onCompleted() {
            Watchdog.this.openStreams.remove(this);
            this.innerStreamObserver.onCompleted();
        }
    }

    class ServerStreamingRPCWatchdogStream<T>
    implements Iterator<T>,
    WatchdogStream {
        private final Object lock = new Object();
        private final Duration waitTimeout;
        private final Iterator<T> innerIterator;
        private final Context.CancellableContext requestContext;
        @GuardedBy(value="lock")
        private long lastActivityAt = Watchdog.access$000(Watchdog.this).millis();
        @GuardedBy(value="lock")
        private State state = State.IDLE;

        ServerStreamingRPCWatchdogStream(Context.CancellableContext requestContext, Iterator<T> iterator, Duration waitTimeout) {
            this.waitTimeout = waitTimeout;
            this.requestContext = requestContext;
            this.innerIterator = iterator;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public T next() {
            T next = this.innerIterator.next();
            Object object = this.lock;
            synchronized (object) {
                this.state = State.DELIVERING;
            }
            return next;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public boolean cancelIfStale() {
            if (this.innerIterator == null || this.requestContext == null) {
                return false;
            }
            TimeoutException throwable = null;
            Object object = this.lock;
            synchronized (object) {
                long waitTime = Watchdog.this.clock.millis() - this.lastActivityAt;
                if (this.state == State.WAITING && !this.waitTimeout.isZero() && waitTime >= this.waitTimeout.toMillis()) {
                    throwable = new TimeoutException("Canceled due to timeout waiting for next response");
                }
            }
            if (throwable != null) {
                this.requestContext.cancel(throwable);
                return true;
            }
            return false;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public boolean hasNext() {
            boolean hasNext = false;
            try {
                Object object = this.lock;
                synchronized (object) {
                    this.lastActivityAt = Watchdog.this.clock.millis();
                    this.state = State.WAITING;
                }
                hasNext = this.innerIterator.hasNext();
            }
            finally {
                if (!hasNext) {
                    Watchdog.this.openStreams.remove(this);
                }
            }
            return hasNext;
        }
    }

    static interface WatchdogStream {
        public boolean cancelIfStale();
    }

    static enum State {
        IDLE,
        WAITING,
        DELIVERING;

    }
}

