/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.hints;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import java.io.File;
import java.net.UnknownHostException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.ParameterizedClass;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.gms.IFailureDetector;
import org.apache.cassandra.hints.Hint;
import org.apache.cassandra.hints.HintsBufferPool;
import org.apache.cassandra.hints.HintsCatalog;
import org.apache.cassandra.hints.HintsDispatchExecutor;
import org.apache.cassandra.hints.HintsDispatchTrigger;
import org.apache.cassandra.hints.HintsServiceDiagnostics;
import org.apache.cassandra.hints.HintsServiceMBean;
import org.apache.cassandra.hints.HintsStore;
import org.apache.cassandra.hints.HintsWriteExecutor;
import org.apache.cassandra.locator.EndpointsForToken;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.locator.ReplicaLayout;
import org.apache.cassandra.metrics.HintedHandoffMetrics;
import org.apache.cassandra.metrics.StorageMetrics;
import org.apache.cassandra.service.StorageProxy;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.MBeanWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class HintsService
implements HintsServiceMBean {
    private static final Logger logger = LoggerFactory.getLogger(HintsService.class);
    public static HintsService instance = new HintsService();
    public static final String MBEAN_NAME = "org.apache.cassandra.hints:type=HintsService";
    private static final int MIN_BUFFER_SIZE = 0x2000000;
    static final ImmutableMap<String, Object> EMPTY_PARAMS = ImmutableMap.of();
    private final HintsCatalog catalog;
    private final HintsWriteExecutor writeExecutor;
    private final HintsBufferPool bufferPool;
    final HintsDispatchExecutor dispatchExecutor;
    final AtomicBoolean isDispatchPaused;
    private volatile boolean isShutDown = false;
    private final ScheduledFuture triggerFlushingFuture;
    private volatile ScheduledFuture triggerDispatchFuture;
    public final HintedHandoffMetrics metrics;

    private HintsService() {
        this(FailureDetector.instance);
    }

    @VisibleForTesting
    HintsService(IFailureDetector failureDetector) {
        File hintsDirectory = DatabaseDescriptor.getHintsDirectory();
        int maxDeliveryThreads = DatabaseDescriptor.getMaxHintsDeliveryThreads();
        this.catalog = HintsCatalog.load(hintsDirectory, HintsService.createDescriptorParams());
        this.writeExecutor = new HintsWriteExecutor(this.catalog);
        int bufferSize = Math.max(DatabaseDescriptor.getMaxMutationSize() * 2, 0x2000000);
        this.bufferPool = new HintsBufferPool(bufferSize, this.writeExecutor::flushBuffer);
        this.isDispatchPaused = new AtomicBoolean(true);
        this.dispatchExecutor = new HintsDispatchExecutor(hintsDirectory, maxDeliveryThreads, this.isDispatchPaused, failureDetector::isAlive);
        int flushPeriod = DatabaseDescriptor.getHintsFlushPeriodInMS();
        this.triggerFlushingFuture = ScheduledExecutors.optionalTasks.scheduleWithFixedDelay(() -> this.writeExecutor.flushBufferPool(this.bufferPool), flushPeriod, flushPeriod, TimeUnit.MILLISECONDS);
        this.metrics = new HintedHandoffMetrics();
    }

    private static ImmutableMap<String, Object> createDescriptorParams() {
        ImmutableMap.Builder builder = ImmutableMap.builder();
        ParameterizedClass compressionConfig = DatabaseDescriptor.getHintsCompression();
        if (compressionConfig != null) {
            ImmutableMap.Builder compressorParams = ImmutableMap.builder();
            compressorParams.put((Object)"class_name", (Object)compressionConfig.class_name);
            if (compressionConfig.parameters != null) {
                compressorParams.put((Object)"parameters", compressionConfig.parameters);
            }
            builder.put((Object)"compression", (Object)compressorParams.build());
        }
        return builder.build();
    }

    public void registerMBean() {
        MBeanWrapper.instance.registerMBean((Object)this, MBEAN_NAME);
    }

    public void write(Collection<UUID> hostIds, Hint hint) {
        if (this.isShutDown) {
            throw new IllegalStateException("HintsService is shut down and can't accept new hints");
        }
        this.catalog.maybeLoadStores(hostIds);
        this.bufferPool.write(hostIds, hint);
        StorageMetrics.totalHints.inc((long)hostIds.size());
    }

    public void write(UUID hostId, Hint hint) {
        this.write(Collections.singleton(hostId), hint);
    }

    void writeForAllReplicas(Hint hint) {
        String keyspaceName = hint.mutation.getKeyspaceName();
        Token token = hint.mutation.key().getToken();
        EndpointsForToken replicas = (EndpointsForToken)ReplicaLayout.forTokenWriteLiveAndDown(Keyspace.open(keyspaceName), token).all();
        List<UUID> hostIds = replicas.stream().filter(StorageProxy::shouldHint).map(replica -> StorageService.instance.getHostIdForEndpoint(replica.endpoint())).collect(Collectors.toList());
        this.write(hostIds, hint);
    }

    public void flushAndFsyncBlockingly(Iterable<UUID> hostIds) {
        Iterable stores = Iterables.filter((Iterable)Iterables.transform(hostIds, this.catalog::getNullable), Objects::nonNull);
        this.writeExecutor.flushBufferPool(this.bufferPool, stores);
        this.writeExecutor.fsyncWritersBlockingly(stores);
    }

    public synchronized void startDispatch() {
        if (this.isShutDown) {
            throw new IllegalStateException("HintsService is shut down and cannot be restarted");
        }
        this.isDispatchPaused.set(false);
        HintsServiceDiagnostics.dispatchingStarted(this);
        HintsDispatchTrigger trigger = new HintsDispatchTrigger(this.catalog, this.writeExecutor, this.dispatchExecutor, this.isDispatchPaused);
        this.triggerDispatchFuture = ScheduledExecutors.scheduledTasks.scheduleWithFixedDelay(trigger, 10L, 10L, TimeUnit.SECONDS);
    }

    @Override
    public void pauseDispatch() {
        logger.info("Paused hints dispatch");
        this.isDispatchPaused.set(true);
        HintsServiceDiagnostics.dispatchingPaused(this);
    }

    @Override
    public void resumeDispatch() {
        logger.info("Resumed hints dispatch");
        this.isDispatchPaused.set(false);
        HintsServiceDiagnostics.dispatchingResumed(this);
    }

    public synchronized void shutdownBlocking() throws ExecutionException, InterruptedException {
        if (this.isShutDown) {
            throw new IllegalStateException("HintsService has already been shut down");
        }
        this.isShutDown = true;
        if (this.triggerDispatchFuture != null) {
            this.triggerDispatchFuture.cancel(false);
        }
        this.pauseDispatch();
        this.triggerFlushingFuture.cancel(false);
        this.writeExecutor.flushBufferPool(this.bufferPool).get();
        this.writeExecutor.closeAllWriters().get();
        this.dispatchExecutor.shutdownBlocking();
        this.writeExecutor.shutdownBlocking();
        HintsServiceDiagnostics.dispatchingShutdown(this);
        this.bufferPool.close();
    }

    @Override
    public void deleteAllHints() {
        this.catalog.deleteAllHints();
    }

    @Override
    public void deleteAllHintsForEndpoint(String address) {
        InetAddressAndPort target;
        try {
            target = InetAddressAndPort.getByName(address);
        }
        catch (UnknownHostException e) {
            throw new IllegalArgumentException(e);
        }
        this.deleteAllHintsForEndpoint(target);
    }

    public void deleteAllHintsForEndpoint(InetAddressAndPort target) {
        UUID hostId = StorageService.instance.getHostIdForEndpoint(target);
        if (hostId == null) {
            throw new IllegalArgumentException("Can't delete hints for unknown address " + target);
        }
        this.catalog.deleteAllHints(hostId);
    }

    public void excise(UUID hostId) {
        HintsStore store = this.catalog.getNullable(hostId);
        if (store == null) {
            return;
        }
        Future<?> flushFuture = this.writeExecutor.flushBufferPool(this.bufferPool, Collections.singleton(store));
        Future<?> closeFuture = this.writeExecutor.closeWriter(store);
        try {
            flushFuture.get();
            closeFuture.get();
        }
        catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException(e);
        }
        this.dispatchExecutor.interruptDispatch(store.hostId);
        this.catalog.exciseStore(hostId);
    }

    public Future transferHints(Supplier<UUID> hostIdSupplier) {
        Future<?> flushFuture = this.writeExecutor.flushBufferPool(this.bufferPool);
        Future<?> closeFuture = this.writeExecutor.closeAllWriters();
        try {
            flushFuture.get();
            closeFuture.get();
        }
        catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException(e);
        }
        this.resumeDispatch();
        this.catalog.stores().forEach(this.dispatchExecutor::completeDispatchBlockingly);
        return this.dispatchExecutor.transfer(this.catalog, hostIdSupplier);
    }

    HintsCatalog getCatalog() {
        return this.catalog;
    }

    public boolean isShutDown() {
        return this.isShutDown;
    }

    @VisibleForTesting
    public boolean isDispatchPaused() {
        return this.isDispatchPaused.get();
    }
}

