/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.binarytuple;

import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Period;
import java.util.UUID;
import org.apache.ignite.internal.binarytuple.BinaryTupleFormatException;
import org.apache.ignite.internal.util.ByteUtils;

public class BinaryTupleParser {
    public static final ByteOrder ORDER = ByteOrder.LITTLE_ENDIAN;
    private static final int UUID_SIZE = 16;
    private final int numElements;
    private final int entrySize;
    private final int entryBase;
    private final int valueBase;
    private final ByteBuffer buffer;

    public BinaryTupleParser(int numElements, ByteBuffer buffer) {
        this.numElements = numElements;
        assert (buffer.order() == ORDER) : "Buffer order must be LITTLE_ENDIAN, actual: " + String.valueOf(buffer.order());
        assert (buffer.position() == 0) : "Buffer position must be 0, actual: " + buffer.position();
        this.buffer = buffer;
        byte flags = buffer.get(0);
        this.entryBase = 1;
        this.entrySize = 1 << (flags & 3);
        this.valueBase = this.entryBase + this.entrySize * numElements;
    }

    public int size() {
        return this.valueBase + this.getOffset(this.valueBase - this.entrySize);
    }

    public int elementCount() {
        return this.numElements;
    }

    public ByteBuffer byteBuffer() {
        return this.buffer.slice().order(ORDER);
    }

    public void fetch(int index, Sink sink) {
        int nextOffset;
        assert (index >= 0);
        assert (index < this.numElements) : "Index out of bounds: " + index + " >= " + this.numElements;
        int entry = this.entryBase + index * this.entrySize;
        int offset = this.valueBase;
        if (index > 0) {
            offset += this.getOffset(entry - this.entrySize);
        }
        if ((nextOffset = this.valueBase + this.getOffset(entry)) < offset) {
            throw new BinaryTupleFormatException("Corrupted offset table");
        }
        sink.nextElement(index, offset, nextOffset);
    }

    public void parse(Sink sink) {
        int entry = this.entryBase;
        int offset = this.valueBase;
        for (int i = 0; i < this.numElements; ++i) {
            int nextOffset = this.valueBase + this.getOffset(entry);
            if (nextOffset < offset) {
                throw new BinaryTupleFormatException("Corrupted offset table");
            }
            sink.nextElement(i, offset, nextOffset);
            offset = nextOffset;
            entry += this.entrySize;
        }
    }

    public final boolean booleanValue(int begin, int end) {
        int len = end - begin;
        if (len == 1) {
            return ByteUtils.byteToBoolean((byte)this.buffer.get(begin));
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    public final byte byteValue(int begin, int end) {
        int len = end - begin;
        switch (len) {
            case 1: {
                return this.buffer.get(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    public final short shortValue(int begin, int end) {
        int len = end - begin;
        switch (len) {
            case 1: {
                return this.buffer.get(begin);
            }
            case 2: {
                return this.buffer.getShort(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    public final int intValue(int begin, int end) {
        int len = end - begin;
        switch (len) {
            case 1: {
                return this.buffer.get(begin);
            }
            case 2: {
                return this.buffer.getShort(begin);
            }
            case 4: {
                return this.buffer.getInt(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    public final long longValue(int begin, int end) {
        int len = end - begin;
        switch (len) {
            case 1: {
                return this.buffer.get(begin);
            }
            case 2: {
                return this.buffer.getShort(begin);
            }
            case 4: {
                return this.buffer.getInt(begin);
            }
            case 8: {
                return this.buffer.getLong(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    public final float floatValue(int begin, int end) {
        int len = end - begin;
        switch (len) {
            case 4: {
                return this.buffer.getFloat(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    public final double doubleValue(int begin, int end) {
        int len = end - begin;
        switch (len) {
            case 4: {
                return this.buffer.getFloat(begin);
            }
            case 8: {
                return this.buffer.getDouble(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    protected BigInteger numberValue(int begin, int end) {
        byte[] bytes;
        int len = end - begin;
        if (len <= 0) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        if (this.buffer.hasArray()) {
            bytes = this.buffer.array();
            begin += this.buffer.arrayOffset();
        } else {
            bytes = this.getBytes(begin, end);
            begin = 0;
        }
        return new BigInteger(bytes, begin, len);
    }

    public final String stringValue(int begin, int end) {
        byte[] bytes;
        int len = end - begin;
        if (len <= 0) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        if (this.buffer.get(begin) == -128) {
            ++begin;
            --len;
        }
        if (this.buffer.hasArray()) {
            bytes = this.buffer.array();
            begin += this.buffer.arrayOffset();
        } else {
            bytes = this.getBytes(begin, end);
            begin = 0;
        }
        return new String(bytes, begin, len, StandardCharsets.UTF_8);
    }

    public final byte[] bytesValue(int begin, int end) {
        int len = end - begin;
        if (len <= 0) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        if (this.buffer.get(begin) == -128) {
            ++begin;
        }
        return this.getBytes(begin, end);
    }

    public final ByteBuffer bytesValueAsBuffer(int begin, int end) {
        int len = end - begin;
        if (len <= 0) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        if (this.buffer.get(begin) == -128) {
            ++begin;
        }
        return this.buffer.duplicate().position(begin).limit(end).slice();
    }

    public final UUID uuidValue(int begin, int end) {
        int len = end - begin;
        if (len != 16) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        long msb = this.buffer.getLong(begin);
        long lsb = this.buffer.getLong(begin + 8);
        return new UUID(msb, lsb);
    }

    public final LocalDate dateValue(int begin, int end) {
        int len = end - begin;
        if (len != 3) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        return this.getDate(begin);
    }

    public final LocalTime timeValue(int begin, int end) {
        int len = end - begin;
        if (len < 4 || len > 6) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        return this.getTime(begin, len);
    }

    public final LocalDateTime dateTimeValue(int begin, int end) {
        int len = end - begin;
        if (len < 7 || len > 9) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        return LocalDateTime.of(this.getDate(begin), this.getTime(begin + 3, len - 3));
    }

    public final Instant timestampValue(int begin, int end) {
        int len = end - begin;
        if (len != 8 && len != 12) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        long seconds = this.buffer.getLong(begin);
        int nanos = len == 8 ? 0 : this.buffer.getInt(begin + 8);
        return Instant.ofEpochSecond(seconds, nanos);
    }

    public final Duration durationValue(int begin, int end) {
        int len = end - begin;
        if (len != 8 && len != 12) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        long seconds = this.buffer.getLong(begin);
        int nanos = len == 8 ? 0 : this.buffer.getInt(begin + 8);
        return Duration.ofSeconds(seconds, nanos);
    }

    public final Period periodValue(int begin, int end) {
        int len = end - begin;
        switch (len) {
            case 3: {
                return Period.of(this.buffer.get(begin), this.buffer.get(begin + 1), this.buffer.get(begin + 2));
            }
            case 6: {
                return Period.of(this.buffer.getShort(begin), this.buffer.getShort(begin + 2), this.buffer.getShort(begin + 4));
            }
            case 12: {
                return Period.of(this.buffer.getInt(begin), this.buffer.getInt(begin + 4), this.buffer.getInt(begin + 8));
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    private int getOffset(int index) {
        switch (this.entrySize) {
            case 1: {
                return Byte.toUnsignedInt(this.buffer.get(index));
            }
            case 2: {
                return Short.toUnsignedInt(this.buffer.getShort(index));
            }
            case 4: {
                int offset = this.buffer.getInt(index);
                if (offset < 0) {
                    throw new BinaryTupleFormatException("Unsupported offset table size");
                }
                return offset;
            }
            case 8: {
                throw new BinaryTupleFormatException("Unsupported offset table size");
            }
        }
        throw new BinaryTupleFormatException("Invalid offset table size");
    }

    private byte[] getBytes(int begin, int end) {
        byte[] bytes = new byte[end - begin];
        this.buffer.duplicate().position(begin).limit(end).get(bytes);
        return bytes;
    }

    private LocalDate getDate(int offset) {
        int date = Short.toUnsignedInt(this.buffer.getShort(offset));
        int day = (date |= this.buffer.get(offset + 2) << 16) & 0x1F;
        int month = date >> 5 & 0xF;
        int year = date >> 9;
        return LocalDate.of(year, month, day);
    }

    private LocalTime getTime(int offset, int length) {
        int nanos;
        long time = Integer.toUnsignedLong(this.buffer.getInt(offset));
        if (length == 4) {
            nanos = ((int)time & 0x3FF) * 1000 * 1000;
            time >>>= 10;
        } else if (length == 5) {
            nanos = ((int)(time |= Byte.toUnsignedLong(this.buffer.get(offset + 4)) << 32) & 0xFFFFF) * 1000;
            time >>>= 20;
        } else {
            nanos = (int)(time |= Short.toUnsignedLong(this.buffer.getShort(offset + 4)) << 32) & 0x3FFFFFFF;
            time >>>= 30;
        }
        int second = (int)time & 0x3F;
        int minute = (int)time >>> 6 & 0x3F;
        int hour = (int)time >>> 12 & 0x1F;
        return LocalTime.of(hour, minute, second, nanos);
    }

    public static interface Sink {
        public void nextElement(int var1, int var2, int var3);
    }
}

