/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.storage.hbase.util;

import java.io.Closeable;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Random;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.TableNotFoundException;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.filter.Filter;
import org.apache.hadoop.hbase.filter.KeyOnlyFilter;
import org.apache.kylin.common.StorageURL;
import org.apache.kylin.common.util.Bytes;
import org.apache.kylin.common.util.Pair;
import org.apache.kylin.shaded.com.google.common.collect.Lists;
import org.apache.kylin.storage.hbase.HBaseConnection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GridTableHBaseBenchmark {
    private static final String TEST_TABLE = "GridTableTest";
    private static final byte[] CF = "F".getBytes(StandardCharsets.UTF_8);
    private static final byte[] QN = "C".getBytes(StandardCharsets.UTF_8);
    private static final int N_ROWS = 10000;
    private static final int CELL_SIZE = 131072;
    private static final double DFT_HIT_RATIO = 0.3;
    private static final double DFT_INDEX_RATIO = 0.1;
    private static final int ROUND = 3;
    private static final Random rand = new Random();
    protected static final Logger logger = LoggerFactory.getLogger(GridTableHBaseBenchmark.class);

    public static void main(String[] args) throws IOException {
        double hitRatio = 0.3;
        try {
            hitRatio = Double.parseDouble(args[0]);
        }
        catch (Exception exception) {
            // empty catch block
        }
        double indexRatio = 0.1;
        try {
            indexRatio = Double.parseDouble(args[1]);
        }
        catch (Exception exception) {
            // empty catch block
        }
        GridTableHBaseBenchmark.testGridTable(hitRatio, indexRatio);
    }

    public static void testGridTable(double hitRatio, double indexRatio) throws IOException {
        logger.info("Testing grid table scanning, hit ratio {}, index ratio {}", (Object)hitRatio, (Object)indexRatio);
        StorageURL hbaseUrl = StorageURL.valueOf("default@hbase");
        Connection conn = HBaseConnection.get(hbaseUrl);
        GridTableHBaseBenchmark.createHTableIfNeeded(conn, TEST_TABLE);
        GridTableHBaseBenchmark.prepareData(conn);
        Hits hits = new Hits(10000, hitRatio, indexRatio);
        for (int i = 0; i < 3; ++i) {
            logger.info("==================================== ROUND {} ========================================", (Object)(i + 1));
            GridTableHBaseBenchmark.testRowScanWithIndex(conn, hits.getHitsForRowScanWithIndex());
            GridTableHBaseBenchmark.testRowScanNoIndexFullScan(conn, hits.getHitsForRowScanNoIndex());
            GridTableHBaseBenchmark.testRowScanNoIndexSkipScan(conn, hits.getHitsForRowScanNoIndex());
            GridTableHBaseBenchmark.testColumnScan(conn, hits.getHitsForColumnScan());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void testColumnScan(Connection conn, List<Pair<Integer, Integer>> colScans) throws IOException {
        Stats stats = new Stats("COLUMN_SCAN");
        Table table = conn.getTable(TableName.valueOf((String)TEST_TABLE));
        try {
            int i;
            stats.markStart();
            int nLogicCols = colScans.size();
            int nLogicRows = colScans.get(0).getSecond() - colScans.get(0).getFirst();
            Scan[] scans = new Scan[nLogicCols];
            ResultScanner[] scanners = new ResultScanner[nLogicCols];
            for (i = 0; i < nLogicCols; ++i) {
                scans[i] = new Scan();
                scans[i].addFamily(CF);
                scanners[i] = table.getScanner(scans[i]);
            }
            for (i = 0; i < nLogicRows; ++i) {
                for (int c = 0; c < nLogicCols; ++c) {
                    Result r = scanners[c].next();
                    stats.consume(r);
                }
                GridTableHBaseBenchmark.dot(i, nLogicRows);
            }
            stats.markEnd();
        }
        finally {
            IOUtils.closeQuietly((Closeable)table);
        }
    }

    private static void testRowScanNoIndexFullScan(Connection conn, boolean[] hits) throws IOException {
        GridTableHBaseBenchmark.fullScan(conn, hits, new Stats("ROW_SCAN_NO_IDX_FULL"));
    }

    private static void testRowScanNoIndexSkipScan(Connection conn, boolean[] hits) throws IOException {
        GridTableHBaseBenchmark.jumpScan(conn, hits, new Stats("ROW_SCAN_NO_IDX_SKIP"));
    }

    private static void testRowScanWithIndex(Connection conn, boolean[] hits) throws IOException {
        GridTableHBaseBenchmark.jumpScan(conn, hits, new Stats("ROW_SCAN_IDX"));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void fullScan(Connection conn, boolean[] hits, Stats stats) throws IOException {
        Table table = conn.getTable(TableName.valueOf((String)TEST_TABLE));
        try {
            stats.markStart();
            Scan scan = new Scan();
            scan.addFamily(CF);
            ResultScanner scanner = table.getScanner(scan);
            int i = 0;
            for (Result r : scanner) {
                if (hits[i]) {
                    stats.consume(r);
                }
                GridTableHBaseBenchmark.dot(i, 10000);
                ++i;
            }
            stats.markEnd();
        }
        finally {
            IOUtils.closeQuietly((Closeable)table);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void jumpScan(Connection conn, boolean[] hits, Stats stats) throws IOException {
        int jumpThreshold = 6;
        Table table = conn.getTable(TableName.valueOf((String)TEST_TABLE));
        try {
            stats.markStart();
            int i = 0;
            while (i < 10000) {
                int start = i;
                while (start + 1 < 10000 && !hits[start]) {
                    ++start;
                }
                int end = start + 1;
                for (int jump = end + 1; jump < 10000 && end + 6 > jump; ++jump) {
                    if (!hits[jump]) continue;
                    end = jump;
                }
                if (start < 10000) {
                    Scan scan = new Scan();
                    scan.setStartRow(Bytes.toBytes(start));
                    scan.setStopRow(Bytes.toBytes(end));
                    scan.addFamily(CF);
                    ResultScanner scanner = table.getScanner(scan);
                    i = start;
                    for (Result r : scanner) {
                        stats.consume(r);
                        GridTableHBaseBenchmark.dot(i, 10000);
                        ++i;
                    }
                }
                i = end;
            }
            stats.markEnd();
        }
        finally {
            IOUtils.closeQuietly((Closeable)table);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void prepareData(Connection conn) throws IOException {
        Table table = conn.getTable(TableName.valueOf((String)TEST_TABLE));
        try {
            int nRows = 0;
            Scan scan = new Scan();
            scan.setFilter((Filter)new KeyOnlyFilter());
            ResultScanner scanner = table.getScanner(scan);
            for (Result r : scanner) {
                r.getRow();
                ++nRows;
            }
            if (nRows > 0) {
                logger.info("{} existing rows", (Object)nRows);
                if (nRows != 10000) {
                    throw new IOException("Expect 10000 rows but it is not");
                }
                return;
            }
            logger.info("Writing {} rows to {}", (Object)10000, (Object)TEST_TABLE);
            long nBytes = 0L;
            for (int i = 0; i < 10000; ++i) {
                byte[] rowkey = Bytes.toBytes(i);
                Put put = new Put(rowkey);
                byte[] cell = GridTableHBaseBenchmark.randomBytes();
                put.addColumn(CF, QN, cell);
                table.put(put);
                nBytes += (long)cell.length;
                GridTableHBaseBenchmark.dot(i, 10000);
            }
            logger.info("Written {} rows, {} bytes", (Object)10000, (Object)nBytes);
        }
        finally {
            IOUtils.closeQuietly((Closeable)table);
        }
    }

    private static void dot(int i, int nRows) {
        if (i % (nRows / 100) == 0) {
            logger.info(".");
        }
    }

    private static byte[] randomBytes() {
        byte[] bytes = new byte[131072];
        rand.nextBytes(bytes);
        return bytes;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void createHTableIfNeeded(Connection conn, String tableName) throws IOException {
        try (Admin hbase = conn.getAdmin();){
            boolean tableExist = false;
            try {
                hbase.getTableDescriptor(TableName.valueOf((String)tableName));
                tableExist = true;
            }
            catch (TableNotFoundException tableNotFoundException) {
                // empty catch block
            }
            if (tableExist) {
                logger.info("HTable '{}' already exists", (Object)tableName);
                return;
            }
            logger.info("Creating HTable '{}'", (Object)tableName);
            HTableDescriptor desc = new HTableDescriptor(TableName.valueOf((String)tableName));
            HColumnDescriptor fd = new HColumnDescriptor(CF);
            fd.setBlocksize(131072);
            desc.addFamily(fd);
            hbase.createTable(desc);
            logger.info("HTable '{}' created", (Object)tableName);
        }
    }

    static class Stats {
        String name;
        long startTime;
        long endTime;
        long rowsRead;
        long bytesRead;

        public Stats(String name) {
            this.name = name;
        }

        public void consume(Result r) {
            this.consume(r, Integer.MAX_VALUE);
        }

        private void consume(Result r, int nBytesToConsume) {
            Cell cell = r.getColumnLatestCell(CF, QN);
            byte mix = 0;
            byte[] valueArray = cell.getValueArray();
            int n = Math.min(nBytesToConsume, cell.getValueLength());
            for (int i = 0; i < n; ++i) {
                mix = (byte)(mix ^ valueArray[i]);
                ++this.bytesRead;
            }
            this.discard(mix);
            ++this.rowsRead;
        }

        private void discard(byte n) {
        }

        public void markStart() {
            logger.info("{} starts", (Object)this.name);
            this.startTime = System.currentTimeMillis();
        }

        public void markEnd() {
            this.endTime = System.currentTimeMillis();
            logger.info("{} ends, {} ms, {} rows read, {} bytes read", new Object[]{this.name, this.endTime - this.startTime, this.rowsRead, this.bytesRead});
        }
    }

    static class Hits {
        boolean[] hitsForRowScanWithIndex;
        boolean[] hitsForRowScanNoIndex;
        List<Pair<Integer, Integer>> hitsForColumnScan;

        public Hits(int nRows, double hitRatio, double indexRatio) {
            Random rand = new Random();
            this.hitsForRowScanWithIndex = new boolean[nRows];
            this.hitsForRowScanNoIndex = new boolean[nRows];
            int blockSize = (int)(1.0 / indexRatio);
            int nBlocks = nRows / blockSize;
            for (int i = 0; i < nBlocks; ++i) {
                if (rand.nextDouble() < hitRatio) {
                    for (int j = 0; j < blockSize; ++j) {
                        this.hitsForRowScanNoIndex[i * blockSize + j] = true;
                        this.hitsForRowScanWithIndex[i * blockSize + j] = true;
                    }
                    continue;
                }
                this.hitsForRowScanNoIndex[i * blockSize] = true;
            }
            this.hitsForColumnScan = Lists.newArrayList();
            int nColumns = 20;
            int logicRows = nRows / nColumns;
            for (int i = 0; i < nColumns; ++i) {
                if (!(rand.nextDouble() < hitRatio)) continue;
                this.hitsForColumnScan.add(Pair.newPair(i * logicRows, (i + 1) * logicRows));
            }
        }

        public boolean[] getHitsForRowScanWithIndex() {
            return this.hitsForRowScanWithIndex;
        }

        public boolean[] getHitsForRowScanNoIndex() {
            return this.hitsForRowScanNoIndex;
        }

        public List<Pair<Integer, Integer>> getHitsForColumnScan() {
            return this.hitsForColumnScan;
        }
    }
}

