/****************************************************************************
 * arch/risc-v/src/k210/k210_irq_dispatch.c
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.  The
 * ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 ****************************************************************************/

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>

#include <stdint.h>
#include <assert.h>

#include <nuttx/irq.h>
#include <nuttx/arch.h>
#include <nuttx/board.h>
#include <arch/board/board.h>

#include "riscv_arch.h"
#include "riscv_internal.h"

#include "group/group.h"

/****************************************************************************
 * Public Data
 ****************************************************************************/

extern void up_fault(int irq, uint64_t *regs);

/****************************************************************************
 * Public Functions
 ****************************************************************************/

/****************************************************************************
 * k210_dispatch_irq
 ****************************************************************************/

void *k210_dispatch_irq(uint64_t vector, uint64_t *regs)
{
  uint32_t  irq = (vector >> (27 + 32)) | (vector & 0xf);
  uint64_t *mepc = regs;

  /* Check if fault happened */

  if (vector < K210_IRQ_ECALLU)
    {
      up_fault((int)irq, regs);
    }

  /* Firstly, check if the irq is machine external interrupt */

  if (K210_IRQ_MEXT == irq)
    {
      uint32_t val = getreg32(K210_PLIC_CLAIM);

      /* Add the value to nuttx irq which is offset to the mext */

      irq += val;
    }

  /* NOTE: In case of ecall, we need to adjust mepc in the context */

  if (K210_IRQ_ECALLM == irq || K210_IRQ_ECALLU == irq)
    {
      *mepc += 4;
    }

  /* Acknowledge the interrupt */

  riscv_ack_irq(irq);

#ifdef CONFIG_SUPPRESS_INTERRUPTS
  PANIC();
#else
  /* Current regs non-zero indicates that we are processing an interrupt;
   * CURRENT_REGS is also used to manage interrupt level context switches.
   *
   * Nested interrupts are not supported
   */

  ASSERT(CURRENT_REGS == NULL);
  CURRENT_REGS = regs;

  /* MEXT means no interrupt */

  if (K210_IRQ_MEXT != irq)
    {
      /* Deliver the IRQ */

      irq_dispatch(irq, regs);
    }

  if (K210_IRQ_MEXT <= irq)
    {
      /* Then write PLIC_CLAIM to clear pending in PLIC */

      putreg32(irq - K210_IRQ_MEXT, K210_PLIC_CLAIM);
    }
#endif

  /* If a context switch occurred while processing the interrupt then
   * CURRENT_REGS may have change value.  If we return any value different
   * from the input regs, then the lower level will know that a context
   * switch occurred during interrupt processing.
   */

  regs = (uint64_t *)CURRENT_REGS;
  CURRENT_REGS = NULL;

  return regs;
}
