/****************************************************************************
 * tools/mkconfig.c
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.  The
 * ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 ****************************************************************************/

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <errno.h>

#include "cfgdefine.h"

/****************************************************************************
 * Pre-processor Definitions
 ****************************************************************************/

#define DEFCONFIG ".config"

/****************************************************************************
 * Private Functions
 ****************************************************************************/

static inline char *getfilepath(const char *name)
{
  snprintf(line, PATH_MAX, "%s/" DEFCONFIG, name);
  line[PATH_MAX] = '\0';
  return strdup(line);
}

static void show_usage(const char *progname)
{
  fprintf(stderr, "USAGE: %s <abs path to .config>\n", progname);
  exit(1);
}

/****************************************************************************
 * Public Functions
 ****************************************************************************/

int main(int argc, char **argv, char **envp)
{
  char *filepath;
  FILE *stream;

  if (argc != 2)
    {
      fprintf(stderr, "Unexpected number of arguments\n");
      show_usage(argv[0]);
    }

  filepath = getfilepath(argv[1]);
  if (!filepath)
    {
      fprintf(stderr, "getfilepath failed\n");
      exit(2);
    }

  stream = fopen(filepath, "r");
  if (!stream)
    {
      fprintf(stderr, "open %s failed: %s\n", filepath, strerror(errno));
      exit(3);
    }

  printf(
    "/* config.h -- Autogenerated! Do not edit. */\n\n"
    "#ifndef __INCLUDE_NUTTX_CONFIG_H\n"
    "#define __INCLUDE_NUTTX_CONFIG_H\n\n"
    "/* General Definitions ***********************************/\n"
    "/* Used to represent the values of tristate options */\n\n"
    "#define CONFIG_y 1\n"
    "#define CONFIG_m 2\n\n"
    "/* Architecture-specific options *************************/\n\n");

  generate_definitions(stream);

  printf(
    "\n/* Sanity Checks *****************************************/\n\n"
    "/* If this is an NXFLAT, external build, then make sure that\n"
    " * NXFLAT support is enabled in the base code.\n"
    " */\n\n"
    "#if defined(__NXFLAT__) && !defined(CONFIG_NXFLAT)\n"
    "# error \"NXFLAT support not enabled in this configuration\"\n"
    "#endif\n\n"
    "/* NXFLAT requires PIC support in the TCBs. */\n\n"
    "#if defined(CONFIG_NXFLAT)\n"
    "#  undef  CONFIG_PIC\n"
    "#  define CONFIG_PIC 1\n"
    "#endif\n\n"
    "/* Binary format support is disabled if no binary formats are\n"
    " * configured (at present, NXFLAT is the only supported binary.\n"
    " * format).\n"
    " */\n\n"
    "#if !defined(CONFIG_NXFLAT) && !defined(CONFIG_ELF) \\\n"
    "    && !defined(CONFIG_BUILTIN)\n"
    "#  undef CONFIG_BINFMT_DISABLE\n"
    "#  define CONFIG_BINFMT_DISABLE 1\n"
    "#endif\n\n"
    "/* The correct way to disable RR scheduling is to set the\n"
    " * timeslice to zero.\n"
    " */\n\n"
    "#ifndef CONFIG_RR_INTERVAL\n"
    "#  define CONFIG_RR_INTERVAL 0\n"
    "#endif\n\n"
    "/* If no file streams are configured, then make certain that\n"
    " * buffered I/O support is disabled\n"
    " */\n\n"
    "#ifndef CONFIG_FILE_STREAM\n"
    "#  undef  CONFIG_STDIO_BUFFER_SIZE\n"
    "#  undef  CONFIG_STDIO_LINEBUFFER\n"
    "#  undef  CONFIG_STDIO_DISABLE_BUFFERING\n"
    "#  define CONFIG_STDIO_DISABLE_BUFFERING 1\n"
    "#endif\n\n"
    "/* If priority inheritance is disabled, then do not allocate any\n"
    " * associated resources.\n"
    " */\n\n"
    "#if !defined(CONFIG_PRIORITY_INHERITANCE) \\\n"
    "    || !defined(CONFIG_SEM_PREALLOCHOLDERS)\n"
    "#  undef CONFIG_SEM_PREALLOCHOLDERS\n"
    "#  define CONFIG_SEM_PREALLOCHOLDERS 0\n"
    "#endif\n\n"
    "#if !defined(CONFIG_PRIORITY_INHERITANCE) \\\n"
    "    || !defined(CONFIG_SEM_NNESTPRIO)\n"
    "#  undef  CONFIG_SEM_NNESTPRIO\n"
    "#  define CONFIG_SEM_NNESTPRIO 0\n"
    "#endif\n\n"
    "/* If the end of RAM is not specified then it is assumed to be\n"
    " * the beginning of RAM plus the RAM size.\n"
    " */\n\n"
    "#ifndef CONFIG_RAM_END\n"
    "#  define CONFIG_RAM_END (CONFIG_RAM_START+CONFIG_RAM_SIZE)\n"
    "#endif\n\n"
    "#ifndef CONFIG_RAM_VEND\n"
    "#  define CONFIG_RAM_VEND (CONFIG_RAM_VSTART+CONFIG_RAM_SIZE)\n"
    "#endif\n\n"
    "/* If the end of FLASH is not specified then it is assumed to be\n"
    " * the beginning of FLASH plus the FLASH size.\n"
    " */\n\n"
    "#ifndef CONFIG_FLASH_END\n"
    "#  define CONFIG_FLASH_END (CONFIG_FLASH_START+CONFIG_FLASH_SIZE)\n"
    "#endif\n\n"
    "/* If the maximum message size is zero, then we assume that\n"
    " * message queues support should be disabled\n"
    " */\n\n"
    "#if !defined(CONFIG_MQ_MAXMSGSIZE) || defined(CONFIG_DISABLE_MQUEUE)\n"
    "#  undef CONFIG_MQ_MAXMSGSIZE\n"
    "#  define CONFIG_MQ_MAXMSGSIZE 0\n"
    "#endif\n\n"
    "#if CONFIG_MQ_MAXMSGSIZE <= 0 && !defined(CONFIG_DISABLE_MQUEUE)\n"
    "#  define CONFIG_DISABLE_MQUEUE 1\n"
    "#endif\n\n"
    "#endif /* __INCLUDE_NUTTX_CONFIG_H */\n");

  fclose(stream);

  /* Exit (without bothering to clean up allocations) */

  free(filepath);
  return 0;
}
