/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.examples.filestore;

import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.ratis.conf.ConfUtils;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.examples.filestore.FileInfo;
import org.apache.ratis.examples.filestore.FileStoreCommon;
import org.apache.ratis.proto.ExamplesProtos;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.util.CollectionUtils;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.LogUtils;
import org.apache.ratis.util.StringUtils;
import org.apache.ratis.util.function.CheckedSupplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileStore
implements Closeable {
    public static final Logger LOG = LoggerFactory.getLogger(FileStore.class);
    private final Supplier<RaftPeerId> idSupplier;
    private final List<Supplier<Path>> rootSuppliers;
    private final FileMap files;
    private final ExecutorService writer;
    private final ExecutorService committer;
    private final ExecutorService reader;
    private final ExecutorService deleter;

    public FileStore(Supplier<RaftPeerId> idSupplier, RaftProperties properties) {
        this.idSupplier = idSupplier;
        this.rootSuppliers = new ArrayList<Supplier<Path>>();
        int writeThreadNum = ConfUtils.getInt(properties::getInt, "example.filestore.statemachine.write.thread.num", 1, LOG::info, new BiConsumer[0]);
        int readThreadNum = ConfUtils.getInt(properties::getInt, "example.filestore.statemachine.read.thread.num", 1, LOG::info, new BiConsumer[0]);
        int commitThreadNum = ConfUtils.getInt(properties::getInt, "example.filestore.statemachine.commit.thread.num", 1, LOG::info, new BiConsumer[0]);
        int deleteThreadNum = ConfUtils.getInt(properties::getInt, "example.filestore.statemachine.delete.thread.num", 1, LOG::info, new BiConsumer[0]);
        this.writer = Executors.newFixedThreadPool(writeThreadNum);
        this.reader = Executors.newFixedThreadPool(readThreadNum);
        this.committer = Executors.newFixedThreadPool(commitThreadNum);
        this.deleter = Executors.newFixedThreadPool(deleteThreadNum);
        List<File> dirs = ConfUtils.getFiles(properties::getFiles, "example.filestore.statemachine.dir", null, LOG::info, new BiConsumer[0]);
        Objects.requireNonNull(dirs, "example.filestore.statemachine.dir is not set.");
        for (File dir : dirs) {
            this.rootSuppliers.add(JavaUtils.memoize(() -> dir.toPath().resolve(this.getId().toString()).normalize().toAbsolutePath()));
        }
        this.files = new FileMap(JavaUtils.memoize(() -> idSupplier.get() + ":files"));
    }

    public RaftPeerId getId() {
        return Objects.requireNonNull(this.idSupplier.get(), () -> JavaUtils.getClassSimpleName(this.getClass()) + " is not initialized.");
    }

    private Path getRoot(Path relative) {
        int hash = relative.toAbsolutePath().toString().hashCode() % this.rootSuppliers.size();
        return this.rootSuppliers.get(Math.abs(hash)).get();
    }

    public List<Path> getRoots() {
        ArrayList<Path> roots = new ArrayList<Path>();
        for (Supplier<Path> s2 : this.rootSuppliers) {
            roots.add(s2.get());
        }
        return roots;
    }

    static Path normalize(String path) {
        Objects.requireNonNull(path, "path == null");
        return Paths.get(path, new String[0]).normalize();
    }

    Path resolve(Path relative) throws IOException {
        Path root = this.getRoot(relative);
        Path full = root.resolve(relative).normalize().toAbsolutePath();
        if (full.equals(root)) {
            throw new IOException("The file path " + relative + " resolved to " + full + " is the root directory " + root);
        }
        if (!full.startsWith(root)) {
            throw new IOException("The file path " + relative + " resolved to " + full + " is not a sub-path under root directory " + root);
        }
        return full;
    }

    CompletableFuture<ExamplesProtos.ReadReplyProto> watch(String relative) {
        FileInfo info = this.files.watch(relative);
        ExamplesProtos.ReadReplyProto reply = ExamplesProtos.ReadReplyProto.newBuilder().setResolvedPath(FileStoreCommon.toByteString(info.getRelativePath())).build();
        if (info instanceof FileInfo.Watch) {
            return ((FileInfo.Watch)info).getFuture().thenApply(uc -> reply);
        }
        return CompletableFuture.completedFuture(reply);
    }

    CompletableFuture<ExamplesProtos.ReadReplyProto> read(String relative, long offset, long length, boolean readCommitted) {
        Supplier<String> name = () -> "read(" + relative + ", " + offset + ", " + length + ") @" + this.getId();
        CheckedSupplier task = LogUtils.newCheckedSupplier(LOG, () -> {
            FileInfo info = this.files.get(relative);
            ExamplesProtos.ReadReplyProto.Builder reply = ExamplesProtos.ReadReplyProto.newBuilder().setResolvedPath(FileStoreCommon.toByteString(info.getRelativePath())).setOffset(offset);
            ByteString bytes = info.read(this::resolve, offset, length, readCommitted);
            return reply.setData(bytes).build();
        }, name);
        return FileStore.submit(task, this.reader);
    }

    CompletableFuture<Path> delete(long index, String relative) {
        Supplier<String> name = () -> "delete(" + relative + ") @" + this.getId() + ":" + index;
        CheckedSupplier task = LogUtils.newCheckedSupplier(LOG, () -> {
            FileInfo info = this.files.remove(relative);
            FileUtils.delete(this.resolve(info.getRelativePath()));
            return info.getRelativePath();
        }, name);
        return FileStore.submit(task, this.deleter);
    }

    static <T> CompletableFuture<T> submit(CheckedSupplier<T, IOException> task, ExecutorService executor) {
        CompletableFuture f = new CompletableFuture();
        executor.submit(() -> {
            try {
                f.complete(task.get());
            }
            catch (IOException e) {
                f.completeExceptionally(new IOException("Failed " + task, e));
            }
        });
        return f;
    }

    CompletableFuture<ExamplesProtos.WriteReplyProto> submitCommit(long index, String relative, boolean close, long offset, int size) {
        FileInfo.UnderConstruction uc;
        Function<FileInfo.UnderConstruction, FileInfo.ReadOnly> converter = close ? this.files::close : null;
        try {
            uc = this.files.get(relative).asUnderConstruction();
        }
        catch (FileNotFoundException e) {
            return FileStoreCommon.completeExceptionally(index, "Failed to submitCommit to " + relative, e);
        }
        return uc.submitCommit(offset, size, converter, this.committer, this.getId(), index).thenApply(n -> ExamplesProtos.WriteReplyProto.newBuilder().setResolvedPath(FileStoreCommon.toByteString(uc.getRelativePath())).setOffset(offset).setLength(n.intValue()).build());
    }

    CompletableFuture<Integer> write(long index, String relative, boolean close, boolean sync, long offset, ByteString data) {
        FileInfo.UnderConstruction uc;
        boolean createNew;
        int size = data != null ? data.size() : 0;
        LOG.trace("write {}, offset={}, size={}, close? {} @{}:{}", relative, offset, size, close, this.getId(), index);
        boolean bl = createNew = offset == 0L;
        if (createNew) {
            uc = new FileInfo.UnderConstruction(FileStore.normalize(relative));
            this.files.putNew(uc);
        } else {
            try {
                uc = this.files.get(relative).asUnderConstruction();
            }
            catch (FileNotFoundException e) {
                return FileStoreCommon.completeExceptionally(index, "Failed to write to " + relative, e);
            }
        }
        return size == 0 && !close ? CompletableFuture.completedFuture(0) : (createNew ? uc.submitCreate(this::resolve, data, close, sync, this.writer, this.getId(), index) : uc.submitWrite(offset, data, close, sync, this.writer, this.getId(), index));
    }

    @Override
    public void close() {
        this.writer.shutdownNow();
        this.committer.shutdownNow();
        this.reader.shutdownNow();
        this.deleter.shutdownNow();
    }

    CompletableFuture<ExamplesProtos.StreamWriteReplyProto> streamCommit(String p, long bytesWritten) {
        return CompletableFuture.supplyAsync(() -> {
            try (RandomAccessFile file = new RandomAccessFile(this.resolve(FileStore.normalize(p)).toFile(), "r");){
                long len = file.length();
                ExamplesProtos.StreamWriteReplyProto streamWriteReplyProto = ExamplesProtos.StreamWriteReplyProto.newBuilder().setIsSuccess(len == bytesWritten).setByteWritten(len).build();
                return streamWriteReplyProto;
            }
            catch (IOException e) {
                throw new CompletionException("Failed to commit stream " + p + " with " + bytesWritten + " B.", e);
            }
        }, this.committer);
    }

    CompletableFuture<?> streamLink(StateMachine.DataStream dataStream) {
        return CompletableFuture.supplyAsync(() -> {
            if (dataStream == null) {
                return JavaUtils.completeExceptionally(new IllegalStateException("Null stream"));
            }
            if (dataStream.getDataChannel().isOpen()) {
                return JavaUtils.completeExceptionally(new IllegalStateException("DataStream: " + dataStream + " is not closed properly"));
            }
            return CompletableFuture.completedFuture(null);
        }, this.committer);
    }

    public CompletableFuture<FileStoreDataChannel> createDataChannel(String p) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                Path full = this.resolve(FileStore.normalize(p));
                return new FileStoreDataChannel(full);
            }
            catch (IOException e) {
                throw new CompletionException("Failed to create " + p, e);
            }
        }, this.writer);
    }

    static class FileStoreDataChannel
    implements StateMachine.DataChannel {
        private final Path path;
        private final RandomAccessFile randomAccessFile;

        FileStoreDataChannel(Path path) throws FileNotFoundException {
            this.path = path;
            this.randomAccessFile = new RandomAccessFile(path.toFile(), "rw");
        }

        @Override
        public void force(boolean metadata) throws IOException {
            LOG.debug("force({}) at {}", (Object)metadata, (Object)this.path);
            this.randomAccessFile.getChannel().force(metadata);
        }

        @Override
        public int write(ByteBuffer src) throws IOException {
            return this.randomAccessFile.getChannel().write(src);
        }

        @Override
        public boolean isOpen() {
            return this.randomAccessFile.getChannel().isOpen();
        }

        @Override
        public void close() throws IOException {
            this.randomAccessFile.close();
        }
    }

    static class FileMap {
        private final Object name;
        private final Map<Path, FileInfo> map = new ConcurrentHashMap<Path, FileInfo>();

        FileMap(Supplier<String> name) {
            this.name = StringUtils.stringSupplierAsObject(name);
        }

        FileInfo get(String relative) throws FileNotFoundException {
            return this.applyFunction(relative, this.map::get);
        }

        FileInfo watch(String relative) {
            try {
                return this.applyFunction(relative, p -> this.map.computeIfAbsent((Path)p, FileInfo.Watch::new));
            }
            catch (FileNotFoundException e) {
                throw new IllegalStateException("Failed to watch " + relative, e);
            }
        }

        FileInfo remove(String relative) throws FileNotFoundException {
            LOG.trace("{}: remove {}", this.name, (Object)relative);
            return this.applyFunction(relative, this.map::remove);
        }

        private FileInfo applyFunction(String relative, Function<Path, FileInfo> f) throws FileNotFoundException {
            FileInfo info = f.apply(FileStore.normalize(relative));
            if (info == null) {
                throw new FileNotFoundException("File " + relative + " not found in " + this.name);
            }
            return info;
        }

        void putNew(FileInfo.UnderConstruction uc) {
            LOG.trace("{}: putNew {}", this.name, (Object)uc.getRelativePath());
            FileInfo previous = this.map.put(uc.getRelativePath(), uc);
            if (previous instanceof FileInfo.Watch) {
                ((FileInfo.Watch)previous).complete(uc);
            }
        }

        FileInfo.ReadOnly close(FileInfo.UnderConstruction uc) {
            LOG.trace("{}: close {}", this.name, (Object)uc.getRelativePath());
            FileInfo.ReadOnly ro = new FileInfo.ReadOnly(uc);
            CollectionUtils.replaceExisting(uc.getRelativePath(), uc, ro, this.map, this.name::toString);
            return ro;
        }
    }
}

