/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.store;

import com.google.common.collect.Lists;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.stream.Stream;
import org.apache.rocketmq.common.UtilAll;
import org.apache.rocketmq.logging.org.slf4j.Logger;
import org.apache.rocketmq.logging.org.slf4j.LoggerFactory;
import org.apache.rocketmq.store.AllocateMappedFileService;
import org.apache.rocketmq.store.ReferenceResource;
import org.apache.rocketmq.store.SelectMappedBufferResult;
import org.apache.rocketmq.store.Swappable;
import org.apache.rocketmq.store.logfile.DefaultMappedFile;
import org.apache.rocketmq.store.logfile.MappedFile;

public class MappedFileQueue
implements Swappable {
    private static final Logger log = LoggerFactory.getLogger((String)"RocketmqStore");
    private static final Logger LOG_ERROR = LoggerFactory.getLogger((String)"RocketmqStoreError");
    protected final String storePath;
    protected final int mappedFileSize;
    protected final CopyOnWriteArrayList<MappedFile> mappedFiles = new CopyOnWriteArrayList();
    protected final AllocateMappedFileService allocateMappedFileService;
    protected long flushedWhere = 0L;
    protected long committedWhere = 0L;
    protected volatile long storeTimestamp = 0L;

    public MappedFileQueue(String storePath, int mappedFileSize, AllocateMappedFileService allocateMappedFileService) {
        this.storePath = storePath;
        this.mappedFileSize = mappedFileSize;
        this.allocateMappedFileService = allocateMappedFileService;
    }

    public void checkSelf() {
        ArrayList<MappedFile> mappedFiles = new ArrayList<MappedFile>(this.mappedFiles);
        if (!mappedFiles.isEmpty()) {
            Iterator iterator = mappedFiles.iterator();
            MappedFile pre = null;
            while (iterator.hasNext()) {
                MappedFile cur = (MappedFile)iterator.next();
                if (pre != null && cur.getFileFromOffset() - pre.getFileFromOffset() != (long)this.mappedFileSize) {
                    LOG_ERROR.error("[BUG]The mappedFile queue's data is damaged, the adjacent mappedFile's offset don't match. pre file {}, cur file {}", (Object)pre.getFileName(), (Object)cur.getFileName());
                }
                pre = cur;
            }
        }
    }

    public MappedFile getMappedFileByTime(long timestamp) {
        Object[] mfs = this.copyMappedFiles(0);
        if (null == mfs) {
            return null;
        }
        for (int i = 0; i < mfs.length; ++i) {
            MappedFile mappedFile = (MappedFile)mfs[i];
            if (mappedFile.getLastModifiedTimestamp() < timestamp) continue;
            return mappedFile;
        }
        return (MappedFile)mfs[mfs.length - 1];
    }

    protected Object[] copyMappedFiles(int reservedMappedFiles) {
        if (this.mappedFiles.size() <= reservedMappedFiles) {
            return null;
        }
        Object[] mfs = this.mappedFiles.toArray();
        return mfs;
    }

    public void truncateDirtyFiles(long offset) {
        ArrayList<MappedFile> willRemoveFiles = new ArrayList<MappedFile>();
        for (MappedFile file : this.mappedFiles) {
            long fileTailOffset = file.getFileFromOffset() + (long)this.mappedFileSize;
            if (fileTailOffset <= offset) continue;
            if (offset >= file.getFileFromOffset()) {
                file.setWrotePosition((int)(offset % (long)this.mappedFileSize));
                file.setCommittedPosition((int)(offset % (long)this.mappedFileSize));
                file.setFlushedPosition((int)(offset % (long)this.mappedFileSize));
                continue;
            }
            file.destroy(1000L);
            willRemoveFiles.add(file);
        }
        this.deleteExpiredFile(willRemoveFiles);
    }

    void deleteExpiredFile(List<MappedFile> files) {
        if (!files.isEmpty()) {
            Iterator<MappedFile> iterator = files.iterator();
            while (iterator.hasNext()) {
                MappedFile cur = iterator.next();
                if (this.mappedFiles.contains(cur)) continue;
                iterator.remove();
                log.info("This mappedFile {} is not contained by mappedFiles, so skip it.", (Object)cur.getFileName());
            }
            try {
                if (!this.mappedFiles.removeAll(files)) {
                    log.error("deleteExpiredFile remove failed.");
                }
            }
            catch (Exception e) {
                log.error("deleteExpiredFile has exception.", (Throwable)e);
            }
        }
    }

    public boolean load() {
        File dir = new File(this.storePath);
        File[] ls = dir.listFiles();
        if (ls != null) {
            return this.doLoad(Arrays.asList(ls));
        }
        return true;
    }

    public boolean doLoad(List<File> files) {
        files.sort(Comparator.comparing(File::getName));
        for (File file : files) {
            if (file.isDirectory()) continue;
            if (file.length() != (long)this.mappedFileSize) {
                log.warn(file + "\t" + file.length() + " length not matched message store config value, please check it manually");
                return false;
            }
            try {
                DefaultMappedFile mappedFile = new DefaultMappedFile(file.getPath(), this.mappedFileSize);
                mappedFile.setWrotePosition(this.mappedFileSize);
                mappedFile.setFlushedPosition(this.mappedFileSize);
                mappedFile.setCommittedPosition(this.mappedFileSize);
                this.mappedFiles.add(mappedFile);
                log.info("load " + file.getPath() + " OK");
            }
            catch (IOException e) {
                log.error("load file " + file + " error", (Throwable)e);
                return false;
            }
        }
        return true;
    }

    public long howMuchFallBehind() {
        MappedFile mappedFile;
        if (this.mappedFiles.isEmpty()) {
            return 0L;
        }
        long committed = this.flushedWhere;
        if (committed != 0L && (mappedFile = this.getLastMappedFile(0L, false)) != null) {
            return mappedFile.getFileFromOffset() + (long)mappedFile.getWrotePosition() - committed;
        }
        return 0L;
    }

    public MappedFile getLastMappedFile(long startOffset, boolean needCreate) {
        long createOffset = -1L;
        MappedFile mappedFileLast = this.getLastMappedFile();
        if (mappedFileLast == null) {
            createOffset = startOffset - startOffset % (long)this.mappedFileSize;
        }
        if (mappedFileLast != null && mappedFileLast.isFull()) {
            createOffset = mappedFileLast.getFileFromOffset() + (long)this.mappedFileSize;
        }
        if (createOffset != -1L && needCreate) {
            return this.tryCreateMappedFile(createOffset);
        }
        return mappedFileLast;
    }

    public boolean isMappedFilesEmpty() {
        return this.mappedFiles.isEmpty();
    }

    public boolean isEmptyOrCurrentFileFull() {
        MappedFile mappedFileLast = this.getLastMappedFile();
        if (mappedFileLast == null) {
            return true;
        }
        return mappedFileLast.isFull();
    }

    public boolean shouldRoll(int msgSize) {
        if (this.isEmptyOrCurrentFileFull()) {
            return true;
        }
        MappedFile mappedFileLast = this.getLastMappedFile();
        return mappedFileLast.getWrotePosition() + msgSize > mappedFileLast.getFileSize();
    }

    public MappedFile tryCreateMappedFile(long createOffset) {
        String nextFilePath = this.storePath + File.separator + UtilAll.offset2FileName((long)createOffset);
        String nextNextFilePath = this.storePath + File.separator + UtilAll.offset2FileName((long)(createOffset + (long)this.mappedFileSize));
        return this.doCreateMappedFile(nextFilePath, nextNextFilePath);
    }

    protected MappedFile doCreateMappedFile(String nextFilePath, String nextNextFilePath) {
        MappedFile mappedFile = null;
        if (this.allocateMappedFileService != null) {
            mappedFile = this.allocateMappedFileService.putRequestAndReturnMappedFile(nextFilePath, nextNextFilePath, this.mappedFileSize);
        } else {
            try {
                mappedFile = new DefaultMappedFile(nextFilePath, this.mappedFileSize);
            }
            catch (IOException e) {
                log.error("create mappedFile exception", (Throwable)e);
            }
        }
        if (mappedFile != null) {
            if (this.mappedFiles.isEmpty()) {
                mappedFile.setFirstCreateInQueue(true);
            }
            this.mappedFiles.add(mappedFile);
        }
        return mappedFile;
    }

    public MappedFile getLastMappedFile(long startOffset) {
        return this.getLastMappedFile(startOffset, true);
    }

    public MappedFile getLastMappedFile() {
        MappedFile[] mappedFiles = this.mappedFiles.toArray(new MappedFile[0]);
        return mappedFiles.length == 0 ? null : mappedFiles[mappedFiles.length - 1];
    }

    public boolean resetOffset(long offset) {
        int maxDiff;
        long lastOffset;
        long diff;
        MappedFile mappedFileLast = this.getLastMappedFile();
        if (mappedFileLast != null && (diff = (lastOffset = mappedFileLast.getFileFromOffset() + (long)mappedFileLast.getWrotePosition()) - offset) > (long)(maxDiff = this.mappedFileSize * 2)) {
            return false;
        }
        ListIterator<MappedFile> iterator = this.mappedFiles.listIterator(this.mappedFiles.size());
        while (iterator.hasPrevious()) {
            mappedFileLast = iterator.previous();
            if (offset >= mappedFileLast.getFileFromOffset()) {
                int where = (int)(offset % (long)mappedFileLast.getFileSize());
                mappedFileLast.setFlushedPosition(where);
                mappedFileLast.setWrotePosition(where);
                mappedFileLast.setCommittedPosition(where);
                break;
            }
            iterator.remove();
        }
        return true;
    }

    public long getMinOffset() {
        if (!this.mappedFiles.isEmpty()) {
            try {
                return this.mappedFiles.get(0).getFileFromOffset();
            }
            catch (IndexOutOfBoundsException indexOutOfBoundsException) {
            }
            catch (Exception e) {
                log.error("getMinOffset has exception.", (Throwable)e);
            }
        }
        return -1L;
    }

    public long getMaxOffset() {
        MappedFile mappedFile = this.getLastMappedFile();
        if (mappedFile != null) {
            return mappedFile.getFileFromOffset() + (long)mappedFile.getReadPosition();
        }
        return 0L;
    }

    public long getMaxWrotePosition() {
        MappedFile mappedFile = this.getLastMappedFile();
        if (mappedFile != null) {
            return mappedFile.getFileFromOffset() + (long)mappedFile.getWrotePosition();
        }
        return 0L;
    }

    public long remainHowManyDataToCommit() {
        return this.getMaxWrotePosition() - this.committedWhere;
    }

    public long remainHowManyDataToFlush() {
        return this.getMaxOffset() - this.flushedWhere;
    }

    public void deleteLastMappedFile() {
        MappedFile lastMappedFile = this.getLastMappedFile();
        if (lastMappedFile != null) {
            lastMappedFile.destroy(1000L);
            this.mappedFiles.remove(lastMappedFile);
            log.info("on recover, destroy a logic mapped file " + lastMappedFile.getFileName());
        }
    }

    public int deleteExpiredFileByTime(long expiredTime, int deleteFilesInterval, long intervalForcibly, boolean cleanImmediately, int deleteFileBatchMax) {
        Object[] mfs = this.copyMappedFiles(0);
        if (null == mfs) {
            return 0;
        }
        int mfsLength = mfs.length - 1;
        int deleteCount = 0;
        ArrayList<MappedFile> files = new ArrayList<MappedFile>();
        int skipFileNum = 0;
        if (null != mfs) {
            this.checkSelf();
            for (int i = 0; i < mfsLength; ++i) {
                MappedFile mappedFile = (MappedFile)mfs[i];
                long liveMaxTimestamp = mappedFile.getLastModifiedTimestamp() + expiredTime;
                if (System.currentTimeMillis() >= liveMaxTimestamp || cleanImmediately) {
                    if (skipFileNum > 0) {
                        log.info("Delete CommitLog {} but skip {} files", (Object)mappedFile.getFileName(), (Object)skipFileNum);
                    }
                    if (!mappedFile.destroy(intervalForcibly)) break;
                    files.add(mappedFile);
                    ++deleteCount;
                    if (files.size() >= deleteFileBatchMax) break;
                    if (deleteFilesInterval <= 0 || i + 1 >= mfsLength) continue;
                    try {
                        Thread.sleep(deleteFilesInterval);
                    }
                    catch (InterruptedException interruptedException) {}
                    continue;
                }
                ++skipFileNum;
                break;
            }
        }
        this.deleteExpiredFile(files);
        return deleteCount;
    }

    public int deleteExpiredFileByOffset(long offset, int unitSize) {
        Object[] mfs = this.copyMappedFiles(0);
        ArrayList<MappedFile> files = new ArrayList<MappedFile>();
        int deleteCount = 0;
        if (null != mfs) {
            int mfsLength = mfs.length - 1;
            for (int i = 0; i < mfsLength; ++i) {
                boolean destroy;
                MappedFile mappedFile = (MappedFile)mfs[i];
                SelectMappedBufferResult result = mappedFile.selectMappedBuffer(this.mappedFileSize - unitSize);
                if (result != null) {
                    long maxOffsetInLogicQueue = result.getByteBuffer().getLong();
                    result.release();
                    boolean bl = destroy = maxOffsetInLogicQueue < offset;
                    if (destroy) {
                        log.info("physic min offset " + offset + ", logics in current mappedFile max offset " + maxOffsetInLogicQueue + ", delete it");
                    }
                } else if (!mappedFile.isAvailable()) {
                    log.warn("Found a hanged consume queue file, attempting to delete it.");
                    destroy = true;
                } else {
                    log.warn("this being not executed forever.");
                    break;
                }
                if (!destroy || !mappedFile.destroy(60000L)) break;
                files.add(mappedFile);
                ++deleteCount;
            }
        }
        this.deleteExpiredFile(files);
        return deleteCount;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int deleteExpiredFileByOffsetForTimerLog(long offset, int checkOffset, int unitSize) {
        Object[] mfs = this.copyMappedFiles(0);
        ArrayList<MappedFile> files = new ArrayList<MappedFile>();
        int deleteCount = 0;
        if (null != mfs) {
            int mfsLength = mfs.length - 1;
            for (int i = 0; i < mfsLength; ++i) {
                MappedFile mappedFile;
                boolean destroy;
                block11: {
                    destroy = false;
                    mappedFile = (MappedFile)mfs[i];
                    SelectMappedBufferResult result = mappedFile.selectMappedBuffer(checkOffset);
                    try {
                        if (result != null) {
                            int position = result.getByteBuffer().position();
                            int size = result.getByteBuffer().getInt();
                            result.getByteBuffer().getLong();
                            int magic = result.getByteBuffer().getInt();
                            if (size == unitSize && (magic | 0xF) == 15) {
                                result.getByteBuffer().position(position + 28);
                                long maxOffsetPy = result.getByteBuffer().getLong();
                                boolean bl = destroy = maxOffsetPy < offset;
                                if (destroy) {
                                    log.info("physic min commitlog offset " + offset + ", current mappedFile's max offset " + maxOffsetPy + ", delete it");
                                }
                            } else {
                                log.warn("Found error data in [{}] checkOffset:{} unitSize:{}", new Object[]{mappedFile.getFileName(), checkOffset, unitSize});
                            }
                            break block11;
                        }
                        if (!mappedFile.isAvailable()) {
                            log.warn("Found a hanged consume queue file, attempting to delete it.");
                            destroy = true;
                            break block11;
                        }
                        log.warn("this being not executed forever.");
                        break;
                    }
                    finally {
                        if (null != result) {
                            result.release();
                        }
                    }
                }
                if (!destroy || !mappedFile.destroy(60000L)) break;
                files.add(mappedFile);
                ++deleteCount;
            }
        }
        this.deleteExpiredFile(files);
        return deleteCount;
    }

    public boolean flush(int flushLeastPages) {
        boolean result = true;
        MappedFile mappedFile = this.findMappedFileByOffset(this.flushedWhere, this.flushedWhere == 0L);
        if (mappedFile != null) {
            long tmpTimeStamp = mappedFile.getStoreTimestamp();
            int offset = mappedFile.flush(flushLeastPages);
            long where = mappedFile.getFileFromOffset() + (long)offset;
            result = where == this.flushedWhere;
            this.flushedWhere = where;
            if (0 == flushLeastPages) {
                this.storeTimestamp = tmpTimeStamp;
            }
        }
        return result;
    }

    public synchronized boolean commit(int commitLeastPages) {
        boolean result = true;
        MappedFile mappedFile = this.findMappedFileByOffset(this.committedWhere, this.committedWhere == 0L);
        if (mappedFile != null) {
            int offset = mappedFile.commit(commitLeastPages);
            long where = mappedFile.getFileFromOffset() + (long)offset;
            result = where == this.committedWhere;
            this.committedWhere = where;
        }
        return result;
    }

    public MappedFile findMappedFileByOffset(long offset, boolean returnFirstOnNotFound) {
        block9: {
            try {
                MappedFile firstMappedFile = this.getFirstMappedFile();
                MappedFile lastMappedFile = this.getLastMappedFile();
                if (firstMappedFile == null || lastMappedFile == null) break block9;
                if (offset < firstMappedFile.getFileFromOffset() || offset >= lastMappedFile.getFileFromOffset() + (long)this.mappedFileSize) {
                    LOG_ERROR.warn("Offset not matched. Request offset: {}, firstOffset: {}, lastOffset: {}, mappedFileSize: {}, mappedFiles count: {}", new Object[]{offset, firstMappedFile.getFileFromOffset(), lastMappedFile.getFileFromOffset() + (long)this.mappedFileSize, this.mappedFileSize, this.mappedFiles.size()});
                } else {
                    int index = (int)(offset / (long)this.mappedFileSize - firstMappedFile.getFileFromOffset() / (long)this.mappedFileSize);
                    MappedFile targetFile = null;
                    try {
                        targetFile = this.mappedFiles.get(index);
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                    if (targetFile != null && offset >= targetFile.getFileFromOffset() && offset < targetFile.getFileFromOffset() + (long)this.mappedFileSize) {
                        return targetFile;
                    }
                    for (MappedFile tmpMappedFile : this.mappedFiles) {
                        if (offset < tmpMappedFile.getFileFromOffset() || offset >= tmpMappedFile.getFileFromOffset() + (long)this.mappedFileSize) continue;
                        return tmpMappedFile;
                    }
                }
                if (returnFirstOnNotFound) {
                    return firstMappedFile;
                }
            }
            catch (Exception e) {
                log.error("findMappedFileByOffset Exception", (Throwable)e);
            }
        }
        return null;
    }

    public MappedFile getFirstMappedFile() {
        MappedFile mappedFileFirst = null;
        if (!this.mappedFiles.isEmpty()) {
            try {
                mappedFileFirst = this.mappedFiles.get(0);
            }
            catch (IndexOutOfBoundsException indexOutOfBoundsException) {
            }
            catch (Exception e) {
                log.error("getFirstMappedFile has exception.", (Throwable)e);
            }
        }
        return mappedFileFirst;
    }

    public MappedFile findMappedFileByOffset(long offset) {
        return this.findMappedFileByOffset(offset, false);
    }

    public long getMappedMemorySize() {
        long size = 0L;
        Object[] mfs = this.copyMappedFiles(0);
        if (mfs != null) {
            for (Object mf : mfs) {
                if (!((ReferenceResource)mf).isAvailable()) continue;
                size += (long)this.mappedFileSize;
            }
        }
        return size;
    }

    public boolean retryDeleteFirstFile(long intervalForcibly) {
        MappedFile mappedFile = this.getFirstMappedFile();
        if (mappedFile != null && !mappedFile.isAvailable()) {
            log.warn("the mappedFile was destroyed once, but still alive, " + mappedFile.getFileName());
            boolean result = mappedFile.destroy(intervalForcibly);
            if (result) {
                log.info("the mappedFile re delete OK, " + mappedFile.getFileName());
                ArrayList<MappedFile> tmpFiles = new ArrayList<MappedFile>();
                tmpFiles.add(mappedFile);
                this.deleteExpiredFile(tmpFiles);
            } else {
                log.warn("the mappedFile re delete failed, " + mappedFile.getFileName());
            }
            return result;
        }
        return false;
    }

    public void shutdown(long intervalForcibly) {
        for (MappedFile mf : this.mappedFiles) {
            mf.shutdown(intervalForcibly);
        }
    }

    public void destroy() {
        for (MappedFile mf : this.mappedFiles) {
            mf.destroy(3000L);
        }
        this.mappedFiles.clear();
        this.flushedWhere = 0L;
        File file = new File(this.storePath);
        if (file.isDirectory()) {
            file.delete();
        }
    }

    @Override
    public void swapMap(int reserveNum, long forceSwapIntervalMs, long normalSwapIntervalMs) {
        Object[] mfs;
        if (this.mappedFiles.isEmpty()) {
            return;
        }
        if (reserveNum < 3) {
            reserveNum = 3;
        }
        if (null == (mfs = this.copyMappedFiles(0))) {
            return;
        }
        for (int i = mfs.length - reserveNum - 1; i >= 0; --i) {
            MappedFile mappedFile = (MappedFile)mfs[i];
            if (System.currentTimeMillis() - mappedFile.getRecentSwapMapTime() > forceSwapIntervalMs) {
                mappedFile.swapMap();
                continue;
            }
            if (System.currentTimeMillis() - mappedFile.getRecentSwapMapTime() <= normalSwapIntervalMs || mappedFile.getMappedByteBufferAccessCountSinceLastSwap() <= 0L) continue;
            mappedFile.swapMap();
        }
    }

    @Override
    public void cleanSwappedMap(long forceCleanSwapIntervalMs) {
        if (this.mappedFiles.isEmpty()) {
            return;
        }
        int reserveNum = 3;
        Object[] mfs = this.copyMappedFiles(0);
        if (null == mfs) {
            return;
        }
        for (int i = mfs.length - reserveNum - 1; i >= 0; --i) {
            MappedFile mappedFile = (MappedFile)mfs[i];
            if (System.currentTimeMillis() - mappedFile.getRecentSwapMapTime() <= forceCleanSwapIntervalMs) continue;
            mappedFile.cleanSwapedMap(false);
        }
    }

    public Object[] snapshot() {
        return this.mappedFiles.toArray();
    }

    public Stream<MappedFile> stream() {
        return this.mappedFiles.stream();
    }

    public Stream<MappedFile> reversedStream() {
        return Lists.reverse(this.mappedFiles).stream();
    }

    public long getFlushedWhere() {
        return this.flushedWhere;
    }

    public void setFlushedWhere(long flushedWhere) {
        this.flushedWhere = flushedWhere;
    }

    public long getStoreTimestamp() {
        return this.storeTimestamp;
    }

    public List<MappedFile> getMappedFiles() {
        return this.mappedFiles;
    }

    public int getMappedFileSize() {
        return this.mappedFileSize;
    }

    public long getCommittedWhere() {
        return this.committedWhere;
    }

    public void setCommittedWhere(long committedWhere) {
        this.committedWhere = committedWhere;
    }

    public long getTotalFileSize() {
        return (long)this.mappedFileSize * (long)this.mappedFiles.size();
    }

    public String getStorePath() {
        return this.storePath;
    }

    public List<MappedFile> range(long from, long to) {
        Object[] mfs = this.copyMappedFiles(0);
        if (null == mfs) {
            return new ArrayList<MappedFile>();
        }
        ArrayList<MappedFile> result = new ArrayList<MappedFile>();
        for (Object mf : mfs) {
            MappedFile mappedFile = (MappedFile)mf;
            if (mappedFile.getFileFromOffset() + (long)mappedFile.getFileSize() <= from) continue;
            if (to <= mappedFile.getFileFromOffset()) break;
            result.add(mappedFile);
        }
        return result;
    }
}

