/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.tieredstore.index;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Stopwatch;
import java.io.File;
import java.nio.ByteBuffer;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.commons.lang3.StringUtils;
import org.apache.rocketmq.common.ServiceThread;
import org.apache.rocketmq.common.UtilAll;
import org.apache.rocketmq.logging.org.slf4j.Logger;
import org.apache.rocketmq.logging.org.slf4j.LoggerFactory;
import org.apache.rocketmq.store.logfile.DefaultMappedFile;
import org.apache.rocketmq.tieredstore.common.AppendResult;
import org.apache.rocketmq.tieredstore.common.TieredMessageStoreConfig;
import org.apache.rocketmq.tieredstore.file.TieredFileAllocator;
import org.apache.rocketmq.tieredstore.file.TieredFlatFile;
import org.apache.rocketmq.tieredstore.index.IndexFile;
import org.apache.rocketmq.tieredstore.index.IndexItem;
import org.apache.rocketmq.tieredstore.index.IndexService;
import org.apache.rocketmq.tieredstore.index.IndexStoreFile;
import org.apache.rocketmq.tieredstore.provider.TieredFileSegment;

public class IndexStoreService
extends ServiceThread
implements IndexService {
    private static final Logger log = LoggerFactory.getLogger((String)"RocketmqTieredStore");
    public static final String FILE_DIRECTORY_NAME = "tiered_index_file";
    public static final String FILE_COMPACTED_DIRECTORY_NAME = "compacting";
    private final TieredMessageStoreConfig storeConfig;
    private final ConcurrentSkipListMap<Long, IndexFile> timeStoreTable;
    private final ReadWriteLock readWriteLock;
    private final AtomicLong compactTimestamp;
    private final String filePath;
    private final TieredFileAllocator fileAllocator;
    private IndexFile currentWriteFile;
    private TieredFlatFile flatFile;

    public IndexStoreService(TieredFileAllocator fileAllocator, String filePath) {
        this.storeConfig = fileAllocator.getStoreConfig();
        this.filePath = filePath;
        this.fileAllocator = fileAllocator;
        this.timeStoreTable = new ConcurrentSkipListMap();
        this.compactTimestamp = new AtomicLong(0L);
        this.readWriteLock = new ReentrantReadWriteLock();
        this.recover();
    }

    private void doConvertOldFormatFile(String filePath) {
        try {
            File file = new File(filePath);
            if (!file.exists()) {
                return;
            }
            DefaultMappedFile mappedFile = new DefaultMappedFile(file.getPath(), (int)file.length());
            long timestamp = mappedFile.getMappedByteBuffer().getLong(4);
            if (timestamp <= 0L) {
                mappedFile.destroy(TimeUnit.SECONDS.toMillis(10L));
            } else {
                mappedFile.renameTo(String.valueOf(new File(file.getParent(), String.valueOf(timestamp))));
                mappedFile.shutdown(TimeUnit.SECONDS.toMillis(10L));
            }
        }
        catch (Exception e) {
            log.error("IndexStoreService do convert old format error, file: {}", (Object)filePath, (Object)e);
        }
    }

    private void recover() {
        Stopwatch stopwatch = Stopwatch.createStarted();
        UtilAll.deleteFile((File)new File(Paths.get(this.storeConfig.getStorePathRootDir(), FILE_DIRECTORY_NAME, FILE_COMPACTED_DIRECTORY_NAME).toString()));
        File dir = new File(Paths.get(this.storeConfig.getStorePathRootDir(), FILE_DIRECTORY_NAME).toString());
        this.doConvertOldFormatFile(Paths.get(dir.getPath(), "0000").toString());
        this.doConvertOldFormatFile(Paths.get(dir.getPath(), "1111").toString());
        File[] files = dir.listFiles();
        if (files != null) {
            List<File> fileList = Arrays.asList(files);
            fileList.sort(Comparator.comparing(File::getName));
            for (File file : fileList) {
                if (file.isDirectory() || !StringUtils.isNumeric((CharSequence)file.getName())) continue;
                try {
                    IndexStoreFile indexFile = new IndexStoreFile(this.storeConfig, Long.parseLong(file.getName()));
                    this.timeStoreTable.put(indexFile.getTimestamp(), indexFile);
                    log.info("IndexStoreService recover load local file, timestamp: {}", (Object)indexFile.getTimestamp());
                }
                catch (Exception e) {
                    log.error("IndexStoreService recover, load local file error", (Throwable)e);
                }
            }
        }
        if (this.timeStoreTable.isEmpty()) {
            this.createNewIndexFile(System.currentTimeMillis());
        }
        this.currentWriteFile = this.timeStoreTable.lastEntry().getValue();
        this.setCompactTimestamp(this.timeStoreTable.firstKey() - 1L);
        this.flatFile = this.fileAllocator.createFlatFileForIndexFile(this.filePath);
        if (this.flatFile.getBaseOffset() == -1L) {
            this.flatFile.setBaseOffset(0L);
        }
        for (TieredFileSegment fileSegment : this.flatFile.getFileSegmentList()) {
            IndexStoreFile indexFile = new IndexStoreFile(this.storeConfig, fileSegment);
            IndexFile localFile = this.timeStoreTable.get(indexFile.getTimestamp());
            if (localFile != null) {
                localFile.destroy();
            }
            this.timeStoreTable.put(indexFile.getTimestamp(), indexFile);
            log.info("IndexStoreService recover load remote file, timestamp: {}", (Object)indexFile.getTimestamp());
        }
        log.info("IndexStoreService recover finished, entrySize: {}, cost: {}ms, directory: {}", new Object[]{this.timeStoreTable.size(), stopwatch.elapsed(TimeUnit.MILLISECONDS), dir.getAbsolutePath()});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void createNewIndexFile(long timestamp) {
        try {
            this.readWriteLock.writeLock().lock();
            IndexFile indexFile = this.currentWriteFile;
            if (this.timeStoreTable.containsKey(timestamp) || indexFile != null && IndexFile.IndexStatusEnum.UNSEALED.equals((Object)indexFile.getFileStatus())) {
                return;
            }
            IndexStoreFile newStoreFile = new IndexStoreFile(this.storeConfig, timestamp);
            this.timeStoreTable.put(timestamp, newStoreFile);
            this.currentWriteFile = newStoreFile;
            log.info("IndexStoreService construct next file, timestamp: {}", (Object)timestamp);
        }
        catch (Exception e) {
            log.error("IndexStoreService construct next file, timestamp: {}", (Object)timestamp, (Object)e);
        }
        finally {
            this.readWriteLock.writeLock().unlock();
        }
    }

    @VisibleForTesting
    public ConcurrentSkipListMap<Long, IndexFile> getTimeStoreTable() {
        return this.timeStoreTable;
    }

    @Override
    public AppendResult putKey(String topic, int topicId, int queueId, Set<String> keySet, long offset, int size, long timestamp) {
        if (StringUtils.isBlank((CharSequence)topic)) {
            return AppendResult.UNKNOWN_ERROR;
        }
        if (keySet == null || keySet.isEmpty()) {
            return AppendResult.SUCCESS;
        }
        for (int i = 0; i < 3; ++i) {
            AppendResult result = this.currentWriteFile.putKey(topic, topicId, queueId, keySet, offset, size, timestamp);
            if (AppendResult.SUCCESS.equals((Object)result)) {
                return AppendResult.SUCCESS;
            }
            if (!AppendResult.FILE_FULL.equals((Object)result)) continue;
            this.createNewIndexFile(timestamp);
        }
        log.error("IndexStoreService put key three times return error, topic: {}, topicId: {}, queueId: {}, keySize: {}, timestamp: {}", new Object[]{topic, topicId, queueId, keySet.size(), timestamp});
        return AppendResult.UNKNOWN_ERROR;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public CompletableFuture<List<IndexItem>> queryAsync(String topic, String key, int maxCount, long beginTime, long endTime) {
        CompletableFuture<List<IndexItem>> future = new CompletableFuture<List<IndexItem>>();
        try {
            this.readWriteLock.readLock().lock();
            NavigableMap pendingMap = this.timeStoreTable.subMap((Object)beginTime, true, (Object)endTime, true);
            ArrayList<CompletionStage> futureList = new ArrayList<CompletionStage>(pendingMap.size());
            ConcurrentHashMap result = new ConcurrentHashMap();
            for (Map.Entry entry : pendingMap.descendingMap().entrySet()) {
                CompletionStage completableFuture = ((IndexFile)entry.getValue()).queryAsync(topic, key, maxCount, beginTime, endTime).thenAccept(itemList -> itemList.forEach(indexItem -> {
                    if (result.size() < maxCount) {
                        result.put(String.format("%d-%d", indexItem.getQueueId(), indexItem.getOffset()), indexItem);
                    }
                }));
                futureList.add(completableFuture);
            }
            CompletableFuture.allOf(futureList.toArray(new CompletableFuture[0])).whenComplete((v, t) -> {
                if (result.isEmpty() && t != null) {
                    future.completeExceptionally((Throwable)t);
                } else {
                    ArrayList resultList = new ArrayList(result.values());
                    future.complete(resultList.subList(0, Math.min(resultList.size(), maxCount)));
                }
            });
        }
        catch (Exception e) {
            future.completeExceptionally(e);
        }
        finally {
            this.readWriteLock.readLock().unlock();
        }
        return future;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void doCompactThenUploadFile(IndexFile indexFile) {
        if (IndexFile.IndexStatusEnum.UPLOAD.equals((Object)indexFile.getFileStatus())) {
            log.error("IndexStoreService file status not correct, so skip, timestamp: {}, status: {}", (Object)indexFile.getTimestamp(), (Object)indexFile.getFileStatus());
            indexFile.destroy();
            return;
        }
        Stopwatch stopwatch = Stopwatch.createStarted();
        ByteBuffer byteBuffer = indexFile.doCompaction();
        if (byteBuffer == null) {
            log.error("IndexStoreService found compaction buffer is null, timestamp: {}", (Object)indexFile.getTimestamp());
            return;
        }
        this.flatFile.append(byteBuffer);
        this.flatFile.commit(true);
        TieredFileSegment fileSegment = this.flatFile.getFileByIndex(this.flatFile.getFileSegmentCount() - 1);
        if (fileSegment == null || fileSegment.getMinTimestamp() != indexFile.getTimestamp()) {
            log.warn("IndexStoreService submit compacted file to server failed, timestamp: {}", (Object)indexFile.getTimestamp());
            return;
        }
        try {
            this.readWriteLock.writeLock().lock();
            IndexStoreFile storeFile = new IndexStoreFile(this.storeConfig, fileSegment);
            this.timeStoreTable.put(indexFile.getTimestamp(), storeFile);
            indexFile.destroy();
        }
        catch (Exception e) {
            log.error("IndexStoreService switch file failed, timestamp: {}, cost: {}ms", new Object[]{indexFile.getTimestamp(), stopwatch.elapsed(TimeUnit.MILLISECONDS), e});
        }
        finally {
            this.readWriteLock.writeLock().unlock();
        }
    }

    public void destroyExpiredFile(long expireTimestamp) {
        this.flatFile.cleanExpiredFile(expireTimestamp);
        this.flatFile.destroyExpiredFile();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void destroy() {
        try {
            this.readWriteLock.writeLock().lock();
            for (Map.Entry<Long, IndexFile> entry : this.timeStoreTable.entrySet()) {
                IndexFile indexFile = entry.getValue();
                if (IndexFile.IndexStatusEnum.UPLOAD.equals((Object)indexFile.getFileStatus())) continue;
                indexFile.destroy();
            }
            if (this.flatFile != null) {
                this.flatFile.destroy();
            }
        }
        catch (Exception e) {
            log.error("IndexStoreService destroy all file error", (Throwable)e);
        }
        finally {
            this.readWriteLock.writeLock().unlock();
        }
    }

    public String getServiceName() {
        return IndexStoreService.class.getSimpleName();
    }

    public void setCompactTimestamp(long timestamp) {
        this.compactTimestamp.set(timestamp);
        log.info("IndexStoreService compact timestamp has been set to: {}", (Object)timestamp);
    }

    protected IndexFile getNextSealedFile() {
        try {
            Map.Entry<Long, IndexFile> entry = this.timeStoreTable.higherEntry(this.compactTimestamp.get());
            if (entry != null && entry.getKey() < this.timeStoreTable.lastKey()) {
                return entry.getValue();
            }
        }
        catch (Throwable e) {
            log.error("Error occurred in " + this.getServiceName(), e);
        }
        return null;
    }

    public void run() {
        log.info(this.getServiceName() + " service started");
        while (!this.isStopped()) {
            long expireTimestamp = System.currentTimeMillis() - TimeUnit.HOURS.toMillis(this.storeConfig.getTieredStoreFileReservedTime());
            this.destroyExpiredFile(expireTimestamp);
            IndexFile indexFile = this.getNextSealedFile();
            if (indexFile == null) {
                this.waitForRunning(TimeUnit.SECONDS.toMillis(10L));
                continue;
            }
            this.doCompactThenUploadFile(indexFile);
            this.setCompactTimestamp(indexFile.getTimestamp());
        }
        log.info(this.getServiceName() + " service shutdown");
    }

    @Override
    public void shutdown() {
        super.shutdown();
        for (Map.Entry<Long, IndexFile> entry : this.timeStoreTable.entrySet()) {
            entry.getValue().shutdown();
        }
        this.timeStoreTable.clear();
        log.info("IndexStoreService shutdown gracefully");
    }
}

