/*
 * Decompiled with CFR 0.152.
 */
package org.apache.seatunnel.common.utils;

import java.util.concurrent.TimeUnit;
import org.apache.seatunnel.common.utils.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RetryUtils {
    private static final Logger log = LoggerFactory.getLogger(RetryUtils.class);

    public static <T> T retryWithException(Execution<T, Exception> execution, RetryMaterial retryMaterial) throws Exception {
        RetryCondition<Exception> retryCondition = retryMaterial.getRetryCondition();
        int retryTimes = retryMaterial.getRetryTimes();
        if (retryMaterial.getRetryTimes() < 0) {
            throw new IllegalArgumentException("Retry times must be greater than 0");
        }
        int i = 0;
        while (true) {
            ++i;
            try {
                return execution.execute();
            }
            catch (Exception e) {
                Exception lastException = e;
                if (retryCondition != null && !retryCondition.canRetry(e)) {
                    if (!retryMaterial.shouldThrowException()) continue;
                    throw e;
                }
                String attemptMessage = "Failed to execute due to {}. Retrying attempt ({}/{}) after backoff of {} ms";
                if (retryMaterial.getSleepTimeMillis() > 0L) {
                    long backoff = retryMaterial.computeRetryWaitTimeMillis(i);
                    log.debug(attemptMessage, ExceptionUtils.getMessage(e), i, retryTimes, backoff);
                    Thread.sleep(backoff);
                    continue;
                }
                log.info(attemptMessage, ExceptionUtils.getMessage(e), i, retryTimes, 0);
                if (i < retryTimes) continue;
                if (retryMaterial.shouldThrowException()) {
                    throw new RuntimeException("Execute given execution failed after retry " + retryTimes + " times", lastException);
                }
                return null;
            }
            break;
        }
    }

    public static interface RetryCondition<T> {
        public boolean canRetry(T var1);
    }

    @FunctionalInterface
    public static interface Execution<T, E extends Exception> {
        public T execute() throws E;
    }

    public static class RetryMaterial {
        public static final long MAX_RETRY_TIME_MS = TimeUnit.SECONDS.toMillis(20L);
        public static final long MAX_RETRY_TIME = 32L;
        private final int retryTimes;
        private final boolean shouldThrowException;
        private final RetryCondition<Exception> retryCondition;
        private final boolean sleepTimeIncrease;
        private final long sleepTimeMillis;

        public RetryMaterial(int retryTimes, boolean shouldThrowException, RetryCondition<Exception> retryCondition) {
            this(retryTimes, shouldThrowException, retryCondition, 0L);
        }

        public RetryMaterial(int retryTimes, boolean shouldThrowException, RetryCondition<Exception> retryCondition, long sleepTimeMillis) {
            this(retryTimes, shouldThrowException, retryCondition, sleepTimeMillis, false);
        }

        public RetryMaterial(int retryTimes, boolean shouldThrowException, RetryCondition<Exception> retryCondition, long sleepTimeMillis, boolean sleepTimeIncrease) {
            this.retryTimes = retryTimes;
            this.shouldThrowException = shouldThrowException;
            this.retryCondition = retryCondition;
            this.sleepTimeMillis = sleepTimeMillis;
            this.sleepTimeIncrease = sleepTimeIncrease;
        }

        public int getRetryTimes() {
            return this.retryTimes;
        }

        public boolean shouldThrowException() {
            return this.shouldThrowException;
        }

        public RetryCondition<Exception> getRetryCondition() {
            return this.retryCondition;
        }

        public long getSleepTimeMillis() {
            return this.sleepTimeMillis;
        }

        public long computeRetryWaitTimeMillis(int retryAttempts) {
            if (this.sleepTimeMillis < 0L) {
                return 0L;
            }
            if (!this.sleepTimeIncrease) {
                return this.sleepTimeMillis;
            }
            if ((long)retryAttempts > 32L) {
                return MAX_RETRY_TIME_MS;
            }
            long result = this.sleepTimeMillis << retryAttempts;
            return result < 0L ? MAX_RETRY_TIME_MS : Math.min(MAX_RETRY_TIME_MS, result);
        }
    }
}

