/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.util;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletionService;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicy;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedStripedBlock;
import org.apache.hadoop.hdfs.security.token.block.BlockTokenIdentifier;
import org.apache.hadoop.security.token.Token;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class StripedBlockUtil {
    public static final Logger LOG = LoggerFactory.getLogger(StripedBlockUtil.class);

    public static LocatedBlock[] parseStripedBlockGroup(LocatedStripedBlock bg, int cellSize, int dataBlkNum, int parityBlkNum) {
        int locatedBGSize = bg.getBlockIndices().length;
        LocatedBlock[] lbs = new LocatedBlock[dataBlkNum + parityBlkNum];
        for (int i = 0; i < locatedBGSize; i = (int)((short)(i + 1))) {
            byte idx = bg.getBlockIndices()[i];
            if (idx >= dataBlkNum + parityBlkNum || lbs[idx] != null) continue;
            lbs[idx] = StripedBlockUtil.constructInternalBlock(bg, i, cellSize, dataBlkNum, idx);
        }
        return lbs;
    }

    public static LocatedBlock constructInternalBlock(LocatedStripedBlock bg, int idxInReturnedLocs, int cellSize, int dataBlkNum, int idxInBlockGroup) {
        ExtendedBlock blk = StripedBlockUtil.constructInternalBlock(bg.getBlock(), cellSize, dataBlkNum, idxInBlockGroup);
        LocatedBlock locatedBlock = idxInReturnedLocs < bg.getLocations().length ? new LocatedBlock(blk, new DatanodeInfo[]{bg.getLocations()[idxInReturnedLocs]}, new String[]{bg.getStorageIDs()[idxInReturnedLocs]}, new StorageType[]{bg.getStorageTypes()[idxInReturnedLocs]}, bg.getStartOffset(), bg.isCorrupt(), null) : new LocatedBlock(blk, null, null, null, bg.getStartOffset(), bg.isCorrupt(), null);
        Token<BlockTokenIdentifier>[] blockTokens = bg.getBlockTokens();
        if (idxInReturnedLocs < blockTokens.length) {
            locatedBlock.setBlockToken(blockTokens[idxInReturnedLocs]);
        }
        return locatedBlock;
    }

    public static ExtendedBlock constructInternalBlock(ExtendedBlock blockGroup, ErasureCodingPolicy ecPolicy, int idxInBlockGroup) {
        return StripedBlockUtil.constructInternalBlock(blockGroup, ecPolicy.getCellSize(), ecPolicy.getNumDataUnits(), idxInBlockGroup);
    }

    public static ExtendedBlock constructInternalBlock(ExtendedBlock blockGroup, int cellSize, int dataBlkNum, int idxInBlockGroup) {
        ExtendedBlock block = new ExtendedBlock(blockGroup);
        block.setBlockId(blockGroup.getBlockId() + (long)idxInBlockGroup);
        block.setNumBytes(StripedBlockUtil.getInternalBlockLength(blockGroup.getNumBytes(), cellSize, dataBlkNum, idxInBlockGroup));
        return block;
    }

    public static long getInternalBlockLength(long dataSize, ErasureCodingPolicy ecPolicy, int idxInBlockGroup) {
        return StripedBlockUtil.getInternalBlockLength(dataSize, ecPolicy.getCellSize(), ecPolicy.getNumDataUnits(), idxInBlockGroup);
    }

    public static long getInternalBlockLength(long dataSize, int cellSize, int numDataBlocks, int idxInBlockGroup) {
        Preconditions.checkArgument((dataSize >= 0L ? 1 : 0) != 0);
        Preconditions.checkArgument((cellSize > 0 ? 1 : 0) != 0);
        Preconditions.checkArgument((numDataBlocks > 0 ? 1 : 0) != 0);
        Preconditions.checkArgument((idxInBlockGroup >= 0 ? 1 : 0) != 0);
        int stripeSize = cellSize * numDataBlocks;
        int lastStripeDataLen = (int)(dataSize % (long)stripeSize);
        if (lastStripeDataLen == 0) {
            return dataSize / (long)numDataBlocks;
        }
        int numStripes = (int)((dataSize - 1L) / (long)stripeSize + 1L);
        return ((long)numStripes - 1L) * (long)cellSize + (long)StripedBlockUtil.lastCellSize(lastStripeDataLen, cellSize, numDataBlocks, idxInBlockGroup);
    }

    public static long getSafeLength(ErasureCodingPolicy ecPolicy, long[] blockLens) {
        int cellSize = ecPolicy.getCellSize();
        int dataBlkNum = ecPolicy.getNumDataUnits();
        Preconditions.checkArgument((blockLens.length >= dataBlkNum ? 1 : 0) != 0);
        int stripeSize = dataBlkNum * cellSize;
        long[] cpy = Arrays.copyOf(blockLens, blockLens.length);
        Arrays.sort(cpy);
        int lastFullStripeIdx = (int)(cpy[cpy.length - dataBlkNum] / (long)cellSize);
        return lastFullStripeIdx * stripeSize;
    }

    private static int lastCellSize(int size, int cellSize, int numDataBlocks, int i) {
        if (i < numDataBlocks && (size -= i * cellSize) < 0) {
            size = 0;
        }
        return size > cellSize ? cellSize : size;
    }

    public static long offsetInBlkToOffsetInBG(int cellSize, int dataBlkNum, long offsetInBlk, int idxInBlockGroup) {
        int cellIdxInBlk = (int)(offsetInBlk / (long)cellSize);
        return (long)(cellIdxInBlk * cellSize * dataBlkNum + idxInBlockGroup * cellSize) + offsetInBlk % (long)cellSize;
    }

    public static StripingChunkReadResult getNextCompletedStripedRead(CompletionService<BlockReadStats> readService, Map<Future<BlockReadStats>, Integer> futures, long timeoutMillis) throws InterruptedException {
        Preconditions.checkArgument((!futures.isEmpty() ? 1 : 0) != 0);
        Future<BlockReadStats> future = null;
        try {
            future = timeoutMillis > 0L ? readService.poll(timeoutMillis, TimeUnit.MILLISECONDS) : readService.take();
            if (future != null) {
                BlockReadStats stats = future.get();
                return new StripingChunkReadResult(futures.remove(future), 1, stats);
            }
            return new StripingChunkReadResult(4);
        }
        catch (ExecutionException e) {
            LOG.debug("Exception during striped read task", (Throwable)e);
            return new StripingChunkReadResult(futures.remove(future), 2);
        }
        catch (CancellationException e) {
            LOG.debug("Exception during striped read task", (Throwable)e);
            return new StripingChunkReadResult(futures.remove(future), 8);
        }
    }

    public static long spaceConsumedByStripedBlock(long numDataBlkBytes, int dataBlkNum, int parityBlkNum, int cellSize) {
        int parityIndex = dataBlkNum + 1;
        long numParityBlkBytes = StripedBlockUtil.getInternalBlockLength(numDataBlkBytes, cellSize, dataBlkNum, parityIndex) * (long)parityBlkNum;
        return numDataBlkBytes + numParityBlkBytes;
    }

    public static AlignedStripe[] divideOneStripe(ErasureCodingPolicy ecPolicy, int cellSize, LocatedStripedBlock blockGroup, long rangeStartInBlockGroup, long rangeEndInBlockGroup, ByteBuffer buf) {
        int dataBlkNum = ecPolicy.getNumDataUnits();
        StripingCell[] cells = StripedBlockUtil.getStripingCellsOfByteRange(ecPolicy, cellSize, blockGroup, rangeStartInBlockGroup, rangeEndInBlockGroup);
        VerticalRange[] ranges = StripedBlockUtil.getRangesForInternalBlocks(ecPolicy, cellSize, cells);
        AlignedStripe[] stripes = StripedBlockUtil.mergeRangesForInternalBlocks(ecPolicy, ranges);
        int bufOffset = (int)(rangeStartInBlockGroup % ((long)cellSize * (long)dataBlkNum));
        for (StripingCell cell : cells) {
            long cellStart = cell.idxInInternalBlk * (long)cellSize + cell.offset;
            long cellEnd = cellStart + (long)cell.size - 1L;
            for (AlignedStripe s : stripes) {
                long stripeEnd = s.getOffsetInBlock() + s.getSpanInBlock() - 1L;
                long overlapStart = Math.max(cellStart, s.getOffsetInBlock());
                long overlapEnd = Math.min(cellEnd, stripeEnd);
                int overLapLen = (int)(overlapEnd - overlapStart + 1L);
                if (overLapLen <= 0) continue;
                Preconditions.checkState((s.chunks[cell.idxInStripe] == null ? 1 : 0) != 0);
                int pos = (int)((long)bufOffset + overlapStart - cellStart);
                buf.position(pos);
                buf.limit(pos + overLapLen);
                s.chunks[((StripingCell)cell).idxInStripe] = new StripingChunk(buf.slice());
            }
            bufOffset += cell.size;
        }
        StripedBlockUtil.prepareAllZeroChunks(blockGroup, stripes, cellSize, dataBlkNum);
        return stripes;
    }

    public static AlignedStripe[] divideByteRangeIntoStripes(ErasureCodingPolicy ecPolicy, int cellSize, LocatedStripedBlock blockGroup, long rangeStartInBlockGroup, long rangeEndInBlockGroup, ByteBuffer buf) {
        int dataBlkNum = ecPolicy.getNumDataUnits();
        StripingCell[] cells = StripedBlockUtil.getStripingCellsOfByteRange(ecPolicy, cellSize, blockGroup, rangeStartInBlockGroup, rangeEndInBlockGroup);
        VerticalRange[] ranges = StripedBlockUtil.getRangesForInternalBlocks(ecPolicy, cellSize, cells);
        AlignedStripe[] stripes = StripedBlockUtil.mergeRangesForInternalBlocks(ecPolicy, ranges);
        StripedBlockUtil.calcualteChunkPositionsInBuf(cellSize, stripes, cells, buf);
        StripedBlockUtil.prepareAllZeroChunks(blockGroup, stripes, cellSize, dataBlkNum);
        return stripes;
    }

    @VisibleForTesting
    private static StripingCell[] getStripingCellsOfByteRange(ErasureCodingPolicy ecPolicy, int cellSize, LocatedStripedBlock blockGroup, long rangeStartInBlockGroup, long rangeEndInBlockGroup) {
        Preconditions.checkArgument((rangeStartInBlockGroup <= rangeEndInBlockGroup && rangeEndInBlockGroup < blockGroup.getBlockSize() ? 1 : 0) != 0, (String)"start=%s end=%s blockSize=%s", (Object)rangeStartInBlockGroup, (Object)rangeEndInBlockGroup, (Object)blockGroup.getBlockSize());
        long len = rangeEndInBlockGroup - rangeStartInBlockGroup + 1L;
        int firstCellIdxInBG = (int)(rangeStartInBlockGroup / (long)cellSize);
        int lastCellIdxInBG = (int)(rangeEndInBlockGroup / (long)cellSize);
        int numCells = lastCellIdxInBG - firstCellIdxInBG + 1;
        StripingCell[] cells = new StripingCell[numCells];
        int firstCellOffset = (int)(rangeStartInBlockGroup % (long)cellSize);
        int firstCellSize = (int)Math.min((long)cellSize - rangeStartInBlockGroup % (long)cellSize, len);
        cells[0] = new StripingCell(ecPolicy, firstCellSize, firstCellIdxInBG, firstCellOffset);
        if (lastCellIdxInBG != firstCellIdxInBG) {
            int lastCellSize = (int)(rangeEndInBlockGroup % (long)cellSize) + 1;
            cells[numCells - 1] = new StripingCell(ecPolicy, lastCellSize, lastCellIdxInBG, 0L);
        }
        for (int i = 1; i < numCells - 1; ++i) {
            cells[i] = new StripingCell(ecPolicy, cellSize, i + firstCellIdxInBG, 0L);
        }
        return cells;
    }

    @VisibleForTesting
    private static VerticalRange[] getRangesForInternalBlocks(ErasureCodingPolicy ecPolicy, int cellSize, StripingCell[] cells) {
        int dataBlkNum = ecPolicy.getNumDataUnits();
        int parityBlkNum = ecPolicy.getNumParityUnits();
        VerticalRange[] ranges = new VerticalRange[dataBlkNum + parityBlkNum];
        long earliestStart = Long.MAX_VALUE;
        long latestEnd = -1L;
        for (StripingCell cell : cells) {
            if (ranges[cell.idxInStripe] == null) {
                ranges[((StripingCell)cell).idxInStripe] = new VerticalRange(cell.idxInInternalBlk * (long)cellSize + cell.offset, cell.size);
            } else {
                ranges[((StripingCell)cell).idxInStripe].spanInBlock += (long)cell.size;
            }
            VerticalRange range = ranges[cell.idxInStripe];
            if (range.offsetInBlock < earliestStart) {
                earliestStart = range.offsetInBlock;
            }
            if (range.offsetInBlock + range.spanInBlock - 1L <= latestEnd) continue;
            latestEnd = range.offsetInBlock + range.spanInBlock - 1L;
        }
        for (int i = dataBlkNum; i < dataBlkNum + parityBlkNum; ++i) {
            ranges[i] = new VerticalRange(earliestStart, latestEnd - earliestStart + 1L);
        }
        return ranges;
    }

    private static AlignedStripe[] mergeRangesForInternalBlocks(ErasureCodingPolicy ecPolicy, VerticalRange[] ranges) {
        int dataBlkNum = ecPolicy.getNumDataUnits();
        int parityBlkNum = ecPolicy.getNumParityUnits();
        ArrayList<AlignedStripe> stripes = new ArrayList<AlignedStripe>();
        TreeSet<Long> stripePoints = new TreeSet<Long>();
        for (VerticalRange r : ranges) {
            if (r == null) continue;
            stripePoints.add(r.offsetInBlock);
            stripePoints.add(r.offsetInBlock + r.spanInBlock);
        }
        long prev = -1L;
        Iterator iterator = stripePoints.iterator();
        while (iterator.hasNext()) {
            long point = (Long)iterator.next();
            if (prev >= 0L) {
                stripes.add(new AlignedStripe(prev, point - prev, dataBlkNum + parityBlkNum));
            }
            prev = point;
        }
        return stripes.toArray(new AlignedStripe[stripes.size()]);
    }

    private static void calcualteChunkPositionsInBuf(int cellSize, AlignedStripe[] stripes, StripingCell[] cells, ByteBuffer buf) {
        int done = 0;
        for (StripingCell cell : cells) {
            long cellStart = cell.idxInInternalBlk * (long)cellSize + cell.offset;
            long cellEnd = cellStart + (long)cell.size - 1L;
            for (AlignedStripe s : stripes) {
                long stripeEnd = s.getOffsetInBlock() + s.getSpanInBlock() - 1L;
                long overlapStart = Math.max(cellStart, s.getOffsetInBlock());
                long overlapEnd = Math.min(cellEnd, stripeEnd);
                int overLapLen = (int)(overlapEnd - overlapStart + 1L);
                if (overLapLen <= 0) continue;
                StripingChunk chunk = s.chunks[cell.idxInStripe];
                if (chunk == null) {
                    s.chunks[((StripingCell)cell).idxInStripe] = chunk = new StripingChunk();
                }
                chunk.getChunkBuffer().addSlice(buf, (int)((long)done + overlapStart - cellStart), overLapLen);
            }
            done += cell.size;
        }
    }

    private static void prepareAllZeroChunks(LocatedStripedBlock blockGroup, AlignedStripe[] stripes, int cellSize, int dataBlkNum) {
        for (AlignedStripe s : stripes) {
            for (int i = 0; i < dataBlkNum; ++i) {
                long internalBlkLen = StripedBlockUtil.getInternalBlockLength(blockGroup.getBlockSize(), cellSize, dataBlkNum, i);
                if (internalBlkLen > s.getOffsetInBlock()) continue;
                Preconditions.checkState((s.chunks[i] == null ? 1 : 0) != 0);
                s.chunks[i] = new StripingChunk(15);
            }
        }
    }

    public static void checkBlocks(ExtendedBlock blockGroup, int i, ExtendedBlock blocki) throws IOException {
        if (!blocki.getBlockPoolId().equals(blockGroup.getBlockPoolId())) {
            throw new IOException("Block pool IDs mismatched: block" + i + "=" + blocki + ", expected block group=" + blockGroup);
        }
        if (blocki.getBlockId() - (long)i != blockGroup.getBlockId()) {
            throw new IOException("Block IDs mismatched: block" + i + "=" + blocki + ", expected block group=" + blockGroup);
        }
        if (blocki.getGenerationStamp() != blockGroup.getGenerationStamp()) {
            throw new IOException("Generation stamps mismatched: block" + i + "=" + blocki + ", expected block group=" + blockGroup);
        }
    }

    public static int getBlockIndex(Block reportedBlock) {
        long BLOCK_GROUP_INDEX_MASK = 15L;
        return (int)(reportedBlock.getBlockId() & BLOCK_GROUP_INDEX_MASK);
    }

    public static class StripeRange {
        final long offsetInBlock;
        final long length;

        public StripeRange(long offsetInBlock, long length) {
            Preconditions.checkArgument((offsetInBlock >= 0L && length >= 0L ? 1 : 0) != 0, (String)"Offset(%s) and length(%s) must be non-negative", (long)offsetInBlock, (long)length);
            this.offsetInBlock = offsetInBlock;
            this.length = length;
        }

        public boolean include(long pos) {
            return pos >= this.offsetInBlock && pos < this.offsetInBlock + this.length;
        }

        public long getLength() {
            return this.length;
        }

        public String toString() {
            return String.format("StripeRange(offsetInBlock=%d, length=%d)", this.offsetInBlock, this.length);
        }
    }

    public static class StripingChunkReadResult {
        public static final int SUCCESSFUL = 1;
        public static final int FAILED = 2;
        public static final int TIMEOUT = 4;
        public static final int CANCELLED = 8;
        public final int index;
        public final int state;
        private final BlockReadStats readStats;

        public StripingChunkReadResult(int state) {
            Preconditions.checkArgument((state == 4 ? 1 : 0) != 0, (Object)"Only timeout result should return negative index.");
            this.index = -1;
            this.state = state;
            this.readStats = null;
        }

        public StripingChunkReadResult(int index, int state) {
            this(index, state, null);
        }

        public StripingChunkReadResult(int index, int state, BlockReadStats stats) {
            Preconditions.checkArgument((state != 4 ? 1 : 0) != 0, (Object)"Timeout result should return negative index.");
            this.index = index;
            this.state = state;
            this.readStats = stats;
        }

        public BlockReadStats getReadStats() {
            return this.readStats;
        }

        public String toString() {
            return "(index=" + this.index + ", state =" + this.state + ", readStats =" + this.readStats + ")";
        }
    }

    public static class ChunkByteBuffer {
        private final List<ByteBuffer> slices = new ArrayList<ByteBuffer>();

        ChunkByteBuffer() {
        }

        public void addSlice(ByteBuffer buffer, int offset, int len) {
            ByteBuffer tmp = buffer.duplicate();
            tmp.position(buffer.position() + offset);
            tmp.limit(buffer.position() + offset + len);
            this.slices.add(tmp.slice());
        }

        public ByteBuffer getSlice(int i) {
            return this.slices.get(i);
        }

        public List<ByteBuffer> getSlices() {
            return this.slices;
        }

        public void copyTo(ByteBuffer target) {
            for (ByteBuffer slice : this.slices) {
                slice.flip();
                target.put(slice);
            }
            target.flip();
        }

        public void copyFrom(ByteBuffer src) {
            for (ByteBuffer slice : this.slices) {
                int len = slice.remaining();
                ByteBuffer tmp = src.duplicate();
                tmp.limit(tmp.position() + len);
                slice.put(tmp);
                src.position(src.position() + len);
            }
        }
    }

    public static class StripingChunk {
        public static final int FETCHED = 1;
        public static final int MISSING = 2;
        public static final int PENDING = 4;
        public static final int REQUESTED = 8;
        public static final int ALLZERO = 15;
        public int state = 8;
        private final ChunkByteBuffer chunkBuffer;
        private final ByteBuffer byteBuffer;

        public StripingChunk() {
            this.chunkBuffer = new ChunkByteBuffer();
            this.byteBuffer = null;
        }

        public StripingChunk(ByteBuffer buf) {
            this.chunkBuffer = null;
            this.byteBuffer = buf;
        }

        public StripingChunk(int state) {
            this.chunkBuffer = null;
            this.byteBuffer = null;
            this.state = state;
        }

        public boolean useByteBuffer() {
            return this.byteBuffer != null;
        }

        public boolean useChunkBuffer() {
            return this.chunkBuffer != null;
        }

        public ByteBuffer getByteBuffer() {
            assert (this.byteBuffer != null);
            return this.byteBuffer;
        }

        public ChunkByteBuffer getChunkBuffer() {
            assert (this.chunkBuffer != null);
            return this.chunkBuffer;
        }
    }

    public static class VerticalRange {
        public long offsetInBlock;
        public long spanInBlock;

        public VerticalRange(long offsetInBlock, long length) {
            Preconditions.checkArgument((offsetInBlock >= 0L && length >= 0L ? 1 : 0) != 0, (String)"OffsetInBlock(%s) and length(%s) must be non-negative", (long)offsetInBlock, (long)length);
            this.offsetInBlock = offsetInBlock;
            this.spanInBlock = length;
        }

        public boolean include(long pos) {
            return pos >= this.offsetInBlock && pos < this.offsetInBlock + this.spanInBlock;
        }

        public String toString() {
            return String.format("VerticalRange(offsetInBlock=%d, spanInBlock=%d)", this.offsetInBlock, this.spanInBlock);
        }
    }

    public static class AlignedStripe {
        public VerticalRange range;
        public final StripingChunk[] chunks;
        public int fetchedChunksNum = 0;
        public int missingChunksNum = 0;

        public AlignedStripe(long offsetInBlock, long length, int width) {
            Preconditions.checkArgument((offsetInBlock >= 0L && length >= 0L ? 1 : 0) != 0, (String)"OffsetInBlock(%s) and length(%s) must be non-negative", (long)offsetInBlock, (long)length);
            this.range = new VerticalRange(offsetInBlock, length);
            this.chunks = new StripingChunk[width];
        }

        public boolean include(long pos) {
            return this.range.include(pos);
        }

        public long getOffsetInBlock() {
            return this.range.offsetInBlock;
        }

        public long getSpanInBlock() {
            return this.range.spanInBlock;
        }

        public String toString() {
            return "AlignedStripe(Offset=" + this.range.offsetInBlock + ", length=" + this.range.spanInBlock + ", fetchedChunksNum=" + this.fetchedChunksNum + ", missingChunksNum=" + this.missingChunksNum + ")";
        }
    }

    @VisibleForTesting
    public static class StripingCell {
        final ErasureCodingPolicy ecPolicy;
        private final long idxInBlkGroup;
        private final long idxInInternalBlk;
        private final int idxInStripe;
        private final long offset;
        private final int size;

        StripingCell(ErasureCodingPolicy ecPolicy, int cellSize, long idxInBlkGroup, long offset) {
            this.ecPolicy = ecPolicy;
            this.idxInBlkGroup = idxInBlkGroup;
            this.idxInInternalBlk = idxInBlkGroup / (long)ecPolicy.getNumDataUnits();
            this.idxInStripe = (int)(idxInBlkGroup - this.idxInInternalBlk * (long)ecPolicy.getNumDataUnits());
            this.offset = offset;
            this.size = cellSize;
        }

        int getIdxInStripe() {
            return this.idxInStripe;
        }

        public String toString() {
            return String.format("StripingCell(idxInBlkGroup=%d, idxInInternalBlk=%d, idxInStrip=%d, offset=%d, size=%d)", this.idxInBlkGroup, this.idxInInternalBlk, this.idxInStripe, this.offset, this.size);
        }
    }

    public static class BlockReadStats {
        private final int bytesRead;
        private final boolean isShortCircuit;
        private final int networkDistance;

        public BlockReadStats(int numBytesRead, boolean shortCircuit, int distance) {
            this.bytesRead = numBytesRead;
            this.isShortCircuit = shortCircuit;
            this.networkDistance = distance;
        }

        public int getBytesRead() {
            return this.bytesRead;
        }

        public boolean isShortCircuit() {
            return this.isShortCircuit;
        }

        public int getNetworkDistance() {
            return this.networkDistance;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("bytesRead=").append(this.bytesRead);
            sb.append(',');
            sb.append("isShortCircuit=").append(this.isShortCircuit);
            sb.append(',');
            sb.append("networkDistance=").append(this.networkDistance);
            return sb.toString();
        }
    }
}

