/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.protocol.datatransfer.sasl;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.Socket;
import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.crypto.SecretKey;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.TextInputCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.sasl.RealmCallback;
import javax.security.sasl.RealmChoiceCallback;
import org.apache.commons.codec.binary.Base64;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.CipherOption;
import org.apache.hadoop.crypto.CipherSuite;
import org.apache.hadoop.hdfs.net.EncryptedPeer;
import org.apache.hadoop.hdfs.net.Peer;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.protocol.datatransfer.IOStreamPair;
import org.apache.hadoop.hdfs.protocol.datatransfer.TrustedChannelResolver;
import org.apache.hadoop.hdfs.protocol.datatransfer.sasl.DataEncryptionKeyFactory;
import org.apache.hadoop.hdfs.protocol.datatransfer.sasl.DataTransferSaslUtil;
import org.apache.hadoop.hdfs.protocol.datatransfer.sasl.SaslParticipant;
import org.apache.hadoop.hdfs.protocol.datatransfer.sasl.SaslResponseWithNegotiatedCipherOption;
import org.apache.hadoop.hdfs.security.token.block.BlockTokenIdentifier;
import org.apache.hadoop.hdfs.security.token.block.DataEncryptionKey;
import org.apache.hadoop.security.SaslPropertiesResolver;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.SecretManager;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.thirdparty.com.google.common.base.Charsets;
import org.apache.hadoop.thirdparty.com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class SaslDataTransferClient {
    private static final Logger LOG = LoggerFactory.getLogger(SaslDataTransferClient.class);
    private final Configuration conf;
    private final AtomicBoolean fallbackToSimpleAuth;
    private final SaslPropertiesResolver saslPropsResolver;
    private final TrustedChannelResolver trustedChannelResolver;
    private String targetQOP;

    public SaslDataTransferClient(Configuration conf, SaslPropertiesResolver saslPropsResolver, TrustedChannelResolver trustedChannelResolver) {
        this(conf, saslPropsResolver, trustedChannelResolver, null);
    }

    public SaslDataTransferClient(Configuration conf, SaslPropertiesResolver saslPropsResolver, TrustedChannelResolver trustedChannelResolver, AtomicBoolean fallbackToSimpleAuth) {
        this.conf = conf;
        this.fallbackToSimpleAuth = fallbackToSimpleAuth;
        this.saslPropsResolver = saslPropsResolver;
        this.trustedChannelResolver = trustedChannelResolver;
    }

    public IOStreamPair newSocketSend(Socket socket, OutputStream underlyingOut, InputStream underlyingIn, DataEncryptionKeyFactory encryptionKeyFactory, Token<BlockTokenIdentifier> accessToken, DatanodeID datanodeId) throws IOException {
        DataEncryptionKey encryptionKey = !this.trustedChannelResolver.isTrusted() ? encryptionKeyFactory.newDataEncryptionKey() : null;
        IOStreamPair ios = this.send(socket.getInetAddress(), underlyingOut, underlyingIn, encryptionKey, accessToken, datanodeId, null);
        return ios != null ? ios : new IOStreamPair(underlyingIn, underlyingOut);
    }

    public Peer peerSend(Peer peer, DataEncryptionKeyFactory encryptionKeyFactory, Token<BlockTokenIdentifier> accessToken, DatanodeID datanodeId) throws IOException {
        IOStreamPair ios = this.checkTrustAndSend(DataTransferSaslUtil.getPeerAddress(peer), peer.getOutputStream(), peer.getInputStream(), encryptionKeyFactory, accessToken, datanodeId, null);
        return ios != null ? new EncryptedPeer(peer, ios) : peer;
    }

    public IOStreamPair socketSend(Socket socket, OutputStream underlyingOut, InputStream underlyingIn, DataEncryptionKeyFactory encryptionKeyFactory, Token<BlockTokenIdentifier> accessToken, DatanodeID datanodeId) throws IOException {
        return this.socketSend(socket, underlyingOut, underlyingIn, encryptionKeyFactory, accessToken, datanodeId, null);
    }

    public IOStreamPair socketSend(Socket socket, OutputStream underlyingOut, InputStream underlyingIn, DataEncryptionKeyFactory encryptionKeyFactory, Token<BlockTokenIdentifier> accessToken, DatanodeID datanodeId, SecretKey secretKey) throws IOException {
        IOStreamPair ios = this.checkTrustAndSend(socket.getInetAddress(), underlyingOut, underlyingIn, encryptionKeyFactory, accessToken, datanodeId, secretKey);
        return ios != null ? ios : new IOStreamPair(underlyingIn, underlyingOut);
    }

    private IOStreamPair checkTrustAndSend(InetAddress addr, OutputStream underlyingOut, InputStream underlyingIn, DataEncryptionKeyFactory encryptionKeyFactory, Token<BlockTokenIdentifier> accessToken, DatanodeID datanodeId, SecretKey secretKey) throws IOException {
        boolean localTrusted = this.trustedChannelResolver.isTrusted();
        boolean remoteTrusted = this.trustedChannelResolver.isTrusted(addr);
        LOG.debug("SASL encryption trust check: localHostTrusted = {}, remoteHostTrusted = {}", (Object)localTrusted, (Object)remoteTrusted);
        if (!localTrusted || !remoteTrusted) {
            DataEncryptionKey encryptionKey = encryptionKeyFactory.newDataEncryptionKey();
            return this.send(addr, underlyingOut, underlyingIn, encryptionKey, accessToken, datanodeId, secretKey);
        }
        LOG.debug("SASL client skipping handshake on trusted connection for addr = {}, datanodeId = {}", (Object)addr, (Object)datanodeId);
        return null;
    }

    private IOStreamPair send(InetAddress addr, OutputStream underlyingOut, InputStream underlyingIn, DataEncryptionKey encryptionKey, Token<BlockTokenIdentifier> accessToken, DatanodeID datanodeId, SecretKey secretKey) throws IOException {
        if (encryptionKey != null) {
            LOG.debug("SASL client doing encrypted handshake for addr = {}, datanodeId = {}", (Object)addr, (Object)datanodeId);
            return this.getEncryptedStreams(addr, underlyingOut, underlyingIn, encryptionKey, accessToken, secretKey);
        }
        if (!UserGroupInformation.isSecurityEnabled()) {
            LOG.debug("SASL client skipping handshake in unsecured configuration for addr = {}, datanodeId = {}", (Object)addr, (Object)datanodeId);
            return null;
        }
        if (SecurityUtil.isPrivilegedPort((int)datanodeId.getXferPort())) {
            LOG.debug("SASL client skipping handshake in secured configuration with privileged port for addr = {}, datanodeId = {}", (Object)addr, (Object)datanodeId);
            return null;
        }
        if (this.fallbackToSimpleAuth != null && this.fallbackToSimpleAuth.get()) {
            LOG.debug("SASL client skipping handshake in secured configuration with unsecured cluster for addr = {}, datanodeId = {}", (Object)addr, (Object)datanodeId);
            return null;
        }
        if (this.saslPropsResolver != null) {
            LOG.debug("SASL client doing general handshake for addr = {}, datanodeId = {}", (Object)addr, (Object)datanodeId);
            return this.getSaslStreams(addr, underlyingOut, underlyingIn, accessToken, secretKey);
        }
        LOG.debug("SASL client skipping handshake in secured configuration with no SASL protection configured for addr = {}, datanodeId = {}", (Object)addr, (Object)datanodeId);
        return null;
    }

    private IOStreamPair getEncryptedStreams(InetAddress addr, OutputStream underlyingOut, InputStream underlyingIn, DataEncryptionKey encryptionKey, Token<BlockTokenIdentifier> accessToken, SecretKey secretKey) throws IOException {
        Map<String, String> saslProps = DataTransferSaslUtil.createSaslPropertiesForEncryption(encryptionKey.encryptionAlgorithm);
        if (secretKey != null) {
            LOG.debug("DataNode overwriting downstream QOP" + saslProps.get("javax.security.sasl.qop"));
            this.updateToken(accessToken, secretKey, saslProps);
        }
        LOG.debug("Client using encryption algorithm {}", (Object)encryptionKey.encryptionAlgorithm);
        String userName = SaslDataTransferClient.getUserNameFromEncryptionKey(encryptionKey);
        char[] password = DataTransferSaslUtil.encryptionKeyToPassword(encryptionKey.encryptionKey);
        SaslClientCallbackHandler callbackHandler = new SaslClientCallbackHandler(userName, password);
        return this.doSaslHandshake(addr, underlyingOut, underlyingIn, userName, saslProps, callbackHandler, accessToken);
    }

    private static String getUserNameFromEncryptionKey(DataEncryptionKey encryptionKey) {
        return encryptionKey.keyId + " " + encryptionKey.blockPoolId + " " + new String(Base64.encodeBase64((byte[])encryptionKey.nonce, (boolean)false), Charsets.UTF_8);
    }

    @VisibleForTesting
    public String getTargetQOP() {
        return this.targetQOP;
    }

    private IOStreamPair getSaslStreams(InetAddress addr, OutputStream underlyingOut, InputStream underlyingIn, Token<BlockTokenIdentifier> accessToken, SecretKey secretKey) throws IOException {
        Map saslProps = this.saslPropsResolver.getClientProperties(addr);
        if (secretKey != null) {
            String newQOP = this.conf.get("dfs.encrypt.data.overwrite.downstream.new.qop");
            if (newQOP != null) {
                saslProps.put("javax.security.sasl.qop", newQOP);
            }
            LOG.debug("DataNode overwriting downstream QOP " + (String)saslProps.get("javax.security.sasl.qop"));
            this.updateToken(accessToken, secretKey, saslProps);
        }
        this.targetQOP = (String)saslProps.get("javax.security.sasl.qop");
        String userName = SaslDataTransferClient.buildUserName(accessToken);
        char[] password = this.buildClientPassword(accessToken);
        SaslClientCallbackHandler callbackHandler = new SaslClientCallbackHandler(userName, password);
        return this.doSaslHandshake(addr, underlyingOut, underlyingIn, userName, saslProps, callbackHandler, accessToken);
    }

    private void updateToken(Token<BlockTokenIdentifier> accessToken, SecretKey secretKey, Map<String, String> saslProps) throws IOException {
        byte[] newSecret = saslProps.get("javax.security.sasl.qop").getBytes(Charsets.UTF_8);
        BlockTokenIdentifier bkid = (BlockTokenIdentifier)accessToken.decodeIdentifier();
        bkid.setHandshakeMsg(newSecret);
        byte[] bkidBytes = bkid.getBytes();
        accessToken.setPassword(SecretManager.createPassword((byte[])bkidBytes, (SecretKey)secretKey));
        accessToken.setID(bkidBytes);
    }

    private static String buildUserName(Token<BlockTokenIdentifier> blockToken) {
        return new String(Base64.encodeBase64((byte[])blockToken.getIdentifier(), (boolean)false), Charsets.UTF_8);
    }

    private char[] buildClientPassword(Token<BlockTokenIdentifier> blockToken) {
        return new String(Base64.encodeBase64((byte[])blockToken.getPassword(), (boolean)false), Charsets.UTF_8).toCharArray();
    }

    private IOStreamPair doSaslHandshake(InetAddress addr, OutputStream underlyingOut, InputStream underlyingIn, String userName, Map<String, String> saslProps, CallbackHandler callbackHandler, Token<BlockTokenIdentifier> accessToken) throws IOException {
        DataOutputStream out = new DataOutputStream(underlyingOut);
        DataInputStream in = new DataInputStream(underlyingIn);
        SaslParticipant sasl = SaslParticipant.createClientSaslParticipant(userName, saslProps, callbackHandler);
        out.writeInt(-559038737);
        out.flush();
        try {
            BlockTokenIdentifier blockTokenIdentifier = (BlockTokenIdentifier)accessToken.decodeIdentifier();
            if (blockTokenIdentifier != null) {
                byte[] handshakeSecret = ((BlockTokenIdentifier)accessToken.decodeIdentifier()).getHandshakeMsg();
                if (handshakeSecret == null || handshakeSecret.length == 0) {
                    LOG.debug("Handshake secret is null, sending without handshake secret.");
                    DataTransferSaslUtil.sendSaslMessage(out, new byte[0]);
                } else {
                    LOG.debug("Sending handshake secret.");
                    BlockTokenIdentifier identifier = new BlockTokenIdentifier();
                    identifier.readFields(new DataInputStream(new ByteArrayInputStream(accessToken.getIdentifier())));
                    String bpid = identifier.getBlockPoolId();
                    DataTransferSaslUtil.sendSaslMessageHandshakeSecret(out, new byte[0], handshakeSecret, bpid);
                }
            } else {
                LOG.debug("Block token id is null, sending without handshake secret.");
                DataTransferSaslUtil.sendSaslMessage(out, new byte[0]);
            }
            byte[] remoteResponse = DataTransferSaslUtil.readSaslMessage(in);
            byte[] localResponse = sasl.evaluateChallengeOrResponse(remoteResponse);
            ArrayList cipherOptions = null;
            String cipherSuites = this.conf.get("dfs.encrypt.data.transfer.cipher.suites");
            if (DataTransferSaslUtil.requestedQopContainsPrivacy(saslProps) && cipherSuites != null && !cipherSuites.isEmpty()) {
                if (!cipherSuites.equals(CipherSuite.AES_CTR_NOPADDING.getName())) {
                    throw new IOException(String.format("Invalid cipher suite, %s=%s", "dfs.encrypt.data.transfer.cipher.suites", cipherSuites));
                }
                CipherOption option = new CipherOption(CipherSuite.AES_CTR_NOPADDING);
                cipherOptions = Lists.newArrayListWithCapacity((int)1);
                cipherOptions.add(option);
            }
            DataTransferSaslUtil.sendSaslMessageAndNegotiationCipherOptions(out, localResponse, cipherOptions);
            SaslResponseWithNegotiatedCipherOption response = DataTransferSaslUtil.readSaslMessageAndNegotiatedCipherOption(in);
            localResponse = sasl.evaluateChallengeOrResponse(response.payload);
            assert (localResponse == null);
            DataTransferSaslUtil.checkSaslComplete(sasl, saslProps);
            CipherOption cipherOption = null;
            if (sasl.isNegotiatedQopPrivacy()) {
                cipherOption = DataTransferSaslUtil.unwrap(response.cipherOption, sasl);
                if (LOG.isDebugEnabled()) {
                    if (cipherOption == null) {
                        if (cipherSuites != null && !cipherSuites.isEmpty()) {
                            LOG.debug("Client accepts cipher suites {}, but server {} does not accept any of them", (Object)cipherSuites, (Object)addr.toString());
                        }
                    } else {
                        LOG.debug("Client using cipher suite {} with server {}", (Object)cipherOption.getCipherSuite().getName(), (Object)addr.toString());
                    }
                }
            }
            return cipherOption != null ? DataTransferSaslUtil.createStreamPair(this.conf, cipherOption, underlyingOut, underlyingIn, false) : sasl.createStreamPair(out, in);
        }
        catch (IOException ioe) {
            DataTransferSaslUtil.sendGenericSaslErrorMessage(out, ioe.getMessage());
            throw ioe;
        }
    }

    private static final class SaslClientCallbackHandler
    implements CallbackHandler {
        private final char[] password;
        private final String userName;

        public SaslClientCallbackHandler(String userName, char[] password) {
            this.password = password;
            this.userName = userName;
        }

        @Override
        public void handle(Callback[] callbacks) throws IOException, UnsupportedCallbackException {
            NameCallback nc = null;
            PasswordCallback pc = null;
            TextInputCallback rc = null;
            for (Callback callback : callbacks) {
                if (callback instanceof NameCallback) {
                    nc = (NameCallback)callback;
                    continue;
                }
                if (callback instanceof PasswordCallback) {
                    pc = (PasswordCallback)callback;
                    continue;
                }
                if (callback instanceof RealmCallback) {
                    rc = (RealmCallback)callback;
                    continue;
                }
                if (callback instanceof RealmChoiceCallback) continue;
                throw new UnsupportedCallbackException(callback, "Unrecognized SASL client callback");
            }
            if (nc != null) {
                nc.setName(this.userName);
            }
            if (pc != null) {
                pc.setPassword(this.password);
            }
            if (rc != null) {
                rc.setText(rc.getDefaultText());
            }
        }
    }
}

