package App::Manoc::Netwalker::Config;
#ABSTRACT: Configuration for Manoc Netwalker

use Moose;

our $VERSION = '2.99.3'; ##TRIAL VERSION

use namespace::autoclean;

use Moose::Util::TypeConstraints;
use App::Manoc::Utils::Datetime qw(str2seconds);
use Cwd;

subtype 'TimeInterval', as 'Int',
    where { $_ > 0 },
    message { "The number you provided, $_, was not a positive number" };

coerce 'TimeInterval', from 'Str', via { str2seconds($_) };


has manoc_config_dir => (
    is      => 'ro',
    isa     => 'Str',
    default => sub { getcwd() },
);


has n_procs => (
    is      => 'rw',
    isa     => 'Int',
    default => 1,
);



has default_vlan => (
    is      => 'rw',
    isa     => 'Int',
    default => 1,
);


has iface_filter => (
    is      => 'rw',
    isa     => 'Int',
    default => 1,
);


has ignore_portchannel => (
    is      => 'rw',
    isa     => 'Bool',
    default => 1,
);


has mat_force_vlan => (
    is      => 'rw',
    isa     => 'Maybe[Int]',
    default => undef,
);


has force_full_update => (
    is      => 'rw',
    isa     => 'Bool',
    default => 0,
);


has snmp_version => (
    is      => 'rw',
    isa     => 'Str',
    default => '2',
);


has snmp_community => (
    is      => 'rw',
    isa     => 'Str',
    default => 'public',
);


has control_port => (
    is      => 'rw',
    isa     => 'Str',
    default => '8001',
);


has remote_control => (
    is      => 'rw',
    isa     => 'Str',
    default => '127.0.0.1',
);


has refresh_interval => (
    is      => 'rw',
    isa     => 'TimeInterval',
    coerce  => 1,
    default => '10m',
);


has full_update_interval => (
    is      => 'rw',
    isa     => 'TimeInterval',
    coerce  => 1,
    default => '1h'
);


has config_update_interval => (
    is      => 'rw',
    isa     => 'TimeInterval',
    coerce  => 1,
    default => '1d',
);


has min_backoff_time => (
    is      => 'rw',
    isa     => 'TimeInterval',
    coerce  => 1,
    default => '5m',
);


has max_backoff_time => (
    is      => 'rw',
    isa     => 'TimeInterval',
    coerce  => 1,
    default => '30m',
);


has 'default_ssh_key' => (
    is      => 'rw',
    isa     => 'Str',
    lazy    => 1,
    builder => '_build_default_ssh_key',
);

sub _build_default_ssh_key {
    my $self = shift;

    my $basedir = $self->manoc_config_dir;
    foreach (qw( id_dsa id_ecdsa id_ed25519 id_rsa )) {
        my $file = File::Spec->catfile( $basedir, $_ );
        -f $file and return $file;
    }
}

no Moose;
__PACKAGE__->meta->make_immutable;

# Local Variables:
# mode: cperl
# indent-tabs-mode: nil
# cperl-indent-level: 4
# cperl-indent-parens-as-block: t
# End:

__END__

=pod

=head1 NAME

App::Manoc::Netwalker::Config - Configuration for Manoc Netwalker

=head1 VERSION

version 2.99.3

=head1 ATTRIBUTES

=head2 manoc_config_dir

used to construct default paths

=head2 n_procs

number of concurrent processes for workers

=head2 default_vlan

The default vlan ID to use when fetching ARP and mac address tables.

=head2 default_vlan

=head2 iface_filter

=head2 ignore_portchannel

=head2 mat_force_vlan

=head2 force_full_update

=head2 snmp_version

=head2 snmp_community

=head2 control_port

=head2 remote_control

=head2 refresh_interval

=head2 full_update_interval

=head2 config_update_interval

=head2 min_backoff_time

=head2 max_backoff_time

=head2 default_ssh_key

Default to id_dsa, id_ecdsa, id_ed25519 or id_rsa file on manoc config
dir.

=head1 AUTHORS

=over 4

=item *

Gabriele Mambrini <gmambro@cpan.org>

=item *

Enrico Liguori

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Gabriele Mambrini.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
