/* BEAST - Bedevilled Audio System
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bstrc.h"
#include	<stdarg.h>
#include        <fcntl.h>
#include        <sys/stat.h>
#include	<unistd.h>
#include	<string.h>

/* --- defines --- */


/* --- variables --- */
static GScanner	*bst_rc_scanner = NULL;
static GList   *first_args_list = NULL;
static GList   *last_args_list = NULL;
static	GScannerConfig	scanner_config =
{
  (
   " \t\n"
   )			/* cset_skip_characters */,
  (
   G_CSET_a_2_z
   "_#"
   G_CSET_A_2_Z
   )			/* cset_identifier_first */,
  (
   G_CSET_a_2_z
   "-+_0123456789"
   G_CSET_A_2_Z
   G_CSET_LATINS
   G_CSET_LATINC
   )			/* cset_identifier_nth */,
  ( ";\n" )		/* cpair_comment_single */,
  
  FALSE			/* case_sensitive */,
  
  TRUE			/* skip_comment_multi */,
  TRUE			/* skip_comment_single */,
  FALSE			/* scan_comment_multi */,
  TRUE			/* scan_identifier */,
  FALSE			/* scan_identifier_1char */,
  FALSE			/* scan_identifier_NULL */,
  TRUE			/* scan_symbols */,
  TRUE			/* scan_binary */,
  TRUE			/* scan_octal */,
  TRUE			/* scan_float */,
  TRUE			/* scan_hex */,
  FALSE			/* scan_hex_dollar */,
  TRUE			/* scan_string_sq */,
  TRUE			/* scan_string_dq */,
  TRUE			/* numbers_2_int */,
  TRUE			/* int_2_float */,
  FALSE			/* identifier_2_string */,
  TRUE			/* char_2_token */,
  FALSE			/* symbol_2_token */,
};


/* --- functions --- */
static void
bst_rc_init (void)
{
  if (!bst_rc_scanner)
    {
      bst_rc_scanner = g_scanner_new (&scanner_config);
    }
}

void
bst_rc_add_arg (BstRcUnion  *arg)
{
  g_return_if_fail (arg != NULL);
  g_return_if_fail (arg->type < BST_RC_LAST);
  bst_rc_init ();
  g_return_if_fail (g_scanner_lookup_symbol (bst_rc_scanner, arg->any.arg_name) == NULL);

  if (arg->type > BST_RC_NONE)
    {
      switch (arg->type)
	{
	case BST_RC_BOOL:
	  arg->abool.value = arg->abool.default_value;
	  break;
	case BST_RC_LONG:
	  arg->along.value = arg->along.default_value;
	  break;
	case BST_RC_DOUBLE:
	  arg->adouble.value = arg->adouble.default_value;
	  break;
	case BST_RC_STRING:
	  arg->astring.value = g_strdup (arg->astring.default_value);
	  if (!arg->astring.value)
	    arg->astring.value = g_strdup ("");
	  break;
	case BST_RC_STRING_LIST:
	  arg->astring_list.value = arg->astring_list.default_value;
	  break;
	case BST_RC_CALLBACK:
	  g_return_if_fail (arg->acallback.parse_arg != NULL);
	  break;
	default:
	  g_assert_not_reached ();
	  break;
	}
      arg->any.flags &= BST_RC_MASK;
      if (arg->any.flags & BST_RC_AUTOMATIC)
	last_args_list = g_list_prepend (last_args_list, arg);
      else
	first_args_list = g_list_prepend (first_args_list, arg);
      g_scanner_add_symbol (bst_rc_scanner, arg->any.arg_name, arg);
    }
  else
    g_scanner_add_symbol (bst_rc_scanner, arg->any.arg_name, NULL);
}

static gboolean
arg_has_default (BstRcUnion *arg)
{
  if (arg->type == BST_RC_CALLBACK &&
      arg->acallback.dump_arg == NULL)
    return TRUE;

  switch (arg->type)
    {
    case BST_RC_BOOL:
      return arg->abool.value == arg->abool.default_value;
    case BST_RC_LONG:
      return arg->along.value == arg->along.default_value;
    case BST_RC_DOUBLE:
      return arg->adouble.value == arg->adouble.default_value;
    case BST_RC_STRING:
      return g_str_equal (arg->astring.value, arg->astring.default_value);
    case BST_RC_STRING_LIST:
      return arg->astring_list.value == arg->astring_list.default_value;
    case BST_RC_CALLBACK:
      return (arg->acallback.has_default && arg->acallback.has_default (&arg->acallback));
    default:
      g_assert_not_reached ();
      return TRUE;
    }
}

BstRcUnion*
bst_rc_get_arg (const gchar    *arg_name)
{
  g_return_val_if_fail (arg_name != NULL, NULL);

  if (!bst_rc_scanner)
    return NULL;

  return g_scanner_lookup_symbol (bst_rc_scanner, arg_name);
}

static guint
parse_arg (GScanner *scanner,
	   BstRcUnion *arg)
{
  gboolean negate = FALSE;

  switch (arg->type)
    {
      GSList *slist;
      gdouble v;
      gchar *s;

    case BST_RC_BOOL:
      scanner->config->scan_symbols = FALSE;
      g_scanner_get_next_token (scanner);
      scanner->config->scan_symbols = TRUE;
      if (scanner->token == G_TOKEN_IDENTIFIER)
	{
	  s = scanner->value.v_identifier;
	  if (*s == '#')
	    s++;
	  arg->abool.value = (*s == 't' || *s == 'T' ||
			      *s == 'j' || *s == 'J' ||
			      *s == 'y' || *s == 'Y');
	}
      else if (scanner->token == G_TOKEN_FLOAT)
	arg->abool.value = scanner->value.v_float >= 0.5 || scanner->value.v_float <= -0.5;
      else
	return G_TOKEN_IDENTIFIER;
      break;

    case BST_RC_LONG:
      g_scanner_get_next_token (scanner);
      while (scanner->token == '-')
	{
	  negate = !negate;
	  g_scanner_get_next_token (scanner);
	}
      if (scanner->token != G_TOKEN_FLOAT)
	return G_TOKEN_INT;
      v = negate ? - scanner->value.v_float : scanner->value.v_float;
      arg->along.value = CLAMP (v, arg->along.minimum, arg->along.maximum);
      break;

    case BST_RC_DOUBLE:
      g_scanner_get_next_token (scanner);
      while (scanner->token == '-')
	{
	  negate = !negate;
	  g_scanner_get_next_token (scanner);
	}
      if (scanner->token != G_TOKEN_FLOAT)
	return G_TOKEN_FLOAT;
      v = negate ? - scanner->value.v_float : scanner->value.v_float;
      arg->adouble.value = CLAMP (v, arg->adouble.minimum, arg->adouble.maximum);
      break;

    case BST_RC_STRING:
      g_scanner_peek_next_token (scanner);
      if (scanner->next_token != G_TOKEN_STRING)
	{
	  g_scanner_get_next_token (scanner);
	  return G_TOKEN_STRING;
	}
      s = g_strdup ("");
      while (scanner->next_token == G_TOKEN_STRING)
	{
	  guint l;
	  gchar *o;

	  g_scanner_get_next_token (scanner);

	  l = strlen (s);
	  o = s;
	  s = g_strconcat (o, l > 0 ? " " : "", scanner->value.v_string, NULL);
	  g_free (o);

	  g_scanner_peek_next_token (scanner);
	}
      g_free (arg->astring.value);
      arg->astring.value = s;
      break;

    case BST_RC_STRING_LIST:
      slist = NULL;
      g_scanner_peek_next_token (scanner);
      if (scanner->next_token != G_TOKEN_STRING)
	{
	  g_scanner_get_next_token (scanner);
	  return G_TOKEN_STRING;
	}
      while (scanner->next_token == G_TOKEN_STRING)
	{
	  g_scanner_get_next_token (scanner);

	  if (scanner->value.v_string && scanner->value.v_string[0])
	    slist = g_slist_prepend (slist, g_strdup (scanner->value.v_string));

	  g_scanner_peek_next_token (scanner);
	}
      slist = g_slist_reverse (slist);
      if (arg->astring_list.value == arg->astring_list.default_value)
	arg->astring_list.value = NULL;
      arg->astring_list.value = g_slist_concat (arg->astring_list.value, slist);
      break;
      
    default:
      g_assert_not_reached ();
      break;
    }

  g_scanner_get_next_token (scanner);

  return scanner->token == ')' ? G_TOKEN_NONE : ')';
}

static void
parse_statement (GScanner            *scanner)
{
  guint expected_token;
  gboolean is_error = FALSE;
  guint parse_level = 0;
  
  g_scanner_get_next_token (scanner);
  if (scanner->token == '(')
    {
      parse_level = 1;
      g_scanner_get_next_token (scanner);
      
      if (scanner->token == G_TOKEN_SYMBOL)
	{
	  if (scanner->value.v_symbol)
	    {
	      BstRcUnion *arg;

	      arg = scanner->value.v_symbol;
	      if (arg->type == BST_RC_CALLBACK)
		expected_token = arg->acallback.parse_arg (scanner, &arg->acallback);
	      else
		expected_token = parse_arg (scanner, arg);
	    }
	  else
	    expected_token = G_TOKEN_NONE;
	}
      else
	expected_token = G_TOKEN_SYMBOL;
    }
  else
    {
      expected_token = '(';
      is_error = TRUE;
    }

  if (expected_token != G_TOKEN_NONE)
    {
      g_scanner_unexp_token (scanner, expected_token, NULL, "argument", NULL, NULL, is_error);
      
      /* skip rest of statement on errrors
       */
      if (parse_level)
	{
	  register guint level;

	  level = parse_level;
	  if (scanner->token == ')')
	    level--;
	  if (scanner->token == '(')
	    level++;
	  
	  while (!g_scanner_eof (scanner) && level > 0)
	    {
	      g_scanner_get_next_token (scanner);
	      
	      if (scanner->token == '(')
		level++;
	      else if (scanner->token == ')')
		level--;
	    }
	}
    }
}

void
bst_rc_parse (const gchar    *file_name)
{
  gint fd;

  g_return_if_fail (file_name != NULL);
  bst_rc_init ();

  fd = open (file_name, O_RDONLY);
  if (fd < 0)
    return;

  g_scanner_input_file (bst_rc_scanner, fd);
  bst_rc_scanner->parse_errors = 0;
  bst_rc_scanner->input_name = file_name;

  g_scanner_peek_next_token (bst_rc_scanner);

  while (bst_rc_scanner->next_token != G_TOKEN_EOF &&
	 bst_rc_scanner->parse_errors == 0)
    {
      parse_statement (bst_rc_scanner);

      g_scanner_peek_next_token (bst_rc_scanner);
    }

  bst_rc_scanner->input_name = NULL;
  g_scanner_input_text (bst_rc_scanner, "", 0);

  close (fd);
}

static void
dump_arg (FILE		*f_out,
	  BstRcUnion	*arg)
{
  gboolean dfl;

  dfl = arg_has_default (arg);
  
  if (dfl && !arg->any.flags & BST_RC_FORCE_DUMP)
    return;

  fputc ('\n', f_out);

  if (arg->any.description)
    {
      gchar *s;

      s = arg->any.description;
      while (s[0] != 0)
	{
	  gchar *p;

	  fputs ("; ", f_out);
	  p = strchr (s, '\n');
	  if (!p)
	    p = s + strlen (s);
	  fwrite (s, 1, p - s, f_out);
	  fputc ('\n', f_out);
	  
	  s = p + (*p == '\n' ? 1 : 0);
	}
    }

  if (dfl && ((arg->type == BST_RC_BOOL) ||
	      (arg->type == BST_RC_LONG) ||
	      (arg->type == BST_RC_DOUBLE) ||
	      (arg->type == BST_RC_STRING)))
    fputc (';', f_out);
  
  fputc ('(', f_out);
  fputs (arg->any.arg_name, f_out);
  switch (arg->type)
    {
      GSList *slist;

    case BST_RC_BOOL:
      fprintf (f_out, " #%c", arg->abool.value ? 't' : 'f');
      break;
    case BST_RC_LONG:
      fprintf (f_out, " %ld", arg->along.value);
      break;
    case BST_RC_DOUBLE:
      fprintf (f_out, " %f", arg->adouble.value);
      break;
    case BST_RC_STRING:
      fprintf (f_out, " \"%s\"", arg->astring.value ? arg->astring.value : "");
      break;
    case BST_RC_STRING_LIST:
      for (slist = arg->astring_list.value; slist; slist = slist->next)
	fprintf (f_out, " \"%s\"", (gchar*) slist->data);
      if (!arg->astring_list.value)
	fprintf (f_out, " \"\"");
      break;
    case BST_RC_CALLBACK:
      fputc (' ', f_out);
      arg->acallback.dump_arg (f_out, &arg->acallback, "  ");
      break;
    default:
      g_assert_not_reached ();
      break;
    }
  fputs (")\n", f_out);
}

void
bst_rc_dump (const gchar	*prg_name,
	     const gchar	*file_name)
{
  FILE *f_out;
  GList *list;

  g_return_if_fail (prg_name != NULL);
  g_return_if_fail (file_name != NULL);
  bst_rc_init ();

  f_out = fopen (file_name, "w");
  if (!f_out)
    return;

  fputs ("; ", f_out);
  fputs (prg_name, f_out);
  fputs (" rc-file         -*- scheme -*-\n", f_out);
  fputs ("; this file will automatically be rewritten at program shutdown\n", f_out);
  fputs (";\n", f_out);

  for (list = g_list_last (first_args_list); list; list = list->prev)
    dump_arg (f_out, list->data);

  if (last_args_list)
    {
      fputs ("\n\n; automatic data\n", f_out);
      fputs (";\n", f_out);
    }
  for (list = last_args_list; list; list = list->next)
    dump_arg (f_out, list->data);

  fclose (f_out);
}
