/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bsemain.h"
#include	"bsesong.h"
#include	"bsesequencer.h"
#include	"bsestream.h"
#include	"bsemixer.h"



/* --- variables --- */
static GSList *song_list = NULL;
static BseMixerValue *mix_buffer = NULL;
static BseSampleValue *clip_buffer = NULL;
static guint mix_buffer_n_values = 0;
static guint main_n_voices = 0;
static BseVoice *main_voices = NULL;
static GSList *stream_list = NULL;
static gboolean main_suspended = FALSE;


/* --- functions --- */
void
bse_main_add_output_stream (BseStream *stream)
{
  g_return_if_fail (stream != NULL);
  g_return_if_fail (stream->opened == TRUE);
  g_return_if_fail (stream->writable == TRUE);
  
  bse_stream_ref (stream);
  stream_list = g_slist_prepend (stream_list, stream);
  
  if (song_list)
    bse_stream_start (stream);
}

void
bse_main_remove_output_stream (BseStream *stream)
{
  GSList *slist;
  
  g_return_if_fail (stream != NULL);
  
  for (slist = stream_list; slist; slist = slist->next)
    {
      if (slist->data == stream)
	{
	  bse_stream_suspend (stream);
	  
	  bse_stream_unref (stream);
	  stream_list = g_slist_remove_link (stream_list, slist);
	  g_slist_free_1 (slist);
	  break;
	}
    }
}

void
bse_main_play_song (BseSong	   *song)
{
  g_return_if_fail (song != NULL);
  
  bse_main_play_song_with_ticks (song, NULL, NULL);
}

void
bse_main_play_song_with_ticks (BseSong	      *song,
			       BseSequencerTickCB callback,
			       gpointer	       callback_data)
{
  g_return_if_fail (song != NULL);
  
  if (song->sequencer)
    {
      g_warning ("BSE: play request for song \"%s\", which is already playing.",
		 song->name);
      return;
    }
  
  if (!song_list)
    {
      GSList *slist;
      
      /* initialize for playing
       */
      bse_mixer_init (2);
      clip_buffer = (BseSampleValue*) bse_mixer_get_buffer (0);
      mix_buffer = bse_mixer_get_buffer (1);
      mix_buffer_n_values = bse_mixer_get_n_buffer_values ();
      
      main_n_voices = 256; /* FIXME: BSE_CHANNEL_2_VOICE_FACTOR */
      main_voices = bse_voice_block_alloc (main_n_voices);
      
      for (slist = stream_list; slist; slist = slist->next)
	bse_stream_start (slist->data);
    }
  
  song_list = g_slist_append (song_list, song);
  bse_song_ref (song);
  
  bse_sequencer_start (song, callback, callback_data);
}

void
bse_main_quit_song (BseSong *song)
{
  GSList *slist;
  
  g_return_if_fail (song != NULL);
  
  for (slist = song_list; slist; slist = slist->next)
    {
      if (slist->data == song)
	{
	  bse_sequencer_stop (song);
	  
	  song_list = g_slist_remove_link (song_list, slist);
	  bse_song_unref (song);
	  
	  g_slist_free_1 (slist);
	  break;
	}
    }
  
  if (!song_list)
    {
      main_n_voices = 0;
      bse_voice_block_free (main_voices);
      main_voices = NULL;
      
      bse_mixer_shutdown ();
      clip_buffer = NULL;
      mix_buffer = NULL;
      mix_buffer_n_values = 0;
      
      for (slist = stream_list; slist; slist = slist->next)
	bse_stream_suspend (slist->data);
    }
}

void
bse_main_trigger_ticks (void)
{
  GSList *slist;
  
  for (slist = song_list; slist; slist = slist->next)
    {
      BseSong *song;
      
      song = slist->data;
      
      bse_sequencer_trigger_tick (song);
    }
}

void
bse_main_iteration (void)
{
  GSList *slsl;
  
  if (!song_list || main_suspended)
    return;
  
  for (slsl = song_list; slsl; slsl = slsl->next)
    {
      BseSong *song;
      GSList *slist;
      guint i;
      
      song = slsl->data;
      
      bse_sequencer_step (song, main_n_voices, main_voices);
      
      /* FIXME: this can only play one song!!!
       */
      bse_mixer_fill_buffer (mix_buffer, main_n_voices, main_voices);
      
      /* clip the values */
      for (i = 0; i < mix_buffer_n_values; i++)
	{
	  register BseMixerValue mix_v;
	  
	  mix_v = mix_buffer[i];
	  // clipping
	  if ( mix_v > 32767 )
	    mix_v = 0x7fff;
	  else if ( mix_v < -32768 )
	    mix_v = 0x8000;
	  clip_buffer[i] = mix_v;
	}
      
      for (slist = stream_list; slist; slist = slist->next)
	{
	  BseStream *stream;
	  
	  stream = slist->data;
	  bse_stream_write (stream, mix_buffer_n_values, clip_buffer);
	}
    }
}

gboolean
bse_main_need_iteration (void)
{
  GSList *slist;
  gboolean need_iteration;
  
  if (!song_list || !stream_list || main_suspended)
    return FALSE;
  
  need_iteration = FALSE;
  for (slist = stream_list; slist; slist = slist->next)
    {
      BseStream *stream;
      
      stream = slist->data;
      
      if (!bse_stream_would_block (stream, mix_buffer_n_values))
	{
	  need_iteration = TRUE;
	  break;
	}
    }
  
  return need_iteration;
}

void
bse_main_suspend (BseSong	 *song)
{
  if (!song_list)
    return;
  
  main_suspended = TRUE;
}

void
bse_main_continue (BseSong	  *song)
{
  main_suspended = FALSE;
}

gboolean
bse_main_can_block (void)
{
  GSList *slist;
  
  
  if (!song_list || !stream_list || main_suspended)
    return FALSE;
  
  for (slist = stream_list; slist; slist = slist->next)
    {
      BseStream *stream;
      
      stream = slist->data;
      
      if (stream->can_block)
	return TRUE;
    }
  
  return FALSE;
}
