/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include "bsemixer.h"



/* --- defines --- */
#define	MIXER_BUFFER_SIZE	(bse_mixer_n_ouput_channels * bse_mixer_n_values)


/* --- prototypes --- */
static void		bse_mixer_add_voice_to_buffer	(BseVoice	*voice,
							 BseMixerValue	*buffer);


/* --- variables --- */
static gboolean		 bse_mixer_is_active = FALSE;
static BseMixerValue	*bse_mixer_buffers = NULL;
static guint		 bse_mixer_n_buffers = 0;
static guint		 bse_mixer_n_values = BSE_DFL_N_MIXER_VALUES;
static guint		 bse_mixer_n_ouput_channels = BSE_DFL_N_MIXER_CHANNELS;
static guint		 bse_mixer_mix_freq = BSE_DFL_MIX_FREQ;


/* --- functions --- */
void
bse_mixer_init (guint n_buffers)
{
  g_return_if_fail (bse_initialized ());
  g_return_if_fail (bse_mixer_is_active == FALSE);
  g_return_if_fail (n_buffers > 0);

  bse_mixer_is_active = TRUE;
  bse_mixer_n_buffers = n_buffers;

  bse_mixer_buffers = g_new0 (BseMixerValue, n_buffers * MIXER_BUFFER_SIZE);
}

/* fuellt den angegebenen buffer mit den aktiven voices
 */
void
bse_mixer_fill_buffer (BseMixerValue  *buffer,
		       guint	       n_voices,
		       BseVoice	      *voices)
{
  guint    i;
  
  g_return_if_fail (bse_mixer_is_active != FALSE);
  g_return_if_fail (buffer != NULL);
  g_return_if_fail (n_voices > 0);
  g_return_if_fail (voices != NULL);
  
  /* clear buffer, we use memset() because this function is
   * most likely hand-optimized for the target machine
   */
  memset (buffer, 0, sizeof (BseMixerValue) * MIXER_BUFFER_SIZE);
  
  /* effekte etc. bearbeiten und evtl. stimmen aktivieren
   * anschliessend die voice zum ausgabebuffer addieren
   */
  
  for (i = 0; i < n_voices; ++i)
    {
      if (voices[i].active)
	{
	  bse_mixer_add_voice_to_buffer (&voices[i], buffer);
	}
    }
}

void
bse_mixer_shutdown (void)
{
  g_return_if_fail (bse_mixer_is_active == TRUE);

  bse_mixer_is_active = FALSE;
  g_free (bse_mixer_buffers);
  bse_mixer_buffers = NULL;
  bse_mixer_n_buffers = 0;
}

gboolean
bse_mixer_active (void)
{
  return bse_mixer_is_active;
}

void
bse_mixer_set_n_output_channels (guint          n_channels)
{
  g_return_if_fail (bse_mixer_is_active == FALSE);
  g_return_if_fail (n_channels >= BSE_MIN_N_MIXER_CHANNELS);
  g_return_if_fail (n_channels <= BSE_MAX_N_MIXER_CHANNELS);

  bse_mixer_n_ouput_channels = n_channels;
}

void
bse_mixer_set_mix_freq (guint          mixing_freq)
{
  g_return_if_fail (bse_mixer_is_active == FALSE);
  g_return_if_fail (mixing_freq >= BSE_MIN_MIX_FREQ);
  g_return_if_fail (mixing_freq <= BSE_MAX_MIX_FREQ);

  bse_mixer_mix_freq = mixing_freq;
}

void
bse_mixer_set_n_values (guint          n_values)
{
  g_return_if_fail (bse_mixer_is_active == FALSE);
  g_return_if_fail (n_values >= BSE_MIN_N_MIXER_VALUES);
  g_return_if_fail (n_values <= BSE_MAX_N_MIXER_VALUES);

  bse_mixer_n_values = n_values;
}

guint
bse_mixer_get_n_output_channels (void)
{
  return bse_mixer_n_ouput_channels;
}

guint
bse_mixer_get_mix_freq (void)
{
  return bse_mixer_mix_freq;
}

guint
bse_mixer_get_n_buffer_values (void)
{
  return MIXER_BUFFER_SIZE;
}

BseMixerValue*
bse_mixer_get_buffer (guint buffer_index)
{
  g_return_val_if_fail (bse_mixer_is_active == TRUE, NULL);
  g_return_val_if_fail (buffer_index < bse_mixer_n_buffers, NULL);
  
  return bse_mixer_buffers + buffer_index * MIXER_BUFFER_SIZE;
}

/* uebernimmt alle daten aus der instrumentstruktur */
/* in die voicestruktur und aktiviert die voice */
/* diese funktion kann unter anderem auch dazu verwendet */
/* werden um eine note ohne sequenzer anzuschlagen */
/* wird eine note ohne instrument angeschlagen d.h. */
/* der pointer zum instrument ist gleich NULL */
/* wird das instrument nicht neu angeschlagen */
void
bse_mixer_activate_voice (BseVoice *voice,
                          BseNote *note)
{
  BseInstrument *instrument = NULL;
  BseSample     *sample = NULL;
  BseMunk       *munk = NULL;
  
  g_return_if_fail (bse_mixer_is_active == TRUE);
  g_return_if_fail (voice != NULL);
  g_return_if_fail (note != NULL);
  
  instrument = note->instrument;
  
  /* nur wenn das instrument != NULL ist wird die note
     neu angeschlagen */ 
  if ( instrument )
    {
      /* uns interessieren nur die samplestimmen */
      if ( instrument->type == BSE_INSTRUMENT_SAMPLE )
	{
	  sample = instrument->sample;
	  bse_critical_assert( sample != NULL );
	  munk = &sample->munk[ voice->sample_munk_index ];
	  
	  /* sampleptr anhand des sampletypes setzten */
	  switch( sample->type )
	    {

	    case BSE_SAMPLE_NOTE_MUNKS:
	      // break;
	    case BSE_SAMPLE_EFFECT_MUNKS:
	    {
	      guint len = 0;
	      
	      voice->sample_pos = munk->values;
	      bse_critical_assert( voice->sample_pos != NULL );
	      
	      len = munk->n_values;
	      
	      voice->sample_end_pos  = voice->sample_pos + len;
	      voice->sample_pos_frac = 0;
	      voice->last_instrument = instrument;
	      voice->last_sample = sample;
	      
	      voice->volume = instrument->volume;
	      voice->balance = instrument->balance;
	      voice->active = TRUE;
	    }
	    break;
	      
	    default:
	      bse_critical_assert( NULL != NULL );
	      break;
	      
	    } /* switch */
	} /* BSE_INSTRUMENT_SAMPLE */
    } /* instrument != 0 */ 
  
  if ( voice->last_instrument )
    {
      sample = voice->last_instrument->sample;
      bse_critical_assert( sample != NULL );
      munk = &sample->munk[ voice->sample_munk_index ];
      
      if ( note->note != BSE_NOTE_VOID )
	{
	  gint note_rate;

	  note_rate = note->note + voice->last_instrument->transpose;
	  note_rate += BSE_KAMMER_NOTE - munk->recording_note;
	  note_rate = CLAMP (note_rate, BSE_MIN_NOTE, BSE_MAX_NOTE);
	  note_rate = bse_halftone_factor_table_fixed[ note_rate ];
	  voice->sample_rate = (float) note_rate *
	    ((float) sample->recording_freq /
	     (float) bse_mixer_mix_freq);
	}
      voice->sample_rate_offset = 0;
    }
}

static void
bse_mixer_add_voice_to_buffer (BseVoice		*voice,
			       BseMixerValue	*buffer)
{
  guint	   buffer_length;
  gint    rate;
  gint    rate_pos;
  gfloat   l_volume;
  gfloat   r_volume;
  BseMixerValue  *l_buffer;
  BseMixerValue  *r_buffer;
  gint16  *l_sample;
  gint16  *r_sample;
  
  bse_critical_assert (voice->last_sample != NULL);
  
  buffer_length = MIXER_BUFFER_SIZE;

  l_buffer = buffer;
  r_buffer = buffer + 1;
  rate = voice->sample_rate;
  rate_pos = voice->sample_pos_frac;
  l_sample = voice->sample_pos;
  r_sample = voice->sample_pos + 1;
  
  /* stereo position */
  if ( voice->balance < 0 )
    {
      l_volume = voice->volume / BSE_FULL_VOLUME_f;
      r_volume = ( voice->volume * ( ( BSE_FULL_VOLUME_f + voice->balance ) /
				     BSE_FULL_VOLUME_f) ) / BSE_FULL_VOLUME_f;
    }
  else
    {
      r_volume = voice->volume / BSE_FULL_VOLUME_f;
      l_volume = ( voice->volume * ( ( BSE_FULL_VOLUME_f - voice->balance ) /
				     BSE_FULL_VOLUME_f) ) / BSE_FULL_VOLUME_f;
    }

  /* hier kommen spaeter die aufrufe fuer die optimierten
   * routinen hin
   */
  
  /* mix voice to buffer
   */
  if ( voice->last_sample->n_channels == 1 )
    { 
      register guint i;
      register guint next_smp_word;

      /* mono samples */
      
      next_smp_word = rate >> 16;
      if (next_smp_word == 0)
	next_smp_word += 1;

      for( i = 0; i < buffer_length; i += bse_mixer_n_ouput_channels )
	{
	  register BseMixerValue pos_low;
	  register BseMixerValue pos_high;
	  register BseMixerValue  l_value;
	  register BseMixerValue  r_value;
	  
	  pos_low  = rate_pos & 0xffff;
	  pos_high = rate_pos >> 16;

	  /* old version
	   * sample_value = l_sample[ rate_pos>>16 ];
	   */

	  l_value =
	    ((BseMixerValue) l_sample[ pos_high ]) * (0x00010000 - pos_low) +
	    ((BseMixerValue) l_sample[ pos_high + next_smp_word ]) * pos_low;
	  r_value = l_value;

	  /* shift *atfer* the volume multiplication
	   */
	  l_value *= l_volume;
	  r_value *= r_volume;
	  l_value >>= 16;
	  r_value >>= 16;

	  l_buffer[i] += l_value;
	  r_buffer[i] += r_value;

	  rate_pos += rate;
	}
      voice->sample_pos += rate_pos >> 16;
      voice->sample_pos_frac = rate_pos & 0xffff;
    }
  else /* we rely on BSE_MAX_SAMPLE_CHANNELS==2 here */
    {
      register guint i;
      register guint next_smp_word;
      
      /* stereo samples */
      
      next_smp_word = rate >> 16;
      if (next_smp_word == 0)
	next_smp_word += 1;
      next_smp_word *= 2;
      
      for( i = 0; i < buffer_length; i += bse_mixer_n_ouput_channels )
	{
	  register BseMixerValue pos_low;
	  register BseMixerValue pos_high;
	  register BseMixerValue l_value;
	  register BseMixerValue r_value;

	  pos_low  = rate_pos & 0xffff;
	  pos_high = rate_pos >> 16;

	  /* old version
	   *  sample_idx = (rate_pos>>16)<<1;
	   *  l_buffer[i] += l_sample[ sample_idx ] * l_volume;
	   *  r_buffer[i] += r_sample[ sample_idx ] * r_volume;
	   */

	  l_value =
	    ((BseMixerValue) l_sample[ pos_high ]) * (0x00010000 - pos_low) +
	    ((BseMixerValue) l_sample[ pos_high + next_smp_word ]) * pos_low;
	  r_value =
	    ((BseMixerValue) l_sample[ pos_high ]) * (0x00010000 - pos_low) +
	    ((BseMixerValue) l_sample[ pos_high + next_smp_word ]) * pos_low;

	  /* shift *after* the volume multiply
	   */
	  l_value *= l_volume;
	  r_value *= r_volume;
	  l_value >>= 16;
	  r_value >>= 16;

	  l_buffer[i] += l_value;
	  r_buffer[i] += r_value;

	  rate_pos += rate;
	}
      voice->sample_pos += (rate_pos >> 16) << 1;
      voice->sample_pos_frac = rate_pos & 0xffff;
    }
  
  if ( voice->sample_pos > voice->sample_end_pos )
    {
      bse_voice_reset (voice);
    }
}
