/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bsebase.h"
#include	"bsepcmstream.h"
#include	<string.h>
#include	<sys/soundcard.h>
#include	<sys/ioctl.h>
#include	<unistd.h>
#include	<errno.h>
#include	<fcntl.h>



/* --- stream prototypes --- */
static BseErrorType bse_pcm_stream_open		   (BseStream	   *stream,
						    gboolean	    read_access,
						    gboolean	    write_access);
static void	    bse_pcm_stream_close	   (BseStream	   *stream);
static BseErrorType bse_pcm_stream_start	   (BseStream	   *stream);
static BseErrorType bse_pcm_stream_suspend	   (BseStream	   *stream);
static gboolean	    bse_pcm_stream_would_block	   (BseStream	   *stream,
						    guint	    n_values);
static BseErrorType bse_pcm_stream_read		   (BseStream	   *stream,
						    guint	    n_values,
						    BseSampleValue *values);
static BseErrorType bse_pcm_stream_write	   (BseStream	   *stream,
						    guint	    n_values,
						    BseSampleValue *values);
static BseErrorType bse_pcm_stream_set_attribs	   (BseStream	   *stream,
						    BseStreamAttribMask mask,
						    BseStreamAttribs *attribs);


/* --- structures --- */
struct _BseStreamPrivate
{
  gint	fd;
  guint block_size;
};


/* --- prototypes --- */
static BseErrorType	devdsp_open	(BseStream	*stream);
static BseErrorType	devdsp_set	(BseStream	*stream,
					 BseStreamAttribMask attrib_mask,
					 BseStreamAttribs *attribs);

/* --- variables --- */
static const BseStreamAttribMask BSE_PCM_SA_MASK = (BSE_SA_N_CHANNELS |
						    BSE_SA_PLAY_FREQUENCY |
						    BSE_SA_REC_FREQUENCY |
						    BSE_SA_FRAGMENT_SIZE);


/* --- functions --- */
BseStream*
bse_pcm_stream_new (void)
{
  BseStream *stream;
  
  stream = bse_null_stream_new ();

  stream->type_name = "PCM";
  
  stream->opened = FALSE;
  stream->readable = FALSE;
  stream->writable = TRUE;
  stream->suspended = TRUE;
  stream->can_block = TRUE;
  
  stream->max_channels = BSE_MAX_SAMPLE_CHANNELS;
  stream->min_play_frequency = 8000;
  stream->max_play_frequency = 48000;
  stream->min_record_frequency = 8000;
  stream->max_record_frequency = 48000;
  stream->min_fragment_size = 128;
  stream->max_fragment_size = 8192;
  
  stream->attribs.n_channels = stream->max_channels;
  stream->attribs.play_frequency = stream->max_play_frequency;
  stream->attribs.record_frequency = stream->max_record_frequency;
  stream->attribs.fragment_size = 1 << bse_bit_storage (MAX (bse_get_obuffer_size (), 256) - 1);
  
  stream->open = bse_pcm_stream_open;
  stream->close = bse_pcm_stream_close;
  stream->start = bse_pcm_stream_start;
  stream->suspend = bse_pcm_stream_suspend;
  stream->would_block = bse_pcm_stream_would_block;
  stream->read = bse_pcm_stream_read;
  stream->write = bse_pcm_stream_write;
  stream->set_attribs = bse_pcm_stream_set_attribs;
  
  return stream;
}

gchar*
bse_pcm_stream_get_default_device (void)
{
  return "/dev/dsp";
}

static BseErrorType
bse_pcm_stream_open (BseStream	    *stream,
		     gboolean	     read_access,
		     gboolean	     write_access)
{
  BseErrorType error;
  
  if (read_access)
    return BSE_ERROR_STREAM_READ_DENIED;
  if (!write_access)
    return BSE_ERROR_STREAM_INTERNAL;
  
  stream->readable = FALSE;
  stream->writable = TRUE;
  stream->can_block = TRUE;
  
  stream->attribs.n_channels = CLAMP (stream->attribs.n_channels, 1, stream->max_channels);
  stream->attribs.play_frequency = CLAMP (stream->attribs.play_frequency,
					  stream->min_play_frequency,
					  stream->max_play_frequency);
  stream->attribs.record_frequency = CLAMP (stream->attribs.record_frequency,
					    stream->min_record_frequency,
					    stream->max_record_frequency);
  stream->attribs.fragment_size = CLAMP (stream->attribs.fragment_size,
					 stream->min_fragment_size,
					 stream->max_fragment_size) & 0xfff0;
  
  stream->private = g_new0 (BseStreamPrivate, 1);
  stream->private->fd = -1;
  
  error = devdsp_open (stream);
  
  if (!error)
    error = bse_pcm_stream_set_attribs (stream,
					BSE_PCM_SA_MASK,
					&stream->attribs);
  if (error != BSE_ERROR_NONE)
    {
      g_free (stream->private);
      stream->private = NULL;
    }
  
  errno = 0;
  return error;
}

static void
bse_pcm_stream_close (BseStream	     *stream)
{
  (void) ioctl (stream->private->fd, SNDCTL_DSP_RESET);
  
  (void) close (stream->private->fd);
  
  g_free (stream->private);
  stream->private = NULL;

  errno = 0;
}

static BseErrorType
bse_pcm_stream_start (BseStream	     *stream)
{
  static BseSampleValue *zeros = NULL;
  static guint z_size = 0;
  gint fd;
  
  /* this is sick!
   * we need to put an amount of zeros into the soundcard prior to atual
   * playing to avoid an extra click-noise.
   * this is with GUS-MAX and the ultra-driver from jaroslav.
   */
  
  if (!zeros)
    {
      z_size = bse_get_obuffer_size ();
      zeros = g_new0 (BseSampleValue, z_size);
    }
  
  fd = stream->private->fd;
  
  (void) ioctl (fd, SNDCTL_DSP_RESET);
  
  while (!bse_pcm_stream_would_block (stream, z_size))
    write (fd, zeros, z_size);
  
  errno = 0;
  return BSE_ERROR_NONE;
}

static BseErrorType
bse_pcm_stream_suspend (BseStream      *stream)
{
  /* (void) ioctl (stream->private->fd, SNDCTL_DSP_SYNC); */
  (void) ioctl (stream->private->fd, SNDCTL_DSP_RESET);
  
  errno = 0;
  return BSE_ERROR_NONE;
}

static BseErrorType
bse_pcm_stream_read (BseStream	    *stream,
		     guint	     n_values,
		     BseSampleValue *values)
{
  memset (values, 0, sizeof (BseSampleValue) * n_values);
  
  errno = 0;
  return BSE_ERROR_NONE;
}

static gboolean
bse_pcm_stream_would_block (BseStream	   *stream,
			    guint	    n_values)
{
  static guint debug = 0;
  audio_buf_info info;
  
  (void) ioctl (stream->private->fd, SNDCTL_DSP_GETOSPACE, &info);
  
  if (debug)
    {
      g_print ("bse_pcm_stream_would_block: fragstotal=%d, fragsize=%d, fragments=%d, would_block=%d\n",
	       info.fragstotal,
	       info.fragsize,
	       info.fragments,
	       info.fragsize * info.fragments < n_values);
      debug = 1;
    }
  
  errno = 0;
  return info.fragsize * info.fragments < n_values * sizeof (BseSampleValue);
}

static BseErrorType
bse_pcm_stream_write (BseStream	     *stream,
		      guint	      n_values,
		      BseSampleValue *values)
{
  do
    {
      write (stream->private->fd, values, n_values * sizeof (BseSampleValue));
      if (errno == EAGAIN)
	g_warning ("BSE: PCM device is busy when blocking?");
    }
  while (errno == EAGAIN);
  
  errno = 0;
  return BSE_ERROR_NONE;
}

static BseErrorType
bse_pcm_stream_set_attribs (BseStream	*stream,
			    BseStreamAttribMask mask,
			    BseStreamAttribs *attribs)
{
  BseErrorType error;
  
  mask &= BSE_PCM_SA_MASK;
  
  error = devdsp_set (stream, mask, attribs);
  
  if (attribs != &stream->attribs)
    g_memmove (attribs, &stream->attribs, sizeof (*attribs));
  
  errno = 0;
  return error;
}

static BseErrorType
devdsp_open (BseStream	*stream)
{
  register gint fd;
  int	d_int;
  
  stream->readable = FALSE;
  stream->writable = TRUE;
  fd = open (stream->name, O_WRONLY);
  if (fd < 0)
    {
      if (errno == EBUSY)
	return BSE_ERROR_STREAM_DEVICE_BUSY;
      else
	return BSE_ERROR_STREAM_IO;
    }
  
  d_int = stream->private->block_size;
  if (ioctl (fd, SNDCTL_DSP_GETBLKSIZE, &d_int) < 0 ||
      d_int < 1024 ||
      d_int > 131072 ||
      d_int != (d_int & 0xffffffe))
    {
      close (fd);
      return BSE_ERROR_STREAM_INTERNAL;
    }
  stream->private->block_size = d_int;
  
  d_int = AFMT_S16_LE;
  if (ioctl (fd, SNDCTL_DSP_GETFMTS, &d_int) < 0 ||
      (d_int & AFMT_S16_LE) != AFMT_S16_LE)
    {
      /* audio format not supported
       */
      close (fd);
      return BSE_ERROR_STREAM_INTERNAL;
    }
  
  d_int = AFMT_S16_LE;
  if (ioctl (fd, SNDCTL_DSP_SETFMT, &d_int) < 0 ||
      d_int != AFMT_S16_LE)
    {
      /* failed to set audio format
       */
      close (fd);
      return BSE_ERROR_STREAM_INTERNAL;
    }
  
  stream->private->fd = fd;
  
  return BSE_ERROR_NONE;
}

static BseErrorType
devdsp_set (BseStream		*stream,
	    BseStreamAttribMask	attrib_mask,
	    BseStreamAttribs	*attribs)
{
  int d_int;
  
  if (attrib_mask & BSE_SA_N_CHANNELS)
    {
      attribs->n_channels = CLAMP (attribs->n_channels, 1, stream->max_channels);
      d_int = attribs->n_channels - 1;
      if (ioctl (stream->private->fd, SNDCTL_DSP_STEREO, &d_int) < 0)
	{
	  /* failed to set audio format
	   */
	  return BSE_ERROR_STREAM_SET_ATTRIB;
	}
      d_int++;
      stream->attribs.n_channels = d_int;
      if (attribs->n_channels > d_int)
	stream->max_channels = d_int;
    }
  
  if (attrib_mask & BSE_SA_PLAY_FREQUENCY ||
      attrib_mask & BSE_SA_REC_FREQUENCY)
    {
      guint freq;
      
      freq = 0;
      if (attrib_mask & BSE_SA_PLAY_FREQUENCY)
	{
	  attribs->play_frequency = CLAMP (attribs->play_frequency,
					   stream->min_play_frequency,
					   stream->max_play_frequency);
	  freq = attribs->play_frequency;
	}
      if (attrib_mask & BSE_SA_REC_FREQUENCY)
	{
	  attribs->record_frequency = CLAMP (attribs->record_frequency,
					     stream->min_record_frequency,
					     stream->max_record_frequency);
	  freq = MAX (freq, attribs->record_frequency);
	}
      d_int = freq;
      if (ioctl (stream->private->fd, SNDCTL_DSP_SPEED, &d_int) < 0)
	{
	  /* failed to set audio format
	   */
	  return BSE_ERROR_STREAM_SET_ATTRIB;
	}
      stream->attribs.play_frequency = d_int;
      stream->attribs.record_frequency = d_int;
      printf ("DEBUG(linuxpcm): play_freq: %d\n", d_int);
    }
  
  if (attrib_mask & BSE_SA_FRAGMENT_SIZE)
    {
      /* Note: fragment = block_count << 16;
       *       fragment |= bse_bit_storage (size - 1);
       */
      attribs->fragment_size = CLAMP (attribs->fragment_size,
				      stream->min_fragment_size,
				      stream->max_fragment_size);
      attribs->fragment_size &= 0xfff0;
      
      d_int = (1024 << 16) | bse_bit_storage (attribs->fragment_size - 1);
      if (ioctl (stream->private->fd, SNDCTL_DSP_SETFRAGMENT, &d_int) < 0)
	{
	  /* failed to set audio format
	   */
	  return BSE_ERROR_STREAM_SET_ATTRIB;
	}
      stream->attribs.fragment_size = 1 << (d_int & 0xffff);
    }

  if (1)
    {
      long block;
      
      if (1)
	block = 0;
      else
	block = O_NONBLOCK;
      (void) fcntl (stream->private->fd, F_SETFL, block);
      stream->can_block = (block == 0);
    }
  
  return BSE_ERROR_NONE;
}
