/* CSL - Common Sound Layer
 * Copyright (C) 2001 Jeff Tranter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * This example illustrates reading and writing in full duplex
 * mode. It creates input and output streams, then spawns two threads,
 * one to handle i/o for each stream.  The underlying sound driver
 * (OSS or aRts) must support full duplex.
 */

#define _REENTRANT
#include <pthread.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <csl/csl.h>

const int size = 1024;
int len;
char *buffer;
CslPcmStream *istream, *ostream;

/* writes data from buffer to stream */
static void writer()
{
	int wp, rc;

	for (wp = 0; wp < len; wp += size)
	{
		fprintf(stderr, "W");
		rc = csl_pcm_write(ostream, size, &buffer[wp]);
		if (rc != size)
			csl_warning("short write of %d bytes", rc);
	}
}


/* read data from stream to buffer */
static void reader()
{
	int rp, rc;

	for (rp = 0; rp < len; rp += size)
	{
		fprintf(stderr, "R");
		rc = csl_pcm_read(istream, size, &buffer[rp]);
		if (rc != size)
			csl_warning("short read of %d bytes", rc);
	}
}

int main (int argc, char **argv)
{
	CslErrorType error;
	CslDriver *driver;
	CslOptions options;
	CslMutex cmutex; /* CSL mutex */
	pthread_t thr1, thr2; /* threads */
	pthread_mutex_t pmutex = PTHREAD_MUTEX_INITIALIZER; /* pthread mutex */

	cmutex.lock = (CslMutexLock) &pthread_mutex_lock;
	cmutex.unlock = (CslMutexUnlock) &pthread_mutex_unlock;
	cmutex.destroy = (CslMutexDestroy) &pthread_mutex_destroy;
	cmutex.user_data = &pmutex;

	csl_options_parse (&options, &argc, &argv);
	csl_set_debug_mask (options.debug_flags);

	/* make a buffer big enough for 10 seconds of audio assuming 16-bit data */
	len = 10 * options.rate * options.n_channels * 2;
	buffer = (char *) csl_malloc(len);

	fprintf(stderr, "Data format: %d channels, %d samples/sec, %s\n",
			options.n_channels, options.rate, csl_describe_pcm_format(options.pcm_format));

	/* init driver */
	error = csl_driver_init_mutex(NULL, CSL_DRIVER_CAP_PCM, &cmutex, &driver); 
	if (error)
		csl_error ("unable to initialize driver: %s", csl_strerror (error));
	
	/* open PCM input and output streams */
	error = csl_pcm_open_input (driver, "cslduplex-in", options.rate, options.n_channels, options.pcm_format, &istream);
	if (error)
		csl_error ("failed to open input device: %s", csl_strerror (error));
	error = csl_pcm_open_output (driver, "cslduplex-out", options.rate, options.n_channels, options.pcm_format, &ostream);
	if (error)
		csl_error ("failed to open output device: %s", csl_strerror (error));

    /* create two threads and start them */
	printf("Main: Creating 2 threads\n");
	pthread_create(&thr1, NULL, (void*)reader, NULL);
	/* Give the reader process a head start */
	sleep(1);
	pthread_create(&thr2, NULL, (void*)writer, NULL);
 
    /* wait for threads to complete */
	pthread_join(thr1, NULL);
	pthread_join(thr2, NULL);

	fprintf(stderr, "\nDone\n");

	/* clean up */
	csl_pcm_close (istream);
	csl_pcm_close (ostream);
	csl_driver_shutdown (driver);
	csl_free(buffer);

	return 0;
}
