static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Copyright 1996 Spencer Kimball                                             \n"
" * Copyright 1997 Elliot Lee                                                  \n"
" * Copyright 2013 Téo Mazars <teo.mazars@ensimag.fr>                         \n"
" */                                                                           \n"
"                                                                              \n"
"/*  Mosaic is a filter which transforms an image into                         \n"
" *  what appears to be a mosaic, composed of small primitives,                \n"
" *  each of constant color and of an approximate size.                        \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"enum_start (gegl_mosaic_tile)                                                 \n"
"  enum_value (GEGL_MOSAIC_TILE_SQUARES,   \"squares\",   N_(\"Squares\"))     \n"
"  enum_value (GEGL_MOSAIC_TILE_HEXAGONS,  \"hexagons\",  N_(\"Hexagons\"))    \n"
"  enum_value (GEGL_MOSAIC_TILE_OCTAGONS,  \"octagons\",  N_(\"Octagons\"))    \n"
"  enum_value (GEGL_MOSAIC_TILE_TRIANGLES, \"triangles\", N_(\"Triangles\"))   \n"
"enum_end (GeglMosaicTile)                                                     \n"
"                                                                              \n"
"property_enum (tile_type, _(\"Tile geometry\"),                               \n"
"    GeglMosaicTile, gegl_mosaic_tile, GEGL_MOSAIC_TILE_HEXAGONS)              \n"
"    description (_(\"What shape to use for tiles\"))                          \n"
"                                                                              \n"
"property_double (tile_size, _(\"Tile size\"), 15.0)                           \n"
"    description (_(\"Average diameter of each tile (in pixels)\"))            \n"
"    value_range (1.0, 1000.0)                                                 \n"
"    ui_range    (5.0, 400.0)                                                  \n"
"    ui_meta     (\"unit\", \"pixel-distance\")                                \n"
"                                                                              \n"
"property_double (tile_height, _(\"Tile height\"), 4.0)                        \n"
"    description (_(\"Apparent height of each tile (in pixels)\"))             \n"
"    value_range (1.0, 1000.0)                                                 \n"
"    ui_range    (1.0, 20.0)                                                   \n"
"                                                                              \n"
"property_double (tile_neatness, _(\"Tile neatness\"), 0.65)                   \n"
"    description (_(\"Deviation from perfectly formed tiles\"))                \n"
"    value_range (0.0, 1.0)                                                    \n"
"                                                                              \n"
"property_double (color_variation, _(\"Tile color variation\"), 0.2)           \n"
"    description ((\"Magnitude of random color variations\"))                  \n"
"    value_range (0.0, 1.0)                                                    \n"
"                                                                              \n"
"property_boolean (color_averaging, _(\"Color averaging\"), TRUE)              \n"
"    description (_(\"Tile color based on average of subsumed pixels\"))       \n"
"                                                                              \n"
"property_boolean (tile_surface, _(\"Rough tile surface\"), FALSE)             \n"
"    description (_(\"Surface characteristics\"))                              \n"
"                                                                              \n"
"property_boolean (tile_allow_split, _(\"Allow splitting tiles\"), TRUE)       \n"
"    description (_(\"Allows splitting tiles at hard edges\"))                 \n"
"                                                                              \n"
"property_double (tile_spacing, _(\"Tile spacing\"), 1.0)                      \n"
"    description (_(\"Inter-tile spacing (in pixels)\"))                       \n"
"    value_range (0.0, 1000.0)                                                 \n"
"    ui_range    (0.5, 30.0)                                                   \n"
"    ui_meta     (\"unit\", \"pixel-distance\")                                \n"
"                                                                              \n"
"property_color (joints_color, _(\"Joints color\"), \"black\")                 \n"
"                                                                              \n"
"property_color (light_color, _(\"Light color\"), \"white\")                   \n"
"                                                                              \n"
"property_double (light_dir, _(\"Light direction\"), 135.0)                    \n"
"    description ((\"Direction of light-source (in degrees)\"))                \n"
"    value_range (0.0, 360.0)                                                  \n"
"    ui_meta     (\"unit\", \"degree\")                                        \n"
"                                                                              \n"
"property_boolean (antialiasing, _(\"Antialiasing\"), TRUE)                    \n"
"    description  (_(\"Enables smoother tile output\"))                        \n"
"                                                                              \n"
"property_seed (seed, _(\"Random seed\"), rand)                                \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_AREA_FILTER                                                   \n"
"#define GEGL_OP_NAME     mosaic                                               \n"
"#define GEGL_OP_C_SOURCE mosaic.c                                             \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"#include <math.h>                                                             \n"
"                                                                              \n"
"#define SUPERSAMPLE       3                                                   \n"
"#define MAG_THRESHOLD     (7.5/255.0)                                         \n"
"#define COUNT_THRESHOLD   0.1                                                 \n"
"#define MAX_POINTS        12                                                  \n"
"#define NB_CPN            4                                                   \n"
"#define STD_DEV           1.0                                                 \n"
"                                                                              \n"
"typedef enum                                                                  \n"
"{                                                                             \n"
"  SQUARES   = 0,                                                              \n"
"  HEXAGONS  = 1,                                                              \n"
"  OCTAGONS  = 2,                                                              \n"
"  TRIANGLES = 3                                                               \n"
"} TileType;                                                                   \n"
"                                                                              \n"
"#define SMOOTH   FALSE                                                        \n"
"#define ROUGH    TRUE                                                         \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  gdouble x, y;                                                               \n"
"} Vertex;                                                                     \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  guint  npts;                                                                \n"
"  Vertex pts[MAX_POINTS];                                                     \n"
"} Polygon;                                                                    \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  gdouble base_x, base_y;                                                     \n"
"  gdouble base_x2, base_y2;                                                   \n"
"  gdouble norm_x, norm_y;                                                     \n"
"  gdouble light;                                                              \n"
"} SpecVec;                                                                    \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  Vertex *vert, *vert_o;                                                      \n"
"  gint    vert_rows;                                                          \n"
"  gint    vert_cols;                                                          \n"
"  gint    row_pad;                                                            \n"
"  gint    col_pad;                                                            \n"
"  gint    vert_multiple;                                                      \n"
"  gint    vert_rowstride;                                                     \n"
"} GridDescriptor;                                                             \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  gdouble        light_x;                                                     \n"
"  gdouble        light_y;                                                     \n"
"  gdouble        scale;                                                       \n"
"  gfloat        *h_grad;                                                      \n"
"  gfloat        *v_grad;                                                      \n"
"  gfloat        *m_grad;                                                      \n"
"  GridDescriptor grid;                                                        \n"
"  gfloat         back[4];                                                     \n"
"  gfloat         fore[4];                                                     \n"
"  gint           width, height;                                               \n"
"} MosaicDatas;                                                                \n"
"                                                                              \n"
"                                                                              \n"
"/* Declare local functions.                                                   \n"
" */                                                                           \n"
"                                                                              \n"
"static gfloat*   mosaic                (GeglOperation       *operation,       \n"
"                                        GeglBuffer          *drawable,        \n"
"                                        const GeglRectangle *result);         \n"
"                                                                              \n"
"/*  gradient finding machinery  */                                            \n"
"static void      find_gradients        (gfloat              *input_buf,       \n"
"                                        gdouble              std_dev,         \n"
"                                        const GeglRectangle *result,          \n"
"                                        MosaicDatas         *mdatas);         \n"
"static void      find_max_gradient     (gfloat              *src_rgn,         \n"
"                                        gfloat              *dest_rgn,        \n"
"                                        gint                 width,           \n"
"                                        gint                 height);         \n"
"                                                                              \n"
"/*  gaussian & 1st derivative  */                                             \n"
"static void      gaussian_deriv        (gfloat              *src_rgn,         \n"
"                                        gfloat              *dest_rgn,        \n"
"                                        GeglOrientation      direction,       \n"
"                                        gdouble              std_dev,         \n"
"                                        const GeglRectangle *result);         \n"
"static void      make_curve            (gfloat              *curve,           \n"
"                                        gfloat              *sum,             \n"
"                                        gdouble              std_dev,         \n"
"                                        gint                 length);         \n"
"static void      make_curve_d          (gfloat              *curve,           \n"
"                                        gfloat              *sum,             \n"
"                                        gdouble              std_dev,         \n"
"                                        gint                 length);         \n"
"                                                                              \n"
"/*  grid creation and localization machinery  */                              \n"
"static void      grid_create_squares   (const GeglRectangle *result,          \n"
"                                        gdouble              tile_size,       \n"
"                                        GridDescriptor      *grid);           \n"
"static void      grid_create_hexagons  (const GeglRectangle *result,          \n"
"                                        gdouble              tile_size,       \n"
"                                        GridDescriptor      *grid);           \n"
"static void      grid_create_octagons  (const GeglRectangle *result,          \n"
"                                        gdouble              tile_size,       \n"
"                                        GridDescriptor      *grid);           \n"
"static void      grid_create_triangles (const GeglRectangle *result,          \n"
"                                        gdouble              tile_size,       \n"
"                                        GridDescriptor      *grid);           \n"
"static void      grid_localize         (const GeglRectangle *result,          \n"
"                                        GeglProperties          *o,           \n"
"                                        MosaicDatas         *mdatas);         \n"
"static gfloat*   grid_render           (gfloat              *input_buf,       \n"
"                                        const GeglRectangle *result,          \n"
"                                        GeglProperties          *o,           \n"
"                                        MosaicDatas         *mdatas);         \n"
"static void      split_poly            (Polygon             *poly,            \n"
"                                        gfloat              *input_buf,       \n"
"                                        gfloat              *output_buf,      \n"
"                                        gdouble             *dir,             \n"
"                                        gdouble              color_vary,      \n"
"                                        const GeglRectangle *result,          \n"
"                                        GeglProperties          *o,           \n"
"                                        MosaicDatas         *mdatas);         \n"
"static void      clip_poly             (gdouble             *vec,             \n"
"                                        gdouble             *pt,              \n"
"                                        Polygon             *poly,            \n"
"                                        Polygon             *new_poly);       \n"
"static void      clip_point            (gdouble             *dir,             \n"
"                                        gdouble             *pt,              \n"
"                                        gdouble              x1,              \n"
"                                        gdouble              y1,              \n"
"                                        gdouble              x2,              \n"
"                                        gdouble              y2,              \n"
"                                        Polygon             *poly);           \n"
"static void      process_poly          (Polygon             *poly,            \n"
"                                        gboolean             allow_split,     \n"
"                                        gfloat              *input_buf,       \n"
"                                        gfloat              *output_buf,      \n"
"                                        const GeglRectangle *result,          \n"
"                                        GeglProperties          *o,           \n"
"                                        MosaicDatas         *mdatas);         \n"
"static void      render_poly           (Polygon             *poly,            \n"
"                                        gfloat              *input_buf,       \n"
"                                        gfloat              *output_buf,      \n"
"                                        gdouble              vary,            \n"
"                                        const GeglRectangle *result,          \n"
"                                        GeglProperties          *o,           \n"
"                                        MosaicDatas         *mdatas);         \n"
"static void      find_poly_dir         (Polygon             *poly,            \n"
"                                        gfloat              *m_gr,            \n"
"                                        gfloat              *h_gr,            \n"
"                                        gfloat              *v_gr,            \n"
"                                        gdouble             *dir,             \n"
"                                        gdouble             *loc,             \n"
"                                        const GeglRectangle *result);         \n"
"static void      find_poly_color       (Polygon             *poly,            \n"
"                                        gfloat              *input_buf,       \n"
"                                        gfloat              *col,             \n"
"                                        double               vary,            \n"
"                                        const GeglRectangle *result);         \n"
"static void      scale_poly            (Polygon             *poly,            \n"
"                                        gdouble              cx,              \n"
"                                        gdouble              cy,              \n"
"                                        gdouble              scale);          \n"
"static void      fill_poly_color       (Polygon             *poly,            \n"
"                                        gfloat              *input_buf,       \n"
"                                        gfloat              *output_buf,      \n"
"                                        gfloat              *col,             \n"
"                                        const GeglRectangle *result,          \n"
"                                        gboolean             antialiasing,    \n"
"                                        gboolean             tile_rough,      \n"
"                                        gdouble              tile_height,     \n"
"                                        MosaicDatas         *mdatas);         \n"
"static void      fill_poly_image       (Polygon             *poly,            \n"
"                                        gfloat              *input_buf,       \n"
"                                        gfloat              *output_buf,      \n"
"                                        gdouble             vary,             \n"
"                                        const GeglRectangle *result,          \n"
"                                        gboolean             antialiasing,    \n"
"                                        gboolean             tile_rough,      \n"
"                                        gdouble              tile_height,     \n"
"                                        MosaicDatas         *mdatas);         \n"
"static void      calc_spec_vec         (SpecVec             *vec,             \n"
"                                        gint                 xs,              \n"
"                                        gint                 ys,              \n"
"                                        gint                 xe,              \n"
"                                        gint                 ye,              \n"
"                                        gdouble              light_x,         \n"
"                                        gdouble              light_y);        \n"
"static gdouble   calc_spec_contrib     (SpecVec             *vec,             \n"
"                                        gint                 n,               \n"
"                                        gdouble              x,               \n"
"                                        gdouble              y,               \n"
"                                        gboolean             tile_rough,      \n"
"                                        gdouble              tile_height);    \n"
"/*  Polygon machinery  */                                                     \n"
"static void      convert_segment       (gint                 x1,              \n"
"                                        gint                 y1,              \n"
"                                        gint                 x2,              \n"
"                                        gint                 y2,              \n"
"                                        gint                 offset,          \n"
"                                        gint                *min,             \n"
"                                        gint                *max);            \n"
"static void      polygon_add_point     (Polygon             *poly,            \n"
"                                        gdouble              x,               \n"
"                                        gdouble              y);              \n"
"static gboolean  polygon_find_center   (Polygon             *poly,            \n"
"                                        gdouble             *x,               \n"
"                                        gdouble             *y);              \n"
"static void      polygon_translate     (Polygon             *poly,            \n"
"                                        gdouble              tx,              \n"
"                                        gdouble              ty);             \n"
"static void      polygon_scale         (Polygon             *poly,            \n"
"                                        gdouble              scale);          \n"
"static gboolean  polygon_extents       (Polygon             *poly,            \n"
"                                        gdouble             *min_x,           \n"
"                                        gdouble             *min_y,           \n"
"                                        gdouble             *max_x,           \n"
"                                        gdouble             *max_y);          \n"
"static void      polygon_reset         (Polygon             *poly);           \n"
"static gfloat    rand_f                (GeglRandom          *rand,            \n"
"                                        gfloat               pos_x,           \n"
"                                        gfloat               pos_y,           \n"
"                                        gfloat               min,             \n"
"                                        gfloat               max);            \n"
"static gint      rand_i                (GeglRandom          *rand,            \n"
"                                        gfloat               pos_x,           \n"
"                                        gfloat               pos_y,           \n"
"                                        gint                 min,             \n"
"                                        gint                 max);            \n"
"static gfloat    distance              (SpecVec             *vec,             \n"
"                                        gfloat               x,               \n"
"                                        gfloat               y);              \n"
"                                                                              \n"
"#define ROUND(x)   ((gint) ((((x) < 0) ? (x) - 0.5 : (x) + 0.5)))             \n"
"                                                                              \n"
"#define CLAMP01(v) ((v) < 0.0 ? 0.0 : ((v) > 1.0 ? 1.0 : (v)))                \n"
"                                                                              \n"
"#define SQR(v)     ((v)*(v))                                                  \n"
"                                                                              \n"
"static gfloat                                                                 \n"
"rand_f (GeglRandom *rand,                                                     \n"
"        gfloat      pos_x,                                                    \n"
"        gfloat      pos_y,                                                    \n"
"        gfloat      min,                                                      \n"
"        gfloat      max)                                                      \n"
"{                                                                             \n"
"  return gegl_random_float_range (rand, ROUND (pos_x), ROUND (pos_y),         \n"
"                                  0, 0, min, max);                            \n"
"}                                                                             \n"
"                                                                              \n"
"static gint                                                                   \n"
"rand_i (GeglRandom *rand,                                                     \n"
"        gfloat      pos_x,                                                    \n"
"        gfloat      pos_y,                                                    \n"
"        gint        min,                                                      \n"
"        gint        max)                                                      \n"
"{                                                                             \n"
"  return gegl_random_int_range (rand, ROUND (pos_x), ROUND (pos_y),           \n"
"                                0, 0, min, max);                              \n"
"}                                                                             \n"
"                                                                              \n"
"static gfloat*                                                                \n"
"mosaic (GeglOperation       *operation,                                       \n"
"        GeglBuffer          *drawable,                                        \n"
"        const GeglRectangle *result)                                          \n"
"{                                                                             \n"
"  GeglProperties          *o;                                                 \n"
"  MosaicDatas          mdatas;                                                \n"
"  gfloat              *rendered;                                              \n"
"  const GeglRectangle *whole_region;                                          \n"
"  gfloat              *input_buf;                                             \n"
"                                                                              \n"
"  o = GEGL_PROPERTIES (operation);                                            \n"
"                                                                              \n"
"  input_buf = g_new (gfloat, NB_CPN * result->width * result->height);        \n"
"                                                                              \n"
"  gegl_buffer_get (drawable, result,                                          \n"
"                   1.0, babl_format (\"R'G'B'A float\"), input_buf,           \n"
"                   GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);                     \n"
"                                                                              \n"
"  {                                                                           \n"
"    gdouble r, g, b, a;                                                       \n"
"    gegl_color_get_rgba (o->light_color, &r, &g, &b, &a);                     \n"
"    mdatas.fore[0] = r;                                                       \n"
"    mdatas.fore[1] = g;                                                       \n"
"    mdatas.fore[2] = b;                                                       \n"
"    mdatas.fore[3] = a;                                                       \n"
"                                                                              \n"
"    gegl_color_get_rgba (o->joints_color, &r, &g, &b, &a);                    \n"
"    mdatas.back[0] = r;                                                       \n"
"    mdatas.back[1] = g;                                                       \n"
"    mdatas.back[2] = b;                                                       \n"
"    mdatas.back[3] = a;                                                       \n"
"  }                                                                           \n"
"                                                                              \n"
"  whole_region = gegl_operation_source_get_bounding_box (operation, \"input\");\n"
"                                                                              \n"
"  mdatas.width  = whole_region->width;                                        \n"
"  mdatas.height = whole_region->height;                                       \n"
"                                                                              \n"
"  /*  Find the gradients  */                                                  \n"
"  find_gradients (input_buf, STD_DEV, result, &mdatas);                       \n"
"                                                                              \n"
"  /*  Create the tile geometry grid  */                                       \n"
"  switch (o->tile_type)                                                       \n"
"    {                                                                         \n"
"    case SQUARES:                                                             \n"
"      grid_create_squares (result, o->tile_size, &(mdatas.grid));             \n"
"      break;                                                                  \n"
"    case HEXAGONS:                                                            \n"
"      grid_create_hexagons (result, o->tile_size, &(mdatas.grid));            \n"
"      break;                                                                  \n"
"    case OCTAGONS:                                                            \n"
"      grid_create_octagons (result, o->tile_size, &(mdatas.grid));            \n"
"      break;                                                                  \n"
"    case TRIANGLES:                                                           \n"
"      grid_create_triangles (result, o->tile_size, &(mdatas.grid));           \n"
"      break;                                                                  \n"
"    default:                                                                  \n"
"      break;                                                                  \n"
"    }                                                                         \n"
"                                                                              \n"
"  /*  Deform the tiles based on image content  */                             \n"
"  grid_localize (result, o, &mdatas);                                         \n"
"                                                                              \n"
"  mdatas.light_x = -cos (o->light_dir * G_PI / 180.0);                        \n"
"  mdatas.light_y =  sin (o->light_dir * G_PI / 180.0);                        \n"
"  mdatas.scale = (o->tile_spacing > o->tile_size / 2.0) ?                     \n"
"    0.5 : 1.0 - o->tile_spacing / o->tile_size;                               \n"
"                                                                              \n"
"  /*  Render the tiles  */                                                    \n"
"  rendered = grid_render (input_buf, result, o, &mdatas);                     \n"
"                                                                              \n"
"  g_free (mdatas.h_grad);                                                     \n"
"  g_free (mdatas.v_grad);                                                     \n"
"  g_free (mdatas.m_grad);                                                     \n"
"  g_free (mdatas.grid.vert_o);                                                \n"
"  g_free (input_buf);                                                         \n"
"                                                                              \n"
"  return rendered;                                                            \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/*                                                                            \n"
" *  Gradient finding machinery                                                \n"
" */                                                                           \n"
"                                                                              \n"
"static void                                                                   \n"
"find_gradients (gfloat              *input_buf,                               \n"
"                gdouble              std_dev,                                 \n"
"                const GeglRectangle *result,                                  \n"
"                MosaicDatas         *mdatas)                                  \n"
"{                                                                             \n"
"  gfloat *dest_rgn;                                                           \n"
"  gint    i, j;                                                               \n"
"  gfloat *gr, *dh, *dv;                                                       \n"
"  gfloat  hmax, vmax;                                                         \n"
"                                                                              \n"
"  /*  allocate the gradient maps  */                                          \n"
"  mdatas->h_grad = g_new (gfloat, result->width * result->height);            \n"
"  mdatas->v_grad = g_new (gfloat, result->width * result->height);            \n"
"  mdatas->m_grad = g_new (gfloat, result->width * result->height);            \n"
"                                                                              \n"
"  dest_rgn = g_new (gfloat, result->width * result->height * NB_CPN);         \n"
"                                                                              \n"
"  gaussian_deriv (input_buf, dest_rgn, GEGL_ORIENTATION_HORIZONTAL, std_dev, result);\n"
"                                                                              \n"
"  find_max_gradient (dest_rgn, mdatas->h_grad, result->width, result->height);\n"
"                                                                              \n"
"  gaussian_deriv (input_buf, dest_rgn, GEGL_ORIENTATION_VERTICAL, std_dev, result);\n"
"                                                                              \n"
"  find_max_gradient (dest_rgn, mdatas->v_grad, result->width, result->height);\n"
"                                                                              \n"
"  /*  fill in the gradient map  */                                            \n"
"  gr = mdatas->m_grad;                                                        \n"
"  dh = mdatas->h_grad;                                                        \n"
"  dv = mdatas->v_grad;                                                        \n"
"                                                                              \n"
"  for (i = 0; i < result->height; i++)                                        \n"
"    {                                                                         \n"
"      for (j = 0; j < result->width; j++, dh++, dv++, gr++)                   \n"
"        {                                                                     \n"
"          /*  Find the gradient  */                                           \n"
"          if (!j || !i || (j == result->width - 1) || (i == result->height - 1))\n"
"            {                                                                 \n"
"              *gr = MAG_THRESHOLD;                                            \n"
"            }                                                                 \n"
"          else                                                                \n"
"            {                                                                 \n"
"              hmax = *dh - 0.5;                                               \n"
"              vmax = *dv - 0.5;                                               \n"
"                                                                              \n"
"              *gr = (gfloat) sqrt (SQR (hmax) + SQR (vmax));                  \n"
"            }                                                                 \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (dest_rgn);                                                          \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"find_max_gradient (gfloat *src_rgn,                                           \n"
"                   gfloat *dest_rgn,                                          \n"
"                   gint    width,                                             \n"
"                   gint    height)                                            \n"
"{                                                                             \n"
"  gfloat *s, *d;                                                              \n"
"  gint    i, j, k;                                                            \n"
"  gfloat  val;                                                                \n"
"  gfloat  max;                                                                \n"
"                                                                              \n"
"  s = src_rgn;                                                                \n"
"  d = dest_rgn;                                                               \n"
"                                                                              \n"
"  for (i = 0; i < height; i++)                                                \n"
"    {                                                                         \n"
"      for (j = 0; j < width; j++)                                             \n"
"        {                                                                     \n"
"          max = 0.5;                                                          \n"
"          for (k = 0; k < NB_CPN; k++)                                        \n"
"            {                                                                 \n"
"              val = *s;                                                       \n"
"              if (fabs (val - 0.5) > fabs (max - 0.5))                        \n"
"                max = val;                                                    \n"
"            }                                                                 \n"
"                                                                              \n"
"          *d++ = max;                                                         \n"
"          s += NB_CPN;                                                        \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gaussian_deriv (gfloat              *src_rgn,                                 \n"
"                gfloat              *dest_rgn,                                \n"
"                GeglOrientation      direction,                               \n"
"                gdouble              std_dev,                                 \n"
"                const GeglRectangle *result)                                  \n"
"{                                                                             \n"
"  gfloat *dp;                                                                 \n"
"  gfloat *sp, *s;                                                             \n"
"  gfloat *buf, *b;                                                            \n"
"  gint    chan;                                                               \n"
"  gint    i, row, col;                                                        \n"
"  gint    start, end;                                                         \n"
"  gfloat  curve_array [9];                                                    \n"
"  gfloat  sum_array [9];                                                      \n"
"  gfloat *curve;                                                              \n"
"  gfloat *sum;                                                                \n"
"  gfloat  val;                                                                \n"
"  gfloat  total;                                                              \n"
"  gint    length;                                                             \n"
"  gfloat  initial_p[4], initial_m[4];                                         \n"
"  gint    width = result->width, height = result->height;                     \n"
"                                                                              \n"
"  /*  allocate buffers for rows/cols  */                                      \n"
"  length = MAX (result->width, result->height) * NB_CPN;                      \n"
"                                                                              \n"
"  length = 3;                                                                 \n"
"                                                                              \n"
"  /*  initialize  */                                                          \n"
"  curve = curve_array + length;                                               \n"
"  sum = sum_array + length;                                                   \n"
"  buf = g_new (gfloat, MAX (width, height) * NB_CPN);                         \n"
"                                                                              \n"
"  if (direction == GEGL_ORIENTATION_VERTICAL)                                 \n"
"    {                                                                         \n"
"      make_curve_d (curve, sum, std_dev, length);                             \n"
"      total = sum[0] * -2;                                                    \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      make_curve (curve, sum, std_dev, length);                               \n"
"      total = sum[length] + curve[length];                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (col = 0; col < width; col++)                                           \n"
"    {                                                                         \n"
"      sp = src_rgn + col * NB_CPN;                                            \n"
"      dp = dest_rgn + col * NB_CPN;;                                          \n"
"      b = buf;                                                                \n"
"                                                                              \n"
"      for (chan = 0; chan < NB_CPN; chan++)                                   \n"
"        {                                                                     \n"
"          initial_p[chan] = sp[chan];                                         \n"
"          initial_m[chan] = sp[width * (height - 1) * NB_CPN + chan];         \n"
"        }                                                                     \n"
"                                                                              \n"
"      for (row = 0; row < height; row++)                                      \n"
"        {                                                                     \n"
"          start = (row < length) ?  - row : -length;                          \n"
"          end = ((height - row - 1) < length) ? (height - row - 1) : length;  \n"
"                                                                              \n"
"          for (chan = 0; chan < NB_CPN; chan++)                               \n"
"            {                                                                 \n"
"              s = sp + width * (start * NB_CPN) + chan;                       \n"
"              val = 0;                                                        \n"
"              i = start;                                                      \n"
"                                                                              \n"
"              if (start != -length)                                           \n"
"                val += initial_p[chan] * (sum[start] - sum[-length]);         \n"
"                                                                              \n"
"              while (i <= end)                                                \n"
"                {                                                             \n"
"                  val += *s * curve[i++];                                     \n"
"                  s += NB_CPN * width;                                        \n"
"                }                                                             \n"
"                                                                              \n"
"              if (end != length)                                              \n"
"                val += initial_m[chan] * (sum[length] + curve[length] - sum[end+1]);\n"
"                                                                              \n"
"              if ( val == 0)                                                  \n"
"                *b++ = val;                                                   \n"
"              else                                                            \n"
"                *b++ = val / total;                                           \n"
"            }                                                                 \n"
"                                                                              \n"
"          sp += NB_CPN * width;                                               \n"
"        }                                                                     \n"
"                                                                              \n"
"      b = buf;                                                                \n"
"      if (direction == GEGL_ORIENTATION_VERTICAL)                             \n"
"        for (row = 0; row < height; row++)                                    \n"
"          {                                                                   \n"
"            for (chan = 0; chan < NB_CPN; chan++)                             \n"
"              {                                                               \n"
"                b[chan] += 0.5;                                               \n"
"                dp[chan] = CLAMP01 (b[chan]);                                 \n"
"              }                                                               \n"
"            b += NB_CPN;                                                      \n"
"            dp += NB_CPN * width;                                             \n"
"          }                                                                   \n"
"      else                                                                    \n"
"        for (row = 0; row < height; row++)                                    \n"
"          {                                                                   \n"
"            for (chan = 0; chan < NB_CPN; chan++)                             \n"
"              {                                                               \n"
"                dp[chan] = CLAMP01 (b[chan]);                                 \n"
"              }                                                               \n"
"            b += NB_CPN;                                                      \n"
"            dp += NB_CPN * width;                                             \n"
"          }                                                                   \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (direction == GEGL_ORIENTATION_HORIZONTAL)                               \n"
"    {                                                                         \n"
"      make_curve_d (curve, sum, std_dev, length);                             \n"
"      total = sum[0] * -2;                                                    \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      make_curve (curve, sum, std_dev, length);                               \n"
"      total = sum[length] + curve[length];                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (row = 0; row < height; row++)                                          \n"
"    {                                                                         \n"
"      sp = dest_rgn + width * row * NB_CPN;                                   \n"
"      dp = dest_rgn + width * row * NB_CPN;                                   \n"
"      b = buf;                                                                \n"
"                                                                              \n"
"                                                                              \n"
"      for (chan = 0; chan < NB_CPN; chan++)                                   \n"
"        {                                                                     \n"
"          initial_p[chan] = sp[chan];                                         \n"
"          initial_m[chan] = sp[(width - 1) * NB_CPN + chan];                  \n"
"        }                                                                     \n"
"                                                                              \n"
"      for (col = 0; col < width; col++)                                       \n"
"        {                                                                     \n"
"          start = (col < length) ?  - col : -length;                          \n"
"          end = (width - col - 1 < length) ? (width - col - 1) : length;      \n"
"                                                                              \n"
"          for (chan = 0; chan < NB_CPN; chan++)                               \n"
"            {                                                                 \n"
"              s = sp + (start * NB_CPN) + chan;                               \n"
"              val = 0;                                                        \n"
"              i = start;                                                      \n"
"                                                                              \n"
"              if (start != -length)                                           \n"
"                val += initial_p[chan] * (sum[start] - sum[-length]);         \n"
"                                                                              \n"
"              while (i <= end)                                                \n"
"                {                                                             \n"
"                  val += *s * curve[i++];                                     \n"
"                  s += NB_CPN;                                                \n"
"                }                                                             \n"
"                                                                              \n"
"              if (end != length)                                              \n"
"                val += initial_m[chan] * (sum[length] + curve[length] - sum[end+1]);\n"
"                                                                              \n"
"              if ( val == 0)                                                  \n"
"                *b++ = val;                                                   \n"
"              else                                                            \n"
"                *b++ = val / total;                                           \n"
"            }                                                                 \n"
"                                                                              \n"
"          sp += NB_CPN;                                                       \n"
"        }                                                                     \n"
"                                                                              \n"
"      b = buf;                                                                \n"
"      if (direction == GEGL_ORIENTATION_HORIZONTAL)                           \n"
"        for (col = 0; col < width; col++)                                     \n"
"          {                                                                   \n"
"            for (chan = 0; chan < NB_CPN; chan++)                             \n"
"              {                                                               \n"
"                b[chan] += 0.5;                                               \n"
"                dp[chan] = CLAMP01 (b[chan]);                                 \n"
"              }                                                               \n"
"            b += NB_CPN;                                                      \n"
"            dp += NB_CPN;                                                     \n"
"          }                                                                   \n"
"      else                                                                    \n"
"        for (col = 0; col < width; col++)                                     \n"
"          {                                                                   \n"
"            for (chan = 0; chan < NB_CPN; chan++)                             \n"
"              {                                                               \n"
"                dp[chan] = CLAMP01 (b[chan]);                                 \n"
"              }                                                               \n"
"            b += NB_CPN;                                                      \n"
"            dp += NB_CPN;                                                     \n"
"          }                                                                   \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (buf);                                                               \n"
"}                                                                             \n"
"                                                                              \n"
"/*                                                                            \n"
" * The equations: g(r) = exp (- r^2 / (2 * sigma^2))                          \n"
" *                   r = sqrt (x^2 + y ^2)                                    \n"
" */                                                                           \n"
"                                                                              \n"
"static void                                                                   \n"
"make_curve (gfloat  *curve,                                                   \n"
"            gfloat  *sum,                                                     \n"
"            gdouble  sigma,                                                   \n"
"            gint     length)                                                  \n"
"{                                                                             \n"
"  gdouble sigma2;                                                             \n"
"  gint    i;                                                                  \n"
"                                                                              \n"
"  sigma2 = sigma * sigma;                                                     \n"
"                                                                              \n"
"  curve[0] = 1.0;                                                             \n"
"  for (i = 1; i <= length; i++)                                               \n"
"    {                                                                         \n"
"      curve[i] = (gfloat) (exp (- (i * i) / (2 * sigma2)));                   \n"
"      curve[-i] = curve[i];                                                   \n"
"    }                                                                         \n"
"                                                                              \n"
"  sum[-length] = 0;                                                           \n"
"  for (i = -length+1; i <= length; i++)                                       \n"
"    sum[i] = sum[i-1] + curve[i-1];                                           \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/*                                                                            \n"
" * The equations: d_g(r) = -r * exp (- r^2 / (2 * sigma^2)) / sigma^2         \n"
" *                   r = sqrt (x^2 + y ^2)                                    \n"
" */                                                                           \n"
"                                                                              \n"
"static void                                                                   \n"
"make_curve_d (gfloat  *curve,                                                 \n"
"              gfloat  *sum,                                                   \n"
"              gdouble  sigma,                                                 \n"
"              gint     length)                                                \n"
"{                                                                             \n"
"  gdouble sigma2;                                                             \n"
"  gint    i;                                                                  \n"
"                                                                              \n"
"  sigma2 = sigma * sigma;                                                     \n"
"                                                                              \n"
"  curve[0] = 0;                                                               \n"
"  for (i = 1; i <= length; i++)                                               \n"
"    {                                                                         \n"
"      curve[i] = (gfloat) ((i * exp (- (i * i) / (2 * sigma2)) / sigma2));    \n"
"      curve[-i] = -curve[i];                                                  \n"
"    }                                                                         \n"
"                                                                              \n"
"  sum[-length] = 0;                                                           \n"
"  sum[0] = 0;                                                                 \n"
"  for (i = 1; i <= length; i++)                                               \n"
"    {                                                                         \n"
"      sum[-length + i] = sum[-length + i - 1] + curve[-length + i - 1];       \n"
"      sum[i] = sum[i - 1] + curve[i - 1];                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/*********************************************/                               \n"
"/*   Functions for grid manipulation         */                               \n"
"/*********************************************/                               \n"
"                                                                              \n"
"static void                                                                   \n"
"grid_create_squares (const GeglRectangle *result,                             \n"
"                     gdouble              tile_size,                          \n"
"                     GridDescriptor      *grid)                               \n"
"{                                                                             \n"
"  gint    rows, cols;                                                         \n"
"  gint    width, height;                                                      \n"
"  gdouble pre_cols, pre_rows;                                                 \n"
"  gdouble offset_x, offset_y;                                                 \n"
"  gint    i, j;                                                               \n"
"  gint    size = (gint) tile_size;                                            \n"
"  Vertex *pt;                                                                 \n"
"                                                                              \n"
"  pre_cols = (result->x / size) * size;                                       \n"
"  pre_rows = (result->y / size) * size;                                       \n"
"                                                                              \n"
"  offset_x = pre_cols - result->x;                                            \n"
"  offset_y = pre_rows - result->y;                                            \n"
"                                                                              \n"
"  width = result->width - offset_x;                                           \n"
"  height = result->height - offset_y;                                         \n"
"                                                                              \n"
"  rows = (height + size - 1) / size;                                          \n"
"  cols = (width + size - 1) / size;                                           \n"
"                                                                              \n"
"  grid->vert_o = grid->vert = g_new (Vertex, (cols + 2) * (rows + 2));        \n"
"  grid->vert += (cols + 2) + 1;                                               \n"
"                                                                              \n"
"  for (i = -1; i <= rows; i++)                                                \n"
"    for (j = -1; j <= cols; j++)                                              \n"
"      {                                                                       \n"
"        pt = grid->vert + (i * (cols + 2) + j);                               \n"
"                                                                              \n"
"        pt->x = j * size + size / 2 + offset_x;                               \n"
"        pt->y = i * size + size / 2 + offset_y;                               \n"
"      }                                                                       \n"
"                                                                              \n"
"  grid->vert_rows = rows;                                                     \n"
"  grid->vert_cols = cols;                                                     \n"
"  grid->row_pad = 1;                                                          \n"
"  grid->col_pad = 1;                                                          \n"
"  grid->vert_multiple = 1;                                                    \n"
"  grid->vert_rowstride = cols + 2;                                            \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"grid_create_hexagons (const GeglRectangle *result,                            \n"
"                      gdouble              tile_size,                         \n"
"                      GridDescriptor      *grid)                              \n"
"{                                                                             \n"
"  gint     rows, cols;                                                        \n"
"  gint     i, j;                                                              \n"
"  gint     width, height;                                                     \n"
"  gdouble  pre_rows, pre_cols;                                                \n"
"  gdouble  hex_l1, hex_l2, hex_l3;                                            \n"
"  gdouble  hex_width;                                                         \n"
"  gdouble  hex_height;                                                        \n"
"  gdouble  offset_x, offset_y;                                                \n"
"  Vertex  *pt;                                                                \n"
"  hex_l1 = tile_size / 2.0;                                                   \n"
"  hex_l2 = hex_l1 * 2.0 / sqrt (3.0);                                         \n"
"  hex_l3 = hex_l1 / sqrt (3.0);                                               \n"
"  hex_width = 6 * hex_l1 / sqrt (3.0);                                        \n"
"  hex_height = tile_size;                                                     \n"
"                                                                              \n"
"  pre_cols = floor (result->x / hex_width) * hex_width;                       \n"
"  pre_rows = floor (result->y / hex_height) * hex_height;                     \n"
"                                                                              \n"
"  offset_x = pre_cols - result->x;                                            \n"
"  offset_y = pre_rows - result->y;                                            \n"
"                                                                              \n"
"  width = result->width - offset_x;                                           \n"
"  height = result->height - offset_y;                                         \n"
"                                                                              \n"
"  rows = ((height + hex_height - 1) / hex_height);                            \n"
"  cols = ((width + hex_width * 2 - 1) / hex_width);                           \n"
"                                                                              \n"
"  grid->vert_o = grid->vert = g_new (Vertex, (cols + 2) * 4 * (rows + 2));    \n"
"  grid->vert += (cols + 2) * 4 + 4;                                           \n"
"                                                                              \n"
"  for (i = -1; i <= rows; i++)                                                \n"
"    for (j = -1; j <= cols; j++)                                              \n"
"      {                                                                       \n"
"        pt = grid->vert + (i * (cols + 2) * 4 + j * 4);                       \n"
"                                                                              \n"
"        pt[0].x = hex_width * j + hex_l3 + offset_x;                          \n"
"        pt[0].y = hex_height * i + offset_y;                                  \n"
"        pt[1].x = pt[0].x + hex_l2;                                           \n"
"        pt[1].y = pt[0].y;                                                    \n"
"        pt[2].x = pt[1].x + hex_l3;                                           \n"
"        pt[2].y = pt[1].y + hex_l1;                                           \n"
"        pt[3].x = pt[0].x - hex_l3;                                           \n"
"        pt[3].y = pt[0].y + hex_l1;                                           \n"
"      }                                                                       \n"
"                                                                              \n"
"  grid->vert_rows = rows;                                                     \n"
"  grid->vert_cols = cols;                                                     \n"
"  grid->row_pad = 1;                                                          \n"
"  grid->col_pad = 1;                                                          \n"
"  grid->vert_multiple = 4;                                                    \n"
"  grid->vert_rowstride = (cols + 2) * 4;                                      \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"grid_create_octagons (const GeglRectangle *result,                            \n"
"                      gdouble              tile_size,                         \n"
"                      GridDescriptor      *grid)                              \n"
"{                                                                             \n"
"  gint     rows, cols;                                                        \n"
"  gint     width, height;                                                     \n"
"  gdouble  pre_rows, pre_cols;                                                \n"
"  gint     i, j;                                                              \n"
"  gdouble  ts, side, leg;                                                     \n"
"  gdouble  oct_size;                                                          \n"
"  Vertex  *pt;                                                                \n"
"  gdouble  offset_x, offset_y;                                                \n"
"                                                                              \n"
"  ts = tile_size;                                                             \n"
"  side = ts / (sqrt (2.0) + 1.0);                                             \n"
"  leg = side * sqrt (2.0) * 0.5;                                              \n"
"  oct_size = ts + side;                                                       \n"
"                                                                              \n"
"  pre_cols = floor (result->x / oct_size) * oct_size;                         \n"
"  pre_rows = floor (result->y / oct_size) * oct_size;                         \n"
"                                                                              \n"
"  offset_x = pre_cols - result->x;                                            \n"
"  offset_y = pre_rows - result->y;                                            \n"
"                                                                              \n"
"  width = result->width - offset_x;                                           \n"
"  height = result->height - offset_y;                                         \n"
"                                                                              \n"
"  rows = ((height + oct_size - 1) / oct_size);                                \n"
"  cols = ((width + oct_size * 2 - 1) / oct_size);                             \n"
"                                                                              \n"
"  grid->vert_o = grid->vert = g_new (Vertex, (cols + 2) * 8 * (rows + 2));    \n"
"  grid->vert += (cols + 2) * 8 + 8;                                           \n"
"                                                                              \n"
"  for (i = -1; i < rows + 1; i++)                                             \n"
"    for (j = -1; j < cols + 1; j++)                                           \n"
"      {                                                                       \n"
"        pt = grid->vert + (i * (cols + 2) * 8 + j * 8);                       \n"
"                                                                              \n"
"        pt[0].x = oct_size * j + offset_x;                                    \n"
"        pt[0].y = oct_size * i + offset_y;                                    \n"
"        pt[1].x = pt[0].x + side;                                             \n"
"        pt[1].y = pt[0].y;                                                    \n"
"        pt[2].x = pt[0].x + leg + side;                                       \n"
"        pt[2].y = pt[0].y + leg;                                              \n"
"        pt[3].x = pt[2].x;                                                    \n"
"        pt[3].y = pt[0].y + leg + side;                                       \n"
"        pt[4].x = pt[1].x;                                                    \n"
"        pt[4].y = pt[0].y + 2 * leg + side;                                   \n"
"        pt[5].x = pt[0].x;                                                    \n"
"        pt[5].y = pt[4].y;                                                    \n"
"        pt[6].x = pt[0].x - leg;                                              \n"
"        pt[6].y = pt[3].y;                                                    \n"
"        pt[7].x = pt[6].x;                                                    \n"
"        pt[7].y = pt[2].y;                                                    \n"
"      }                                                                       \n"
"                                                                              \n"
"  grid->vert_rows = rows;                                                     \n"
"  grid->vert_cols = cols;                                                     \n"
"  grid->row_pad = 1;                                                          \n"
"  grid->col_pad = 1;                                                          \n"
"  grid->vert_multiple = 8;                                                    \n"
"  grid->vert_rowstride = (cols + 2) * 8;                                      \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"grid_create_triangles (const GeglRectangle *result,                           \n"
"                       gdouble              tile_size,                        \n"
"                       GridDescriptor      *grid)                             \n"
"{                                                                             \n"
"  gint     rows, cols;                                                        \n"
"  gdouble  pre_rows, pre_cols;                                                \n"
"  gint     width, height;                                                     \n"
"  gint     i, j;                                                              \n"
"  gdouble  tri_mid, tri_height;                                               \n"
"  Vertex  *pt;                                                                \n"
"  gdouble  offset_x, offset_y;                                                \n"
"                                                                              \n"
"  tri_mid    = tile_size / 2.0;              /* cos 60 */                     \n"
"  tri_height = tile_size / 2.0 * sqrt (3.0); /* sin 60 */                     \n"
"                                                                              \n"
"  pre_cols = floor (result->x / tile_size) * tile_size;                       \n"
"  pre_rows = floor (result->y / (2 * tri_height)) * (2 * tri_height);         \n"
"                                                                              \n"
"  offset_x = pre_cols - result->x;                                            \n"
"  offset_y = pre_rows - result->y;                                            \n"
"                                                                              \n"
"  width = result->width - offset_x;                                           \n"
"  height = result->height - offset_y;                                         \n"
"                                                                              \n"
"  rows = (height + 2 * tri_height - 1) / (2 * tri_height);                    \n"
"  cols = (width + tile_size - 1) / tile_size;                                 \n"
"                                                                              \n"
"  grid->vert_o = grid->vert = g_new (Vertex, (cols + 2) * 2 * (rows + 2));    \n"
"  grid->vert += (cols + 2) * 2 + 2;                                           \n"
"                                                                              \n"
"  for (i = -1; i <= rows; i++)                                                \n"
"    for (j = -1; j <= cols; j++)                                              \n"
"      {                                                                       \n"
"        pt = grid->vert + (i * (cols + 2) * 2 + j * 2);                       \n"
"                                                                              \n"
"        pt[0].x = tile_size * j + offset_x;                                   \n"
"        pt[0].y = (tri_height * 2) * i + offset_y;                            \n"
"        pt[1].x = pt[0].x + tri_mid;                                          \n"
"        pt[1].y = pt[0].y + tri_height;                                       \n"
"      }                                                                       \n"
"                                                                              \n"
"  grid->vert_rows = rows;                                                     \n"
"  grid->vert_cols = cols;                                                     \n"
"  grid->row_pad = 1;                                                          \n"
"  grid->col_pad = 1;                                                          \n"
"  grid->vert_multiple = 2;                                                    \n"
"  grid->vert_rowstride = (cols + 2) * 2;                                      \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"grid_localize (const GeglRectangle *result,                                   \n"
"               GeglProperties          *o,                                    \n"
"               MosaicDatas         *mdatas)                                   \n"
"{                                                                             \n"
"  gint           i, j;                                                        \n"
"  gint           k, l;                                                        \n"
"  gint           x3, y3, x4, y4;                                              \n"
"  gint           size;                                                        \n"
"  gint           max_x, max_y;                                                \n"
"  gfloat         max;                                                         \n"
"  gfloat        *data;                                                        \n"
"  gdouble        rand_localize;                                               \n"
"  Vertex        *pt;                                                          \n"
"  GridDescriptor grid;                                                        \n"
"                                                                              \n"
"  grid = mdatas->grid;                                                        \n"
"                                                                              \n"
"  size = (gint) o->tile_size;                                                 \n"
"  rand_localize = size * (1.0 - o->tile_neatness);                            \n"
"                                                                              \n"
"  for (i = -grid.row_pad; i < grid.vert_rows + grid.row_pad; i++)             \n"
"    for (j = -grid.col_pad * grid.vert_multiple;                              \n"
"         j < (grid.vert_cols + grid.col_pad) * grid.vert_multiple;            \n"
"         j++)                                                                 \n"
"      {                                                                       \n"
"        pt = grid.vert + (i * grid.vert_rowstride + j);                       \n"
"                                                                              \n"
"        if (result->x + pt->x < 0 || result->x + pt->x >= mdatas->width ||    \n"
"            result->y + pt->y < 0 || result->y + pt->y >= mdatas->height)     \n"
"          continue;                                                           \n"
"                                                                              \n"
"        if (rand_localize > GEGL_FLOAT_EPSILON)                               \n"
"          {                                                                   \n"
"            /* Rely on global values to make gegl's tiles seamless */         \n"
"            max_x = pt->x + (gint) rand_f (o->rand,                           \n"
"                                           result->x + pt->x, result->y + pt->y,\n"
"                                           - rand_localize / 2.0, rand_localize / 2.0);\n"
"            max_y = pt->y + (gint) rand_f (o->rand,                           \n"
"                                           result->y + pt->y, result->x + pt->x, /*sic*/\n"
"                                           - rand_localize / 2.0, rand_localize / 2.0);\n"
"          }                                                                   \n"
"        else                                                                  \n"
"          {                                                                   \n"
"            max_x = pt->x;                                                    \n"
"            max_y = pt->y;                                                    \n"
"          }                                                                   \n"
"                                                                              \n"
"        x3 = pt->x - (gint) (rand_localize / 2.0);                            \n"
"        y3 = pt->y - (gint) (rand_localize / 2.0);                            \n"
"        x4 = x3 + (gint) rand_localize;                                       \n"
"        y4 = y3 + (gint) rand_localize;                                       \n"
"                                                                              \n"
"        x3 = CLAMP (x3, 0, result->width  - 1);                               \n"
"        y3 = CLAMP (y3, 0, result->height - 1);                               \n"
"        x4 = CLAMP (x4, 0, result->width  - 1);                               \n"
"        y4 = CLAMP (y4, 0, result->height - 1);                               \n"
"                                                                              \n"
"        max = * (mdatas->m_grad + y3 * result->width + x3);                   \n"
"        data = mdatas->m_grad + result->width * y3;                           \n"
"                                                                              \n"
"        for (k = y3; k <= y4; k++)                                            \n"
"          {                                                                   \n"
"            for (l = x3; l <= x4; l++)                                        \n"
"              {                                                               \n"
"                if (data[l] > max)                                            \n"
"                  {                                                           \n"
"                    max_y = k;                                                \n"
"                    max_x = l;                                                \n"
"                    max = data[l];                                            \n"
"                  }                                                           \n"
"              }                                                               \n"
"            data += result->width;                                            \n"
"          }                                                                   \n"
"                                                                              \n"
"        pt->x = max_x;                                                        \n"
"        pt->y = max_y;                                                        \n"
"      }                                                                       \n"
"}                                                                             \n"
"                                                                              \n"
"static gfloat*                                                                \n"
"grid_render (gfloat              *input_buf,                                  \n"
"             const GeglRectangle *result,                                     \n"
"             GeglProperties          *o,                                      \n"
"             MosaicDatas         *mdatas)                                     \n"
"{                                                                             \n"
"  gint            i, j, k, index;                                             \n"
"  Polygon         poly;                                                       \n"
"  gfloat         *output_buf, *o_buf;                                         \n"
"  GridDescriptor  grid;                                                       \n"
"                                                                              \n"
"  grid = mdatas->grid;                                                        \n"
"                                                                              \n"
"  output_buf = g_new (gfloat, NB_CPN * (result->width) * (result->height));   \n"
"                                                                              \n"
"  o_buf = output_buf;                                                         \n"
"                                                                              \n"
"  for (i = 0; i < result->height; i++)                                        \n"
"    {                                                                         \n"
"      for (j = 0; j < result->width; j++)                                     \n"
"        {                                                                     \n"
"          for (k = 0; k < NB_CPN; k++)                                        \n"
"            o_buf[k] = mdatas->back[k];                                       \n"
"          o_buf += NB_CPN;                                                    \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (i = -grid.row_pad; i < grid.vert_rows; i++)                            \n"
"    for (j = -grid.col_pad; j < grid.vert_cols; j++)                          \n"
"      {                                                                       \n"
"        index = i * grid.vert_rowstride + j * grid.vert_multiple;             \n"
"                                                                              \n"
"        switch (o->tile_type)                                                 \n"
"          {                                                                   \n"
"          case SQUARES:                                                       \n"
"            polygon_reset (&poly);                                            \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index].x,                            \n"
"                               grid.vert[index].y);                           \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + 1].x,                        \n"
"                               grid.vert[index + 1].y);                       \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_rowstride + 1].x,  \n"
"                               grid.vert[index + grid.vert_rowstride + 1].y); \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_rowstride].x,      \n"
"                               grid.vert[index + grid.vert_rowstride].y);     \n"
"                                                                              \n"
"            process_poly (&poly, o->tile_allow_split, input_buf, output_buf,  \n"
"                          result, o, mdatas);                                 \n"
"            break;                                                            \n"
"                                                                              \n"
"          case HEXAGONS:                                                      \n"
"            /*  The main hexagon  */                                          \n"
"            polygon_reset (&poly);                                            \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index].x,                            \n"
"                               grid.vert[index].y);                           \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + 1].x,                        \n"
"                               grid.vert[index + 1].y);                       \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + 2].x,                        \n"
"                               grid.vert[index + 2].y);                       \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_rowstride + 1].x,  \n"
"                               grid.vert[index + grid.vert_rowstride + 1].y); \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_rowstride].x,      \n"
"                               grid.vert[index + grid.vert_rowstride].y);     \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + 3].x,                        \n"
"                               grid.vert[index + 3].y);                       \n"
"            process_poly (&poly, o->tile_allow_split, input_buf, output_buf,  \n"
"                          result, o, mdatas);                                 \n"
"                                                                              \n"
"            /*  The auxiliary hexagon  */                                     \n"
"            polygon_reset (&poly);                                            \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + 2].x,                        \n"
"                               grid.vert[index + 2].y);                       \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_multiple * 2 - 1].x,\n"
"                               grid.vert[index + grid.vert_multiple * 2 - 1].y);\n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_rowstride + grid.vert_multiple].x,\n"
"                               grid.vert[index + grid.vert_rowstride + grid.vert_multiple].y);\n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_rowstride + grid.vert_multiple + 3].x,\n"
"                               grid.vert[index + grid.vert_rowstride + grid.vert_multiple + 3].y);\n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_rowstride + 2].x,  \n"
"                               grid.vert[index + grid.vert_rowstride + 2].y); \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_rowstride + 1].x,  \n"
"                               grid.vert[index + grid.vert_rowstride + 1].y); \n"
"            process_poly (&poly, o->tile_allow_split, input_buf, output_buf,  \n"
"                          result, o, mdatas);                                 \n"
"            break;                                                            \n"
"                                                                              \n"
"          case OCTAGONS:                                                      \n"
"            /*  The main octagon  */                                          \n"
"            polygon_reset (&poly);                                            \n"
"            for (k = 0; k < 8; k++)                                           \n"
"              polygon_add_point (&poly,                                       \n"
"                                 grid.vert[index + k].x,                      \n"
"                                 grid.vert[index + k].y);                     \n"
"            process_poly (&poly, o->tile_allow_split, input_buf, output_buf,  \n"
"                          result, o, mdatas);                                 \n"
"                                                                              \n"
"            /*  The auxiliary octagon  */                                     \n"
"            polygon_reset (&poly);                                            \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + 3].x,                        \n"
"                               grid.vert[index + 3].y);                       \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_multiple * 2 - 2].x,\n"
"                               grid.vert[index + grid.vert_multiple * 2 - 2].y);\n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_multiple * 2 - 3].x,\n"
"                               grid.vert[index + grid.vert_multiple * 2 - 3].y);\n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_rowstride + grid.vert_multiple].x,\n"
"                               grid.vert[index + grid.vert_rowstride + grid.vert_multiple].y);\n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_rowstride + grid.vert_multiple * 2 - 1].x,\n"
"                               grid.vert[index + grid.vert_rowstride + grid.vert_multiple * 2 - 1].y);\n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_rowstride + 2].x,  \n"
"                               grid.vert[index + grid.vert_rowstride + 2].y); \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_rowstride + 1].x,  \n"
"                               grid.vert[index + grid.vert_rowstride + 1].y); \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + 4].x,                        \n"
"                               grid.vert[index + 4].y);                       \n"
"            process_poly (&poly, o->tile_allow_split, input_buf, output_buf,  \n"
"                          result, o, mdatas);                                 \n"
"                                                                              \n"
"            /*  The main square  */                                           \n"
"            polygon_reset (&poly);                                            \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + 2].x,                        \n"
"                               grid.vert[index + 2].y);                       \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_multiple * 2 - 1].x,\n"
"                               grid.vert[index + grid.vert_multiple * 2 - 1].y);\n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_multiple * 2 - 2].x,\n"
"                               grid.vert[index + grid.vert_multiple * 2 - 2].y);\n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + 3].x,                        \n"
"                               grid.vert[index + 3].y);                       \n"
"            process_poly (&poly, FALSE, input_buf, output_buf,                \n"
"                          result, o, mdatas);                                 \n"
"                                                                              \n"
"            /*  The auxiliary square  */                                      \n"
"            polygon_reset (&poly);                                            \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + 5].x,                        \n"
"                               grid.vert[index + 5].y);                       \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + 4].x,                        \n"
"                               grid.vert[index + 4].y);                       \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_rowstride + 1].x,  \n"
"                               grid.vert[index + grid.vert_rowstride + 1].y); \n"
"            polygon_add_point (&poly,                                         \n"
"                               grid.vert[index + grid.vert_rowstride].x,      \n"
"                               grid.vert[index + grid.vert_rowstride].y);     \n"
"            process_poly (&poly, FALSE, input_buf, output_buf,                \n"
"                          result, o, mdatas);                                 \n"
"            break;                                                            \n"
"          case TRIANGLES:                                                     \n"
"            /*  Lower left  */                                                \n"
"            polygon_reset (&poly);                                            \n"
"            polygon_add_point (&poly,                                         \n"
"                                grid.vert[index].x,                           \n"
"                                grid.vert[index].y);                          \n"
"            polygon_add_point (&poly,                                         \n"
"                                grid.vert[index + grid.vert_multiple].x,      \n"
"                                grid.vert[index + grid.vert_multiple].y);     \n"
"            polygon_add_point (&poly,                                         \n"
"                                grid.vert[index + 1].x,                       \n"
"                                grid.vert[index + 1].y);                      \n"
"            process_poly (&poly, o->tile_allow_split, input_buf, output_buf,  \n"
"                          result, o, mdatas);                                 \n"
"                                                                              \n"
"            /*  lower right  */                                               \n"
"            polygon_reset (&poly);                                            \n"
"            polygon_add_point (&poly,                                         \n"
"                                grid.vert[index + 1].x,                       \n"
"                                grid.vert[index + 1].y);                      \n"
"            polygon_add_point (&poly,                                         \n"
"                                grid.vert[index + grid.vert_multiple].x,      \n"
"                                grid.vert[index + grid.vert_multiple].y);     \n"
"            polygon_add_point (&poly,                                         \n"
"                                grid.vert[index + grid.vert_multiple + 1].x,  \n"
"                                grid.vert[index + grid.vert_multiple + 1].y); \n"
"            process_poly (&poly, o->tile_allow_split, input_buf, output_buf,  \n"
"                          result, o, mdatas);                                 \n"
"                                                                              \n"
"            /*  upper left  */                                                \n"
"            polygon_reset (&poly);                                            \n"
"            polygon_add_point (&poly,                                         \n"
"                                grid.vert[index + 1].x,                       \n"
"                                grid.vert[index + 1].y);                      \n"
"            polygon_add_point (&poly,                                         \n"
"                                grid.vert[index + grid.vert_multiple + grid.vert_rowstride].x,\n"
"                                grid.vert[index + grid.vert_multiple + grid.vert_rowstride].y);\n"
"            polygon_add_point (&poly,                                         \n"
"                                grid.vert[index + grid.vert_rowstride].x,     \n"
"                                grid.vert[index + grid.vert_rowstride].y);    \n"
"            process_poly (&poly, o->tile_allow_split, input_buf, output_buf,  \n"
"                          result, o, mdatas);                                 \n"
"                                                                              \n"
"            /*  upper right  */                                               \n"
"            polygon_reset (&poly);                                            \n"
"            polygon_add_point (&poly,                                         \n"
"                                grid.vert[index + 1].x,                       \n"
"                                grid.vert[index + 1].y);                      \n"
"            polygon_add_point (&poly,                                         \n"
"                                grid.vert[index + grid.vert_multiple +1 ].x,  \n"
"                                grid.vert[index + grid.vert_multiple +1 ].y); \n"
"            polygon_add_point (&poly,                                         \n"
"                                grid.vert[index + grid.vert_multiple + grid.vert_rowstride].x,\n"
"                                grid.vert[index + grid.vert_multiple + grid.vert_rowstride].y);\n"
"            process_poly (&poly, o->tile_allow_split, input_buf, output_buf,  \n"
"                          result, o, mdatas);                                 \n"
"            break;                                                            \n"
"                                                                              \n"
"          }                                                                   \n"
"      }                                                                       \n"
"  return output_buf;                                                          \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"process_poly (Polygon             *poly,                                      \n"
"              gboolean             allow_split,                               \n"
"              gfloat              *input_buf,                                 \n"
"              gfloat              *output_buf,                                \n"
"              const GeglRectangle *result,                                    \n"
"              GeglProperties          *o,                                     \n"
"              MosaicDatas         *mdatas)                                    \n"
"{                                                                             \n"
"  gdouble  dir[2];                                                            \n"
"  gdouble  loc[2];                                                            \n"
"  gdouble  cx = 0.0, cy = 0.0;                                                \n"
"  gdouble  magnitude;                                                         \n"
"  gdouble  distance;                                                          \n"
"  gboolean vary;                                                              \n"
"  gint     size, frac_size;                                                   \n"
"  gdouble  color_vary = 0;                                                    \n"
"                                                                              \n"
"  /*  Determine direction of edges inside polygon, if any  */                 \n"
"  find_poly_dir (poly, mdatas->m_grad, mdatas->h_grad, mdatas->v_grad,        \n"
"                 dir, loc, result);                                           \n"
"                                                                              \n"
"  magnitude = sqrt (SQR (dir[0] - 0.5) + SQR (dir[1] - 0.5));                 \n"
"                                                                              \n"
"  /*  Find the center of the polygon  */                                      \n"
"  polygon_find_center (poly, &cx, &cy);                                       \n"
"  distance = sqrt (SQR (loc[0] - cx) + SQR (loc[1] - cy));                    \n"
"                                                                              \n"
"  /* Here we rely on global values and coordinates to get predictable results,\n"
"   * and to make gegl's tiles seamless */                                     \n"
"  size = (mdatas->width * mdatas->height);                                    \n"
"  frac_size = size * o->color_variation;                                      \n"
"                                                                              \n"
"  vary = rand_i (o->rand, result->x + cx, result->y + cy, 0, size) < frac_size;\n"
"                                                                              \n"
"  /*  determine the variation of tile color based on tile number  */          \n"
"  if (vary)                                                                   \n"
"    {                                                                         \n"
"      color_vary = rand_f (o->rand, result->x + cx, result->y + cy,           \n"
"                           - o->color_variation * 0.5, o->color_variation * 0.5);\n"
"    }                                                                         \n"
"                                                                              \n"
"  /*  If the magnitude of direction inside the polygon is greater than        \n"
"   *  THRESHOLD, split the polygon into two new polygons                      \n"
"   */                                                                         \n"
"  if (magnitude > MAG_THRESHOLD &&                                            \n"
"      (2 * distance / o->tile_size) < 0.5 && allow_split)                     \n"
"    {                                                                         \n"
"      split_poly (poly, input_buf, output_buf,                                \n"
"                  dir, color_vary, result, o, mdatas);                        \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      /*  Otherwise, render the original polygon  */                          \n"
"      render_poly (poly, input_buf, output_buf,                               \n"
"                   color_vary, result, o, mdatas);                            \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"render_poly (Polygon             *poly,                                       \n"
"             gfloat              *input_buf,                                  \n"
"             gfloat              *output_buf,                                 \n"
"             gdouble              vary,                                       \n"
"             const GeglRectangle *result,                                     \n"
"             GeglProperties          *o,                                      \n"
"             MosaicDatas         *mdatas)                                     \n"
"{                                                                             \n"
"  gdouble cx = 0.0;                                                           \n"
"  gdouble cy = 0.0;                                                           \n"
"  gfloat  col[4];                                                             \n"
"                                                                              \n"
"  polygon_find_center (poly, &cx, &cy);                                       \n"
"                                                                              \n"
"  if (o->color_averaging)                                                     \n"
"    find_poly_color (poly, input_buf, col, vary, result);                     \n"
"                                                                              \n"
"  scale_poly (poly, cx, cy, mdatas->scale);                                   \n"
"                                                                              \n"
"  if (o->color_averaging)                                                     \n"
"    fill_poly_color (poly, input_buf, output_buf, col, result,                \n"
"                     o->antialiasing, o->tile_surface, o->tile_height, mdatas);\n"
"  else                                                                        \n"
"    fill_poly_image (poly, input_buf, output_buf, vary, result,               \n"
"                     o->antialiasing, o->tile_surface, o->tile_height, mdatas);\n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"split_poly (Polygon             *poly,                                        \n"
"            gfloat              *input_buf,                                   \n"
"            gfloat              *output_buf,                                  \n"
"            gdouble             *dir,                                         \n"
"            gdouble              vary,                                        \n"
"            const GeglRectangle *result,                                      \n"
"            GeglProperties          *o,                                       \n"
"            MosaicDatas         *mdatas)                                      \n"
"{                                                                             \n"
"  Polygon new_poly;                                                           \n"
"  gdouble spacing;                                                            \n"
"  gdouble cx = 0.0;                                                           \n"
"  gdouble cy = 0.0;                                                           \n"
"  gdouble magnitude;                                                          \n"
"  gdouble vec[2];                                                             \n"
"  gdouble pt[2];                                                              \n"
"  gfloat  col[4];                                                             \n"
"                                                                              \n"
"  spacing = o->tile_spacing / (2.0 * mdatas->scale);                          \n"
"                                                                              \n"
"  polygon_find_center (poly, &cx, &cy);                                       \n"
"  polygon_translate (poly, -cx, -cy);                                         \n"
"                                                                              \n"
"  magnitude = sqrt (SQR (dir[0] - 0.5) + SQR (dir[1] - 0.5));                 \n"
"  vec[0] = - (dir[1] - 0.5) / magnitude;                                      \n"
"  vec[1] =   (dir[0] - 0.5) / magnitude;                                      \n"
"  pt[0] = -vec[1] * spacing;                                                  \n"
"  pt[1] =  vec[0] * spacing;                                                  \n"
"                                                                              \n"
"  polygon_reset (&new_poly);                                                  \n"
"  clip_poly (vec, pt, poly, &new_poly);                                       \n"
"  polygon_translate (&new_poly, cx, cy);                                      \n"
"                                                                              \n"
"  if (new_poly.npts)                                                          \n"
"    {                                                                         \n"
"      if (o->color_averaging)                                                 \n"
"        find_poly_color (&new_poly, input_buf, col, vary, result);            \n"
"                                                                              \n"
"      scale_poly (&new_poly, cx, cy, mdatas->scale);                          \n"
"                                                                              \n"
"      if (o->color_averaging)                                                 \n"
"        fill_poly_color (&new_poly, input_buf, output_buf, col, result,       \n"
"                         o->antialiasing, o->tile_surface, o->tile_height,    \n"
"                         mdatas);                                             \n"
"      else                                                                    \n"
"        fill_poly_image (&new_poly, input_buf, output_buf, vary, result,      \n"
"                         o->antialiasing, o->tile_surface, o->tile_height,    \n"
"                         mdatas);                                             \n"
"    }                                                                         \n"
"                                                                              \n"
"  vec[0] = -vec[0];                                                           \n"
"  vec[1] = -vec[1];                                                           \n"
"  pt[0] = -pt[0];                                                             \n"
"  pt[1] = -pt[1];                                                             \n"
"                                                                              \n"
"  polygon_reset (&new_poly);                                                  \n"
"  clip_poly (vec, pt, poly, &new_poly);                                       \n"
"  polygon_translate (&new_poly, cx, cy);                                      \n"
"                                                                              \n"
"  if (new_poly.npts)                                                          \n"
"    {                                                                         \n"
"      if (o->color_averaging)                                                 \n"
"        find_poly_color (&new_poly, input_buf, col, vary, result);            \n"
"                                                                              \n"
"      scale_poly (&new_poly, cx, cy, mdatas->scale);                          \n"
"                                                                              \n"
"      if (o->color_averaging)                                                 \n"
"        fill_poly_color (&new_poly, input_buf, output_buf, col, result,       \n"
"                         o->antialiasing, o->tile_surface, o->tile_height,    \n"
"                         mdatas);                                             \n"
"      else                                                                    \n"
"        fill_poly_image (&new_poly, input_buf, output_buf, vary, result,      \n"
"                         o->antialiasing, o->tile_surface, o->tile_height,    \n"
"                         mdatas);                                             \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"clip_poly (gdouble *dir,                                                      \n"
"           gdouble *pt,                                                       \n"
"           Polygon *poly,                                                     \n"
"           Polygon *poly_new)                                                 \n"
"{                                                                             \n"
"  gint    i;                                                                  \n"
"  gdouble x1, y1, x2, y2;                                                     \n"
"                                                                              \n"
"  for (i = 0; i < poly->npts; i++)                                            \n"
"    {                                                                         \n"
"      x1 = (i) ? poly->pts[i-1].x : poly->pts[poly->npts-1].x;                \n"
"      y1 = (i) ? poly->pts[i-1].y : poly->pts[poly->npts-1].y;                \n"
"      x2 = poly->pts[i].x;                                                    \n"
"      y2 = poly->pts[i].y;                                                    \n"
"                                                                              \n"
"      clip_point (dir, pt, x1, y1, x2, y2, poly_new);                         \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"clip_point (gdouble *dir,                                                     \n"
"            gdouble *pt,                                                      \n"
"            gdouble  x1,                                                      \n"
"            gdouble  y1,                                                      \n"
"            gdouble  x2,                                                      \n"
"            gdouble  y2,                                                      \n"
"            Polygon *poly_new)                                                \n"
"{                                                                             \n"
"  gdouble det, m11, m12;                                                      \n"
"  gdouble side1, side2;                                                       \n"
"  gdouble t;                                                                  \n"
"  gdouble vec[2];                                                             \n"
"                                                                              \n"
"  x1 -= pt[0]; x2 -= pt[0];                                                   \n"
"  y1 -= pt[1]; y2 -= pt[1];                                                   \n"
"                                                                              \n"
"  side1 = x1 * -dir[1] + y1 * dir[0];                                         \n"
"  side2 = x2 * -dir[1] + y2 * dir[0];                                         \n"
"                                                                              \n"
"  /*  If both points are to be clipped, ignore  */                            \n"
"  if (side1 < 0.0 && side2 < 0.0)                                             \n"
"    {                                                                         \n"
"      return;                                                                 \n"
"    }                                                                         \n"
"  /*  If both points are non-clipped, set point  */                           \n"
"  else if (side1 >= 0.0 && side2 >= 0.0)                                      \n"
"    {                                                                         \n"
"      polygon_add_point (poly_new, x2 + pt[0], y2 + pt[1]);                   \n"
"      return;                                                                 \n"
"    }                                                                         \n"
"  /*  Otherwise, there is an intersection...  */                              \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      vec[0] = x1 - x2;                                                       \n"
"      vec[1] = y1 - y2;                                                       \n"
"      det = dir[0] * vec[1] - dir[1] * vec[0];                                \n"
"                                                                              \n"
"      if (det == 0.0)                                                         \n"
"        {                                                                     \n"
"          polygon_add_point (poly_new, x2 + pt[0], y2 + pt[1]);               \n"
"          return;                                                             \n"
"        }                                                                     \n"
"                                                                              \n"
"      m11 = vec[1] / det;                                                     \n"
"      m12 = -vec[0] / det;                                                    \n"
"                                                                              \n"
"      t = m11 * x1 + m12 * y1;                                                \n"
"                                                                              \n"
"      /*  If the first point is clipped, set intersection and point  */       \n"
"      if (side1 < 0.0 && side2 > 0.0)                                         \n"
"        {                                                                     \n"
"          polygon_add_point (poly_new, dir[0] * t + pt[0], dir[1] * t + pt[1]);\n"
"          polygon_add_point (poly_new, x2 + pt[0], y2 + pt[1]);               \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          polygon_add_point (poly_new, dir[0] * t + pt[0], dir[1] * t + pt[1]);\n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"find_poly_dir (Polygon             *poly,                                     \n"
"               gfloat              *m_gr,                                     \n"
"               gfloat              *h_gr,                                     \n"
"               gfloat              *v_gr,                                     \n"
"               gdouble             *dir,                                      \n"
"               gdouble             *loc,                                      \n"
"               const GeglRectangle *result)                                   \n"
"{                                                                             \n"
"  gdouble dmin_x = 0.0, dmin_y = 0.0;                                         \n"
"  gdouble dmax_x = 0.0, dmax_y = 0.0;                                         \n"
"  gint    xs, ys;                                                             \n"
"  gint    xe, ye;                                                             \n"
"  gint    min_x, min_y;                                                       \n"
"  gint    max_x, max_y;                                                       \n"
"  gint    size_y;                                                             \n"
"  gint   *max_scanlines;                                                      \n"
"  gint   *min_scanlines;                                                      \n"
"  gfloat *dm, *dv, *dh;                                                       \n"
"  gint    count, total;                                                       \n"
"  gint    rowstride;                                                          \n"
"  gint    i, j;                                                               \n"
"                                                                              \n"
"  rowstride = result->width;                                                  \n"
"  count = 0;                                                                  \n"
"  total = 0;                                                                  \n"
"  dir[0] = 0.0;                                                               \n"
"  dir[1] = 0.0;                                                               \n"
"  loc[0] = 0.0;                                                               \n"
"  loc[1] = 0.0;                                                               \n"
"                                                                              \n"
"  polygon_extents (poly, &dmin_x, &dmin_y, &dmax_x, &dmax_y);                 \n"
"                                                                              \n"
"  min_x = (gint) dmin_x;                                                      \n"
"  min_y = (gint) dmin_y;                                                      \n"
"  max_x = (gint) dmax_x;                                                      \n"
"  max_y = (gint) dmax_y;                                                      \n"
"                                                                              \n"
"  size_y = max_y - min_y;                                                     \n"
"                                                                              \n"
"  min_scanlines = g_new (gint, size_y);                                       \n"
"  max_scanlines = g_new (gint, size_y);                                       \n"
"                                                                              \n"
"  for (i = 0; i < size_y; i++)                                                \n"
"    {                                                                         \n"
"      min_scanlines[i] = max_x;                                               \n"
"      max_scanlines[i] = min_x;                                               \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (i = 0; i < poly->npts; i++)                                            \n"
"    {                                                                         \n"
"      xs = (gint) ((i) ? poly->pts[i-1].x : poly->pts[poly->npts-1].x);       \n"
"      ys = (gint) ((i) ? poly->pts[i-1].y : poly->pts[poly->npts-1].y);       \n"
"      xe = (gint) poly->pts[i].x;                                             \n"
"      ye = (gint) poly->pts[i].y;                                             \n"
"                                                                              \n"
"      convert_segment (xs, ys, xe, ye, min_y,                                 \n"
"                       min_scanlines, max_scanlines);                         \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (i = 0; i < size_y; i++)                                                \n"
"    {                                                                         \n"
"      if ((i + min_y) >= 0 && (i + min_y) < result->height)                   \n"
"        {                                                                     \n"
"          dm = m_gr + (i + min_y) * rowstride;                                \n"
"          dh = h_gr + (i + min_y) * rowstride;                                \n"
"          dv = v_gr + (i + min_y) * rowstride;                                \n"
"                                                                              \n"
"          for (j = min_scanlines[i]; j < max_scanlines[i]; j++)               \n"
"            {                                                                 \n"
"              if (j >= 0 && j < result->width)                                \n"
"                {                                                             \n"
"                  if (dm[j] > MAG_THRESHOLD)                                  \n"
"                    {                                                         \n"
"                      dir[0] += dh[j];                                        \n"
"                      dir[1] += dv[j];                                        \n"
"                      loc[0] += j;                                            \n"
"                      loc[1] += i + min_y;                                    \n"
"                      count++;                                                \n"
"                    }                                                         \n"
"                  total++;                                                    \n"
"                }                                                             \n"
"            }                                                                 \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (!total)                                                                 \n"
"    {                                                                         \n"
"      g_free (max_scanlines);                                                 \n"
"      g_free (min_scanlines);                                                 \n"
"      return;                                                                 \n"
"    }                                                                         \n"
"                                                                              \n"
"  if ((gdouble) count / (gdouble) total > COUNT_THRESHOLD)                    \n"
"    {                                                                         \n"
"      dir[0] /= count;                                                        \n"
"      dir[1] /= count;                                                        \n"
"      loc[0] /= count;                                                        \n"
"      loc[1] /= count;                                                        \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      dir[0] = 0.5;                                                           \n"
"      dir[1] = 0.5;                                                           \n"
"      loc[0] = 0.0;                                                           \n"
"      loc[1] = 0.0;                                                           \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (min_scanlines);                                                     \n"
"  g_free (max_scanlines);                                                     \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"find_poly_color (Polygon             *poly,                                   \n"
"                 gfloat              *input_buf,                              \n"
"                 gfloat              *col,                                    \n"
"                 gdouble              color_var,                              \n"
"                 const GeglRectangle *result)                                 \n"
"{                                                                             \n"
"  gdouble       dmin_x = 0.0, dmin_y = 0.0;                                   \n"
"  gdouble       dmax_x = 0.0, dmax_y = 0.0;                                   \n"
"  gint          xs, ys, xe, ye;                                               \n"
"  gint          min_x, min_y, max_x, max_y;                                   \n"
"  gint          size_y;                                                       \n"
"  gint         *max_scanlines, *min_scanlines;                                \n"
"  gfloat        col_sum[4] = {0, 0, 0, 0};                                    \n"
"  gint          b, count;                                                     \n"
"  gint          i, j, y;                                                      \n"
"                                                                              \n"
"  count = 0;                                                                  \n"
"                                                                              \n"
"  polygon_extents (poly, &dmin_x, &dmin_y, &dmax_x, &dmax_y);                 \n"
"                                                                              \n"
"  min_x = (gint) dmin_x;                                                      \n"
"  min_y = (gint) dmin_y;                                                      \n"
"  max_x = (gint) dmax_x;                                                      \n"
"  max_y = (gint) dmax_y;                                                      \n"
"                                                                              \n"
"  size_y = max_y - min_y;                                                     \n"
"                                                                              \n"
"  min_scanlines = g_new (gint, size_y);                                       \n"
"  max_scanlines = g_new (gint, size_y);                                       \n"
"                                                                              \n"
"  for (i = 0; i < size_y; i++)                                                \n"
"    {                                                                         \n"
"      min_scanlines[i] = max_x;                                               \n"
"      max_scanlines[i] = min_x;                                               \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (i = 0; i < poly->npts; i++)                                            \n"
"    {                                                                         \n"
"      xs = (gint) ((i) ? poly->pts[i-1].x : poly->pts[poly->npts-1].x);       \n"
"      ys = (gint) ((i) ? poly->pts[i-1].y : poly->pts[poly->npts-1].y);       \n"
"      xe = (gint) poly->pts[i].x;                                             \n"
"      ye = (gint) poly->pts[i].y;                                             \n"
"                                                                              \n"
"      convert_segment (xs, ys, xe, ye, min_y,                                 \n"
"                       min_scanlines, max_scanlines);                         \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (i = 0; i < size_y; i++)                                                \n"
"    {                                                                         \n"
"      y = i + min_y;                                                          \n"
"                                                                              \n"
"      if (y >= 0 && y < result->height)                                       \n"
"        {                                                                     \n"
"          for (j = min_scanlines[i]; j < max_scanlines[i]; j++)               \n"
"            {                                                                 \n"
"              if (j >= 0 && j < result->width)                                \n"
"                {                                                             \n"
"                  gint id = (j + y * result->width) * NB_CPN;                 \n"
"                                                                              \n"
"                  for (b = 0; b < NB_CPN; b++)                                \n"
"                    col_sum[b] += input_buf[id + b];                          \n"
"                                                                              \n"
"                  count++;                                                    \n"
"                }                                                             \n"
"            }                                                                 \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (count)                                                                  \n"
"    {                                                                         \n"
"      /* but alpha */                                                         \n"
"      for (b = 0; b < NB_CPN-1; b++)                                          \n"
"        {                                                                     \n"
"          col_sum[b] = (col_sum[b] / count + color_var);                      \n"
"          col[b] = CLAMP01 (col_sum[b]);                                      \n"
"        }                                                                     \n"
"      col_sum[NB_CPN -1] = (col_sum[NB_CPN - 1] / count);                     \n"
"      col[NB_CPN -1] = CLAMP01 (col_sum[NB_CPN -1]);                          \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (min_scanlines);                                                     \n"
"  g_free (max_scanlines);                                                     \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"scale_poly (Polygon *poly,                                                    \n"
"            gdouble  tx,                                                      \n"
"            gdouble  ty,                                                      \n"
"            gdouble  poly_scale)                                              \n"
"{                                                                             \n"
"  polygon_translate (poly, -tx, -ty);                                         \n"
"  polygon_scale (poly, poly_scale);                                           \n"
"  polygon_translate (poly, tx, ty);                                           \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"fill_poly_color (Polygon             *poly,                                   \n"
"                 gfloat              *input_buf,                              \n"
"                 gfloat              *output_buf,                             \n"
"                 gfloat              *col,                                    \n"
"                 const GeglRectangle *result,                                 \n"
"                 gboolean             antialiasing,                           \n"
"                 gboolean             tile_rough,                             \n"
"                 gdouble              tile_height,                            \n"
"                 MosaicDatas         *mdatas)                                 \n"
"{                                                                             \n"
"  gdouble    dmin_x = 0.0, dmin_y = 0.0;                                      \n"
"  gdouble    dmax_x = 0.0, dmax_y = 0.0;                                      \n"
"  gint       xs, ys, xe, ye;                                                  \n"
"  gint       min_x, min_y, max_x, max_y;                                      \n"
"  gint       size_x, size_y;                                                  \n"
"  gint      *max_scanlines, *max_scanlines_iter;                              \n"
"  gint      *min_scanlines, *min_scanlines_iter;                              \n"
"  gfloat    *vals;                                                            \n"
"  gfloat     val, pixel;                                                      \n"
"  gfloat     buf[NB_CPN];                                                     \n"
"  gint       b, i, j, k, x, y;                                                \n"
"  gdouble    contrib;                                                         \n"
"  gdouble    xx, yy;                                                          \n"
"  gint       supersample, supersample2;                                       \n"
"  Vertex    *pts_tmp;                                                         \n"
"  const gint poly_npts = poly->npts;                                          \n"
"  SpecVec    vecs[MAX_POINTS];                                                \n"
"                                                                              \n"
"  /*  Determine antialiasing  */                                              \n"
"  if (antialiasing)                                                           \n"
"    {                                                                         \n"
"      supersample = SUPERSAMPLE;                                              \n"
"      supersample2 = SQR (supersample);                                       \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      supersample = supersample2 = 1;                                         \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (poly_npts)                                                              \n"
"    {                                                                         \n"
"      pts_tmp = poly->pts;                                                    \n"
"      xs = (gint) pts_tmp[poly_npts - 1].x;                                   \n"
"      ys = (gint) pts_tmp[poly_npts - 1].y;                                   \n"
"      xe = (gint) pts_tmp->x;                                                 \n"
"      ye = (gint) pts_tmp->y;                                                 \n"
"                                                                              \n"
"      calc_spec_vec (vecs, xs, ys, xe, ye,                                    \n"
"                     mdatas->light_x, mdatas->light_y);                       \n"
"                                                                              \n"
"      for (i = 1; i < poly_npts; i++)                                         \n"
"        {                                                                     \n"
"          xs = (gint) (pts_tmp->x);                                           \n"
"          ys = (gint) (pts_tmp->y);                                           \n"
"          pts_tmp++;                                                          \n"
"          xe = (gint) pts_tmp->x;                                             \n"
"          ye = (gint) pts_tmp->y;                                             \n"
"                                                                              \n"
"          calc_spec_vec (vecs+i, xs, ys, xe, ye,                              \n"
"                         mdatas->light_x, mdatas->light_y);                   \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  polygon_extents (poly, &dmin_x, &dmin_y, &dmax_x, &dmax_y);                 \n"
"                                                                              \n"
"  min_x = (gint) dmin_x;                                                      \n"
"  min_y = (gint) dmin_y;                                                      \n"
"  max_x = (gint) dmax_x;                                                      \n"
"  max_y = (gint) dmax_y;                                                      \n"
"                                                                              \n"
"  size_y = (max_y - min_y) * supersample;                                     \n"
"  size_x = (max_x - min_x) * supersample;                                     \n"
"                                                                              \n"
"  min_scanlines = min_scanlines_iter = g_new (gint, size_y);                  \n"
"  max_scanlines = max_scanlines_iter = g_new (gint, size_y);                  \n"
"                                                                              \n"
"  for (i = 0; i < size_y; i++)                                                \n"
"    {                                                                         \n"
"      min_scanlines[i] = max_x * supersample;                                 \n"
"      max_scanlines[i] = min_x * supersample;                                 \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (poly_npts)                                                              \n"
"    {                                                                         \n"
"      pts_tmp = poly->pts;                                                    \n"
"      xs = (gint) pts_tmp[poly_npts-1].x;                                     \n"
"      ys = (gint) pts_tmp[poly_npts-1].y;                                     \n"
"      xe = (gint) pts_tmp->x;                                                 \n"
"      ye = (gint) pts_tmp->y;                                                 \n"
"                                                                              \n"
"      xs *= supersample;                                                      \n"
"      ys *= supersample;                                                      \n"
"      xe *= supersample;                                                      \n"
"      ye *= supersample;                                                      \n"
"                                                                              \n"
"      convert_segment (xs, ys, xe, ye, min_y * supersample,                   \n"
"                       min_scanlines, max_scanlines);                         \n"
"                                                                              \n"
"      for (i = 1; i < poly_npts; i++)                                         \n"
"        {                                                                     \n"
"          xs = (gint) pts_tmp->x;                                             \n"
"          ys = (gint) pts_tmp->y;                                             \n"
"          pts_tmp++;                                                          \n"
"          xe = (gint) pts_tmp->x;                                             \n"
"          ye = (gint) pts_tmp->y;                                             \n"
"                                                                              \n"
"          xs *= supersample;                                                  \n"
"          ys *= supersample;                                                  \n"
"          xe *= supersample;                                                  \n"
"          ye *= supersample;                                                  \n"
"                                                                              \n"
"          convert_segment (xs, ys, xe, ye, min_y * supersample,               \n"
"                           min_scanlines, max_scanlines);                     \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  vals = g_new (gfloat, size_x);                                              \n"
"  for (i = 0; i < size_y; i++, min_scanlines_iter++, max_scanlines_iter++)    \n"
"    {                                                                         \n"
"      if (! (i % supersample))                                                \n"
"        memset (vals, 0, sizeof (gfloat) * size_x);                           \n"
"                                                                              \n"
"      yy = (gdouble) i / (gdouble) supersample + min_y;                       \n"
"                                                                              \n"
"      for (j = *min_scanlines_iter; j < *max_scanlines_iter; j++)             \n"
"        {                                                                     \n"
"          x = j - min_x * supersample;                                        \n"
"          vals[x] += 1.0;                                                     \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (! ((i + 1) % supersample))                                          \n"
"        {                                                                     \n"
"          y = (i / supersample) + min_y;                                      \n"
"                                                                              \n"
"          if (y >= 0 && y < result->height)                                   \n"
"            {                                                                 \n"
"              for (j = 0; j < size_x; j += supersample)                       \n"
"                {                                                             \n"
"                  x = (j / supersample) + min_x;                              \n"
"                                                                              \n"
"                  if (x >= 0 && x < result->width)                            \n"
"                    {                                                         \n"
"                      val = 0;                                                \n"
"                                                                              \n"
"                      for (k = 0; k < supersample; k++)                       \n"
"                        val += vals[j + k];                                   \n"
"                                                                              \n"
"                      val /= supersample2;                                    \n"
"                                                                              \n"
"                      if (val > 0)                                            \n"
"                        {                                                     \n"
"                          xx = (gdouble) j / (gdouble) supersample + min_x;   \n"
"                                                                              \n"
"                          contrib = calc_spec_contrib (vecs, poly_npts,       \n"
"                                                       xx, yy, tile_rough,    \n"
"                                                       tile_height);          \n"
"                                                                              \n"
"                          for (b = 0; b < NB_CPN; b++)                        \n"
"                            {                                                 \n"
"                              pixel = col[b];                                 \n"
"                                                                              \n"
"                              if (contrib < 0.0)                              \n"
"                                pixel += (col[b] - mdatas->back[b]) * contrib;\n"
"                              else                                            \n"
"                                pixel += (mdatas->fore[b] - col[b]) * contrib;\n"
"                                                                              \n"
"                              buf[b] = ((pixel * val) + (mdatas->back[b] * (1.0 - val)));\n"
"                            }                                                 \n"
"                          memcpy (output_buf + (y * result->width + x) * NB_CPN,\n"
"                                  buf, NB_CPN * sizeof (gfloat));             \n"
"                        }                                                     \n"
"                    }                                                         \n"
"                }                                                             \n"
"            }                                                                 \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (vals);                                                              \n"
"  g_free (min_scanlines);                                                     \n"
"  g_free (max_scanlines);                                                     \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"fill_poly_image (Polygon             *poly,                                   \n"
"                 gfloat              *input_buf,                              \n"
"                 gfloat              *output_buf,                             \n"
"                 gdouble              vary,                                   \n"
"                 const GeglRectangle *result,                                 \n"
"                 gboolean             antialiasing,                           \n"
"                 gboolean             tile_rough,                             \n"
"                 gdouble              tile_height,                            \n"
"                 MosaicDatas         *mdatas)                                 \n"
"{                                                                             \n"
"  gdouble   dmin_x = 0.0, dmin_y = 0.0;                                       \n"
"  gdouble   dmax_x = 0.0, dmax_y = 0.0;                                       \n"
"  gint      xs, ys, xe, ye;                                                   \n"
"  gint      min_x, min_y, max_x, max_y;                                       \n"
"  gint      size_x, size_y;                                                   \n"
"  gint     *max_scanlines, *min_scanlines;                                    \n"
"  gfloat   *vals;                                                             \n"
"  gfloat    val;                                                              \n"
"  gfloat    pixel;                                                            \n"
"  gfloat    buf[NB_CPN];                                                      \n"
"  gint      b, i, j, k, x, y;                                                 \n"
"  gdouble   contrib;                                                          \n"
"  gdouble   xx, yy;                                                           \n"
"  gint      supersample, supersample2;                                        \n"
"  SpecVec   vecs[MAX_POINTS];                                                 \n"
"                                                                              \n"
"  /*  Determine antialiasing  */                                              \n"
"  if (antialiasing)                                                           \n"
"    {                                                                         \n"
"      supersample = SUPERSAMPLE;                                              \n"
"      supersample2 = SQR (supersample);                                       \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      supersample = supersample2 = 1;                                         \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (i = 0; i < poly->npts; i++)                                            \n"
"    {                                                                         \n"
"      xs = (gint) ((i) ? poly->pts[i-1].x : poly->pts[poly->npts-1].x);       \n"
"      ys = (gint) ((i) ? poly->pts[i-1].y : poly->pts[poly->npts-1].y);       \n"
"      xe = (gint) poly->pts[i].x;                                             \n"
"      ye = (gint) poly->pts[i].y;                                             \n"
"                                                                              \n"
"      calc_spec_vec (vecs+i, xs, ys, xe, ye,                                  \n"
"                     mdatas->light_x, mdatas->light_y);                       \n"
"    }                                                                         \n"
"                                                                              \n"
"  polygon_extents (poly, &dmin_x, &dmin_y, &dmax_x, &dmax_y);                 \n"
"                                                                              \n"
"  min_x = (gint) dmin_x;                                                      \n"
"  min_y = (gint) dmin_y;                                                      \n"
"  max_x = (gint) dmax_x;                                                      \n"
"  max_y = (gint) dmax_y;                                                      \n"
"                                                                              \n"
"  size_y = (max_y - min_y) * supersample;                                     \n"
"  size_x = (max_x - min_x) * supersample;                                     \n"
"                                                                              \n"
"  min_scanlines = g_new (gint, size_y);                                       \n"
"  max_scanlines = g_new (gint, size_y);                                       \n"
"                                                                              \n"
"  for (i = 0; i < size_y; i++)                                                \n"
"    {                                                                         \n"
"      min_scanlines[i] = max_x * supersample;                                 \n"
"      max_scanlines[i] = min_x * supersample;                                 \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (i = 0; i < poly->npts; i++)                                            \n"
"    {                                                                         \n"
"      xs = (gint) ((i) ? poly->pts[i-1].x : poly->pts[poly->npts-1].x);       \n"
"      ys = (gint) ((i) ? poly->pts[i-1].y : poly->pts[poly->npts-1].y);       \n"
"      xe = (gint) poly->pts[i].x;                                             \n"
"      ye = (gint) poly->pts[i].y;                                             \n"
"                                                                              \n"
"      xs *= supersample;                                                      \n"
"      ys *= supersample;                                                      \n"
"      xe *= supersample;                                                      \n"
"      ye *= supersample;                                                      \n"
"                                                                              \n"
"      convert_segment (xs, ys, xe, ye, min_y * supersample,                   \n"
"                       min_scanlines, max_scanlines);                         \n"
"    }                                                                         \n"
"                                                                              \n"
"  vals = g_new (gfloat, size_x);                                              \n"
"                                                                              \n"
"  for (i = 0; i < size_y; i++)                                                \n"
"    {                                                                         \n"
"      if (! (i % supersample))                                                \n"
"        memset (vals, 0, sizeof (gfloat) * size_x);                           \n"
"                                                                              \n"
"      yy = (gdouble) i / (gdouble) supersample + min_y;                       \n"
"                                                                              \n"
"      for (j = min_scanlines[i]; j < max_scanlines[i]; j++)                   \n"
"        {                                                                     \n"
"          x = j - min_x * supersample;                                        \n"
"          vals[x] += 1.0;                                                     \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (! ((i + 1) % supersample))                                          \n"
"        {                                                                     \n"
"          y = (i / supersample) + min_y;                                      \n"
"                                                                              \n"
"          if (y >= 0 && y < result->height)                                   \n"
"            {                                                                 \n"
"              for (j = 0; j < size_x; j += supersample)                       \n"
"                {                                                             \n"
"                  x = (j / supersample) + min_x;                              \n"
"                                                                              \n"
"                  if (x >= 0 && x < result->width)                            \n"
"                    {                                                         \n"
"                      val = 0;                                                \n"
"                      for (k = 0; k < supersample; k++)                       \n"
"                        val += vals[j + k];                                   \n"
"                      val /= supersample2;                                    \n"
"                                                                              \n"
"                      if (val > 0)                                            \n"
"                        {                                                     \n"
"                          xx = (gdouble) j / (gdouble) supersample + min_x;   \n"
"                                                                              \n"
"                          contrib = calc_spec_contrib (vecs,                  \n"
"                                                       poly->npts, xx, yy,    \n"
"                                                       tile_rough, tile_height);\n"
"                                                                              \n"
"                          memcpy (buf,                                        \n"
"                                  input_buf + (x + y*result->width) * NB_CPN, \n"
"                                  NB_CPN * sizeof (gfloat));                  \n"
"                                                                              \n"
"                          for (b = 0; b < NB_CPN; b++)                        \n"
"                            {                                                 \n"
"                              if (contrib < 0.0)                              \n"
"                                pixel = buf[b] + (buf[b] - mdatas->back[b]) * contrib;\n"
"                              else                                            \n"
"                                pixel = buf[b] + (mdatas->fore[b] - buf[b]) * contrib;\n"
"                                                                              \n"
"                              /* factor in per-tile intensity variation but alpha */\n"
"                              if (b < NB_CPN - 1)                             \n"
"                                pixel += vary;                                \n"
"                                                                              \n"
"                              pixel = CLAMP (pixel, 0.0, 1.0);                \n"
"                                                                              \n"
"                              buf[b] = (mdatas->back[b]) + (pixel - mdatas->back[b]) * val;\n"
"                            }                                                 \n"
"                                                                              \n"
"                          memcpy (output_buf + (y * result->width + x) * NB_CPN,\n"
"                                  buf, NB_CPN * sizeof (gfloat));             \n"
"                        }                                                     \n"
"                    }                                                         \n"
"                }                                                             \n"
"            }                                                                 \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (vals);                                                              \n"
"  g_free (min_scanlines);                                                     \n"
"  g_free (max_scanlines);                                                     \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"calc_spec_vec (SpecVec *vec,                                                  \n"
"               gint     x1,                                                   \n"
"               gint     y1,                                                   \n"
"               gint     x2,                                                   \n"
"               gint     y2,                                                   \n"
"               gdouble  light_x,                                              \n"
"               gdouble  light_y)                                              \n"
"{                                                                             \n"
"  gdouble r;                                                                  \n"
"                                                                              \n"
"  vec->base_x = x1;                                                           \n"
"  vec->base_y = y1;                                                           \n"
"  vec->base_x2 = x2;                                                          \n"
"  vec->base_y2 = y2;                                                          \n"
"                                                                              \n"
"  r = sqrt (SQR (x2 - x1) + SQR (y2 - y1));                                   \n"
"                                                                              \n"
"  if (r > GEGL_FLOAT_EPSILON)                                                 \n"
"    {                                                                         \n"
"      vec->norm_x = - (y2 - y1) / r;                                          \n"
"      vec->norm_y =   (x2 - x1) / r;                                          \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      vec->norm_x = 0;                                                        \n"
"      vec->norm_y = 0;                                                        \n"
"    }                                                                         \n"
"                                                                              \n"
"  vec->light = vec->norm_x * light_x + vec->norm_y * light_y;                 \n"
"}                                                                             \n"
"                                                                              \n"
"static gfloat                                                                 \n"
"distance (SpecVec *vec,                                                       \n"
"          gfloat   x,                                                         \n"
"          gfloat   y)                                                         \n"
"{                                                                             \n"
"  gfloat l2, t;                                                               \n"
"  gfloat pv_x, pv_y;                                                          \n"
"  gfloat pw_x, pw_y;                                                          \n"
"  gfloat wv_x, wv_y;                                                          \n"
"  gfloat proj_x, proj_y;                                                      \n"
"                                                                              \n"
"  l2 = SQR (vec->base_x - vec->base_x2) + SQR (vec->base_y - vec->base_y2);   \n"
"                                                                              \n"
"  if (l2 < GEGL_FLOAT_EPSILON)                                                \n"
"    return sqrt (SQR (vec->base_x - x) + SQR (vec->base_y - y));              \n"
"                                                                              \n"
"  pv_x = x - vec->base_x;                                                     \n"
"  pv_y = y - vec->base_y;                                                     \n"
"                                                                              \n"
"  pw_x = x - vec->base_x2;                                                    \n"
"  pw_y = y - vec->base_y2;                                                    \n"
"                                                                              \n"
"  wv_x = vec->base_x2 - vec->base_x;                                          \n"
"  wv_y = vec->base_y2 - vec->base_y;                                          \n"
"                                                                              \n"
"  t = (pv_x * wv_x + pv_y * wv_y) / l2;                                       \n"
"                                                                              \n"
"  if (t < 0.0)                                                                \n"
"    return sqrt (SQR (pv_x) + SQR (pv_y));                                    \n"
"                                                                              \n"
"  else if (t > 1.0)                                                           \n"
"    return sqrt (SQR (pw_x) + SQR (pw_y));                                    \n"
"                                                                              \n"
"  proj_x = vec->base_x + t * wv_x;                                            \n"
"  proj_y = vec->base_y + t * wv_y;                                            \n"
"                                                                              \n"
"  return sqrt (SQR (x - proj_x) + SQR (y - proj_y));                          \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static double                                                                 \n"
"calc_spec_contrib (SpecVec *vecs,                                             \n"
"                   gint     n,                                                \n"
"                   gdouble  x,                                                \n"
"                   gdouble  y,                                                \n"
"                   gboolean tile_rough,                                       \n"
"                   gdouble  tile_height)                                      \n"
"{                                                                             \n"
"  gint i;                                                                     \n"
"  gdouble contrib = 0;                                                        \n"
"                                                                              \n"
"  for (i = 0; i < n; i++)                                                     \n"
"    {                                                                         \n"
"      gfloat dist;                                                            \n"
"                                                                              \n"
"      dist = distance (vecs+i, x, y);                                         \n"
"                                                                              \n"
"      if (tile_rough)                                                         \n"
"        {                                                                     \n"
"          /*  If the surface is rough, randomly perturb the distance  */      \n"
"          dist -= dist * g_random_double ();                                  \n"
"        }                                                                     \n"
"                                                                              \n"
"      /*  If the distance to an edge is less than the tile_spacing, there     \n"
"       *  will be no highlight as the tile blends to background here          \n"
"       */                                                                     \n"
"      if (dist < 1.0)                                                         \n"
"        {                                                                     \n"
"          contrib += vecs[i].light;                                           \n"
"        }                                                                     \n"
"      else if (dist <= tile_height)                                           \n"
"        {                                                                     \n"
"          contrib += vecs[i].light * (1.0 - (dist / tile_height));            \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  return contrib / 4.0;                                                       \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"convert_segment (gint  x1,                                                    \n"
"                 gint  y1,                                                    \n"
"                 gint  x2,                                                    \n"
"                 gint  y2,                                                    \n"
"                 gint  offset,                                                \n"
"                 gint *min,                                                   \n"
"                 gint *max)                                                   \n"
"{                                                                             \n"
"  gint    ydiff, y, tmp;                                                      \n"
"  gdouble xinc, xstart;                                                       \n"
"                                                                              \n"
"  if (y1 > y2)                                                                \n"
"    {                                                                         \n"
"      tmp = y2; y2 = y1; y1 = tmp;                                            \n"
"      tmp = x2; x2 = x1; x1 = tmp;                                            \n"
"    }                                                                         \n"
"                                                                              \n"
"  ydiff = y2 - y1;                                                            \n"
"                                                                              \n"
"  if (ydiff)                                                                  \n"
"    {                                                                         \n"
"      xinc = (gdouble) (x2 - x1) / (gdouble) ydiff;                           \n"
"      xstart = x1 + 0.5 * xinc;                                               \n"
"                                                                              \n"
"      for (y = y1; y < y2; y++)                                               \n"
"        {                                                                     \n"
"          min[y - offset] = MIN (min[y - offset], xstart);                    \n"
"          max[y - offset] = MAX (max[y - offset], xstart);                    \n"
"                                                                              \n"
"          xstart += xinc;                                                     \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"polygon_add_point (Polygon *poly,                                             \n"
"                   gdouble  x,                                                \n"
"                   gdouble  y)                                                \n"
"{                                                                             \n"
"  if (poly->npts < 12)                                                        \n"
"    {                                                                         \n"
"      poly->pts[poly->npts].x = x;                                            \n"
"      poly->pts[poly->npts].y = y;                                            \n"
"      poly->npts++;                                                           \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      g_warning (\"can't add more points\");                                  \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"polygon_find_center (Polygon *poly,                                           \n"
"                     gdouble *cx,                                             \n"
"                     gdouble *cy)                                             \n"
"{                                                                             \n"
"  guint i;                                                                    \n"
"                                                                              \n"
"  if (!poly->npts)                                                            \n"
"    return FALSE;                                                             \n"
"                                                                              \n"
"  *cx = 0.0;                                                                  \n"
"  *cy = 0.0;                                                                  \n"
"                                                                              \n"
"  for (i = 0; i < poly->npts; i++)                                            \n"
"    {                                                                         \n"
"      *cx += poly->pts[i].x;                                                  \n"
"      *cy += poly->pts[i].y;                                                  \n"
"    }                                                                         \n"
"                                                                              \n"
"  *cx /= poly->npts;                                                          \n"
"  *cy /= poly->npts;                                                          \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"polygon_translate (Polygon *poly,                                             \n"
"                   gdouble  tx,                                               \n"
"                   gdouble  ty)                                               \n"
"{                                                                             \n"
"  guint i;                                                                    \n"
"                                                                              \n"
"  for (i = 0; i < poly->npts; i++)                                            \n"
"    {                                                                         \n"
"      poly->pts[i].x += tx;                                                   \n"
"      poly->pts[i].y += ty;                                                   \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"polygon_scale (Polygon *poly,                                                 \n"
"               gdouble  poly_scale)                                           \n"
"{                                                                             \n"
"  guint i;                                                                    \n"
"                                                                              \n"
"  for (i = 0; i < poly->npts; i++)                                            \n"
"    {                                                                         \n"
"      poly->pts[i].x *= poly_scale;                                           \n"
"      poly->pts[i].y *= poly_scale;                                           \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"polygon_extents (Polygon *poly,                                               \n"
"                 gdouble *min_x,                                              \n"
"                 gdouble *min_y,                                              \n"
"                 gdouble *max_x,                                              \n"
"                 gdouble *max_y)                                              \n"
"{                                                                             \n"
"  guint i;                                                                    \n"
"                                                                              \n"
"  if (!poly->npts)                                                            \n"
"    return FALSE;                                                             \n"
"                                                                              \n"
"  *min_x = *max_x = poly->pts[0].x;                                           \n"
"  *min_y = *max_y = poly->pts[0].y;                                           \n"
"                                                                              \n"
"  for (i = 1; i < poly->npts; i++)                                            \n"
"    {                                                                         \n"
"      *min_x = MIN (*min_x, poly->pts[i].x);                                  \n"
"      *max_x = MAX (*max_x, poly->pts[i].x);                                  \n"
"      *min_y = MIN (*min_y, poly->pts[i].y);                                  \n"
"      *max_y = MAX (*max_y, poly->pts[i].y);                                  \n"
"    }                                                                         \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"polygon_reset (Polygon *poly)                                                 \n"
"{                                                                             \n"
"  poly->npts = 0;                                                             \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"prepare (GeglOperation *operation)                                            \n"
"{                                                                             \n"
"  GeglOperationAreaFilter *area = GEGL_OPERATION_AREA_FILTER (operation);     \n"
"  GeglProperties              *o    = GEGL_PROPERTIES (operation);            \n"
"                                                                              \n"
"  /* Should be slightly larger than needed */                                 \n"
"  area->left = area->right = 2 * ceil (o->tile_size) + 1;                     \n"
"  area->top = area->bottom = 2 * ceil (o->tile_size) + 1;                     \n"
"                                                                              \n"
"  gegl_operation_set_format (operation, \"input\",                            \n"
"                             babl_format (\"R'G'B'A float\"));                \n"
"  gegl_operation_set_format (operation, \"output\",                           \n"
"                             babl_format (\"R'G'B'A float\"));                \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"process (GeglOperation       *operation,                                      \n"
"         GeglBuffer          *input,                                          \n"
"         GeglBuffer          *output,                                         \n"
"         const GeglRectangle *result,                                         \n"
"         gint                 level)                                          \n"
"{                                                                             \n"
"                                                                              \n"
"  GeglRectangle            working_region;                                    \n"
"  GeglRectangle           *whole_region;                                      \n"
"  GeglOperationAreaFilter *area;                                              \n"
"  gfloat                  *res;                                               \n"
"  gint                     rowstride, offset;                                 \n"
"                                                                              \n"
"  area = GEGL_OPERATION_AREA_FILTER (operation);                              \n"
"                                                                              \n"
"  whole_region = gegl_operation_source_get_bounding_box (operation, \"input\");\n"
"                                                                              \n"
"  working_region.x      = result->x - area->left;                             \n"
"  working_region.width  = result->width + area->left + area->right;           \n"
"  working_region.y      = result->y - area->top;                              \n"
"  working_region.height = result->height + area->top + area->bottom;          \n"
"                                                                              \n"
"  /* Don't try to compute things outside the input extent,                    \n"
"   * otherwise the behaviour is undefined */                                  \n"
"  gegl_rectangle_intersect (&working_region, &working_region, whole_region);  \n"
"                                                                              \n"
"  res = mosaic (operation, input, &working_region);                           \n"
"                                                                              \n"
"  /* The region computed is larger than needed,                               \n"
"   * so we extract the roi from the result of mosaic() */                     \n"
"  rowstride = (working_region.width) * sizeof (gfloat) * 4;                   \n"
"                                                                              \n"
"  offset  = (result->y - working_region.y) * working_region.width;            \n"
"  offset += (result->x - working_region.x);                                   \n"
"  offset *= 4;                                                                \n"
"                                                                              \n"
"  gegl_buffer_set (output, result, 0, babl_format (\"R'G'B'A float\"),        \n"
"                   res + offset, rowstride);                                  \n"
"                                                                              \n"
"  g_free (res);                                                               \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"get_bounding_box (GeglOperation *operation)                                   \n"
"{                                                                             \n"
"  GeglRectangle  result = { 0, 0, 0, 0 };                                     \n"
"  GeglRectangle *in_rect;                                                     \n"
"                                                                              \n"
"  in_rect = gegl_operation_source_get_bounding_box (operation, \"input\");    \n"
"                                                                              \n"
"  if (in_rect)                                                                \n"
"    {                                                                         \n"
"      result = *in_rect;                                                      \n"
"    }                                                                         \n"
"                                                                              \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GeglOperationClass       *operation_class;                                  \n"
"  GeglOperationFilterClass *filter_class;                                     \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  filter_class    = GEGL_OPERATION_FILTER_CLASS (klass);                      \n"
"                                                                              \n"
"  operation_class->prepare          = prepare;                                \n"
"  operation_class->get_bounding_box = get_bounding_box;                       \n"
"  filter_class->process             = process;                                \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"    \"name\",        \"gegl:mosaic\",                                         \n"
"    \"title\",       _(\"Mosaic\"),                                           \n"
"    \"categories\",  \"artistic:scramble\",                                   \n"
"    \"license\",     \"GPL3+\",                                               \n"
"    \"description\", _(\"Mosaic is a filter which transforms an image into \" \n"
"                     \"what appears to be a mosaic, composed of small primitives, \"\n"
"                     \"each of constant color and of an approximate size.\"), \n"
"    NULL);                                                                    \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
