static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Copyright 2015 Thomas Manni <thomas.manni@free.fr>                         \n"
" *                                                                            \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"#include <stdlib.h>                                                           \n"
"#include <math.h>                                                             \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"enum_start (gegl_median_blur_neighborhood)                                    \n"
"  enum_value (GEGL_MEDIAN_BLUR_NEIGHBORHOOD_SQUARE,  \"square\",  N_(\"Square\"))\n"
"  enum_value (GEGL_MEDIAN_BLUR_NEIGHBORHOOD_CIRCLE,  \"circle\",  N_(\"Circle\"))\n"
"  enum_value (GEGL_MEDIAN_BLUR_NEIGHBORHOOD_DIAMOND, \"diamond\", N_(\"Diamond\"))\n"
"enum_end (GeglMedianBlurNeighborhood)                                         \n"
"                                                                              \n"
"property_enum (neighborhood, _(\"Neighborhood\"),                             \n"
"               GeglMedianBlurNeighborhood, gegl_median_blur_neighborhood,     \n"
"               GEGL_MEDIAN_BLUR_NEIGHBORHOOD_CIRCLE)                          \n"
"  description (_(\"Neighborhood type\"))                                      \n"
"                                                                              \n"
"property_int  (radius, _(\"Radius\"), 3)                                      \n"
"  value_range (0, 100)                                                        \n"
"  ui_meta     (\"unit\", \"pixel-distance\")                                  \n"
"  description (_(\"Neighborhood radius\"))                                    \n"
"                                                                              \n"
"property_double  (percentile, _(\"Percentile\"), 50)                          \n"
"  value_range (0, 100)                                                        \n"
"  description (_(\"Neighborhood color percentile\"))                          \n"
"                                                                              \n"
"property_double  (alpha_percentile, _(\"Alpha percentile\"), 50)              \n"
"  value_range (0, 100)                                                        \n"
"  description (_(\"Neighborhood alpha percentile\"))                          \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_AREA_FILTER                                                   \n"
"#define GEGL_OP_NAME         median_blur                                      \n"
"#define GEGL_OP_C_SOURCE     median-blur.c                                    \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"                                                                              \n"
"#define N_BINS  1024                                                          \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  gint bins[N_BINS];                                                          \n"
"  gint last_median;                                                           \n"
"  gint last_median_sum;                                                       \n"
"} HistogramComponent;                                                         \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  HistogramComponent components[4];                                           \n"
"  gint               count;                                                   \n"
"  gint               size;                                                    \n"
"} Histogram;                                                                  \n"
"                                                                              \n"
"typedef enum                                                                  \n"
"{                                                                             \n"
"  LEFT_TO_RIGHT,                                                              \n"
"  RIGHT_TO_LEFT,                                                              \n"
"  TOP_TO_BOTTOM                                                               \n"
"} Direction;                                                                  \n"
"                                                                              \n"
"static inline gfloat                                                          \n"
"histogram_get_median (Histogram *hist,                                        \n"
"                      gint       component,                                   \n"
"                      gdouble    percentile)                                  \n"
"{                                                                             \n"
"  gint                count = hist->count;                                    \n"
"  HistogramComponent *comp  = &hist->components[component];                   \n"
"  gint                i     = comp->last_median;                              \n"
"  gint                sum   = comp->last_median_sum;                          \n"
"                                                                              \n"
"  if (component == 3)                                                         \n"
"    count = hist->size;                                                       \n"
"                                                                              \n"
"  if (count == 0)                                                             \n"
"    return 0.0f;                                                              \n"
"                                                                              \n"
"  count = (gint) ceil (count * percentile);                                   \n"
"  count = MAX (count, 1);                                                     \n"
"                                                                              \n"
"  if (sum < count)                                                            \n"
"    {                                                                         \n"
"      while ((sum += comp->bins[++i]) < count);                               \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      while ((sum -= comp->bins[i--]) >= count);                              \n"
"      sum += comp->bins[++i];                                                 \n"
"    }                                                                         \n"
"                                                                              \n"
"  comp->last_median     = i;                                                  \n"
"  comp->last_median_sum = sum;                                                \n"
"                                                                              \n"
"  return ((gfloat) i + .5f) / (gfloat) N_BINS;                                \n"
"}                                                                             \n"
"                                                                              \n"
"static inline void                                                            \n"
"histogram_modify_val (Histogram    *hist,                                     \n"
"                      const gint32 *src,                                      \n"
"                      gboolean      has_alpha,                                \n"
"                      gint          diff)                                     \n"
"{                                                                             \n"
"  gint alpha = diff;                                                          \n"
"  gint c;                                                                     \n"
"                                                                              \n"
"  if (has_alpha)                                                              \n"
"    alpha *= src[3];                                                          \n"
"                                                                              \n"
"  for (c = 0; c < 3; c++)                                                     \n"
"    {                                                                         \n"
"      HistogramComponent *comp = &hist->components[c];                        \n"
"      gint                bin  = src[c];                                      \n"
"                                                                              \n"
"      comp->bins[bin] += alpha;                                               \n"
"                                                                              \n"
"      /* this is shorthand for:                                               \n"
"       *                                                                      \n"
"       *   if (bin <= comp->last_median)                                      \n"
"       *     comp->last_median_sum += alpha;                                  \n"
"       *                                                                      \n"
"       * but with a notable speed boost.                                      \n"
"       */                                                                     \n"
"      comp->last_median_sum += (bin <= comp->last_median) * alpha;            \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (has_alpha)                                                              \n"
"    {                                                                         \n"
"      HistogramComponent *comp = &hist->components[3];                        \n"
"      gint                bin  = src[3];                                      \n"
"                                                                              \n"
"      comp->bins[bin] += diff;                                                \n"
"                                                                              \n"
"      comp->last_median_sum += (bin <= comp->last_median) * diff;             \n"
"    }                                                                         \n"
"                                                                              \n"
"  hist->count += alpha;                                                       \n"
"}                                                                             \n"
"                                                                              \n"
"static inline void                                                            \n"
"histogram_modify_vals (Histogram    *hist,                                    \n"
"                       const gint32 *src,                                     \n"
"                       gint          bpp,                                     \n"
"                       gint          stride,                                  \n"
"                       gboolean      has_alpha,                               \n"
"                       gint          xmin,                                    \n"
"                       gint          ymin,                                    \n"
"                       gint          xmax,                                    \n"
"                       gint          ymax,                                    \n"
"                       gint          diff)                                    \n"
"{                                                                             \n"
"  gint x;                                                                     \n"
"  gint y;                                                                     \n"
"                                                                              \n"
"  if (xmin > xmax || ymin > ymax)                                             \n"
"    return;                                                                   \n"
"                                                                              \n"
"  src += ymin * stride + xmin * bpp;                                          \n"
"                                                                              \n"
"  for (y = ymin; y <= ymax; y++, src += stride)                               \n"
"    {                                                                         \n"
"      const gint32 *pixel = src;                                              \n"
"                                                                              \n"
"      for (x = xmin; x <= xmax; x++, pixel += bpp)                            \n"
"        {                                                                     \n"
"          histogram_modify_val (hist, pixel, has_alpha, diff);                \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static inline void                                                            \n"
"histogram_update (Histogram                  *hist,                           \n"
"                  const gint32               *src,                            \n"
"                  gint                        bpp,                            \n"
"                  gint                        stride,                         \n"
"                  gboolean                    has_alpha,                      \n"
"                  GeglMedianBlurNeighborhood  neighborhood,                   \n"
"                  gint                        radius,                         \n"
"                  const gint                 *neighborhood_outline,           \n"
"                  Direction                   dir)                            \n"
"{                                                                             \n"
"  gint i;                                                                     \n"
"                                                                              \n"
"  switch (neighborhood)                                                       \n"
"    {                                                                         \n"
"    case GEGL_MEDIAN_BLUR_NEIGHBORHOOD_SQUARE:                                \n"
"      switch (dir)                                                            \n"
"        {                                                                     \n"
"          case LEFT_TO_RIGHT:                                                 \n"
"            histogram_modify_vals (hist, src, bpp, stride, has_alpha,         \n"
"                                   -radius - 1, -radius,                      \n"
"                                   -radius - 1, +radius,                      \n"
"                                   -1);                                       \n"
"                                                                              \n"
"            histogram_modify_vals (hist, src, bpp, stride, has_alpha,         \n"
"                                   +radius, -radius,                          \n"
"                                   +radius, +radius,                          \n"
"                                   +1);                                       \n"
"            break;                                                            \n"
"                                                                              \n"
"          case RIGHT_TO_LEFT:                                                 \n"
"            histogram_modify_vals (hist, src, bpp, stride, has_alpha,         \n"
"                                   +radius + 1, -radius,                      \n"
"                                   +radius + 1, +radius,                      \n"
"                                   -1);                                       \n"
"                                                                              \n"
"            histogram_modify_vals (hist, src, bpp, stride, has_alpha,         \n"
"                                   -radius, -radius,                          \n"
"                                   -radius, +radius,                          \n"
"                                   +1);                                       \n"
"            break;                                                            \n"
"                                                                              \n"
"          case TOP_TO_BOTTOM:                                                 \n"
"            histogram_modify_vals (hist, src, bpp, stride, has_alpha,         \n"
"                                   -radius, -radius - 1,                      \n"
"                                   +radius, -radius - 1,                      \n"
"                                   -1);                                       \n"
"                                                                              \n"
"            histogram_modify_vals (hist, src, bpp, stride, has_alpha,         \n"
"                                   -radius, +radius,                          \n"
"                                   +radius, +radius,                          \n"
"                                   +1);                                       \n"
"            break;                                                            \n"
"        }                                                                     \n"
"      break;                                                                  \n"
"                                                                              \n"
"    default:                                                                  \n"
"      switch (dir)                                                            \n"
"        {                                                                     \n"
"          case LEFT_TO_RIGHT:                                                 \n"
"            for (i = 0; i < radius; i++)                                      \n"
"              {                                                               \n"
"                histogram_modify_vals (hist, src, bpp, stride, has_alpha,     \n"
"                                       -i - 1, -neighborhood_outline[i],      \n"
"                                       -i - 1, -neighborhood_outline[i + 1] - 1,\n"
"                                       -1);                                   \n"
"                histogram_modify_vals (hist, src, bpp, stride, has_alpha,     \n"
"                                       -i - 1, +neighborhood_outline[i + 1] + 1,\n"
"                                       -i - 1, +neighborhood_outline[i],      \n"
"                                       -1);                                   \n"
"                                                                              \n"
"                histogram_modify_vals (hist, src, bpp, stride, has_alpha,     \n"
"                                       +i, -neighborhood_outline[i],          \n"
"                                       +i, -neighborhood_outline[i + 1] - 1,  \n"
"                                       +1);                                   \n"
"                histogram_modify_vals (hist, src, bpp, stride, has_alpha,     \n"
"                                       +i, +neighborhood_outline[i + 1] + 1,  \n"
"                                       +i, +neighborhood_outline[i],          \n"
"                                       +1);                                   \n"
"              }                                                               \n"
"                                                                              \n"
"            histogram_modify_vals (hist, src, bpp, stride, has_alpha,         \n"
"                                   -i - 1, -neighborhood_outline[i],          \n"
"                                   -i - 1, +neighborhood_outline[i],          \n"
"                                   -1);                                       \n"
"                                                                              \n"
"            histogram_modify_vals (hist, src, bpp, stride, has_alpha,         \n"
"                                   +i, -neighborhood_outline[i],              \n"
"                                   +i, +neighborhood_outline[i],              \n"
"                                   +1);                                       \n"
"                                                                              \n"
"            break;                                                            \n"
"                                                                              \n"
"          case RIGHT_TO_LEFT:                                                 \n"
"            for (i = 0; i < radius; i++)                                      \n"
"              {                                                               \n"
"                histogram_modify_vals (hist, src, bpp, stride, has_alpha,     \n"
"                                       +i + 1, -neighborhood_outline[i],      \n"
"                                       +i + 1, -neighborhood_outline[i + 1] - 1,\n"
"                                       -1);                                   \n"
"                histogram_modify_vals (hist, src, bpp, stride, has_alpha,     \n"
"                                       +i + 1, +neighborhood_outline[i + 1] + 1,\n"
"                                       +i + 1, +neighborhood_outline[i],      \n"
"                                       -1);                                   \n"
"                                                                              \n"
"                histogram_modify_vals (hist, src, bpp, stride, has_alpha,     \n"
"                                       -i, -neighborhood_outline[i],          \n"
"                                       -i, -neighborhood_outline[i + 1] - 1,  \n"
"                                       +1);                                   \n"
"                histogram_modify_vals (hist, src, bpp, stride, has_alpha,     \n"
"                                       -i, +neighborhood_outline[i + 1] + 1,  \n"
"                                       -i, +neighborhood_outline[i],          \n"
"                                       +1);                                   \n"
"              }                                                               \n"
"                                                                              \n"
"            histogram_modify_vals (hist, src, bpp, stride, has_alpha,         \n"
"                                   +i + 1, -neighborhood_outline[i],          \n"
"                                   +i + 1, +neighborhood_outline[i],          \n"
"                                   -1);                                       \n"
"                                                                              \n"
"            histogram_modify_vals (hist, src, bpp, stride, has_alpha,         \n"
"                                   -i, -neighborhood_outline[i],              \n"
"                                   -i, +neighborhood_outline[i],              \n"
"                                   +1);                                       \n"
"                                                                              \n"
"            break;                                                            \n"
"                                                                              \n"
"          case TOP_TO_BOTTOM:                                                 \n"
"            for (i = 0; i < radius; i++)                                      \n"
"              {                                                               \n"
"                histogram_modify_vals (hist, src, bpp, stride, has_alpha,     \n"
"                                       -neighborhood_outline[i],         -i - 1,\n"
"                                       -neighborhood_outline[i + 1] - 1, -i - 1,\n"
"                                       -1);                                   \n"
"                histogram_modify_vals (hist, src, bpp, stride, has_alpha,     \n"
"                                       +neighborhood_outline[i + 1] + 1, -i - 1,\n"
"                                       +neighborhood_outline[i],         -i - 1,\n"
"                                       -1);                                   \n"
"                                                                              \n"
"                histogram_modify_vals (hist, src, bpp, stride, has_alpha,     \n"
"                                       -neighborhood_outline[i],         +i,  \n"
"                                       -neighborhood_outline[i + 1] - 1, +i,  \n"
"                                       +1);                                   \n"
"                histogram_modify_vals (hist, src, bpp, stride, has_alpha,     \n"
"                                       +neighborhood_outline[i + 1] + 1, +i,  \n"
"                                       +neighborhood_outline[i],         +i,  \n"
"                                       +1);                                   \n"
"              }                                                               \n"
"                                                                              \n"
"            histogram_modify_vals (hist, src, bpp, stride, has_alpha,         \n"
"                                   -neighborhood_outline[i], -i - 1,          \n"
"                                   +neighborhood_outline[i], -i - 1,          \n"
"                                   -1);                                       \n"
"                                                                              \n"
"            histogram_modify_vals (hist, src, bpp, stride, has_alpha,         \n"
"                                   -neighborhood_outline[i], +i,              \n"
"                                   +neighborhood_outline[i], +i,              \n"
"                                   +1);                                       \n"
"                                                                              \n"
"            break;                                                            \n"
"        }                                                                     \n"
"      break;                                                                  \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"init_neighborhood_outline (GeglMedianBlurNeighborhood  neighborhood,          \n"
"                           gint                        radius,                \n"
"                           gint                       *neighborhood_outline)  \n"
"{                                                                             \n"
"  gint i;                                                                     \n"
"                                                                              \n"
"  for (i = 0; i <= radius; i++)                                               \n"
"    {                                                                         \n"
"      switch (neighborhood)                                                   \n"
"        {                                                                     \n"
"        case GEGL_MEDIAN_BLUR_NEIGHBORHOOD_SQUARE:                            \n"
"          neighborhood_outline[i] = radius;                                   \n"
"          break;                                                              \n"
"                                                                              \n"
"        case GEGL_MEDIAN_BLUR_NEIGHBORHOOD_CIRCLE:                            \n"
"          neighborhood_outline[i] =                                           \n"
"            (gint) sqrt ((radius + .5) * (radius + .5) - i * i);              \n"
"          break;                                                              \n"
"                                                                              \n"
"        case GEGL_MEDIAN_BLUR_NEIGHBORHOOD_DIAMOND:                           \n"
"          neighborhood_outline[i] = radius - i;                               \n"
"          break;                                                              \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"convert_values_to_bins (gint32   *src,                                        \n"
"                        gint      bpp,                                        \n"
"                        gboolean  has_alpha,                                  \n"
"                        gint      count)                                      \n"
"{                                                                             \n"
"  gint components = 3;                                                        \n"
"  gint c;                                                                     \n"
"                                                                              \n"
"  if (has_alpha)                                                              \n"
"    components++;                                                             \n"
"                                                                              \n"
"  while (count--)                                                             \n"
"    {                                                                         \n"
"      for (c = 0; c < components; c++)                                        \n"
"        {                                                                     \n"
"          gfloat value = ((gfloat *) src)[c];                                 \n"
"          gint   bin;                                                         \n"
"                                                                              \n"
"          bin = (gint) (CLAMP (value, 0.0f, 1.0f) * N_BINS);                  \n"
"          bin = MIN (bin, N_BINS - 1);                                        \n"
"                                                                              \n"
"          src[c] = bin;                                                       \n"
"        }                                                                     \n"
"                                                                              \n"
"      src += bpp;                                                             \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"prepare (GeglOperation *operation)                                            \n"
"{                                                                             \n"
"  GeglOperationAreaFilter *area      = GEGL_OPERATION_AREA_FILTER (operation);\n"
"  GeglProperties          *o         = GEGL_PROPERTIES (operation);           \n"
"  const Babl              *in_format = gegl_operation_get_source_format (operation, \"input\");\n"
"  const Babl              *format    = babl_format (\"RGB float\");           \n"
"                                                                              \n"
"  area->left   =                                                              \n"
"  area->right  =                                                              \n"
"  area->top    =                                                              \n"
"  area->bottom = o->radius;                                                   \n"
"                                                                              \n"
"  o->user_data = g_renew (gint, o->user_data, o->radius + 1);                 \n"
"  init_neighborhood_outline (o->neighborhood, o->radius, o->user_data);       \n"
"                                                                              \n"
"  if (in_format)                                                              \n"
"    {                                                                         \n"
"      if (babl_format_has_alpha (in_format))                                  \n"
"        format = babl_format (\"RGBA float\");                                \n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_operation_set_format (operation, \"input\", format);                   \n"
"  gegl_operation_set_format (operation, \"output\", format);                  \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"get_bounding_box (GeglOperation *operation)                                   \n"
"{                                                                             \n"
"  GeglRectangle  result = { 0, 0, 0, 0 };                                     \n"
"  GeglRectangle *in_rect;                                                     \n"
"                                                                              \n"
"  in_rect = gegl_operation_source_get_bounding_box (operation, \"input\");    \n"
"  if (in_rect)                                                                \n"
"  {                                                                           \n"
"    result = *in_rect;                                                        \n"
"  }                                                                           \n"
"                                                                              \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"process (GeglOperation       *operation,                                      \n"
"         GeglBuffer          *input,                                          \n"
"         GeglBuffer          *output,                                         \n"
"         const GeglRectangle *roi,                                            \n"
"         gint                 level)                                          \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES (operation);                            \n"
"                                                                              \n"
"  gdouble         percentile           = o->percentile       / 100.0;         \n"
"  gdouble         alpha_percentile     = o->alpha_percentile / 100.0;         \n"
"  const gint     *neighborhood_outline = o->user_data;                        \n"
"                                                                              \n"
"  const Babl     *format               = gegl_operation_get_format (operation, \"input\");\n"
"  gint            n_components         = babl_format_get_n_components (format);\n"
"  gboolean        has_alpha            = babl_format_has_alpha (format);      \n"
"                                                                              \n"
"  G_STATIC_ASSERT (sizeof (gint32) == sizeof (gfloat));                       \n"
"  gint32         *src_buf;                                                    \n"
"  gfloat         *dst_buf;                                                    \n"
"  GeglRectangle   src_rect;                                                   \n"
"  gint            src_stride;                                                 \n"
"  gint            dst_stride;                                                 \n"
"  gint            n_pixels;                                                   \n"
"                                                                              \n"
"  Histogram      *hist;                                                       \n"
"                                                                              \n"
"  const gint32   *src;                                                        \n"
"  gfloat         *dst;                                                        \n"
"  gint            dst_x, dst_y;                                               \n"
"  Direction       dir;                                                        \n"
"                                                                              \n"
"  gint            i;                                                          \n"
"  gint            c;                                                          \n"
"                                                                              \n"
"  src_rect   = gegl_operation_get_required_for_output (operation, \"input\", roi);\n"
"  src_stride = src_rect.width * n_components;                                 \n"
"  dst_stride = roi->width * n_components;                                     \n"
"  n_pixels   = roi->width * roi->height;                                      \n"
"  dst_buf = g_new0 (gfloat, n_pixels                         * n_components); \n"
"  src_buf = g_new0 (gint32, src_rect.width * src_rect.height * n_components); \n"
"                                                                              \n"
"  gegl_buffer_get (input, &src_rect, 1.0, format, src_buf,                    \n"
"                   GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_CLAMP);                    \n"
"  convert_values_to_bins (src_buf, n_components, has_alpha,                   \n"
"                          src_rect.width * src_rect.height);                  \n"
"                                                                              \n"
"  hist = g_slice_new0 (Histogram);                                            \n"
"                                                                              \n"
"  src = src_buf + o->radius * (src_rect.width + 1) * n_components;            \n"
"  dst = dst_buf;                                                              \n"
"                                                                              \n"
"  /* compute the first window */                                              \n"
"                                                                              \n"
"  for (i = -o->radius; i <= o->radius; i++)                                   \n"
"    {                                                                         \n"
"      histogram_modify_vals (hist, src, n_components, src_stride, has_alpha,  \n"
"                             i, -neighborhood_outline[abs (i)],               \n"
"                             i, +neighborhood_outline[abs (i)],               \n"
"                             +1);                                             \n"
"                                                                              \n"
"      hist->size += 2 * neighborhood_outline[abs (i)] + 1;                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (c = 0; c < 3; c++)                                                     \n"
"    dst[c] = histogram_get_median (hist, c, percentile);                      \n"
"                                                                              \n"
"  if (has_alpha)                                                              \n"
"    dst[3] = histogram_get_median (hist, 3, alpha_percentile);                \n"
"                                                                              \n"
"  dst_x = 0;                                                                  \n"
"  dst_y = 0;                                                                  \n"
"                                                                              \n"
"  n_pixels--;                                                                 \n"
"  dir = LEFT_TO_RIGHT;                                                        \n"
"                                                                              \n"
"  while (n_pixels--)                                                          \n"
"    {                                                                         \n"
"      /* move the src coords based on current direction and positions */      \n"
"      if (dir == LEFT_TO_RIGHT)                                               \n"
"        {                                                                     \n"
"          if (dst_x != roi->width - 1)                                        \n"
"            {                                                                 \n"
"              dst_x++;                                                        \n"
"              src += n_components;                                            \n"
"              dst += n_components;                                            \n"
"            }                                                                 \n"
"          else                                                                \n"
"            {                                                                 \n"
"              dst_y++;                                                        \n"
"              src += src_stride;                                              \n"
"              dst += dst_stride;                                              \n"
"              dir = TOP_TO_BOTTOM;                                            \n"
"            }                                                                 \n"
"        }                                                                     \n"
"      else if (dir == TOP_TO_BOTTOM)                                          \n"
"        {                                                                     \n"
"          if (dst_x == 0)                                                     \n"
"            {                                                                 \n"
"              dst_x++;                                                        \n"
"              src += n_components;                                            \n"
"              dst += n_components;                                            \n"
"              dir = LEFT_TO_RIGHT;                                            \n"
"            }                                                                 \n"
"          else                                                                \n"
"            {                                                                 \n"
"              dst_x--;                                                        \n"
"              src -= n_components;                                            \n"
"              dst -= n_components;                                            \n"
"              dir = RIGHT_TO_LEFT;                                            \n"
"            }                                                                 \n"
"        }                                                                     \n"
"      else if (dir == RIGHT_TO_LEFT)                                          \n"
"        {                                                                     \n"
"          if (dst_x != 0)                                                     \n"
"            {                                                                 \n"
"              dst_x--;                                                        \n"
"              src -= n_components;                                            \n"
"              dst -= n_components;                                            \n"
"            }                                                                 \n"
"          else                                                                \n"
"            {                                                                 \n"
"              dst_y++;                                                        \n"
"              src += src_stride;                                              \n"
"              dst += dst_stride;                                              \n"
"              dir = TOP_TO_BOTTOM;                                            \n"
"            }                                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      histogram_update (hist, src, n_components, src_stride, has_alpha,       \n"
"                        o->neighborhood, o->radius, neighborhood_outline,     \n"
"                        dir);                                                 \n"
"                                                                              \n"
"      for (c = 0; c < 3; c++)                                                 \n"
"        dst[c] = histogram_get_median (hist, c, percentile);                  \n"
"                                                                              \n"
"      if (has_alpha)                                                          \n"
"        dst[3] = histogram_get_median (hist, 3, alpha_percentile);            \n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_buffer_set (output, roi, 0, format, dst_buf, GEGL_AUTO_ROWSTRIDE);     \n"
"                                                                              \n"
"  g_slice_free (Histogram, hist);                                             \n"
"  g_free (dst_buf);                                                           \n"
"  g_free (src_buf);                                                           \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"finalize (GObject *object)                                                    \n"
"{                                                                             \n"
"  GeglOperation  *op = (void*) object;                                        \n"
"  GeglProperties *o  = GEGL_PROPERTIES (op);                                  \n"
"                                                                              \n"
"  if (o->user_data)                                                           \n"
"    {                                                                         \n"
"      g_free (o->user_data);                                                  \n"
"      o->user_data = NULL;                                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  G_OBJECT_CLASS (gegl_op_parent_class)->finalize (object);                   \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GObjectClass             *object_class;                                     \n"
"  GeglOperationClass       *operation_class;                                  \n"
"  GeglOperationFilterClass *filter_class;                                     \n"
"                                                                              \n"
"  object_class    = G_OBJECT_CLASS (klass);                                   \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  filter_class    = GEGL_OPERATION_FILTER_CLASS (klass);                      \n"
"                                                                              \n"
"  object_class->finalize            = finalize;                               \n"
"  filter_class->process             = process;                                \n"
"  operation_class->prepare          = prepare;                                \n"
"  operation_class->get_bounding_box = get_bounding_box;                       \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"    \"name\",        \"gegl:median-blur\",                                    \n"
"    \"title\",       _(\"Median Blur\"),                                      \n"
"    \"categories\",  \"blur\",                                                \n"
"    \"description\", _(\"Blur resulting from computing the median \"          \n"
"                     \"color in the neighborhood of each pixel.\"),           \n"
"    NULL);                                                                    \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
