/* GIMP Plugin Preview Widget                                                 
 * Copyright (C) 1998-1999 Shawn T. Amundson                
 * 
 * Some of the checkerboard & image combination code is based of
 * code Copyright 1997 (C) Federico Mena Quintero and is used
 * here by permission.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.           
 *                                                                            
 * This library is distributed in the hope that it will be useful,             
 * but WITHOUT ANY WARRANTY; without even the implied warranty of              
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU           
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */      
 
#include <stdio.h>
#include <math.h>
#include <gtk/gtk.h>
#include "gimppreview.h"


static void gimp_preview_init            (GimpPreview *preview);
static void gimp_preview_class_init      (GimpPreviewClass *klass);

static void gimp_preview_plus_callback   (GtkWidget *widget,
				          gpointer data);
static void gimp_preview_minus_callback  (GtkWidget *widget,
				          gpointer data);
static gint gimp_preview_event           (GtkWidget *widget, GdkEvent *event, 
					  gpointer data);
static void gimp_preview_recompute_sizes (GimpPreview *preview,
					  gfloat newscale);
static void gimp_preview_update_preview  (GimpPreview *preview);

#define PREVIEW_SIZE 100
 
enum {
  UPDATE_PREVIEW,
  PREVIEW_CHANGED,
  LAST_SIGNAL
};

#define PREVIEW_SCALE_DEFAULT 5
#define PREVIEW_SCALE_LAST 20
static const gdouble preview_scale[PREVIEW_SCALE_LAST+1] = {  1/6.0, 1/5.0,  1/4.0,
							      1/3.0, 1/2.0,  1.00,
							      2.00,   3.00,  4.00,  
							      5.00,   6.00,  7.00,  
							      8.00,   9.00, 10.00, 
							      11.00, 12.00, 13.00, 
							      14.00, 15.00, 16.00 };

 /* Constants for transparency checkerboard */
/* CHECK_SIZE must be power of 2 */
#define CHECK_SIZE (16)
#define CHECK_DARK  ((guchar) (1.3 / 3.0 * 255))
#define CHECK_LIGHT ((guchar) (1.8 / 3.0 * 255))

static GtkVBox *parent_class;
static guint gimp_preview_signals[LAST_SIGNAL] = { 0 };


#define PREVIEW_DATA(preview) ((GimpPreviewData*)(GIMP_PREVIEW (preview)->private_data))

typedef struct {
  gint scale_n;

  GtkWidget *label;
  GtkWidget *button_minus;
  GtkWidget *button_plus;

  gboolean in_drag;		/* Is true while the preview is dragged */

  gint drag_x;			/* Mouse x-position when dragging started */
  gint drag_y;			/* Mouse y-position when dragging started */

  gint orig_prev_x;		/* Value of prev_x when dragging started */
  gint orig_prev_y;		/* Value of prev_y when dragging started */

  guchar *preview_buffer_na[2];	/* Buffers for storing one row for preview */
  
} GimpPreviewData;


GtkType
gimp_preview_get_type (void)
{
  static GtkType preview_type = 0;

  if (!preview_type)
    {
      GtkTypeInfo preview_info =
      {
	"GimpPreview",
	sizeof (GimpPreview),
	sizeof (GimpPreviewClass),
	(GtkClassInitFunc) gimp_preview_class_init,
	(GtkObjectInitFunc) gimp_preview_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };

      preview_type = gtk_type_unique (gtk_vbox_get_type (), &preview_info);
    }

  return preview_type;
}

static void
gimp_preview_class_init (GimpPreviewClass *klass)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;
  GtkVBoxClass *vbox_class;

  object_class = GTK_OBJECT_CLASS (klass);
  widget_class = GTK_WIDGET_CLASS (klass);
  container_class = GTK_CONTAINER_CLASS (klass);
  vbox_class = GTK_VBOX_CLASS (klass);

  parent_class = gtk_type_class (gtk_vbox_get_type ());

  gimp_preview_signals[UPDATE_PREVIEW] =
    gtk_signal_new("update_preview",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET (GimpPreviewClass, update_preview),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_POINTER);

  gimp_preview_signals[PREVIEW_CHANGED] =
    gtk_signal_new("preview_changed",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET (GimpPreviewClass, preview_changed),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals (object_class, gimp_preview_signals, 
				LAST_SIGNAL);

  klass->update_preview = NULL;
  klass->preview_changed = NULL;
}

static void
gimp_preview_init (GimpPreview *preview)
{
  gchar buffer[10];

  preview->private_data = g_malloc(sizeof(GimpPreviewData));

  PREVIEW_DATA (preview)->scale_n = PREVIEW_SCALE_DEFAULT;
  preview->scale = preview_scale[PREVIEW_DATA (preview)->scale_n];

  PREVIEW_DATA (preview)->label = gtk_label_new("");
  sprintf(buffer, "%d%%", (gint) (preview->scale*100));
  gtk_label_set_text (GTK_LABEL (PREVIEW_DATA (preview)->label), buffer);

  PREVIEW_DATA (preview)->in_drag = FALSE;
  PREVIEW_DATA (preview)->drag_x = 0;
  PREVIEW_DATA (preview)->drag_y = 0;
  PREVIEW_DATA (preview)->orig_prev_x = 0;
  PREVIEW_DATA (preview)->orig_prev_y = 0;

  /* These buffers can each contain one row of pixels for the preview. */
  /* Two buffers are needed for gimp_preview_draw_unscaled_row */
  PREVIEW_DATA (preview)->preview_buffer_na[0] = g_malloc (sizeof (guchar)
							   * PREVIEW_SIZE 
							   * 3);
  PREVIEW_DATA (preview)->preview_buffer_na[1] = g_malloc (sizeof (guchar)
							   * PREVIEW_SIZE 
							   * 3);
  /* The buffer that holds scaled image data */
  preview->buffer = g_malloc (sizeof (guchar)
			      * (PREVIEW_SIZE + 16)
			      * (PREVIEW_SIZE + 16) 
			      * 4);

  preview->preview = NULL;
  preview->progress_bar = NULL;
}

GtkWidget*
gimp_preview_new (GimpDrawable *drawable)
{
  GimpPreview *preview;
  GtkWidget *frame;
  GtkWidget *packer;
  GtkWidget *hbox;
  guchar *color_cube;

  preview = GIMP_PREVIEW (gtk_type_new (gimp_preview_get_type ()));
  preview->drawable = drawable;
  preview->drawable_has_alpha = gimp_drawable_has_alpha(drawable->id);

  frame = gtk_frame_new (NULL);
  gtk_box_pack_start (GTK_BOX (preview), frame, FALSE, FALSE, 0);
  
  packer = gtk_packer_new ();
  gtk_container_add(GTK_CONTAINER (frame), packer);

  preview->preview = gtk_preview_new (GTK_PREVIEW_COLOR);
  gtk_preview_set_expand(GTK_PREVIEW(preview->preview), 0);
  gtk_preview_size(GTK_PREVIEW (preview->preview), PREVIEW_SIZE, PREVIEW_SIZE);
  gtk_container_add(GTK_CONTAINER (packer), preview->preview);
  
  /* Some of this may get moved up to GimpPluginWindow widget instead. */
  gtk_preview_set_gamma (gimp_gamma ());
  gtk_preview_set_install_cmap (gimp_install_cmap());
  color_cube = gimp_color_cube();
  gtk_preview_set_color_cube(color_cube[0], color_cube[1], color_cube[3],
			     color_cube[4]); 
  gtk_widget_set_default_visual (gtk_preview_get_visual ());
  gtk_widget_set_default_colormap (gtk_preview_get_cmap ());

  gtk_widget_set_events (preview->preview,
			 GDK_BUTTON_PRESS_MASK |
			 GDK_BUTTON_RELEASE_MASK |
			 GDK_POINTER_MOTION_HINT_MASK |
			 GDK_BUTTON_MOTION_MASK);
  gtk_signal_connect (GTK_OBJECT (preview->preview), "event",
		      GTK_SIGNAL_FUNC (gimp_preview_event),
		      (gpointer) preview);

  preview->progress_bar = gtk_progress_bar_new ();
  gtk_widget_set_usize (preview->progress_bar, PREVIEW_SIZE, 10);
  gtk_box_pack_start (GTK_BOX (preview), preview->progress_bar, FALSE, FALSE, 0);

  hbox = gtk_hbox_new (0, 0);
  gtk_container_border_width (GTK_CONTAINER (hbox), 4);
  gtk_box_pack_start (GTK_BOX (preview), hbox, FALSE, FALSE, 0);

  PREVIEW_DATA (preview)->button_minus = gtk_button_new_with_label ("-");
  PREVIEW_DATA (preview)->button_plus = gtk_button_new_with_label ("+");

  gtk_box_pack_start (GTK_BOX (hbox), PREVIEW_DATA (preview)->button_minus, TRUE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), PREVIEW_DATA (preview)->label, TRUE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), PREVIEW_DATA (preview)->button_plus, TRUE, FALSE, 0);

  gtk_signal_connect (GTK_OBJECT (PREVIEW_DATA (preview)->button_minus), "clicked",
		      GTK_SIGNAL_FUNC (gimp_preview_minus_callback),
		      (gpointer) preview);
  gtk_signal_connect (GTK_OBJECT (PREVIEW_DATA (preview)->button_plus), "clicked",
		      GTK_SIGNAL_FUNC (gimp_preview_plus_callback),
		      (gpointer) preview);

  gtk_widget_show (PREVIEW_DATA (preview)->button_minus);
  gtk_widget_show (PREVIEW_DATA (preview)->label);
  gtk_widget_show (PREVIEW_DATA (preview)->button_plus);
  gtk_widget_show (hbox);
  gtk_widget_show (preview->progress_bar);
  gtk_widget_show (preview->preview);
  gtk_widget_show (packer);
  gtk_widget_show (frame);

  gimp_preview_recompute_sizes (preview, preview->scale);
  gimp_preview_update_preview (preview);

  return GTK_WIDGET (preview);
}

void
gimp_preview_update (GimpPreview *preview)
{
  gimp_preview_recompute_sizes (preview, preview->scale);
  gimp_preview_update_preview (preview);
}

static void
gimp_preview_plus_callback (GtkWidget *widget, gpointer data)
{
  GimpPreview *preview;
  gchar buffer[10];
  gfloat new_scale;

  preview = GIMP_PREVIEW (data);
  if (PREVIEW_DATA (preview)->scale_n == PREVIEW_SCALE_LAST)
    return;

  PREVIEW_DATA (preview)->scale_n++;
  new_scale = preview_scale[PREVIEW_DATA (preview)->scale_n];
  sprintf(buffer, "%d%%", (gint) (new_scale*100));
  gtk_label_set_text (GTK_LABEL (PREVIEW_DATA (preview)->label), buffer);

  if (PREVIEW_DATA (preview)->scale_n == PREVIEW_SCALE_LAST)
    gtk_widget_set_sensitive (widget, FALSE);

  if (PREVIEW_DATA (preview)->scale_n == 1)
    gtk_widget_set_sensitive (PREVIEW_DATA (preview)->button_minus, TRUE);

  gimp_preview_recompute_sizes (preview, new_scale);
  gimp_preview_update_preview (preview);
}

static void
gimp_preview_minus_callback (GtkWidget *widget, gpointer data)
{
  GimpPreview *preview;
  gchar buffer[10];
  gfloat new_scale;

  preview = GIMP_PREVIEW (data);
  if (PREVIEW_DATA (preview)->scale_n == 0)
    return;

  PREVIEW_DATA (preview)->scale_n--;
  new_scale = preview_scale[PREVIEW_DATA (preview)->scale_n];
  sprintf(buffer, "%d%%", (gint) (new_scale*100));
  gtk_label_set_text (GTK_LABEL (PREVIEW_DATA (preview)->label), buffer);

  if (PREVIEW_DATA (preview)->scale_n == 0)
    gtk_widget_set_sensitive (widget, FALSE);

  if (PREVIEW_DATA (preview)->scale_n == PREVIEW_SCALE_LAST - 1)
    gtk_widget_set_sensitive (PREVIEW_DATA (preview)->button_plus, TRUE);

  gimp_preview_recompute_sizes (preview, new_scale);
  gimp_preview_update_preview (preview);
}

static gint
gimp_preview_event (GtkWidget *widget, GdkEvent *event, gpointer data)
{
  GimpPreview *preview;
  GdkEventButton *button_event;
  gint x, y;
  gint dx, dy;

  preview = GIMP_PREVIEW (data);
  button_event = (GdkEventButton*) event;

  switch (event->type)
    {
    case GDK_BUTTON_PRESS:
      if (button_event->button == 1)
	{
	  gtk_widget_get_pointer(widget, &x, &y);

	  PREVIEW_DATA (preview)->in_drag = TRUE;
	  PREVIEW_DATA (preview)->drag_x = x;
	  PREVIEW_DATA (preview)->drag_y = y;

	  PREVIEW_DATA (preview)->orig_prev_x = preview->prev_x;
	  PREVIEW_DATA (preview)->orig_prev_y = preview->prev_y;

	  gtk_grab_add (widget);

	  gimp_preview_update_preview (preview);
	}
      break;

    case GDK_BUTTON_RELEASE:
      if (PREVIEW_DATA (preview)->in_drag && button_event->button == 1)
	{
	  gtk_grab_remove (widget);
	  PREVIEW_DATA (preview)->in_drag = FALSE;

	  gimp_preview_update_preview (preview);
	}
      
      break;

    case GDK_MOTION_NOTIFY:
      if (PREVIEW_DATA (preview)->in_drag)
	{
	  gtk_widget_get_pointer(widget, &x, &y);
	  
	  dx = x - PREVIEW_DATA (preview)->drag_x;
	  dy = y - PREVIEW_DATA (preview)->drag_y;
	  
	  preview->prev_x =
	    CLAMP(PREVIEW_DATA (preview)->orig_prev_x - dx,
		  0,
		  preview->image_width - preview->width);
	  preview->prev_y =
	    CLAMP(PREVIEW_DATA (preview)->orig_prev_y - dy,
		  0,
		  preview->image_height - preview->height);
	  
	  /* The following is a bit redundant since image_x and image_y
	     can always be recomputed from prev_x and prev_y */
	  preview->image_x = (gint) preview->prev_x / preview->scale;
	  preview->image_y = (gint) preview->prev_y / preview->scale;

	  gimp_preview_update_preview (preview);
	}
      break;

    default:
      break;
    }
  
  return FALSE;
}

static void
gimp_preview_clear_border(GimpPreview *preview)
{
  const gint allocation_width =
    GTK_WIDGET (preview->preview)->allocation.width;
  const gint allocation_height =
    GTK_WIDGET (preview->preview)->allocation.height;

  /* If no window, we're in init stages and don't need to clear anything.  */
  if (preview->preview->window == NULL )
	return;

  if (preview->width < PREVIEW_SIZE || preview->height < PREVIEW_SIZE)
    {
      gdk_window_clear_area (preview->preview->window,
			     0,
			     0,
			     (allocation_width - preview->width)/2,
			     allocation_height);
      gdk_window_clear_area (preview->preview->window,
			     0,
			     0,
			     allocation_width,
			     (allocation_height - preview->height)/2);
      /* Used ceil for rounding up to correct problems when 
	 allocation_width - preview->width is odd. Is it correct this way? */
      gdk_window_clear_area (preview->preview->window,
			     allocation_width -
			     ceil((allocation_width - preview->width)/2),
			     0,
			     allocation_width,
			     allocation_height);
      gdk_window_clear_area (preview->preview->window,
			     0,
			     allocation_height -
			     ceil((allocation_height - preview->height)/2),
			     allocation_width,
			     allocation_height);
    }
}

/* Recompute the size and location of the display and clear the
 * borders. 
 * new_scale is the new value of the scale. It must be supplied
 * as argument because otherwise we don't have enough information
 * to keep the same location in the image at the centre of the preview.
 * This function is also used for initializing the preview.
 */
static void
gimp_preview_recompute_sizes(GimpPreview *preview, gfloat new_scale)
{
  const gboolean scale_changed = (preview->scale != new_scale);
  /* The center of the preview in image coordinates */
  /* We try to keep the center at the same position when changing scales */
  const gfloat im_center_x =
    (preview->prev_x + preview->width / 2.0) / preview->scale;
  const gfloat im_center_y =
    (preview->prev_y + preview->height / 2.0) / preview->scale;

  preview->scale = new_scale;

  preview->image_width = ceil(preview->drawable->width * preview->scale);
  preview->image_height = ceil(preview->drawable->height * preview->scale);
  preview->width = MIN(PREVIEW_SIZE, preview->image_width);
  preview->height = MIN(PREVIEW_SIZE, preview->image_height);

  if (scale_changed)
    {
      preview->prev_x =
	CLAMP((gint)(im_center_x * new_scale - preview->width / 2),
	      0,
	      preview->image_width - preview->width);
      preview->prev_y =
	CLAMP((gint)(im_center_y * new_scale - preview->height / 2),
	      0,
	      preview->image_height - preview->height);
      preview->image_x = (gint) preview->prev_x / preview->scale;
      preview->image_y = (gint) preview->prev_y / preview->scale;
    }

  gtk_preview_size (GTK_PREVIEW (preview->preview), 
		    preview->width, preview->height);

  gimp_preview_clear_border(preview);
}

/* Update the preview. 
 * First scale the original image.
 * If we are currently dragging the preview is redrawn.
 * Otherwise an event is generated to signal the user that
 * the preview must be updated.
 * This function is the main performance bottleneck in the
 * PreviewWidget. Be careful with modifications!
 * This implementation uses explicit subscripting with loop variables.
 * This makes this version faster than a version that uses pointers
 * to step through source and destination!
 */
static void 
gimp_preview_update_preview  (GimpPreview *preview)
{
  GimpPixelRgn region;
  const gint image_x = preview->image_x;
  const gint image_y = preview->image_y;
  const gint prev_x = preview->prev_x;
  const gint prev_y = preview->prev_y;
  const gint width = preview->width;
  const gint height = preview->height;
  const gfloat inverse_scale = 1 / preview->scale;

  const gint image_width =
    (gint) ((preview->prev_x + preview->width - 1) / preview->scale)
    - preview->image_x + 1;
  const gint image_height =
    (gint) ((preview->prev_y + preview->height - 1) / preview->scale)
    - preview->image_y + 1;
  guchar *image_data = NULL;
  
  gimp_pixel_rgn_init (&region, preview->drawable,
		       image_x, 
		       image_y,
		       image_width,
		       image_height,
		       FALSE, FALSE);

  image_data = g_malloc (sizeof (guchar) * preview->drawable->bpp
			 * image_width * image_height);

  gimp_pixel_rgn_get_rect (&region, image_data,
			   image_x, image_y, image_width, image_height);

  /* Scale and convert to RGBA */

  if (preview->drawable_has_alpha && (preview->drawable->bpp == 4))
    { 
      gint dy;
      guchar * const dest_ptr = preview->buffer;
      const gint bpp = 4;
      
      for (dy = height - 1; dy >= 0; dy--) 
	{
	  gint dx;
	  const gint sy = (dy + prev_y) * inverse_scale;
	  const guchar * const src_ptr =
	    image_data + ((sy - image_y) * image_width - image_x) * bpp;
	  guchar * const dest_row_ptr = dest_ptr + dy * width * 4;
	  
	  for (dx = width - 1; dx >= 0; dx--) 
	    {
	      const gint sx = (dx + prev_x) * inverse_scale;
	      
	      dest_row_ptr[dx * 4 + 0] = src_ptr[sx * bpp + 0];
	      dest_row_ptr[dx * 4 + 1] = src_ptr[sx * bpp + 1];
	      dest_row_ptr[dx * 4 + 2] = src_ptr[sx * bpp + 2];
	      dest_row_ptr[dx * 4 + 3] = src_ptr[sx * bpp + 3];
	    }
	}
    } 
  else if (!preview->drawable_has_alpha && (preview->drawable->bpp == 3))
    {
      gint dy;
      guchar * const dest_ptr = preview->buffer;
      const gint bpp = 3;
      
      for (dy = height - 1; dy >= 0; dy--) 
	{
	  gint dx;
	  const gint sy = (dy + prev_y) * inverse_scale;
	  const guchar * const src_ptr =
	    image_data + ((sy - image_y) * image_width - image_x) * bpp;
	  guchar* const dest_row_ptr = dest_ptr + dy * width * 4;

	  for (dx = width - 1; dx >= 0; dx--) 
	    {
	      const gint sx = (dx + prev_x) * inverse_scale;

	      dest_row_ptr[dx * 4 + 0] = src_ptr[sx * bpp + 0];
	      dest_row_ptr[dx * 4 + 1] = src_ptr[sx * bpp + 1];
	      dest_row_ptr[dx * 4 + 2] = src_ptr[sx * bpp + 2];
	      dest_row_ptr[dx * 4 + 3] = 255;
	    }
	}
    } else if (preview->drawable_has_alpha && (preview->drawable->bpp == 2))
      {
	gint dy;
	guchar * const dest_ptr = preview->buffer;
	const gint bpp = 2;

	for (dy = height - 1; dy >= 0; dy--) 
	  {
	    gint dx;
	    const gint sy = (dy + prev_y) * inverse_scale;
	    const guchar * const src_ptr =
	      image_data + ((sy - image_y) * image_width - image_x) * bpp;
	    guchar * const dest_row_ptr = dest_ptr + dy * width * 4;
	    
	    for (dx = width - 1; dx >= 0; dx--) 
	      {
		const gint sx = (dx + prev_x) * inverse_scale;
		
		dest_row_ptr[dx * 4 + 0] =
		  dest_row_ptr[dx * 4 + 1] =
		  dest_row_ptr[dx * 4 + 2] = src_ptr[sx * bpp + 0];
		dest_row_ptr[dx * 4 + 3] = src_ptr[sx * bpp + 1];
	      }
	  }
      } else if (!preview->drawable_has_alpha && (preview->drawable->bpp == 1))
	{
	  gint dy;
	  guchar * const dest_ptr = preview->buffer;
	  const gint bpp = 1;
	  
	  for (dy = height - 1; dy >= 0; dy--) 
	    {
	      gint dx;
	      const gint sy = (dy + prev_y) * inverse_scale;
	      const guchar * const src_ptr =
		image_data + ((sy - image_y) * image_width - image_x) * bpp;
	      guchar * const dest_row_ptr = dest_ptr + dy * width * 4;
	      
	      for (dx = width - 1; dx >= 0; dx--) 
		{
		  const gint sx = (dx + prev_x) * inverse_scale;

		  dest_row_ptr[dx * 4 + 0] =
		    dest_row_ptr[dx * 4 + 1] =
		    dest_row_ptr[dx * 4 + 2] = src_ptr[sx * bpp + 0];
		  dest_row_ptr[dx * 4 + 3] = 255;
		}
	    }
	}

  if (PREVIEW_DATA (preview)->in_drag) 
    {
      /* We are dragging. Now redraw the preview */
      guint row;

      for (row = 0; row < preview->height; row++) 
	{
	  gimp_preview_draw_row (preview, GIMP_RGBA_IMAGE, row, 
				 &(preview->buffer[row * preview->width * 4]));
	}
    } 
  else 
    {
      /* Not preview->in_drag */
      /* Signal the user that the preview must be updated */
      GimpPreviewEvent *preview_event = g_new (GimpPreviewEvent, 1);

      preview_event->scale = preview->scale;
      preview_event->x = preview->image_x;
      preview_event->y = preview->image_y;
      preview_event->width = image_width;
      preview_event->height = image_height; 
      preview_event->scaled_data = preview->buffer;
      
      gtk_signal_emit (GTK_OBJECT (preview), 
		       gimp_preview_signals[UPDATE_PREVIEW],
		       preview_event);

      g_free (preview_event);
    }

  gimp_preview_force_redraw (preview);

  g_free (image_data);
}

void
gimp_preview_force_redraw (GimpPreview *preview)
{
  GdkRectangle req;

  req.x = (GTK_WIDGET (preview->preview)->allocation.width
	   - GTK_PREVIEW (preview->preview)->buffer_width)/2;
  req.y = ((GTK_WIDGET (preview->preview)->allocation.height
	    - GTK_PREVIEW (preview->preview)->buffer_height)/2);
  req.width = GTK_PREVIEW (preview->preview)->buffer_width;
  req.height = GTK_PREVIEW (preview->preview)->buffer_height;

  gtk_widget_draw (GTK_WIDGET (preview->preview), &req);
}

 /*
   Draw one scaled row of data in the preview.
   type is the format of the data (e.g. GIMP_RGBA_IMAGE).
   row is the relative number of the row within the preview.
   The top row of the preview is number 0.
   data must have a length of preview->width pixels.
   This function handles conversion to RGB and checkerboarding of
   transparent areas.
   This implementation uses explicit subscripting with loop variables.
   This makes this version faster than a version that uses pointers
   to step through source and destination!
*/
void 
gimp_preview_draw_row (GimpPreview *preview, GimpImageType type,
		       const gint row, const guchar *const data)
{
  const gint width = preview->width;
  guchar * buf_start = PREVIEW_DATA (preview)->preview_buffer_na[0];
  const gint check_offset =
    ((preview->prev_y + row) & CHECK_SIZE) ?
    preview->prev_x : preview->prev_x + CHECK_SIZE;
  gint j;

  if ((row < 0) || (row >= preview->height))
    return;
  
  switch(type)
    {
    case GIMP_RGBA_IMAGE:
      {
	guchar *dest_ptr = buf_start;
	const guchar * src_ptr = data;
	
	for (j = width - 1; j >= 0; j--) 
	  {
	    const guchar check = ((j + check_offset) & CHECK_SIZE) ?
	      CHECK_LIGHT : CHECK_DARK;
	  
	    dest_ptr[3 * j] =
	      check + (((src_ptr[4 * j + 0] - check) * src_ptr[4 * j + 3]) >> 8);
	    dest_ptr[3 * j + 1] =
	      check + (((src_ptr[4 * j + 1] - check) * src_ptr[4 * j + 3]) >> 8);
	    dest_ptr[3 * j + 2] =
	      check + (((src_ptr[4 * j + 2] - check) * src_ptr[4 * j + 3]) >> 8);
	  }
      }
      break;
    case GIMP_RGB_IMAGE:
      {
	guchar *dest_ptr = buf_start;
	const guchar * src_ptr = data;

	/* Yes, you're right. We could use memcpy here */
	for (j = width - 1; j >= 0; j--) 
	  {
	    dest_ptr[3 * j] = src_ptr[3 * j];
	    dest_ptr[3 * j + 1] = src_ptr[3 * j + 1];
	    dest_ptr[3 * j + 2] = src_ptr[3 * j + 2];
	  }
      }
      break;
    case GIMP_GRAYA_IMAGE:
      {
	guchar *dest_ptr = buf_start;
	const guchar * src_ptr = data;

	for (j = width - 1; j >= 0; j--) 
	  {
	    const guchar check = ((j + check_offset) & CHECK_SIZE) ?
	      CHECK_LIGHT : CHECK_DARK;

	    dest_ptr[3 * j] =
	      dest_ptr[3 * j + 1] =
	      dest_ptr[3 * j + 2] =
	      check + (((src_ptr[2 * j] - check) * src_ptr[2 * j + 1]) >> 8);
	  }
      }
      break;
    case GIMP_GRAY_IMAGE:
      {
	guchar *dest_ptr = buf_start;
	const guchar * src_ptr = data;

	for (j = width - 1; j >= 0; j--) 
	  {
	    
	    dest_ptr[3 * j] =
	      dest_ptr[3 * j + 1] =
	      dest_ptr[3 * j + 2] = src_ptr[j];
	  }
      }
      break;
    default:
      /* Type argument is wrong */
      break;
    }

  gtk_preview_draw_row (GTK_PREVIEW (preview->preview), buf_start, 0, row, 
			width);
}

/*
  Draw one unscaled row of data in the preview.
  type is the format of the data (e.g. GIMP_RGBA_IMAGE).
  row is the relative position of the row w.r.t. preview->image_y
  The first row of data is the row preview->image_y in the image.
  data must have a length of preview_event->width pixels.
  The first pixel in data has x-coordinate preview->image_x in the image.
  This function handles scaling, conversion to RGB and checkerboarding
  of transparent areas.
  A nice feature is that this function will draw several lines of
  the preview when scale > 1.
  There is a small complication with transparent data when scale > 1
  that is caused by the checkerboarding. Since there are two different
  checkerboard patterns we may have to compute two different versions
  for the preview.
  This implementation uses explicit subscripting with loop variables.
  This makes this version faster than a version that uses pointers
  to step through source and destination!
*/
void 
gimp_preview_draw_unscaled_row (GimpPreview *preview, GimpImageType type,
		       const gint row, const guchar *const data)
{
  const gint width = preview->width;
  guchar * buf_start[2];
  /* Indicate whether we must compute the odd and/or the even version 
     for checkerboarding transparent areas */
  gboolean need_check_idx[2] = {TRUE, TRUE}; 
  gint check_idx;

  /* Compute the range of the lines in the preview that we can draw*/
  const gint y_start = CLAMP(ceil((preview->image_y + row) * preview->scale),
			     preview->prev_y,
			     preview->prev_y +preview->height -1);
  const gint y_end = CLAMP(ceil((preview->image_y + row + 1) * preview->scale),
			   preview->prev_y,
			   preview->prev_y + preview->height);

  const gint prev_x = preview->prev_x;
  const gint image_x = preview->image_x;
  const gfloat scale = preview->scale;
  gint x, y;

  if (y_start >= y_end)
    return;

  if ((type == GIMP_RGBA_IMAGE) || (type == GIMP_GRAYA_IMAGE))
    {
      /* This checkerboard version must always be computed */
      const gint first_check_idx =
	((y_start - preview->prev_y) / CHECK_SIZE) & 1;
      
      /* See if we must also compute the other checkerboard version */
      if (y_start + CHECK_SIZE < y_end)
	need_check_idx[1 - first_check_idx] = FALSE;
      buf_start[0] =
	PREVIEW_DATA (preview)->preview_buffer_na[0];
      buf_start[1] =
	PREVIEW_DATA (preview)->preview_buffer_na[1];
    } 
  else 
    {
      buf_start[0] = buf_start[1] = 
	PREVIEW_DATA (preview)->preview_buffer_na[0];
    }
  
  switch(type)
    {
    case GIMP_RGBA_IMAGE:
      for (check_idx = 0; check_idx < 2; check_idx++)
	{
	  buf_start[check_idx] =
	    PREVIEW_DATA (preview)->preview_buffer_na[check_idx];
	  if (need_check_idx[check_idx])
	    {
	      guchar * const dest_ptr = buf_start[check_idx];
	      const gint check_offset =
		preview->prev_x + CHECK_SIZE * (1 - check_idx);

	      for (x = width - 1; x >= 0; x--) 
		{
		  const guchar check = ((x + check_offset) & CHECK_SIZE) ?
		    CHECK_LIGHT : CHECK_DARK;
		  const guchar * const src_ptr = data +
		    (((gint)((prev_x + x)/scale)) - image_x) * 4;
      
		  dest_ptr[3 * x + 0] = 
		    check + (((src_ptr[0] - check) * src_ptr[3]) >> 8);
		  dest_ptr[3 * x + 1] = 
		    check + (((src_ptr[1] - check) * src_ptr[3]) >> 8);
		  dest_ptr[3 * x + 2] = 
		    check + (((src_ptr[2] - check) * src_ptr[3]) >> 8);
		}
	    }
	}
      break;
    case GIMP_RGB_IMAGE:
      {
	guchar* const dest_ptr = buf_start[0];

	for (x = width - 1; x >= 0; x--) 
	  {
	    const guchar * const src_ptr = data +
	      (((gint)((prev_x + x)/scale)) - image_x) * 3;

	    dest_ptr[3 * x + 0] = src_ptr[0];
	    dest_ptr[3 * x + 1] = src_ptr[1];
	    dest_ptr[3 * x + 2] = src_ptr[2];
	  }
      }
      break;
    case GIMP_GRAYA_IMAGE: 
      for (check_idx = 0; check_idx < 2; check_idx++)
	{
	  buf_start[check_idx] =
	    PREVIEW_DATA (preview)->preview_buffer_na[check_idx];
	  if (need_check_idx[check_idx])
	    {
	      guchar * const dest_ptr = buf_start[check_idx];
	      const gint check_offset =
		preview->prev_x + CHECK_SIZE * (1 - check_idx);

	      for (x = width - 1; x >= 0; x--) 
		{
		  const guchar check = ((x + check_offset) & CHECK_SIZE) ?
		    CHECK_LIGHT : CHECK_DARK;
		  const guchar * const src_ptr = data +
		    (((gint)((prev_x + x)/scale)) - image_x) * 2;
      
		  dest_ptr[3 * x + 0] = 
		    dest_ptr[3 * x + 1] = 
		    dest_ptr[3 * x + 2] =
		    check + (((src_ptr[0] - check) * src_ptr[1]) >> 8);
		}
	    }
	}
      break;
    case GIMP_GRAY_IMAGE:
      {
	guchar* const dest_ptr = buf_start[0];

	for (x = width - 1; x >= 0; x--) 
	  {
	    const guchar * const src_ptr = data +
	      (((gint)((prev_x + x)/scale)) - image_x);

	    dest_ptr[3 * x + 0] = 
	      dest_ptr[3 * x + 1] = 
	      dest_ptr[3 * x + 2] = src_ptr[0];
	  }
      }
      break;
    default:
      /* Type argument is wrong */
      break;
    }

  for(y = y_start; y < y_end; y++)
    {
      const gint check_idx = (y  / CHECK_SIZE) & 1;

      gtk_preview_draw_row (GTK_PREVIEW (preview->preview),
			    buf_start[check_idx],
			    0, y - preview->prev_y, width);
    }
}

