/* tga.c release 1.1 1997-09-19
 *
 * TrueVision Targa loading and saving file filter for the Gimp
 *
 * Release 1.1, 1997-09-19, Gordon Matzigkeit <gord@gnu.ai.mit.edu>:
 *   - Preserve alpha channels.  For indexed images, this can only be
 *     done if there is at least one free colormap entry.
 *
 * Release 1.0, 1997-09-06, Gordon Matzigkeit <gord@gnu.ai.mit.edu>:
 *   - Handle loading all image types from the 2.0 specification.
 *   - Fix many alignment and endianness problems.
 *   - Use tiles for lower memory consumption and better speed.
 *   - Rewrite RLE code for clarity and speed.
 *   - Handle saving with RLE.
 *
 * Release 0.9, 1997-06-18, Raphael FRANCOIS <fraph@ibm.net>:
 *   - Can load 24 and 32-bit Truecolor images, with and without RLE.
 *   - Saving currently only works without RLE.
 *
 *
 * TODO:
 *   - Handle loading images that aren't 8 bits per channel.
 *   - Maybe handle special features in developer and extension sections
 *     (the `save' dialogue would give access to them).
 *   - The GIMP stores the indexed alpha channel as a separate byte,
 *     one for each pixel.  The TGA file format spec requires that the
 *     alpha channel be stored as part of the colormap, not with each
 *     individual pixel.  This means that we have no good way of
 *     saving and loading INDEXEDA images that use alpha channel values
 *     other than 0 and 255.
 */

/*
 * The Targa reading and writing code was written from scratch by
 * Raphael FRANCOIS <fraph@ibm.net> and Gordon Matzigkeit
 * <gord@gnu.ai.mit.edu> based on the TrueVision TGA File Format
 * Specification, Version 2.0:
 *
 *   <URL:ftp://ftp.truevision.com/pub/TGA.File.Format.Spec/>
 *
 * It does not contain any code written for other TGA file loaders,
 * including netpbm or pbmplus.  Not even the RLE handling. ;)
 */

#define SAVE_ID_STRING "CREATOR: The GIMP's TGA Filter Version 1.1"

/* #define PROFILE 1 */
#ifdef PROFILE
# include <sys/times.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gtk/gtk.h"
#include "libgimp/gimp.h"

#define FALSE 0
#define TRUE 1

typedef struct _TgaSaveVals
{
  gint rle;
} TgaSaveVals;

static TgaSaveVals tsvals =
{
  1,    /* rle */
};

typedef struct _TgaSaveInterface
{
  unsigned char run;
} TgaSaveInterface;

static TgaSaveInterface tsint =
{
  FALSE                /*  run  */
};

static struct
{
  guint8 idLength;
  guint8 colorMapType;

  /* The image type. */
#define TGA_TYPE_MAPPED 1
#define TGA_TYPE_COLOR 2
#define TGA_TYPE_GRAY 3
#define TGA_TYPE_MAPPED_RLE 9
#define TGA_TYPE_COLOR_RLE 10
#define TGA_TYPE_GRAY_RLE 11
  guint8 imageType;

  /* Color Map Specification. */
  /* We need to separately specify high and low bytes to avoid endianness
     and alignment problems. */
  guint8 colorMapIndexLo, colorMapIndexHi;
  guint8 colorMapLengthLo, colorMapLengthHi;
  guint8 colorMapSize;

  /* Image Specification. */
  guint8 xOriginLo, xOriginHi;
  guint8 yOriginLo, yOriginHi;

  guint8 widthLo, widthHi;
  guint8 heightLo, heightHi;

  guint8 bpp;

  /* Image descriptor.
     3-0: attribute bpp
     4:   left-to-right ordering
     5:   top-to-bottom ordering
     7-6: zero
     */
#define TGA_DESC_ABITS 0x0f
#define TGA_DESC_HORIZONTAL 0x10
#define TGA_DESC_VERTICAL 0x20
  guint8 descriptor;


} tga_header;

static struct
{
  guint32 extensionAreaOffset;
  guint32 developerDirectoryOffset;
#define TGA_SIGNATURE "TRUEVISION-XFILE"
  gchar signature[16];
  gchar dot;
  gchar null;
} tga_footer;


/* Declare some local functions.
 */
static void   query      (void);
static void   run        (char    *name,
                          int      nparams,
                          GParam  *param,
                          int     *nreturn_vals,
                          GParam **return_vals);
static gint32 load_image (char   *filename);
static gint   save_image (char   *filename,
			  gint32  image_ID,
			  gint32  drawable_ID);

static gint   save_dialog ();

static void   save_close_callback  (GtkWidget *widget,
				    gpointer   data);
static void   save_ok_callback     (GtkWidget *widget,
				    gpointer   data);
static void   save_toggle_update   (GtkWidget *widget,
				    gpointer   data);

GPlugInInfo PLUG_IN_INFO =
{
  NULL,    /* init_proc */
  NULL,    /* quit_proc */
  query,   /* query_proc */
  run,     /* run_proc */
};


MAIN ();

static int verbose = 0;

static void
query ()
{
  static GParamDef load_args[] =
  {
    { PARAM_INT32, "run_mode", "Interactive, non-interactive" },
    { PARAM_STRING, "filename", "The name of the file to load" },
    { PARAM_STRING, "raw_filename", "The name entered" },
  };
  static GParamDef load_return_vals[] =
  {
    { PARAM_IMAGE, "image", "Output image" },
  };
  static int nload_args = sizeof (load_args) / sizeof (load_args[0]);
  static int nload_return_vals = sizeof (load_return_vals) / sizeof (load_return_vals[0]);


  static GParamDef save_args[] =
  {
    { PARAM_INT32, "run_mode", "Interactive, non-interactive" },
    { PARAM_IMAGE, "image", "Input image" },
    { PARAM_DRAWABLE, "drawable", "Drawable to save" },
    { PARAM_STRING, "filename", "The name of the file to save the image in" },
    { PARAM_STRING, "raw_filename", "The name of the file to save the image in" },
    { PARAM_INT32, "rle", "Enable RLE compression" },
  } ;
  static int nsave_args = sizeof (save_args) / sizeof (save_args[0]);

  gimp_install_procedure ("file_tga_load",
                          "Loads files of Targa file format",
                          "FIXME: write help for tga_load",
                          "Raphael FRANCOIS, Gordon Matzigkeit",
                          "Raphael FRANCOIS, Gordon Matzigkeit",
                          "1997",
                          "<Load>/TGA",
			  NULL,
                          PROC_PLUG_IN,
                          nload_args, nload_return_vals,
                          load_args, load_return_vals);


  gimp_install_procedure ("file_tga_save",
                          "saves files in the Targa file format",
                          "FIXME: write help for tga_save",
			  "Raphael FRANCOIS, Gordon Matzigkeit",
                          "Raphael FRANCOIS, Gordon Matzigkeit",
                          "1997",
                          "<Save>/TGA",
			  "RGB*, GRAY*, INDEXED*",
                          PROC_PLUG_IN,
                          nsave_args, 0,
                          save_args, NULL);

  gimp_register_magic_load_handler ("file_tga_load", "tga", "",
				    "0&,byte,10,2&,byte,1,3&,byte,>0,3,byte,<9");
  gimp_register_save_handler ("file_tga_save", "tga", "");

}

static void
run (char    *name,
     int      nparams,
     GParam  *param,
     int     *nreturn_vals,
     GParam **return_vals)
{
  static GParam values[2];
  GStatusType status = STATUS_SUCCESS;
  GRunModeType run_mode;
  gint32 image_ID;

#ifdef PROFILE
  struct tms tbuf1, tbuf2;
#endif

  run_mode = param[0].data.d_int32;

  *nreturn_vals = 1;
  *return_vals = values;
  values[0].type = PARAM_STATUS;
  values[0].data.d_status = STATUS_CALLING_ERROR;

  if (verbose)
    printf ("TGA: RUN %s\n", name);

  if (strcmp (name, "file_tga_load") == 0)
    {
#ifdef PROFILE
      times (&tbuf1);
#endif
      image_ID = load_image (param[1].data.d_string);

      if (image_ID != -1)
        {
          *nreturn_vals = 2;
          values[0].data.d_status = STATUS_SUCCESS;
          values[1].type = PARAM_IMAGE;
          values[1].data.d_image = image_ID;
        }
      else
        {
          values[0].data.d_status = STATUS_EXECUTION_ERROR;
        }
    }
  else if (strcmp (name, "file_tga_save") == 0)
    {
      switch (run_mode)
	{
	case RUN_INTERACTIVE:
	  /*  Possibly retrieve data  */
	  gimp_get_data ("file_tga_save", &tsvals);

	  /*  First acquire information with a dialog  */
	  if (! save_dialog ())
	    return;

	  break;

	case RUN_NONINTERACTIVE:
	  /*  Make sure all the arguments are there!  */
	  if (nparams != 6)
	    status = STATUS_CALLING_ERROR;
	  if (status == STATUS_SUCCESS)
	    {
	      tsvals.rle = param[5].data.d_int32;
	    }

	case RUN_WITH_LAST_VALS:
	  /*  Possibly retrieve data  */
	  gimp_get_data ("file_tga_save", &tsvals);
	  break;

	default:
	  break;
	}

#ifdef PROFILE
      times (&tbuf1);
#endif
      *nreturn_vals = 1;
      if (save_image (param[3].data.d_string, param[1].data.d_int32, param[2].data.d_int32))
	{
	  /*  Store psvals data  */
	  gimp_set_data ("file_tga_save", &tsvals, sizeof (tsvals));

	  values[0].data.d_status = STATUS_SUCCESS;
	}
      else
	values[0].data.d_status = STATUS_EXECUTION_ERROR;
    }

#ifdef PROFILE
  times (&tbuf2);
  printf ("TGA: %s profile: %ld user %ld system\n", name,
	  (long) tbuf2.tms_utime - tbuf1.tms_utime,
	  (long) tbuf2.tms_stime - tbuf2.tms_stime);
#endif
}

static gint32 ReadImage (FILE *fp, char *filename);

static gint32
load_image (char *filename)
{
  FILE *fp;
  char * name_buf;

  gint32 image_ID = -1;

  name_buf = g_malloc (strlen (filename) + 11);
  sprintf (name_buf, "Loading %s:", filename);
  gimp_progress_init (name_buf);
  g_free (name_buf);

  fp = fopen (filename, "rb");
  if (!fp) {
      printf ("TGA: can't open %s\n", filename);
      return -1;
    }

  /* Check the footer. */
  if (fseek (fp, 0L - (sizeof (tga_footer)), SEEK_END)
      || fread (&tga_footer, sizeof (tga_footer), 1, fp) != 1)
    {
      printf ("TGA: Cannot read footer from \"%s\"\n", filename);
      return -1;
    }

  /* Check the signature. */
  if (memcmp (tga_footer.signature, TGA_SIGNATURE,
	      sizeof (tga_footer.signature)) == 0)
    {
      if (verbose)
	printf ("TGA: found New TGA\n");
    }
  else if (verbose)
    printf ("TGA: found Original TGA\n");

  if (fseek (fp, 0, SEEK_SET) ||
      fread (&tga_header, sizeof (tga_header), 1, fp) != 1) {
    printf("TGA: Cannot read header from \"%s\"\n", filename);
    return -1;
  }

  /* Skip the image ID field. */
  if (tga_header.idLength && fseek (fp, tga_header.idLength, SEEK_CUR))
    {
      printf ("TGA: Cannot skip ID field in \"%s\"\n", filename);
      return -1;
    }

  image_ID = ReadImage (fp, filename);
  return image_ID;
}


static int totbytes = 0;
static int
std_fread (guchar *buf, int datasize, int nelems, FILE *fp)
{
  if (verbose > 1)
    {
      totbytes += nelems * datasize;
      printf ("TGA: std_fread %d (total %d)\n",
	      nelems * datasize, totbytes);
    }

  return fread (buf, datasize, nelems, fp);
}

static int
std_fwrite (guchar *buf, int datasize, int nelems, FILE *fp)
{
  if (verbose > 1)
    {
      totbytes += nelems * datasize;
      printf ("TGA: std_fwrite %d (total %d)\n",
	      nelems * datasize, totbytes);
    }

  return fwrite (buf, datasize, nelems, fp);
}

#define RLE_PACKETSIZE 0x80

/* Decode a bufferful of file. */
static int
rle_fread (guchar *buf, int datasize, int nelems, FILE *fp)
{
  static guchar *statebuf = 0;
  static int statelen = 0;
  static int laststate = 0;

  int j, k;
  int buflen, count, bytes, curbytes;
  guchar *p;

  /* Scale the buffer length. */
  buflen = nelems * datasize;

  j = 0;
  curbytes = totbytes;
  while (j < buflen)
    {
      if (laststate < statelen)
	{
	  /* Copy bytes from our previously decoded buffer. */
	  bytes = MIN (buflen - j, statelen - laststate);
	  memcpy (buf + j, statebuf + laststate, bytes);
	  j += bytes;
	  laststate += bytes;

	  /* If we used up all of our state bytes, then reset them. */
	  if (laststate >= statelen)
	    {
	      laststate = 0;
	      statelen = 0;
	    }

	  /* If we filled the buffer, then exit the loop. */
	  if (j >= buflen)
	    break;
	}

      /* Decode the next packet. */
      count = fgetc (fp);
      if (count == EOF)
	{
	  if (verbose)
	    printf ("TGA: hit EOF while looking for count\n");
	  return 0;
	}

      /* Scale the byte length to the size of the data. */
      bytes = ((count & ~RLE_PACKETSIZE) + 1) * datasize;

      if (j + bytes <= buflen)
	{
	  /* We can copy directly into the image buffer. */
	  p = buf + j;
	}
      else {
#ifdef PROFILE
	printf ("TGA: needed to use statebuf for %d bytes\n", buflen - j);
#endif
	/* Allocate the state buffer if we haven't already. */
	if (!statebuf)
	  statebuf = (guchar *) g_malloc (RLE_PACKETSIZE * datasize);
	p = statebuf;
      }

      if (count & RLE_PACKETSIZE)
	{
	  /* Fill the buffer with the next value. */
	  if (fread (p, datasize, 1, fp) != 1)
	    {
	      if (verbose)
		printf ("TGA: EOF while reading %d/%d element RLE packet\n",
			bytes, datasize);
	      return 0;
	    }

	  /* Optimized case for single-byte encoded data. */
	  if (datasize == 1)
	    memset (p + 1, *p, bytes - 1);
	  else
	    for (k = datasize; k < bytes; k += datasize)
	      memcpy (p + k, p, datasize);
	}
      else
	{
	  /* Read in the buffer. */
	  if (fread (p, bytes, 1, fp) != 1)
	    {
	      if (verbose)
		printf ("TGA: EOF while reading %d/%d element raw packet\n",
			bytes, datasize);
	      return 0;
	    }
	}

      if (verbose > 1)
	{
	  totbytes += bytes;
	  if (verbose > 2)
	    printf ("TGA: %s packet %d/%d\n",
		    (count & RLE_PACKETSIZE) ? "RLE" : "raw",
		    bytes, totbytes);
	}

      /* We may need to copy bytes from the state buffer. */
      if (p == statebuf)
	statelen = bytes;
      else
	j += bytes;
    }

  if (verbose > 1)
    {
      printf ("TGA: rle_fread %d/%d (total %d)\n",
	      nelems * datasize, totbytes - curbytes, totbytes);
    }
  return nelems;
}


/* This function is stateless, which means that we always finish packets
   on buffer boundaries.  As a beneficial side-effect, rle_fread
   never has to allocate a state buffer when it loads our files, provided
   it is called using the same buffer lengths!

   So, we get better compression than line-by-line encoders, and better
   loading performance than whole-stream images. */
/* RunLength Encode a bufferful of file. */
static int
rle_fwrite (guchar *buf, int datasize, int nelems, FILE *fp)
{
  /* Now runlength-encode the whole buffer. */
  int count, j, buflen, curbytes;
  guchar *begin;

  /* Scale the buffer length. */
  buflen = datasize * nelems;

  begin = buf;
  j = datasize;
  curbytes = totbytes;
  while (j < buflen)
    {
      /* BUF[J] is our lookahead element, BEGIN is the beginning of this
	 run, and COUNT is the number of elements in this run. */
      if (memcmp (buf + j, begin, datasize) == 0)
	{
	  /* We have a run of identical characters. */
	  count = 1;
	  do
	    {
	      j += datasize;
	      count ++;
	    }
	  while (j < buflen && count < RLE_PACKETSIZE &&
		 memcmp (buf + j, begin, datasize) == 0);

	  /* Write out the run. */
	  if (fputc ((count - 1) | RLE_PACKETSIZE, fp) == EOF ||
	      fwrite (begin, datasize, 1, fp) != 1)
	    return 0;

	  if (verbose > 1)
	    {
	      totbytes += count * datasize;
	      if (verbose > 2)
		printf ("TGA: RLE packet %d/%d\n", count * datasize, totbytes);
	    }
	}
      else
	{
	  /* We have a run of raw characters. */
	  count = 0;
	  do
	    {
	      j += datasize;
	      count ++;
	    }
	  while (j < buflen && count < RLE_PACKETSIZE &&
		 memcmp (buf + j - datasize, buf + j, datasize) != 0);

	  /* Back up to the previous element if the lookahead is still
	     in the buffer. */
	  if (j < buflen)
	    j -= datasize;

	  /* Write out the raw packet. */
	  if (fputc (count - 1, fp) == EOF ||
	      fwrite (begin, datasize, count, fp) != count)
	    return 0;

	  if (verbose > 1)
	    {
	      totbytes += count * datasize;
	      if (verbose > 2)
		printf ("TGA: raw packet %d/%d\n", count * datasize, totbytes);
	    }
	}

      /* Set the beginning of the next run and the next lookahead. */
      begin = buf + j;
      j += datasize;
    }

  /* If we didn't encode all the elements, write one last packet. */
  if (begin < buf + buflen)
    {
      if (verbose > 1)
	{
	  totbytes += datasize;
	  if (verbose > 2)
	    printf ("TGA: FINAL raw packet %d/%d\n", datasize, totbytes);
	}
      if (fputc (0, fp) == EOF ||
	  fwrite (begin, datasize, 1, fp) != 1)
	return 0;
    }

  if (verbose > 1)
    {
      printf ("TGA: rle_fwrite %d/%d (total %d)\n",
	      totbytes - curbytes, nelems * datasize, totbytes);
    }
  return nelems;
}




static gint32
ReadImage (FILE *fp, char *filename)

{
  static gint32 image_ID;
  gint32 layer_ID;
  /*  gint32 mask_ID; */
  GPixelRgn pixel_rgn;
  GDrawable *drawable;
  guchar *data, *buffer;
  GDrawableType dtype;
  GImageType itype;
  guchar *alphas;

  int width, height, bpp, abpp, pbpp;
  int i, j, k;
  int pelbytes, tileheight, wbytes, bsize, npels;
  int rle;

  int (*myfread)(guchar *, int, int, FILE *);

  gchar horzrev = tga_header.descriptor & TGA_DESC_HORIZONTAL;
  gchar vertrev = !(tga_header.descriptor & TGA_DESC_VERTICAL);

  width = (tga_header.widthHi << 8) | tga_header.widthLo;
  height = (tga_header.heightHi << 8) | tga_header.heightLo;

  bpp = tga_header.bpp;
  abpp = tga_header.descriptor & TGA_DESC_ABITS;

  if (bpp < abpp)
    {
      printf ("TGA: %d attribute bit size is greater than %d total pixel size\n",
	      abpp, bpp);
      /* Assume that alpha bits were set incorrectly. */
      pbpp = bpp;
      abpp = 0;
    }
  else
    pbpp = bpp - abpp;

  rle = 0;
  switch (tga_header.imageType)
    {
    case TGA_TYPE_MAPPED_RLE:
      rle = 1;
    case TGA_TYPE_MAPPED:
      itype = INDEXED;

      /* Find the size of palette elements. */
      pbpp = MIN (tga_header.colorMapSize / 3, 8) * 3;
      if (pbpp < tga_header.colorMapSize)
	abpp = tga_header.colorMapSize - pbpp;
      else
	abpp = 0;

      if (verbose)
	printf ("TGA: %d bit indexed image, %d bit alpha (%d bit indices)\n",
		pbpp, abpp, bpp);

      if (bpp != 8)
	{
	  /* We can only cope with 8-bit indices. */
	  printf ("TGA: index sizes other than 8 bits are unimplemented\n");
	  return -1;
	}

      if (abpp)
	dtype = INDEXEDA_IMAGE;
      else
	dtype = INDEXED_IMAGE;
      break;

    case TGA_TYPE_GRAY_RLE:
      rle = 1;
    case TGA_TYPE_GRAY:
      itype = GRAY;
      if (verbose)
	printf ("TGA: %d bit grayscale image, %d bit alpha\n",
		pbpp, abpp);

      if (abpp)
	dtype = GRAYA_IMAGE;
      else
	dtype = GRAY_IMAGE;
      break;

    case TGA_TYPE_COLOR_RLE:
      rle = 1;
    case TGA_TYPE_COLOR:
      itype = RGB;
      printf ("TGA: %d bit color image, %d bit alpha\n", pbpp, abpp);

      if (abpp)
	dtype = RGBA_IMAGE;
      else
	dtype = RGB_IMAGE;
      break;

    default:
      printf ("TGA: unrecognized image type %d\n", tga_header.imageType);
      return -1;
    }

  if ((abpp && abpp != 8) ||
      ((itype == RGB || itype == INDEXED) && pbpp != 24) ||
      (itype == GRAY && pbpp != 8))
    {
      /* FIXME: We haven't implemented bit-packed fields yet. */
      printf ("TGA: channel sizes other than 8 bits are unimplemented\n");
      return -1;
    }

  /* Check that we have a color map only when we need it. */
  if (itype == INDEXED)
    {
      if (tga_header.colorMapType != 1)
	{
	  printf ("TGA: indexed image has invalid color map type %d\n",
		  tga_header.colorMapType);
	  return -1;
	}
    }
  else if (tga_header.colorMapType != 0)
    {
      printf ("TGA: non-indexed image has invalid color map type %d\n",
	      tga_header.colorMapType);
      return -1;
    }

  image_ID = gimp_image_new (width, height, itype);
  gimp_image_set_filename (image_ID, filename);

  layer_ID = gimp_layer_new (image_ID,
			     "Background",
			     width, height,
			     dtype,
			     100,
			     NORMAL_MODE);

  gimp_image_add_layer (image_ID, layer_ID, 0);

  drawable = gimp_drawable_get (layer_ID);

#if 0
  verbose = 2;
  printf ("TGA: debug %d\n", getpid ());
  kill (getpid (), 19);
#endif

  alphas = 0;
  if (tga_header.colorMapType == 1)
    {
      /* We need to read in the colormap. */
      int index, length, colors;

      guchar *cmap;

      index = (tga_header.colorMapIndexHi << 8) | tga_header.colorMapIndexLo;
      length = (tga_header.colorMapLengthHi << 8) |
	tga_header.colorMapLengthLo;

      if (verbose)
	printf ("TGA: reading color map (%d + %d) * (%d / 8)\n",
		index, length, tga_header.colorMapSize);
      if (length == 0)
	{
	  printf ("TGA: invalid color map length %d\n", length);
	  return -1;
	}

      pelbytes = tga_header.colorMapSize / 8;
      colors = length + index;
      cmap = g_malloc (colors * pelbytes);

      /* Zero the entries up to the beginning of the map. */
      memset (cmap, 0, index * pelbytes);

      /* Read in the rest of the colormap. */
      if (fread (cmap + (index * pelbytes), pelbytes, length, fp) != length)
	{
	  printf ("TGA: error reading colormap (ftell == %ld)\n", ftell (fp));
	  return -1;
	}

      /* If we have an alpha channel, then create a mapping to the alpha
	 values. */
      if (pelbytes > 3)
	{
	  int tmp, k, a;
	  alphas = (guchar *) g_malloc (colors);

	  /* Take the alpha values out of the colormap. */
	  k = 0;
	  a = 0;
	  for (j = 0; j < colors * pelbytes; j += pelbytes)
	    {
	      /* Swap from BGR to RGB. */
	      tmp = cmap[j];
	      cmap[k ++] = cmap[j + 2];
	      cmap[k ++] = cmap[j + 1];
	      cmap[k ++] = tmp;

	      /* Grab the alpha value. */
	      alphas[a ++] = cmap[j + 3];
	    }

	  /* If the last color was transparent, then omit it. */
	  if (alphas[colors - 1] == 0)
	    colors --;
	}
      else
	{
	  /* Rearrange the colors from BGR to RGB. */
	  int tmp;
	  for (j = index; j < length * pelbytes; j += pelbytes)
	    {
	      tmp = cmap[j];
	      cmap[j] = cmap[j + 2];
	      cmap[j + 2] = tmp;
	    }
	}

      /* Set the colormap. */
      gimp_image_set_cmap (image_ID, cmap, colors);
      g_free (cmap);

      /* Now pretend as if we only have 8 bpp. */
      pbpp = 8;
    }

  /* Prepare the pixel region. */
  gimp_pixel_rgn_init (&pixel_rgn, drawable, 0, 0, width, height, TRUE, FALSE);

  /* Calculate number of GIMP bytes per pixel. */
  pelbytes = (pbpp + abpp + 7) / 8;

  /* Calculate TGA bytes per pixel. */
  bpp = alphas ? (pbpp + 7) / 8 : pelbytes;

  /* Allocate the data. */
  tileheight = gimp_tile_height ();
  data = (guchar *) g_malloc (width * tileheight * pelbytes);

  /* Maybe we need to reverse the data. */
  buffer = 0;
  if (horzrev || vertrev)
    buffer = (guchar *) g_malloc (width * tileheight * pelbytes);

  if (rle)
    myfread = rle_fread;
  else
    myfread = std_fread;

  wbytes = width * pelbytes;
  for (i = 0; i < height; i += tileheight)
    {
      tileheight = MIN (tileheight, height - i);

      if (verbose > 1)
	printf ("TGA: reading %dx(%d+%d)x%d pixel region\n", width,
		(vertrev ? (height - (i + tileheight)) : i),
		tileheight, pelbytes);

      npels = width * tileheight;
      bsize = wbytes * tileheight;

      /* Suck in the data one tileheight at a time. */
      if ((*myfread) (data, bpp, npels, fp) != npels)
	{
	  /* Probably premature end of file. */
	  printf ("TGA: error reading (ftell == %ld)\n", ftell (fp));
	  return -1;
	}

      /* If we have indexed alphas, then set them. */
      if (alphas)
	{
	  int k;

	  /* Start at the end of the buffer, and work backwards. */
	  k = (npels - 1) * bpp;
	  for (j = bsize - pelbytes; j >= 0; j -= pelbytes)
	    {
	      /* Find the alpha for this index. */
	      data[j + 1] = alphas[data[k]];
	      data[j] = data[k --];
	    }
	}
      else if (pelbytes >= 3)
	{
	  /* Rearrange the colors from BGR to RGB. */
	  int tmp;
	  for (j = 0; j < bsize; j += pelbytes)
	    {
	      tmp = data[j];
	      data[j] = data[j + 2];
	      data[j + 2] = tmp;
	    }
	}


      if (horzrev || vertrev)
	{
	  guchar *tmp;
	  if (vertrev)
	    {
	      /* We need to mirror only vertically. */
	      for (j = 0; j < bsize; j += wbytes)
		memcpy (buffer + j,
			data + bsize - (j + wbytes), wbytes);
	    }
	  else if (horzrev)
	    {
	      /* We need to mirror only horizontally. */
	      for (j = 0; j < bsize; j += wbytes)
		for (k = 0; k < wbytes; k += pelbytes)
		  memcpy (buffer + k + j,
			  data + (j + wbytes) - (k + pelbytes), pelbytes);
	    }
	  else
	    {
	      /* Completely reverse the pixels in the buffer. */
	      for (j = 0; j < bsize; j += pelbytes)
		memcpy (buffer + j,
			data + bsize - (j + pelbytes), pelbytes);
	    }

	  /* Swap the buffers because we modified them. */
	  tmp = buffer;
	  buffer = data;
	  data = tmp;
	}

      gimp_progress_update ((double) (i + tileheight) / (double) height);

      /* If vertically reversed, put the data at the end. */
      gimp_pixel_rgn_set_rect
	(&pixel_rgn, data,
	 0, (vertrev ? (height - (i + tileheight)) : i),
	 width, tileheight);
    }

  if (fgetc (fp) != EOF)
    printf ("TGA: too much input data, ignoring extra...\n");

  g_free (data);
  g_free (buffer);

  if (alphas)
    g_free (alphas);

  gimp_drawable_flush (drawable);
  gimp_drawable_detach (drawable);

  return image_ID;
}  /*read_image*/


gint
save_image (char   *filename,
	    gint32  image_ID,
	    gint32  drawable_ID)
{
  GPixelRgn pixel_rgn;
  GDrawable *drawable;
  GDrawableType dtype;
  int width, height;
  FILE *fp;
  guchar *name_buf;
  int i, j;
  int npels, tileheight, pelbytes, bsize;
  int transparent;

  int (*myfwrite)(guchar *, int, int, FILE *);
  guchar *data;

  drawable = gimp_drawable_get(drawable_ID);
  dtype = gimp_drawable_type(drawable_ID);
  width = drawable->width;
  height = drawable->height;

  gimp_pixel_rgn_init(&pixel_rgn, drawable, 0, 0, width, height, FALSE, FALSE);

  name_buf = (guchar *) g_malloc(strlen(filename) + 11);
  sprintf(name_buf, "Saving %s:", filename);
  gimp_progress_init(name_buf);
  g_free(name_buf);

  memset (&tga_header, 0, sizeof (tga_header));
  /* We like our images top-to-bottom, thank you! */
  tga_header.descriptor |= TGA_DESC_VERTICAL;

  /* Choose the imageType based on our drawable and compression option. */
  switch (dtype)
    {
    case INDEXEDA_IMAGE:
    case INDEXED_IMAGE:
      tga_header.bpp = 8;
      tga_header.imageType = TGA_TYPE_MAPPED;
      break;

    case GRAYA_IMAGE:
      tga_header.bpp = 8;
      tga_header.descriptor |= 8;
    case GRAY_IMAGE:
      tga_header.bpp += 8;
      tga_header.imageType = TGA_TYPE_GRAY;
      break;

    case RGBA_IMAGE:
      tga_header.bpp = 8;
      tga_header.descriptor |= 8;
    case RGB_IMAGE:
      tga_header.bpp += 24;
      tga_header.imageType = TGA_TYPE_COLOR;
      break;

    default:
      return FALSE;
    }

  if (tsvals.rle)
    {
      /* Here we take advantage of the fact that the RLE image type codes
	 are exactly 8 greater than the non-RLE. */
      tga_header.imageType += 8;
      myfwrite = rle_fwrite;
    }
  else
    myfwrite = std_fwrite;

  tga_header.widthLo = (width & 0xff);
  tga_header.widthHi = (width >> 8);

  tga_header.heightLo = (height & 0xff);
  tga_header.heightHi = (height >> 8);

  if((fp = fopen(filename, "wb")) == NULL) {
    printf("TGA: Can't open \"%s\"\n", filename);
    return FALSE;
  }

  /* Mark our save ID. */
  tga_header.idLength = strlen (SAVE_ID_STRING);

  /* See if we have to write out the colour map. */
  transparent = 0;
  if (tga_header.imageType == TGA_TYPE_MAPPED_RLE ||
      tga_header.imageType == TGA_TYPE_MAPPED)
    {
      guchar *cmap;
      int colors;

      tga_header.colorMapType = 1;
      cmap = gimp_image_get_cmap (image_ID, &colors);
      if (colors > 256)
	{
	  printf ("TGA: cannot handle colormap with more than 256 colors (got %d)\n", colors);
	  return FALSE;
	}

      /* If we already have more than 256 colors, then ignore the
	 alpha channel.  Otherwise, create an entry for any completely
	 transparent pixels. */
      if (dtype == INDEXEDA_IMAGE && colors < 256)
	{
	  transparent = colors;
	  tga_header.colorMapSize = 32;
	  tga_header.colorMapLengthLo = ((colors + 1) & 0xff);
	  tga_header.colorMapLengthHi = ((colors + 1) >> 8);
	}
      else
	{
	  tga_header.colorMapSize = 24;
	  tga_header.colorMapLengthLo = (colors & 0xff);
	  tga_header.colorMapLengthHi = (colors >> 8);
	}

      /* Write the header. */
      if (fwrite(&tga_header, sizeof(tga_header), 1, fp) != 1)
	return FALSE;
      if (fwrite (SAVE_ID_STRING, tga_header.idLength, 1, fp) != 1)
	return FALSE;

      pelbytes = tga_header.colorMapSize / 8;
      if (transparent)
	{
	  guchar *newcmap;
	  int k;

	  /* Reallocate our colormap to have an alpha channel and
	     a fully transparent color. */
	  newcmap = (guchar *) g_malloc ((colors + 1) * pelbytes);

	  k = 0;
	  for (j = 0; j < colors * 3; j += 3)
	    {
	      /* Rearrange from RGB to BGR. */
	      newcmap[k ++] = cmap[j + 2];
	      newcmap[k ++] = cmap[j + 1];
	      newcmap[k ++] = cmap[j];

	      /* Set to maximum opacity. */
	      newcmap[k ++] = -1;
	    }

	  /* Add the transparent color. */
	  memset (newcmap + k, 0, pelbytes);

	  /* Write out the colormap. */
	  if (fwrite (newcmap, pelbytes, colors + 1, fp) != colors + 1)
	    return FALSE;
	  g_free (newcmap);
	}
      else
	{
	  /* Rearrange the colors from RGB to BGR. */
	  int tmp;
	  for (j = 0; j < colors * pelbytes; j += pelbytes)
	    {
	      tmp = cmap[j];
	      cmap[j] = cmap[j + 2];
	      cmap[j + 2] = tmp;
	    }

	  /* Write out the colormap. */
	  if (fwrite (cmap, pelbytes, colors, fp) != colors)
	    return FALSE;
	}
    }
  /* Just write the header. */
  else
    {
      if (fwrite(&tga_header, sizeof(tga_header), 1, fp) != 1)
	return FALSE;
      if (fwrite (SAVE_ID_STRING, tga_header.idLength, 1, fp) != 1)
	return FALSE;
    }

  /* Allocate a new set of pixels. */
  tileheight = gimp_tile_height ();
  data = (guchar *) g_malloc(width * tileheight * drawable->bpp);

  /* Write out the pixel data. */
  pelbytes = tga_header.bpp / 8;
  for (i = 0; i < height; i += tileheight)
    {
      /* Get a horizontal slice of the image. */
      tileheight = MIN(tileheight, height - i);
      gimp_pixel_rgn_get_rect(&pixel_rgn, data, 0, i, width, tileheight);

      if (verbose > 1)
	printf ("TGA: writing %dx(%d+%d)x%d pixel region\n",
		width, i, tileheight, pelbytes);

      npels = width * tileheight;
      bsize = npels * pelbytes;

      /* If the GIMP's bpp does not match the TGA format, strip
	 the excess bytes.  This is how we handle indexed images with
	 an alpha channel. */
      if (drawable->bpp > pelbytes)
	{
	  int k, nbytes, difference, fullbsize;

	  j = k = 0;
	  fullbsize = npels * drawable->bpp;

	  difference = drawable->bpp - pelbytes;
	  while (j < fullbsize)
	    {
	      nbytes = 0;
	      for (nbytes = 0; nbytes < pelbytes; nbytes ++)
		/* Be careful to handle overlapping pixels. */
		data[k ++] = data[j ++];

	      /* If this is an indexed image, and data[j] (alpha
		 channel) is zero, then we should write our transparent
		 pixel's index. */
	      if (dtype == INDEXEDA_IMAGE && transparent && data[j] == 0)
		data[k - 1] = transparent;

	      /* Increment J to the next pixel. */
	      j += difference;
	    }
	}

      if (pelbytes >= 3)
	{
	  /* Rearrange the colors from RGB to BGR. */
	  int tmp;
	  for (j = 0; j < bsize; j += pelbytes)
	    {
	      tmp = data[j];
	      data[j] = data[j + 2];
	      data[j + 2] = tmp;
	    }
	}

      if ((*myfwrite) (data, pelbytes, npels, fp) != npels)
	return FALSE;

      gimp_progress_update ((double) (i + tileheight) / (double) height);
    }

  gimp_drawable_detach(drawable);
  g_free (data);

  fclose(fp);
  return TRUE;
}


static gint
save_dialog ()
{
  GtkWidget *dlg;
  GtkWidget *button;
  GtkWidget *toggle;
  GtkWidget *frame;
  GtkWidget *vbox;
  gchar **argv;
  gint argc;

  argc = 1;
  argv = g_new (gchar *, 1);
  argv[0] = g_strdup ("save");

  gtk_init (&argc, &argv);
  gtk_rc_parse (gimp_gtkrc ());

  dlg = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dlg), "Save as Tga");
  gtk_window_position (GTK_WINDOW (dlg), GTK_WIN_POS_MOUSE);
  gtk_signal_connect (GTK_OBJECT (dlg), "destroy",
		      (GtkSignalFunc) save_close_callback,
		      NULL);

  /*  Action area  */
  button = gtk_button_new_with_label ("OK");
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      (GtkSignalFunc) save_ok_callback,
                      dlg);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->action_area), button, TRUE, TRUE, 0);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  button = gtk_button_new_with_label ("Cancel");
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     (GtkSignalFunc) gtk_widget_destroy,
			     GTK_OBJECT (dlg));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->action_area), button, TRUE, TRUE, 0);
  gtk_widget_show (button);

  /* regular tga parameter settings */
  frame = gtk_frame_new ("Targa Options");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_border_width (GTK_CONTAINER (frame), 10);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), frame, TRUE, TRUE, 0);
  vbox = gtk_vbox_new (FALSE, 5);
  gtk_container_border_width (GTK_CONTAINER (vbox), 5);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  /*  rle  */
  toggle = gtk_check_button_new_with_label ("RLE compression");
  gtk_box_pack_start (GTK_BOX (vbox), toggle, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		      (GtkSignalFunc) save_toggle_update,
		      &tsvals.rle);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle), tsvals.rle);
  gtk_widget_show (toggle);

  gtk_widget_show (vbox);
  gtk_widget_show (frame);

  gtk_widget_show (dlg);

  gtk_main ();
  gdk_flush ();

  return tsint.run;
}

/*  Save interface functions  */

static void
save_close_callback (GtkWidget *widget,
		     gpointer   data)
{
  gtk_main_quit ();
}

static void
save_ok_callback (GtkWidget *widget,
		  gpointer   data)
{
  tsint.run = TRUE;
  gtk_widget_destroy (GTK_WIDGET (data));
}

static void
save_toggle_update (GtkWidget *widget,
		    gpointer   data)
{
  int *toggle_val;

  toggle_val = (int *) data;

  if (GTK_TOGGLE_BUTTON (widget)->active)
    *toggle_val = TRUE;
  else
    *toggle_val = FALSE;
}

/* The End */

/*
Local Variables:
compile-command:"gcc -Wall -g -O -o tga tga.c -lgimp -lgtk -lgdk -lglib -lm"
End:
*/
