(define-module (gimple)
  :use-module (gtk dynlink)
  :use-module (gtk gdk)
  :use-module (gtk gtk))
(merge-compiled-code "gimple_init" "libgimple")

; we want the shorter :kw syntax for keywords
(read-set! keywords 'prefix)

(define (make-gparam type value) (cons (gparam-type type) value))
(define (gparam:type gp) (gparam-type (car gp)))
(define gparam-value cdr)

; debug thingy


; this allows easy debugging outputs.. just add a comma.
(defmacro unquote (x)
  (define tmp (gensym))
  `(with-output-to-port (current-error-port)
     (lambda ()
       (define ,tmp ,x) 
       (display "Expression: ")
       (write ',x)
       (display "\nValue: ")
       (write ,tmp)
       (newline)
       ,tmp)))
;;;
;;; Conversion
;;;
,'start!
(define (arg-defs-to-gparamdefs argdefs)
  (map (lambda (argdef)
	 (list (gparam-type (arg-gparam-type (arg-def-type argdef)))
	       (arg-def-name argdef)
	       (arg-def-description argdef)))
	 argdefs))

(define (gparam-to-arg gparam argdef)
  (define argtype (arg-def-type argdef))
  (if (eq? (arg-gparam-type argtype) (gparam:type gparam))
      ((arg-input-converter argtype) (gparam-value gparam))
      (throw 'pdb-calling-error)))
  
(define (gparams-to-args gparams args)
  (map gparam-to-arg gparams args))

;;;
;;; Misc utils
;;;

(define (convert_ str)
  (let ((i (string-index str #\_)))
    (if i (begin
	    (string-set! str i #\-)
	    (convert_ str))
	  str)))


;; enum handling

(define (make-enum vec)
  (lambda (val)
    (cond ((integer? val) (vector-ref vec val))
          ((symbol? val)
           (vector-index vec val))
          (else #f))))

(define (vector-index vec x)
  (define len (vector-length vec))
  (let loop ((i 0))
    (cond ((= i len) #f)
          ((eq? (vector-ref vec i) x) i)
          (else (loop (+ i 1))))))


(define grun-mode (make-enum '#(interactive noninteractive with-last-vals)))

(define glayer-mode (make-enum 
                     '#(normal dissolve #f multiply screen
                              overlay difference addition subtract
                              darken-only lighten-only hue saturation
                              color value)))

(define gparam-type (make-enum '#(int32 int16 int8 float string int32array
                                  int16array int8array floatarray stringarray
                                 color
                                 region
                                 display
                                 image
                                 layer
                                 channel
                                 drawable
                                 selection
                                 boundary
                                 path
                                 status)))

(define gimage (make-enum '#(rgb gray indexed)))

(define gdrawable (make-enum '#(rgb rgba gray graya indexed indexeda)))

(define gstatus (make-enum
                 '#(execution-error calling-error pass-through success)))

(define gprocedure-type (make-enum '#(#f plug-in extension temporary)))

;;;
;;; PDB
;;;

; run a pdb proc, verify it was successful and remove status from return values
(define (run-pdb-assert procname . params)
  (define r (apply gimple-run-pdb procname params))
  (case (gstatus (gparam-value (car r)))
    ((success) (cdr r))
    ((calling-error) (throw 'pdb-calling-error procname params))
    ((execution-error) (throw 'pdb-execution-error procname params))
    ; this shouldn't occur in practice..
    ((pass-through) (throw 'pdb-pass-through procname params))))

(define (proc-param-types procname)
  (define proc (make-gparam 'string procname))
  (let loop ((idx (gparam-value (list-ref (run-pdb-assert
                                           "gimp_procedural_db_proc_info"
                                           proc)
                                          6)))
             (accum '()))
    (if (zero? idx)
        (map (lambda (x)
               (gparam-type (gparam-value (car x))))
             accum)
        (loop (- idx 1)
              (cons (run-pdb-assert "gimp_procedural_db_proc_arg"
                                     proc
                                     (make-gparam 'int32 (- idx 1)))
                    accum)))))

(define (add-procedure procname)
  (define paramtypes (proc-param-types procname))
  (define (wrapper . args)
    (if (not (= (length args) (length paramtypes)))
	(error procname "wrong number of args"))
    (define pdbargs
      ,(map (lambda (paramtype arg)
	     (make-gparam paramtype
                          arg))
	   paramtypes
	   args))
    (define pdbretvals ,(apply run-pdb-assert procname pdbargs))
    (map gparam-value pdbretvals))
  (define gname (symbol (convert_ (string-copy procname))))
  (intern-symbol #f gname)
  (symbol-set! #f gname wrapper))

(define (create-pdb-bindings)
  (define x (make-gparam 'string ".*"))
  (define procnames (vector->list (gparam-value (cadr (run-pdb-assert "gimp_procedural_db_query" x x x x x x x)))))
  (for-each add-procedure procnames))





;;;
;;; The default interface
;;;

(define (standard-interface procname argdefs)
  ,'standard-interface
  (define title (string-append "Gimple: " procname))
  (define dlg (gtk-dialog-new))
  (define retvals #f)
  (gtk-window-set-title dlg title)
  (gtk-signal-connect dlg "destroy" gtk-main-quit)
  (define button (gtk-button-new-with-label "OK"))
  (gtk-signal-connect button "clicked" 
		      (lambda ()
			(set! retvals (map (lambda (querier widget)
                                             (,querier ,widget))
                                           queriers
                                           input-widgets))
			(gtk-main-quit)))
  (gtk-box-pack-start (gtk-dialog-action-area dlg) button)
  (gtk-widget-show button)
  (define button (gtk-button-new-with-label "Cancel"))
  (gtk-signal-connect button "clicked" gtk-main-quit)
  (gtk-box-pack-start (gtk-dialog-action-area dlg)  button)
  (gtk-widget-show button)
  (define frame (gtk-frame-new "Script Arguments"))
  (gtk-box-pack-start (gtk-dialog-vbox dlg) frame)
  (gtk-widget-show frame)
  (define table (gtk-table-new (length argdefs) 2 #f))
  (gtk-container-add frame table)
  (gtk-widget-show table)
  (define input-widgets
    (map (lambda (argdef)
           ((arg-widget-maker (arg-def-type argdef))
            (arg-def-default argdef)))
	 argdefs))
  ,'input-widgets-made
  (define descs (map arg-def-description argdefs))
  (define queriers (map (lambda (argdef)
                          (arg-widget-querier (arg-def-type argdef)))
                        argdefs))
  
  (let ((row 0))
    (for-each (lambda (desc widget)
		(define label (gtk-label-new desc))
		(gtk-table-attach table label 0 1 row (+ row 1) '() '())
		(gtk-widget-show label)
		(gtk-table-attach table widget 1 2 row (+ row 1))
		(gtk-widget-show-all widget)
		(set! row (+ row 1)))
	      descs
	      input-widgets))
  (gtk-widget-show dlg)
  ,'entering-gtk-main
  (gtk-main)
  (gtk-widget-destroy dlg)
  (gdk-flush)
  retvals)

,'arg-definitions

(define arg-type
  (make-record-type "GimpleArgType" '(widget-maker
                                      widget-querier
                                      gparam-type
                                      input-converter
                                      output-converter
                                      validator)))



(define make-arg-type (record-constructor arg-type))
(define arg-widget-maker
  (record-accessor arg-type 'widget-maker))
(define arg-widget-querier
  (record-accessor arg-type 'widget-querier))
(define arg-gparam-type
  (record-accessor arg-type 'gparam-type))
(define arg-input-converter
  (record-accessor arg-type 'input-converter))
(define arg-output-converter
  (record-accessor arg-type 'output-converter))
(define arg-validator
  (record-accessor arg-type 'validator))

(define arg-def
  (make-record-type "GimpleArgDef" '(type
                                     name
                                     description
                                     default)))
(define make-arg-def
  (record-constructor arg-def))
(define arg-def-type
  (record-accessor arg-def 'type))
(define arg-def-name
  (record-accessor arg-def 'name))
(define arg-def-description
  (record-accessor arg-def 'description))
(define arg-def-default
  (record-accessor arg-def 'default))







;;;
;;; Argtype definitions
;;;

; A string type

(define string-arg
  (make-arg-type
   (lambda (default)
     (let ((entry (gtk-entry-new)))
       (gtk-entry-set-text entry default)
       entry))
   gtk-entry-get-text
   'string
   id
   id
   string?))

(define boolean-arg
  (make-arg-type
   (lambda (default)
     (let ((toggle (gtk-check-button-new)))
       (gtk-toggle-button-set-state toggle default)
       toggle))
   gtk-toggle-button-active
   'int32
   (lambda (x)
     (not (zero? x)))
   (lambda (x)
     (if x 1 0))
   boolean?))


(define arg-integer
  (make-arg-type
   #f
   #f
   'int32
   id
   id
   #f))


(define arg-image 
  (make-arg-type
   #f
   #f
   'image
   id
   id
   #f))

(define arg-drawable
  (make-arg-type
   #f
   #f
   'drawable
   id
   id
   #f))

(define arg-run-type
  (make-arg-type
   #f
   #f
   'int32
   grun-mode
   grun-mode
   #f))


(define (make-enum-arg . values)
  ,'make-enum-arg
  (define valist (do ((vals values (cddr vals))
                      (al '() (cons (cons (car vals) (cadr vals)) al)))
                     ((null? vals) (reverse al))))
  (define len (length ,valist))
  (make-arg-type
   (lambda (default)
     ,'making
     (define box (gtk-vbox-new #t 0))
     (let loop
         ((options valist)
          (prev #f))
       (if (null? options)
           box
           (let ((button
                  (gtk-radio-button-new-with-label prev (cdar options))))
             (gtk-signal-connect button "clicked" (lambda ()
                                                    (set-object-property!
                                                     box
                                                     'gimple-value
                                                     (caar options))))
             (gtk-toggle-button-set-state button 
                                          (eq? (caar options) default))
             (gtk-box-pack-start box button)
             (loop (cdr options) button)))))
   (lambda (box)
            (object-property box 'gimple-value))
   'int32
   (lambda (i) (car (list-ref valist i)))
   #f
   #f))



(define null-syntax (procedure->syntax (lambda (exp env) *unspecified*)))

(define end-decl #f)
(define define-proc #f)

,'query-def

(define (query-define-proc procname
                           blurb
                           description
                           help
                           author
                           copyright
                           date
                           menupath
                           imgtypes
                           type
                           retvaldefs
                           paramdefs
                           func)
  (gimple-install-proc (gprocedure-type type)
                       ,(arg-defs-to-gparamdefs
                         (cons run-type-def
                               (real-param-defs paramdefs imgtypes)))
                       ,(arg-defs-to-gparamdefs retvaldefs)
                       procname
                       blurb
                       help
                       author
                       copyright
                       date

                       menupath
                       imgtypes))

(define (query)
  (call-with-current-continuation
   (lambda (c)
     (set! define-proc query-define-proc)
     (load app-script))))







(define (call-proc proc retval-types args)
  (define calling-error (gensym))
  (catch
   #t
   (lambda ()
     (define gimple-return-values (apply ,proc ,args))
     (define gparam-return-values
       (map (lambda (retval argtype)
              ((arg-output-converter argtype) retval))
            gimple-return-values
            retval-types))
     (cons (cons (gparam-type 'status) (gstatus 'success)) gparam-return-values))
   (lambda (key . args)
     (with-output-to-port (current-error-port)
       (lambda ()
         (display "Uncaught exception: ")
         (display key)
         (write args)
         (newline)))
     (list (cons (gparam-type 'status)
                 (gstatus (if (eq? key calling-error)
                              'calling-error
                              'execution-error)))))))

; encodes an expression to a vector of ints

(define (encode exp)
  (list->vector
   (map char->integer
        (string->list
         (call-with-output-string
           (lambda (p)
             (write exp p)))))))

(define (decode cod)
  (call-with-input-string
   (list->string
    (map integer->char
         (vector->list cod)))
  read))

(define run-type-def
  (make-arg-def
   arg-run-type
   "run-type"
   "interactive, non-interactive perhaps?"
   #f))

(define img-arg-defs
  (list
   (make-arg-def
    arg-image
    "image"
    "The image"
    #f)
   (make-arg-def
    arg-drawable
    "drawable"
    "The drawable"
    #f)))


(defmacro defproc


(define (real-param-defs paramdefs imgtypes)
  (append (if imgtypes img-arg-defs '())
          paramdefs))

(define (run run-proc run-type-param . params)
  (call-with-current-continuation
   (lambda (pdb-return)
     (define run-type (gparam-to-arg ,run-type-param ,run-type-def))
     (define (run-define-proc procname	;string
                              blurb	;string
                              description	;string
                              help		;string
                              author	;string
                              copyright	;string
                              date
                              menupath
                              imgtypes
                              type		;gprocedure-type
                              retvaldefs	;list of arg-def
                              paramdefs	;list of arg-def
                              ;or #f for interactive-only
                              func)	;procedure
       (if (string=? procname run-proc)
           (let ()
             (define real-paramdefs (real-param-defs paramdefs imgtypes))
             (define args
               (append
                (if imgtypes
                    (gparams-to-args (list-head params 2) img-arg-defs)
                    '())
                (case run-type
                  ((interactive) (if (not (null? paramdefs))
                                     (standard-interface procname paramdefs)
                                      '()))
                  ((noninteractive) (gparams-to-args (list-tail params 2)
                                                     paramdefs))
                  (else (error "with-last-args not yet handled")))))
             (define retval-types
               (map arg-def-type retvaldefs))
             (pdb-return
              ,(call-proc func retval-types args)))))
     (set! define-proc run-define-proc)
     ,'mark1
     (create-pdb-bindings)
     ,'mark2
     (load app-script))))


  
(define app-script (car (command-line)))


,'entering-gimple-main
(gimple-main (command-line)
             query
             run
             #f)

