/*
 * Copyright (C) 1995-1999 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <gtk/gtk.h>

#include "libgimpbase/gimpbase.h"

#include "tools-types.h"

#include "config/gimpconfig.h"

#include "core/gimptoolinfo.h"

#include "tool_options.h"

#include "gimp-intl.h"


enum
{
  PROP_0,
  PROP_TOOL_INFO
};


static void   gimp_tool_options_init       (GimpToolOptions      *options);
static void   gimp_tool_options_class_init (GimpToolOptionsClass *options_class);

static void   gimp_tool_options_set_property (GObject         *object,
                                              guint            property_id,
                                              const GValue    *value,
                                              GParamSpec      *pspec);
static void   gimp_tool_options_get_property (GObject         *object,
                                              guint            property_id,
                                              GValue          *value,
                                              GParamSpec      *pspec);

static void   gimp_tool_options_real_reset   (GimpToolOptions *tool_options);


static GimpContextClass *parent_class = NULL;


GType
gimp_tool_options_get_type (void)
{
  static GType type = 0;

  if (! type)
    {
      static const GTypeInfo info =
      {
        sizeof (GimpToolOptionsClass),
	(GBaseInitFunc) NULL,
	(GBaseFinalizeFunc) NULL,
	(GClassInitFunc) gimp_tool_options_class_init,
	NULL,           /* class_finalize */
	NULL,           /* class_data     */
	sizeof (GimpToolOptions),
	0,              /* n_preallocs    */
	(GInstanceInitFunc) gimp_tool_options_init,
      };

      type = g_type_register_static (GIMP_TYPE_CONTEXT,
                                     "GimpToolOptions",
                                     &info, 0);
    }

  return type;
}

static void 
gimp_tool_options_class_init (GimpToolOptionsClass *klass)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);

  object_class->set_property = gimp_tool_options_set_property;
  object_class->get_property = gimp_tool_options_get_property;

  klass->reset               = gimp_tool_options_real_reset;

  g_object_class_install_property (object_class, PROP_TOOL_INFO,
                                   g_param_spec_object ("tool-info",
                                                        NULL, NULL,
                                                        GIMP_TYPE_TOOL_INFO,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT_ONLY));

}

static void
gimp_tool_options_init (GimpToolOptions *options)
{
  options->tool_info = NULL;
}

static void
gimp_tool_options_set_property (GObject      *object,
                                guint         property_id,
                                const GValue *value,
                                GParamSpec   *pspec)
{
  GimpToolOptions *options;

  options = GIMP_TOOL_OPTIONS (object);

  switch (property_id)
    {
    case PROP_TOOL_INFO:
      options->tool_info = g_value_get_object (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}

static void
gimp_tool_options_get_property (GObject    *object,
                                guint       property_id,
                                GValue     *value,
                                GParamSpec *pspec)
{
  GimpToolOptions *options;

  options = GIMP_TOOL_OPTIONS (object);

  switch (property_id)
    {
    case PROP_TOOL_INFO:
      g_value_set_object (value, options->tool_info);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}

static void
gimp_tool_options_real_reset (GimpToolOptions *tool_options)
{
  gimp_config_reset (G_OBJECT (tool_options));
}

void
gimp_tool_options_reset (GimpToolOptions *tool_options)
{
  g_return_if_fail (GIMP_IS_TOOL_OPTIONS (tool_options));

  GIMP_TOOL_OPTIONS_GET_CLASS (tool_options)->reset (tool_options);
}

static gchar *
gimp_tool_options_build_filename (GimpToolOptions *tool_options,
                                  const gchar     *extension)
{
  gchar *basename;
  gchar *filename;

  if (extension)
    {
      basename = g_strconcat (GIMP_OBJECT (tool_options->tool_info)->name,
                              ".", extension, NULL);

      filename = g_build_filename (gimp_directory (),
                                   "tool-options",
                                   basename,
                                   NULL);
      g_free (basename);
    }
  else
    {
      filename = g_build_filename (gimp_directory (),
                                   "tool-options",
                                   GIMP_OBJECT (tool_options->tool_info)->name,
                                   NULL);      
    }

  return filename;
}

gboolean
gimp_tool_options_serialize (GimpToolOptions  *tool_options,
                             const gchar      *extension,
                             GError          **error)
{
  gchar    *filename;
  gboolean  retval;

  g_return_val_if_fail (GIMP_IS_TOOL_OPTIONS (tool_options), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  filename = gimp_tool_options_build_filename (tool_options, extension);

  retval = gimp_config_serialize (G_OBJECT (tool_options),
                                  filename,
                                  "GIMP tool options",
                                  "end of tool options",
                                  NULL,
                                  error);

  g_free (filename);

  return retval;
}

gboolean
gimp_tool_options_deserialize (GimpToolOptions  *tool_options,
                               const gchar      *extension,
                               GError          **error)
{
  gchar    *filename;
  gboolean  retval;

  g_return_val_if_fail (GIMP_IS_TOOL_OPTIONS (tool_options), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  filename = gimp_tool_options_build_filename (tool_options, extension);

  retval = gimp_config_deserialize (G_OBJECT (tool_options),
                                    filename,
                                    NULL,
                                    error);

  g_free (filename);

  return retval;
}

gpointer
gimp_tool_options_gui (GimpToolOptions *tool_options)
{
  GtkWidget *vbox;

  vbox = gtk_vbox_new (FALSE, 2);

  return vbox;
}
