/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <gtk/gtk.h>

#include "libgimpwidgets/gimpwidgets.h"

#include "tools-types.h"

#include "base/gimphistogram.h"
#include "base/threshold.h"

#include "config/gimpbaseconfig.h"

#include "core/gimp.h"
#include "core/gimpdrawable.h"
#include "core/gimpdrawable-histogram.h"
#include "core/gimpimage.h"
#include "core/gimpimagemap.h"
#include "core/gimptoolinfo.h"

#include "widgets/gimphistogrambox.h"
#include "widgets/gimphistogramview.h"

#include "display/gimpdisplay.h"

#include "gimpthresholdtool.h"
#include "tool_manager.h"

#include "gimp-intl.h"


#define LOW        0x1
#define HIGH       0x2
#define HISTOGRAM  0x4
#define ALL       (LOW | HIGH | HISTOGRAM)


/*  local function prototypes  */

static void   gimp_threshold_tool_class_init (GimpThresholdToolClass *klass);
static void   gimp_threshold_tool_init       (GimpThresholdTool      *threshold_tool);

static void   gimp_threshold_tool_finalize   (GObject          *object);

static void   gimp_threshold_tool_initialize (GimpTool         *tool,
					      GimpDisplay      *gdisp);

static void   gimp_threshold_tool_map        (GimpImageMapTool *image_map_tool);
static void   gimp_threshold_tool_dialog     (GimpImageMapTool *image_map_tool);
static void   gimp_threshold_tool_reset      (GimpImageMapTool *image_map_tool);

static void   gimp_threshold_tool_histogram_range (GimpHistogramView *view,
                                                   gint               start,
                                                   gint               end,
                                                   GimpThresholdTool *t_tool);


static GimpImageMapToolClass *parent_class = NULL;


/*  public functions  */

void
gimp_threshold_tool_register (GimpToolRegisterCallback  callback,
                              gpointer                  data)
{
  (* callback) (GIMP_TYPE_THRESHOLD_TOOL,
                G_TYPE_NONE, NULL,
                FALSE,
                "gimp-threshold-tool",
                _("Threshold"),
                _("Reduce image to two colors using a threshold"),
                N_("/Layer/Colors/Threshold..."), NULL,
                NULL, "tools/threshold.html",
                GIMP_STOCK_TOOL_THRESHOLD,
                data);
}

GType
gimp_threshold_tool_get_type (void)
{
  static GType tool_type = 0;

  if (! tool_type)
    {
      static const GTypeInfo tool_info =
      {
        sizeof (GimpThresholdToolClass),
	(GBaseInitFunc) NULL,
	(GBaseFinalizeFunc) NULL,
	(GClassInitFunc) gimp_threshold_tool_class_init,
	NULL,           /* class_finalize */
	NULL,           /* class_data     */
	sizeof (GimpThresholdTool),
	0,              /* n_preallocs    */
	(GInstanceInitFunc) gimp_threshold_tool_init,
      };

      tool_type = g_type_register_static (GIMP_TYPE_IMAGE_MAP_TOOL,
					  "GimpThresholdTool", 
                                          &tool_info, 0);
    }

  return tool_type;
}


/*  private functions  */

static void
gimp_threshold_tool_class_init (GimpThresholdToolClass *klass)
{
  GObjectClass          *object_class;
  GimpToolClass         *tool_class;
  GimpImageMapToolClass *image_map_tool_class;

  object_class         = G_OBJECT_CLASS (klass);
  tool_class           = GIMP_TOOL_CLASS (klass);
  image_map_tool_class = GIMP_IMAGE_MAP_TOOL_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);

  object_class->finalize       = gimp_threshold_tool_finalize;

  tool_class->initialize       = gimp_threshold_tool_initialize;

  image_map_tool_class->map    = gimp_threshold_tool_map;
  image_map_tool_class->dialog = gimp_threshold_tool_dialog;
  image_map_tool_class->reset  = gimp_threshold_tool_reset;
}

static void
gimp_threshold_tool_init (GimpThresholdTool *t_tool)
{
  GimpImageMapTool *image_map_tool;

  image_map_tool = GIMP_IMAGE_MAP_TOOL (t_tool);

  image_map_tool->shell_desc  = _("Apply Threshold");

  t_tool->threshold = g_new0 (Threshold, 1);
  t_tool->hist      = NULL;

  t_tool->threshold->low_threshold  = 127;
  t_tool->threshold->high_threshold = 255;
}

static void
gimp_threshold_tool_finalize (GObject *object)
{
  GimpThresholdTool *t_tool;

  t_tool = GIMP_THRESHOLD_TOOL (object);

  if (t_tool->threshold)
    {
      g_free (t_tool->threshold);
      t_tool->threshold = NULL;
    }

  if (t_tool->hist)
    {
      gimp_histogram_free (t_tool->hist);
      t_tool->hist = NULL;
    }

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gimp_threshold_tool_initialize (GimpTool    *tool,
				GimpDisplay *gdisp)
{
  GimpThresholdTool *t_tool;
  GimpDrawable      *drawable;

  t_tool = GIMP_THRESHOLD_TOOL (tool);

  drawable = gimp_image_active_drawable (gdisp->gimage);

  if (gimp_drawable_is_indexed (drawable))
    {
      g_message (_("Threshold does not operate on indexed drawables."));
      return;
    }

  if (!t_tool->hist)
    {
      Gimp *gimp = GIMP_TOOL (t_tool)->tool_info->gimp;

      t_tool->hist = gimp_histogram_new (GIMP_BASE_CONFIG (gimp->config));
    }

  t_tool->threshold->color          = gimp_drawable_is_rgb (drawable);
  t_tool->threshold->low_threshold  = 127;
  t_tool->threshold->high_threshold = 255;

  GIMP_TOOL_CLASS (parent_class)->initialize (tool, gdisp);

  gimp_drawable_calculate_histogram (drawable, t_tool->hist);

  g_signal_handlers_block_by_func (t_tool->histogram_box->histogram,
                                   gimp_threshold_tool_histogram_range,
                                   t_tool);
  gimp_histogram_view_set_histogram (t_tool->histogram_box->histogram,
                                     t_tool->hist);
  gimp_histogram_view_set_range (t_tool->histogram_box->histogram,
                                 t_tool->threshold->low_threshold,
                                 t_tool->threshold->high_threshold);
  g_signal_handlers_unblock_by_func (t_tool->histogram_box->histogram,
                                     gimp_threshold_tool_histogram_range,
                                     t_tool);

  gimp_image_map_tool_preview (GIMP_IMAGE_MAP_TOOL (t_tool));
}

static void
gimp_threshold_tool_map (GimpImageMapTool *image_map_tool)
{
  GimpThresholdTool *t_tool;

  t_tool = GIMP_THRESHOLD_TOOL (image_map_tool);

  gimp_image_map_apply (image_map_tool->image_map,
                        threshold,
                        t_tool->threshold);
}


/**********************/
/*  Threshold dialog  */
/**********************/

static void
gimp_threshold_tool_dialog (GimpImageMapTool *image_map_tool)
{
  GimpThresholdTool *t_tool;
  GtkWidget         *box;

  t_tool = GIMP_THRESHOLD_TOOL (image_map_tool);

  box = gimp_histogram_box_new (_("Threshold Range:"));
  gtk_container_add (GTK_CONTAINER (image_map_tool->main_vbox), box);
  gtk_widget_show (box);

  t_tool->histogram_box = GIMP_HISTOGRAM_BOX (box);

  g_signal_connect (t_tool->histogram_box->histogram, "range_changed",
                    G_CALLBACK (gimp_threshold_tool_histogram_range),
                    t_tool);
}

static void
gimp_threshold_tool_reset (GimpImageMapTool *image_map_tool)
{
  GimpThresholdTool *t_tool;

  t_tool = GIMP_THRESHOLD_TOOL (image_map_tool);

  t_tool->threshold->low_threshold  = 127.0;
  t_tool->threshold->high_threshold = 255.0;

  gimp_histogram_view_set_range (t_tool->histogram_box->histogram,
                                 t_tool->threshold->low_threshold,
                                 t_tool->threshold->high_threshold);
}

static void
gimp_threshold_tool_histogram_range (GimpHistogramView *widget,
                                     gint               start,
                                     gint               end,
                                     GimpThresholdTool *t_tool)
{
  if (start != t_tool->threshold->low_threshold ||
      end   != t_tool->threshold->high_threshold)
    {
      t_tool->threshold->low_threshold  = start;
      t_tool->threshold->high_threshold = end;

      gimp_image_map_tool_preview (GIMP_IMAGE_MAP_TOOL (t_tool));
    }
}
