/* The GIMP -- an image manipulation program
 * Copyright (C) 1995-2000 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* NOTE: This file is autogenerated by pdbgen.pl. */

#include "config.h"


#include <glib-object.h>

#include "libgimpbase/gimpbasetypes.h"

#include "pdb-types.h"
#include "procedural_db.h"

#include "base/color-balance.h"
#include "base/curves.h"
#include "base/gimphistogram.h"
#include "base/gimplut.h"
#include "base/hue-saturation.h"
#include "base/levels.h"
#include "base/lut-funcs.h"
#include "base/pixel-processor.h"
#include "base/pixel-region.h"
#include "base/threshold.h"
#include "config/gimpbaseconfig.h"
#include "core/gimp.h"
#include "core/gimpdrawable-desaturate.h"
#include "core/gimpdrawable-equalize.h"
#include "core/gimpdrawable-invert.h"
#include "core/gimpdrawable.h"
#include "core/gimpimage.h"
#include "gimp-intl.h"

static ProcRecord brightness_contrast_proc;
static ProcRecord levels_proc;
static ProcRecord posterize_proc;
static ProcRecord desaturate_proc;
static ProcRecord equalize_proc;
static ProcRecord invert_proc;
static ProcRecord curves_spline_proc;
static ProcRecord curves_explicit_proc;
static ProcRecord color_balance_proc;
static ProcRecord histogram_proc;
static ProcRecord hue_saturation_proc;
static ProcRecord threshold_proc;

void
register_color_procs (Gimp *gimp)
{
  procedural_db_register (gimp, &brightness_contrast_proc);
  procedural_db_register (gimp, &levels_proc);
  procedural_db_register (gimp, &posterize_proc);
  procedural_db_register (gimp, &desaturate_proc);
  procedural_db_register (gimp, &equalize_proc);
  procedural_db_register (gimp, &invert_proc);
  procedural_db_register (gimp, &curves_spline_proc);
  procedural_db_register (gimp, &curves_explicit_proc);
  procedural_db_register (gimp, &color_balance_proc);
  procedural_db_register (gimp, &histogram_proc);
  procedural_db_register (gimp, &hue_saturation_proc);
  procedural_db_register (gimp, &threshold_proc);
}

static Argument *
brightness_contrast_invoker (Gimp     *gimp,
                             Argument *args)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gint32 brightness;
  gint32 contrast;
  GimpImage *gimage;
  GimpLut *lut;
  PixelRegion srcPR, destPR;
  int x1, y1, x2, y2;

  drawable = (GimpDrawable *) gimp_item_get_by_ID (gimp, args[0].value.pdb_int);
  if (! GIMP_IS_DRAWABLE (drawable))
    success = FALSE;

  brightness = args[1].value.pdb_int;
  if (brightness < -127 || brightness > 127)
    success = FALSE;

  contrast = args[2].value.pdb_int;
  if (contrast < -127 || contrast > 127)
    success = FALSE;

  if (success)
    {
      if (gimp_drawable_is_indexed (drawable))
	success = FALSE;
      else
	{
	  gimage = gimp_item_get_image (GIMP_ITEM (drawable));
    
	  lut = brightness_contrast_lut_new (brightness / 255.0,
					     contrast / 127.0,
					     gimp_drawable_bytes (drawable));
    
	  /* The application should occur only within selection bounds */
	  gimp_drawable_mask_bounds (drawable, &x1, &y1, &x2, &y2);
    
	  pixel_region_init (&srcPR, gimp_drawable_data (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), FALSE);
	  pixel_region_init (&destPR, gimp_drawable_shadow (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), TRUE);
    
	  pixel_regions_process_parallel ((p_func) gimp_lut_process, lut, 2,
					  &srcPR, &destPR);
    
	  gimp_lut_free (lut);
	  gimp_drawable_merge_shadow (drawable, TRUE, _("Brightness-Contrast"));
	  gimp_drawable_update (drawable, x1, y1, (x2 - x1), (y2 - y1));
	}
    }

  return procedural_db_return_args (&brightness_contrast_proc, success);
}

static ProcArg brightness_contrast_inargs[] =
{
  {
    GIMP_PDB_DRAWABLE,
    "drawable",
    "The drawable"
  },
  {
    GIMP_PDB_INT32,
    "brightness",
    "Brightness adjustment: (-127 <= brightness <= 127)"
  },
  {
    GIMP_PDB_INT32,
    "contrast",
    "Contrast adjustment: (-127 <= contrast <= 127)"
  }
};

static ProcRecord brightness_contrast_proc =
{
  "gimp_brightness_contrast",
  "Modify brightness/contrast in the specified drawable.",
  "This procedures allows the brightness and contrast of the specified drawable to be modified. Both 'brightness' and 'contrast' parameters are defined between -127 and 127.",
  "Spencer Kimball & Peter Mattis",
  "Spencer Kimball & Peter Mattis",
  "1997",
  GIMP_INTERNAL,
  3,
  brightness_contrast_inargs,
  0,
  NULL,
  { { brightness_contrast_invoker } }
};

static Argument *
levels_invoker (Gimp     *gimp,
                Argument *args)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gint32 channel;
  gint32 low_input;
  gint32 high_input;
  gdouble gamma;
  gint32 low_output;
  gint32 high_output;
  PixelRegion srcPR, destPR;
  Levels l;
  gint x1, y1, x2, y2;
  GimpLut *lut;

  drawable = (GimpDrawable *) gimp_item_get_by_ID (gimp, args[0].value.pdb_int);
  if (! GIMP_IS_DRAWABLE (drawable))
    success = FALSE;

  channel = args[1].value.pdb_int;
  if (channel < GIMP_VALUE_LUT || channel > GIMP_ALPHA_LUT)
    success = FALSE;

  low_input = args[2].value.pdb_int;
  if (low_input < 0 || low_input > 255)
    success = FALSE;

  high_input = args[3].value.pdb_int;
  if (high_input < 0 || high_input > 255)
    success = FALSE;

  gamma = args[4].value.pdb_float;
  if (gamma < 0.1 || gamma > 10.0)
    success = FALSE;

  low_output = args[5].value.pdb_int;
  if (low_output < 0 || low_output > 255)
    success = FALSE;

  high_output = args[6].value.pdb_int;
  if (high_output < 0 || high_output > 255)
    success = FALSE;

  if (success)
    {
      if (gimp_drawable_is_indexed (drawable) ||
	  (!gimp_drawable_has_alpha (drawable) && channel == GIMP_ALPHA_LUT) ||
	  (gimp_drawable_is_gray (drawable) && channel != GIMP_GRAY_LUT
					    && channel != GIMP_ALPHA_LUT))
	success = FALSE;
      else
	{
	  lut = gimp_lut_new ();
    
	  levels_init (&l);
    
	  l.low_input[channel]   = low_input;
	  l.high_input[channel]  = high_input;
	  l.gamma[channel]       = gamma;
	  l.low_output[channel]  = low_output;
	  l.high_output[channel] = high_output;
    
	  /* setup the lut */
	  gimp_lut_setup (lut,
			  (GimpLutFunc) levels_lut_func,
			  &l,
			  gimp_drawable_bytes (drawable));
    
	  /* The application should occur only within selection bounds */
	  gimp_drawable_mask_bounds (drawable, &x1, &y1, &x2, &y2);
    
	  pixel_region_init (&srcPR, gimp_drawable_data (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), FALSE);
	  pixel_region_init (&destPR, gimp_drawable_shadow (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), TRUE);
    
	  pixel_regions_process_parallel ((p_func) gimp_lut_process, lut, 2,
					  &srcPR, &destPR);
    
	  gimp_lut_free (lut);
	  gimp_drawable_merge_shadow (drawable, TRUE, _("Levels"));
	  gimp_drawable_update (drawable, x1, y1, (x2 - x1), (y2 - y1));
	}
    }

  return procedural_db_return_args (&levels_proc, success);
}

static ProcArg levels_inargs[] =
{
  {
    GIMP_PDB_DRAWABLE,
    "drawable",
    "The drawable"
  },
  {
    GIMP_PDB_INT32,
    "channel",
    "The channel to modify: { GIMP_VALUE_LUT (0), GIMP_RED_LUT (1), GIMP_GREEN_LUT (2), GIMP_BLUE_LUT (3), GIMP_ALPHA_LUT (4) }"
  },
  {
    GIMP_PDB_INT32,
    "low_input",
    "Intensity of lowest input: (0 <= low_input <= 255)"
  },
  {
    GIMP_PDB_INT32,
    "high_input",
    "Intensity of highest input: (0 <= high_input <= 255)"
  },
  {
    GIMP_PDB_FLOAT,
    "gamma",
    "Gamma correction factor: (0.1 <= gamma <= 10)"
  },
  {
    GIMP_PDB_INT32,
    "low_output",
    "Intensity of lowest output: (0 <= low_output <= 255)"
  },
  {
    GIMP_PDB_INT32,
    "high_output",
    "Intensity of highest output: (0 <= high_output <= 255)"
  }
};

static ProcRecord levels_proc =
{
  "gimp_levels",
  "Modifies intensity levels in the specified drawable.",
  "This tool allows intensity levels in the specified drawable to be remapped according to a set of parameters. The low/high input levels specify an initial mapping from the source intensities. The gamma value determines how intensities between the low and high input intensities are interpolated. A gamma value of 1.0 results in a linear interpolation. Higher gamma values result in more high-level intensities. Lower gamma values result in more low-level intensities. The low/high output levels constrain the final intensity mapping--that is, no final intensity will be lower than the low output level and no final intensity will be higher than the high output level. This tool is only valid on RGB color and grayscale images. It will not operate on indexed drawables.",
  "Spencer Kimball & Peter Mattis",
  "Spencer Kimball & Peter Mattis",
  "1995-1996",
  GIMP_INTERNAL,
  7,
  levels_inargs,
  0,
  NULL,
  { { levels_invoker } }
};

static Argument *
posterize_invoker (Gimp     *gimp,
                   Argument *args)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gint32 levels;
  GimpImage *gimage;
  GimpLut *lut;
  PixelRegion srcPR, destPR;
  int x1, y1, x2, y2;

  drawable = (GimpDrawable *) gimp_item_get_by_ID (gimp, args[0].value.pdb_int);
  if (! GIMP_IS_DRAWABLE (drawable))
    success = FALSE;

  levels = args[1].value.pdb_int;
  if (levels < 2 || levels > 255)
    success = FALSE;

  if (success)
    {
      if (gimp_drawable_is_indexed (drawable))
	success = FALSE;
      else
	{
	  gimage = gimp_item_get_image (GIMP_ITEM (drawable));
    
	  lut = posterize_lut_new (levels, gimp_drawable_bytes (drawable)); 
    
	  /* The application should occur only within selection bounds */
	  gimp_drawable_mask_bounds (drawable, &x1, &y1, &x2, &y2);
    
	  pixel_region_init (&srcPR, gimp_drawable_data (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), FALSE);
	  pixel_region_init (&destPR, gimp_drawable_shadow (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), TRUE);
    
	  pixel_regions_process_parallel ((p_func) gimp_lut_process, lut, 2,
					  &srcPR, &destPR);
    
	  gimp_lut_free (lut);
	  gimp_drawable_merge_shadow (drawable, TRUE, _("Posterize"));
	  gimp_drawable_update (drawable, x1, y1, (x2 - x1), (y2 - y1));
	}
    }

  return procedural_db_return_args (&posterize_proc, success);
}

static ProcArg posterize_inargs[] =
{
  {
    GIMP_PDB_DRAWABLE,
    "drawable",
    "The drawable"
  },
  {
    GIMP_PDB_INT32,
    "levels",
    "Levels of posterization: (2 <= levels <= 255)"
  }
};

static ProcRecord posterize_proc =
{
  "gimp_posterize",
  "Posterize the specified drawable.",
  "This procedures reduces the number of shades allows in each intensity channel to the specified 'levels' parameter.",
  "Spencer Kimball & Peter Mattis",
  "Spencer Kimball & Peter Mattis",
  "1997",
  GIMP_INTERNAL,
  2,
  posterize_inargs,
  0,
  NULL,
  { { posterize_invoker } }
};

static Argument *
desaturate_invoker (Gimp     *gimp,
                    Argument *args)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;

  drawable = (GimpDrawable *) gimp_item_get_by_ID (gimp, args[0].value.pdb_int);
  if (! GIMP_IS_DRAWABLE (drawable))
    success = FALSE;

  if (success)
    {
      if (gimp_drawable_is_rgb (drawable))
	gimp_drawable_desaturate (drawable);
      else
	success = FALSE;
    }

  return procedural_db_return_args (&desaturate_proc, success);
}

static ProcArg desaturate_inargs[] =
{
  {
    GIMP_PDB_DRAWABLE,
    "drawable",
    "The drawable"
  }
};

static ProcRecord desaturate_proc =
{
  "gimp_desaturate",
  "Desaturate the contents of the specified drawable.",
  "This procedure desaturates the contents of the specified drawable. This procedure only works on drawables of type RGB color.",
  "Spencer Kimball & Peter Mattis",
  "Spencer Kimball & Peter Mattis",
  "1995-1996",
  GIMP_INTERNAL,
  1,
  desaturate_inargs,
  0,
  NULL,
  { { desaturate_invoker } }
};

static Argument *
equalize_invoker (Gimp     *gimp,
                  Argument *args)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gboolean mask_only;

  drawable = (GimpDrawable *) gimp_item_get_by_ID (gimp, args[0].value.pdb_int);
  if (! GIMP_IS_DRAWABLE (drawable))
    success = FALSE;

  mask_only = args[1].value.pdb_int ? TRUE : FALSE;

  if (success)
    {
      if (! gimp_drawable_is_indexed (drawable))
	gimp_drawable_equalize (drawable, mask_only);
      else
	success = FALSE;
    }

  return procedural_db_return_args (&equalize_proc, success);
}

static ProcArg equalize_inargs[] =
{
  {
    GIMP_PDB_DRAWABLE,
    "drawable",
    "The drawable"
  },
  {
    GIMP_PDB_INT32,
    "mask_only",
    "Equalization option"
  }
};

static ProcRecord equalize_proc =
{
  "gimp_equalize",
  "Equalize the contents of the specified drawable.",
  "This procedure equalizes the contents of the specified drawable. Each intensity channel is equalizeed independently. The equalized intensity is given as inten' = (255 - inten). Indexed color drawables are not valid for this operation. The 'mask_only' option specifies whether to adjust only the area of the image within the selection bounds, or the entire image based on the histogram of the selected area. If there is no selection, the entire image is adjusted based on the histogram for the entire image.",
  "Spencer Kimball & Peter Mattis",
  "Spencer Kimball & Peter Mattis",
  "1995-1996",
  GIMP_INTERNAL,
  2,
  equalize_inargs,
  0,
  NULL,
  { { equalize_invoker } }
};

static Argument *
invert_invoker (Gimp     *gimp,
                Argument *args)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;

  drawable = (GimpDrawable *) gimp_item_get_by_ID (gimp, args[0].value.pdb_int);
  if (! GIMP_IS_DRAWABLE (drawable))
    success = FALSE;

  if (success)
    {
      if (! gimp_drawable_is_indexed (drawable))
	gimp_drawable_invert (drawable);
      else
	success = FALSE;
    }

  return procedural_db_return_args (&invert_proc, success);
}

static ProcArg invert_inargs[] =
{
  {
    GIMP_PDB_DRAWABLE,
    "drawable",
    "The drawable"
  }
};

static ProcRecord invert_proc =
{
  "gimp_invert",
  "Invert the contents of the specified drawable.",
  "This procedure inverts the contents of the specified drawable. Each intensity channel is inverted independently. The inverted intensity is given as inten' = (255 - inten). Indexed color drawables are not valid for this operation.",
  "Spencer Kimball & Peter Mattis",
  "Spencer Kimball & Peter Mattis",
  "1995-1996",
  GIMP_INTERNAL,
  1,
  invert_inargs,
  0,
  NULL,
  { { invert_invoker } }
};

static Argument *
curves_spline_invoker (Gimp     *gimp,
                       Argument *args)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gint32 channel;
  gint32 num_points;
  guint8 *control_pts;
  Curves c;
  gint x1, y1, x2, y2;
  gint j;
  PixelRegion srcPR, destPR;
  GimpLut *lut;

  drawable = (GimpDrawable *) gimp_item_get_by_ID (gimp, args[0].value.pdb_int);
  if (! GIMP_IS_DRAWABLE (drawable))
    success = FALSE;

  channel = args[1].value.pdb_int;
  if (channel < GIMP_VALUE_LUT || channel > GIMP_ALPHA_LUT)
    success = FALSE;

  num_points = args[2].value.pdb_int;
  if (num_points <= 3 || num_points > 34)
    success = FALSE;

  control_pts = (guint8 *) args[3].value.pdb_pointer;

  if (success)
    {
      if (gimp_drawable_is_indexed (drawable) || (num_points & 1) ||
	  (gimp_drawable_is_gray (drawable) && channel != GIMP_GRAY_LUT))
	success = FALSE;
      else
	{
	  lut = gimp_lut_new ();
    
	  curves_init (&c);
    
	  for (j = 0; j < num_points / 2; j++)
	    {
	      c.points[channel][j][0] = control_pts[j * 2];
	      c.points[channel][j][1] = control_pts[j * 2 + 1];
	    }
    
	  curves_calculate_curve (&c, channel);
    
	  gimp_lut_setup (lut,
			  (GimpLutFunc) curves_lut_func,
			  &c,
			  gimp_drawable_bytes (drawable));
    
	  /* The application should occur only within selection bounds */
	  gimp_drawable_mask_bounds (drawable, &x1, &y1, &x2, &y2);
    
	  pixel_region_init (&srcPR, gimp_drawable_data (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), FALSE);
	  pixel_region_init (&destPR, gimp_drawable_shadow (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), TRUE);
    
	  pixel_regions_process_parallel ((p_func) gimp_lut_process, lut, 2,
					  &srcPR, &destPR);
    
	  gimp_lut_free (lut);
	  gimp_drawable_merge_shadow (drawable, TRUE, _("Curves"));
	  gimp_drawable_update (drawable, x1, y1, (x2 - x1), (y2 - y1));
	}
    }

  return procedural_db_return_args (&curves_spline_proc, success);
}

static ProcArg curves_spline_inargs[] =
{
  {
    GIMP_PDB_DRAWABLE,
    "drawable",
    "The drawable"
  },
  {
    GIMP_PDB_INT32,
    "channel",
    "The channel to modify: { GIMP_VALUE_LUT (0), GIMP_RED_LUT (1), GIMP_GREEN_LUT (2), GIMP_BLUE_LUT (3), GIMP_ALPHA_LUT (4) }"
  },
  {
    GIMP_PDB_INT32,
    "num_points",
    "The number of values in the control point array (3 < num_points <= 34)"
  },
  {
    GIMP_PDB_INT8ARRAY,
    "control_pts",
    "The spline control points: { cp1.x, cp1.y, cp2.x, cp2.y, ... }"
  }
};

static ProcRecord curves_spline_proc =
{
  "gimp_curves_spline",
  "Modifies the intensity curve(s) for specified drawable.",
  "Modifies the intensity mapping for one channel in the specified drawable. The drawable must be either grayscale or RGB, and the channel can be either an intensity component, or the value. The 'control_pts' parameter is an array of integers which define a set of control points which describe a Catmull Rom spline which yields the final intensity curve. Use the 'gimp_curves_explicit' function to explicitly modify intensity levels.",
  "Spencer Kimball & Peter Mattis",
  "Spencer Kimball & Peter Mattis",
  "1995-1996",
  GIMP_INTERNAL,
  4,
  curves_spline_inargs,
  0,
  NULL,
  { { curves_spline_invoker } }
};

static Argument *
curves_explicit_invoker (Gimp     *gimp,
                         Argument *args)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gint32 channel;
  gint32 num_bytes;
  guint8 *curve;
  Curves c;
  gint x1, y1, x2, y2;
  gint j;
  PixelRegion srcPR, destPR;
  GimpLut *lut;

  drawable = (GimpDrawable *) gimp_item_get_by_ID (gimp, args[0].value.pdb_int);
  if (! GIMP_IS_DRAWABLE (drawable))
    success = FALSE;

  channel = args[1].value.pdb_int;
  if (channel < GIMP_VALUE_LUT || channel > GIMP_ALPHA_LUT)
    success = FALSE;

  num_bytes = args[2].value.pdb_int;
  if (num_bytes <= 0)
    success = FALSE;

  curve = (guint8 *) args[3].value.pdb_pointer;

  if (success)
    {
      if (gimp_drawable_is_indexed (drawable) || (num_bytes != 256) ||
	  (gimp_drawable_is_gray (drawable) && channel != GIMP_GRAY_LUT))
	success = FALSE;
      else 
	{
	  lut = gimp_lut_new ();
    
	  curves_init (&c);
    
	  for (j = 0; j < 256; j++)
	    c.curve[channel][j] = curve[j];
    
	  gimp_lut_setup (lut,
			  (GimpLutFunc) curves_lut_func,
			  &c,
			  gimp_drawable_bytes (drawable));
	  
	  /* The application should occur only within selection bounds */
	  gimp_drawable_mask_bounds (drawable, &x1, &y1, &x2, &y2);
	  pixel_region_init (&srcPR, gimp_drawable_data (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), FALSE);
	  pixel_region_init (&destPR, gimp_drawable_shadow (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), TRUE);
    
	  pixel_regions_process_parallel ((p_func) gimp_lut_process, lut, 2,
					  &srcPR, &destPR);
    
	  gimp_lut_free (lut);
	  gimp_drawable_merge_shadow (drawable, TRUE, _("Curves"));
	  gimp_drawable_update (drawable, x1, y1, (x2 - x1), (y2 - y1));
	}
    }

  return procedural_db_return_args (&curves_explicit_proc, success);
}

static ProcArg curves_explicit_inargs[] =
{
  {
    GIMP_PDB_DRAWABLE,
    "drawable",
    "The drawable"
  },
  {
    GIMP_PDB_INT32,
    "channel",
    "The channel to modify: { GIMP_VALUE_LUT (0), GIMP_RED_LUT (1), GIMP_GREEN_LUT (2), GIMP_BLUE_LUT (3), GIMP_ALPHA_LUT (4) }"
  },
  {
    GIMP_PDB_INT32,
    "num_bytes",
    "The number of bytes in the new curve (always 256)"
  },
  {
    GIMP_PDB_INT8ARRAY,
    "curve",
    "The explicit curve"
  }
};

static ProcRecord curves_explicit_proc =
{
  "gimp_curves_explicit",
  "Modifies the intensity curve(s) for specified drawable.",
  "Modifies the intensity mapping for one channel in the specified drawable. The drawable must be either grayscale or RGB, and the channel can be either an intensity component, or the value. The 'curve' parameter is an array of bytes which explicitly defines how each pixel value in the drawable will be modified. Use the 'gimp_curves_spline' function to modify intensity levels with Catmull Rom splines.",
  "Spencer Kimball & Peter Mattis",
  "Spencer Kimball & Peter Mattis",
  "1995-1996",
  GIMP_INTERNAL,
  4,
  curves_explicit_inargs,
  0,
  NULL,
  { { curves_explicit_invoker } }
};

static Argument *
color_balance_invoker (Gimp     *gimp,
                       Argument *args)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gint32 transfer_mode;
  gboolean preserve_lum;
  gdouble cyan_red;
  gdouble magenta_green;
  gdouble yellow_blue;
  ColorBalance cb;
  PixelRegionIterator *pr;
  PixelRegion srcPR, destPR;
  gint x1, y1, x2, y2;

  drawable = (GimpDrawable *) gimp_item_get_by_ID (gimp, args[0].value.pdb_int);
  if (! GIMP_IS_DRAWABLE (drawable))
    success = FALSE;

  transfer_mode = args[1].value.pdb_int;
  if (transfer_mode < GIMP_SHADOWS || transfer_mode > GIMP_HIGHLIGHTS)
    success = FALSE;

  preserve_lum = args[2].value.pdb_int ? TRUE : FALSE;

  cyan_red = args[3].value.pdb_float;
  if (cyan_red < -100.0 || cyan_red > 100.0)
    success = FALSE;

  magenta_green = args[4].value.pdb_float;
  if (magenta_green < -100.0 || magenta_green > 100.0)
    success = FALSE;

  yellow_blue = args[5].value.pdb_float;
  if (yellow_blue < -100.0 || yellow_blue > 100.0)
    success = FALSE;

  if (success)
    {
      if (gimp_drawable_is_indexed (drawable))
	success = FALSE;
      else
	{
	  color_balance_init (&cb);
    
	  cb.preserve_luminosity = preserve_lum;
    
	  cb.cyan_red[transfer_mode]      = cyan_red;
	  cb.magenta_green[transfer_mode] = magenta_green;
	  cb.yellow_blue[transfer_mode]   = yellow_blue;
    
	  color_balance_create_lookup_tables (&cb);
    
	  /* The application should occur only within selection bounds */
	  gimp_drawable_mask_bounds (drawable, &x1, &y1, &x2, &y2);
    
	  pixel_region_init (&srcPR, gimp_drawable_data (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), FALSE);
	  pixel_region_init (&destPR, gimp_drawable_shadow (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), TRUE);
    
	  for (pr = pixel_regions_register (2, &srcPR, &destPR);
	       pr;
	       pr = pixel_regions_process (pr))
	    {
	      color_balance (&srcPR, &destPR, &cb);
	    }
    
	  gimp_drawable_merge_shadow (drawable, TRUE, _("Color Balance"));
	  gimp_drawable_update (drawable, x1, y1, (x2 - x1), (y2 - y1));
	}
    }

  return procedural_db_return_args (&color_balance_proc, success);
}

static ProcArg color_balance_inargs[] =
{
  {
    GIMP_PDB_DRAWABLE,
    "drawable",
    "The drawable"
  },
  {
    GIMP_PDB_INT32,
    "transfer_mode",
    "Transfer mode: { GIMP_SHADOWS (0), GIMP_MIDTONES (1), GIMP_HIGHLIGHTS (2) }"
  },
  {
    GIMP_PDB_INT32,
    "preserve_lum",
    "Preserve luminosity values at each pixel"
  },
  {
    GIMP_PDB_FLOAT,
    "cyan_red",
    "Cyan-Red color balance: (-100 <= cyan_red <= 100)"
  },
  {
    GIMP_PDB_FLOAT,
    "magenta_green",
    "Magenta-Green color balance: (-100 <= magenta_green <= 100)"
  },
  {
    GIMP_PDB_FLOAT,
    "yellow_blue",
    "Yellow-Blue color balance: (-100 <= yellow_blue <= 100)"
  }
};

static ProcRecord color_balance_proc =
{
  "gimp_color_balance",
  "Modify the color balance of the specified drawable.",
  "Modify the color balance of the specified drawable. There are three axis which can be modified: cyan-red, magenta-green, and yellow-blue. Negative values increase the amount of the former, positive values increase the amount of the latter. Color balance can be controlled with the 'transfer_mode' setting, which allows shadows, midtones, and highlights in an image to be affected differently. The 'preserve_lum' parameter, if non-zero, ensures that the luminosity of each pixel remains fixed.",
  "Spencer Kimball & Peter Mattis",
  "Spencer Kimball & Peter Mattis",
  "1997",
  GIMP_INTERNAL,
  6,
  color_balance_inargs,
  0,
  NULL,
  { { color_balance_invoker } }
};

static Argument *
histogram_invoker (Gimp     *gimp,
                   Argument *args)
{
  gboolean success = TRUE;
  Argument *return_args;
  GimpDrawable *drawable;
  gint32 channel;
  gint32 start_range;
  gint32 end_range;
  gdouble mean = 0;
  gdouble std_dev = 0;
  gdouble median = 0;
  gdouble pixels = 0;
  gdouble count = 0;
  gdouble percentile = 0;
  PixelRegion srcPR, maskPR;
  int x1, y1, x2, y2;
  GimpHistogram *histogram;
  int off_x, off_y;
  gboolean no_mask;
  GimpChannel *mask;

  drawable = (GimpDrawable *) gimp_item_get_by_ID (gimp, args[0].value.pdb_int);
  if (! GIMP_IS_DRAWABLE (drawable))
    success = FALSE;

  channel = args[1].value.pdb_int;
  if (channel < GIMP_VALUE_LUT || channel > GIMP_ALPHA_LUT)
    success = FALSE;

  start_range = args[2].value.pdb_int;
  if (start_range < 0 || start_range >= 256)
    success = FALSE;

  end_range = args[3].value.pdb_int;
  if (end_range < 0 || end_range >= 256)
    success = FALSE;

  if (success)
    {
      if (gimp_drawable_is_indexed (drawable) ||
	  (gimp_drawable_is_gray (drawable) && channel != GIMP_GRAY_LUT))
	success = FALSE;
      else
	{
	  /* The information collection should occur only within selection bounds */
	  no_mask = (gimp_drawable_mask_bounds (drawable, &x1, &y1, &x2, &y2) == FALSE);
	  gimp_item_offsets (GIMP_ITEM (drawable), &off_x, &off_y);
    
	  /* Configure the src from the drawable data */
	  pixel_region_init (&srcPR, gimp_drawable_data (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), FALSE);
    
	  /*  Configure the mask from the gimage's selection mask */
	  mask = gimp_image_get_mask (gimp_item_get_image (GIMP_ITEM (drawable)));
	  pixel_region_init (&maskPR, gimp_drawable_data (GIMP_DRAWABLE (mask)),
			     x1 + off_x, y1 + off_y, (x2 - x1), (y2 - y1), FALSE);
    
	  /* Apply the image transformation to the pixels */
	  histogram = gimp_histogram_new (GIMP_BASE_CONFIG (gimp->config));
    
	  if (no_mask)
	    gimp_histogram_calculate (histogram, &srcPR, NULL);
	  else
	    gimp_histogram_calculate (histogram, &srcPR, &maskPR);
    
	  /* Calculate the statistics */
    
	  mean       = gimp_histogram_get_mean (histogram, channel,
						 start_range, end_range);
	  std_dev    = gimp_histogram_get_std_dev (histogram, channel,
						   start_range, end_range);
	  median     = gimp_histogram_get_median (histogram, channel,
						  start_range, end_range);
	  pixels     = gimp_histogram_get_count (histogram, 0, 255);
	  count      = gimp_histogram_get_count (histogram,
						 start_range, end_range);
	  percentile = count / pixels;
    
	  gimp_histogram_free (histogram);
	}
    }

  return_args = procedural_db_return_args (&histogram_proc, success);

  if (success)
    {
      return_args[1].value.pdb_float = mean;
      return_args[2].value.pdb_float = std_dev;
      return_args[3].value.pdb_float = median;
      return_args[4].value.pdb_float = pixels;
      return_args[5].value.pdb_float = count;
      return_args[6].value.pdb_float = percentile;
    }

  return return_args;
}

static ProcArg histogram_inargs[] =
{
  {
    GIMP_PDB_DRAWABLE,
    "drawable",
    "The drawable"
  },
  {
    GIMP_PDB_INT32,
    "channel",
    "The channel to modify: { GIMP_VALUE_LUT (0), GIMP_RED_LUT (1), GIMP_GREEN_LUT (2), GIMP_BLUE_LUT (3), GIMP_ALPHA_LUT (4) }"
  },
  {
    GIMP_PDB_INT32,
    "start_range",
    "Start of the intensity measurement range"
  },
  {
    GIMP_PDB_INT32,
    "end_range",
    "End of the intensity measurement range"
  }
};

static ProcArg histogram_outargs[] =
{
  {
    GIMP_PDB_FLOAT,
    "mean",
    "Mean intensity value"
  },
  {
    GIMP_PDB_FLOAT,
    "std_dev",
    "Standard deviation of intensity values"
  },
  {
    GIMP_PDB_FLOAT,
    "median",
    "Median intensity value"
  },
  {
    GIMP_PDB_FLOAT,
    "pixels",
    "Alpha-weighted pixel count for entire image"
  },
  {
    GIMP_PDB_FLOAT,
    "count",
    "Alpha-weighted pixel count for range"
  },
  {
    GIMP_PDB_FLOAT,
    "percentile",
    "Percentile that range falls under"
  }
};

static ProcRecord histogram_proc =
{
  "gimp_histogram",
  "Returns information on the intensity histogram for the specified drawable.",
  "This tool makes it possible to gather information about the intensity histogram of a drawable. A channel to examine is first specified. This can be either value, red, green, or blue, depending on whether the drawable is of type color or grayscale. The drawable may not be indexed. Second, a range of intensities are specified. The gimp_histogram function returns statistics based on the pixels in the drawable that fall under this range of values. Mean, standard deviation, median, number of pixels, and percentile are all returned. Additionally, the total count of pixels in the image is returned. Counts of pixels are weighted by any associated alpha values and by the current selection mask. That is, pixels that lie outside an active selection mask will not be counted. Similarly, pixels with transparent alpha values will not be counted.",
  "Spencer Kimball & Peter Mattis",
  "Spencer Kimball & Peter Mattis",
  "1995-1996",
  GIMP_INTERNAL,
  4,
  histogram_inargs,
  6,
  histogram_outargs,
  { { histogram_invoker } }
};

static Argument *
hue_saturation_invoker (Gimp     *gimp,
                        Argument *args)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gint32 hue_range;
  gdouble hue_offset;
  gdouble lightness;
  gdouble saturation;
  HueSaturation hs;
  PixelRegionIterator *pr;
  PixelRegion srcPR, destPR;
  gint x1, y1, x2, y2;

  drawable = (GimpDrawable *) gimp_item_get_by_ID (gimp, args[0].value.pdb_int);
  if (! GIMP_IS_DRAWABLE (drawable))
    success = FALSE;

  hue_range = args[1].value.pdb_int;
  if (hue_range < GIMP_ALL_HUES || hue_range > GIMP_MAGENTA_HUES)
    success = FALSE;

  hue_offset = args[2].value.pdb_float;
  if (hue_offset < -180.0 || hue_offset > 180.0)
    success = FALSE;

  lightness = args[3].value.pdb_float;
  if (lightness < -100.0 || lightness > 100.0)
    success = FALSE;

  saturation = args[4].value.pdb_float;
  if (saturation < -100.0 || saturation > 100.0)
    success = FALSE;

  if (success)
    {
      if (gimp_drawable_is_indexed (drawable))
	success = FALSE;
      else
	{
	  hue_saturation_init (&hs);
    
	  hs.hue[hue_range]        = hue_offset;
	  hs.lightness[hue_range]  = lightness;
	  hs.saturation[hue_range] = saturation;
    
	  /* Calculate the transfer arrays */
	  hue_saturation_calculate_transfers (&hs);
    
	  /* The application should occur only within selection bounds */
	  gimp_drawable_mask_bounds (drawable, &x1, &y1, &x2, &y2);
    
	  pixel_region_init (&srcPR, gimp_drawable_data (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), FALSE);
	  pixel_region_init (&destPR, gimp_drawable_shadow (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), TRUE);
    
	  for (pr = pixel_regions_register (2, &srcPR, &destPR);
	       pr;
	       pr = pixel_regions_process (pr))
	    {
	      hue_saturation (&srcPR, &destPR, &hs);
	    }
    
	  gimp_drawable_merge_shadow (drawable, TRUE, _("Hue-Saturation"));
	  gimp_drawable_update (drawable, x1, y1, (x2 - x1), (y2 - y1));
	}
    }

  return procedural_db_return_args (&hue_saturation_proc, success);
}

static ProcArg hue_saturation_inargs[] =
{
  {
    GIMP_PDB_DRAWABLE,
    "drawable",
    "The drawable"
  },
  {
    GIMP_PDB_INT32,
    "hue_range",
    "Range of affected hues: { GIMP_ALL_HUES (0), GIMP_RED_HUES (1), GIMP_YELLOW_HUES (2), GIMP_GREEN_HUES (3), GIMP_CYAN_HUES (4), GIMP_BLUE_HUES (5), GIMP_MAGENTA_HUES (6) }"
  },
  {
    GIMP_PDB_FLOAT,
    "hue_offset",
    "Hue offset in degrees: (-180 <= hue_offset <= 180)"
  },
  {
    GIMP_PDB_FLOAT,
    "lightness",
    "lightness modification: (-100 <= lightness <= 100)"
  },
  {
    GIMP_PDB_FLOAT,
    "saturation",
    "saturation modification: (-100 <= saturation <= 100)"
  }
};

static ProcRecord hue_saturation_proc =
{
  "gimp_hue_saturation",
  "Modify hue, lightness, and saturation in the specified drawable.",
  "This procedures allows the hue, lightness, and saturation in the specified drawable to be modified. The 'hue_range' parameter provides the capability to limit range of affected hues.",
  "Spencer Kimball & Peter Mattis",
  "Spencer Kimball & Peter Mattis",
  "1997",
  GIMP_INTERNAL,
  5,
  hue_saturation_inargs,
  0,
  NULL,
  { { hue_saturation_invoker } }
};

static Argument *
threshold_invoker (Gimp     *gimp,
                   Argument *args)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gint32 low_threshold;
  gint32 high_threshold;
  Threshold tr;
  gint x1, y1, x2, y2;
  PixelRegion srcPR, destPR;

  drawable = (GimpDrawable *) gimp_item_get_by_ID (gimp, args[0].value.pdb_int);
  if (! GIMP_IS_DRAWABLE (drawable))
    success = FALSE;

  low_threshold = args[1].value.pdb_int;
  if (low_threshold < 0 || low_threshold > 255)
    success = FALSE;

  high_threshold = args[2].value.pdb_int;
  if (high_threshold < 0 || high_threshold > 255)
    success = FALSE;

  if (success)
    {
      if (gimp_drawable_is_indexed (drawable) || (low_threshold >= high_threshold))
	success = FALSE;
      else
	{
	  tr.color          = gimp_drawable_is_rgb (drawable);
	  tr.low_threshold  = low_threshold;
	  tr.high_threshold = high_threshold;
    
	  /* The application should occur only within selection bounds */
	  gimp_drawable_mask_bounds (drawable, &x1, &y1, &x2, &y2);
    
	  pixel_region_init (&srcPR, gimp_drawable_data (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), FALSE);
	  pixel_region_init (&destPR, gimp_drawable_shadow (drawable),
			     x1, y1, (x2 - x1), (y2 - y1), TRUE);
    
	  pixel_regions_process_parallel ((p_func) threshold_2, &tr, 2,
					  &srcPR, &destPR);
    
	  gimp_drawable_merge_shadow (drawable, TRUE, _("Threshold"));
	  gimp_drawable_update (drawable, x1, y1, (x2 - x1), (y2 - y1));
	}
    }

  return procedural_db_return_args (&threshold_proc, success);
}

static ProcArg threshold_inargs[] =
{
  {
    GIMP_PDB_DRAWABLE,
    "drawable",
    "The drawable"
  },
  {
    GIMP_PDB_INT32,
    "low_threshold",
    "The low threshold value: 0 <= low_threshold <= 255"
  },
  {
    GIMP_PDB_INT32,
    "high_threshold",
    "The high threshold value: 0 <= high_threshold <= 255"
  }
};

static ProcRecord threshold_proc =
{
  "gimp_threshold",
  "Threshold the specified drawable.",
  "This procedures generates a threshold map of the specified drawable. All pixels between the values of 'low_threshold' and 'high_threshold' are replaced with white, and all other pixels with black.",
  "Spencer Kimball & Peter Mattis",
  "Spencer Kimball & Peter Mattis",
  "1997",
  GIMP_INTERNAL,
  3,
  threshold_inargs,
  0,
  NULL,
  { { threshold_invoker } }
};
