/* GIMP - The GNU Image Manipulation Program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "config.h"

#include <gegl.h>

#include "core-types.h"

#include "base/pixel-region.h"

#include "paint-funcs/paint-funcs.h"

#include "gimp.h"
#include "gimpchannel.h"
#include "gimpguide.h"
#include "gimpimage.h"
#include "gimpimage-colormap.h"
#include "gimpimage-duplicate.h"
#include "gimpimage-grid.h"
#include "gimpimage-guides.h"
#include "gimpimage-sample-points.h"
#include "gimplayer.h"
#include "gimplayer-floating-sel.h"
#include "gimpparasitelist.h"
#include "gimpsamplepoint.h"

#include "vectors/gimpvectors.h"


static void          gimp_image_duplicate_resolution      (GimpImage     *image,
                                                           GimpImage     *new_image);
static void          gimp_image_duplicate_save_source_uri (GimpImage     *image,
                                                           GimpImage     *new_image);
static void          gimp_image_duplicate_colormap        (GimpImage     *image,
                                                           GimpImage     *new_image);
static GimpLayer   * gimp_image_duplicate_layers          (GimpImage     *image,
                                                           GimpImage     *new_image,
                                                           GimpLayer    **new_floating_layer,
                                                           GimpDrawable **floating_sel_drawable,
                                                           GimpDrawable **new_floating_sel_drawable);
static GimpChannel * gimp_image_duplicate_channels        (GimpImage     *image,
                                                           GimpImage     *new_image,
                                                           GimpDrawable  *floating_sel_drawable,
                                                           GimpDrawable **new_floating_sel_drawable);
static GimpVectors * gimp_image_duplicate_vectors         (GimpImage     *image,
                                                           GimpImage     *new_image);
static void          gimp_image_duplicate_mask            (GimpImage     *image,
                                                           GimpImage     *new_image);
static void          gimp_image_duplicate_components      (GimpImage     *image,
                                                           GimpImage     *new_image);
static void          gimp_image_duplicate_guides          (GimpImage     *image,
                                                           GimpImage     *new_image);
static void          gimp_image_duplicate_sample_points   (GimpImage     *image,
                                                           GimpImage     *new_image);
static void          gimp_image_duplicate_grid            (GimpImage     *image,
                                                           GimpImage     *new_image);
static void          gimp_image_duplicate_quick_mask      (GimpImage     *image,
                                                           GimpImage     *new_image);
static void          gimp_image_duplicate_parasites       (GimpImage     *image,
                                                           GimpImage     *new_image);


GimpImage *
gimp_image_duplicate (GimpImage *image)
{
  GimpImage    *new_image;
  GimpLayer    *active_layer;
  GimpChannel  *active_channel;
  GimpVectors  *active_vectors;
  GimpLayer    *new_floating_layer        = NULL;
  GimpDrawable *floating_sel_drawable     = NULL;
  GimpDrawable *new_floating_sel_drawable = NULL;

  g_return_val_if_fail (GIMP_IS_IMAGE (image), NULL);

  gimp_set_busy_until_idle (image->gimp);

  /*  Create a new image  */
  new_image = gimp_create_image (image->gimp,
                                 gimp_image_get_width  (image),
                                 gimp_image_get_height (image),
                                 gimp_image_base_type (image),
                                 FALSE);
  gimp_image_undo_disable (new_image);

  /*  Store the source uri to be used by the save dialog  */
  gimp_image_duplicate_save_source_uri (image, new_image);


  /*  Copy the colormap if necessary  */
  gimp_image_duplicate_colormap (image, new_image);

  /*  Copy resolution information  */
  gimp_image_duplicate_resolution (image, new_image);

  /*  Copy the layers  */
  active_layer = gimp_image_duplicate_layers (image, new_image,
                                              &new_floating_layer,
                                              &floating_sel_drawable,
                                              &new_floating_sel_drawable);

  /*  Copy the channels  */
  active_channel = gimp_image_duplicate_channels (image, new_image,
                                                  floating_sel_drawable,
                                                  &new_floating_sel_drawable);

  /*  Copy any vectors  */
  active_vectors = gimp_image_duplicate_vectors (image, new_image);

  /*  Copy the selection mask  */
  gimp_image_duplicate_mask (image, new_image);

  if (new_floating_layer)
    floating_sel_attach (new_floating_layer, new_floating_sel_drawable);

  /*  Set active layer, active channel, active vectors  */
  if (active_layer)
    gimp_image_set_active_layer (new_image, active_layer);

  if (active_channel)
    gimp_image_set_active_channel (new_image, active_channel);

  if (active_vectors)
    gimp_image_set_active_vectors (new_image, active_vectors);

  /*  Copy state of all color channels  */
  gimp_image_duplicate_components (image, new_image);

  /*  Copy any guides  */
  gimp_image_duplicate_guides (image, new_image);

  /*  Copy any sample points  */
  gimp_image_duplicate_sample_points (image, new_image);

  /*  Copy the grid  */
  gimp_image_duplicate_grid (image, new_image);

  /*  Copy the quick mask info  */
  gimp_image_duplicate_quick_mask (image, new_image);

  /*  Copy parasites  */
  gimp_image_duplicate_parasites (image, new_image);

  gimp_image_undo_enable (new_image);

  return new_image;
}

static void
gimp_image_duplicate_resolution (GimpImage *image,
                                 GimpImage *new_image)
{
  gdouble xres;
  gdouble yres;

  gimp_image_get_resolution (image, &xres, &yres);
  gimp_image_set_resolution (new_image, xres, yres);
  gimp_image_set_unit (new_image, gimp_image_get_unit (image));
}

static void
gimp_image_duplicate_save_source_uri (GimpImage *image,
                                      GimpImage *new_image)
{
  g_object_set_data_full (G_OBJECT (new_image), "gimp-image-source-uri",
                          g_strdup (gimp_object_get_name (GIMP_OBJECT (image))),
                          (GDestroyNotify) g_free);
}

static void
gimp_image_duplicate_colormap (GimpImage *image,
                               GimpImage *new_image)
{
  if (gimp_image_base_type (new_image) == GIMP_INDEXED)
    gimp_image_set_colormap (new_image,
                             gimp_image_get_colormap (image),
                             gimp_image_get_colormap_size (image),
                             FALSE);
}

static GimpLayer *
gimp_image_duplicate_layers (GimpImage     *image,
                             GimpImage     *new_image,
                             GimpLayer    **floating_layer,
                             GimpDrawable **floating_sel_drawable,
                             GimpDrawable **new_floating_sel_drawable)
{
  GimpLayer *active_layer = NULL;
  GimpLayer *floating_selection;
  GList     *all_layers;
  GList     *list;
  gint       count;

  /*  Copy floating layer  */
  floating_selection = gimp_image_get_floating_selection (image);

  if (floating_selection)
    *floating_sel_drawable = gimp_layer_get_floating_sel_drawable (floating_selection);

  all_layers = gimp_image_get_layer_list (image);

  for (list = all_layers, count = 0;
       list;
       list = g_list_next (list))
    {
      GimpLayer *layer = list->data;
      GimpLayer *new_layer;

      new_layer = GIMP_LAYER (gimp_item_convert (GIMP_ITEM (layer),
                                                 new_image,
                                                 G_TYPE_FROM_INSTANCE (layer)));

      g_object_set_data (G_OBJECT (layer), "gimp-image-duplicate-item",
                         new_layer);

      /*  Make sure the copied layer doesn't say: "<old layer> copy"  */
      gimp_object_set_name (GIMP_OBJECT (new_layer),
                            gimp_object_get_name (GIMP_OBJECT (layer)));

      /*  Make sure that if the layer has a layer mask,
       *  its name isn't screwed up
       */
      if (new_layer->mask)
        gimp_object_set_name (GIMP_OBJECT (new_layer->mask),
                              gimp_object_get_name (GIMP_OBJECT (layer->mask)));

      if (gimp_image_get_active_layer (image) == layer)
        active_layer = new_layer;

      if (gimp_image_get_floating_selection (image) == layer)
        *floating_layer = new_layer;

      if (*floating_sel_drawable == GIMP_DRAWABLE (layer))
        *new_floating_sel_drawable = GIMP_DRAWABLE (new_layer);

      if (*floating_layer != new_layer)
        {
          GimpViewable *parent;
          GimpLayer    *new_parent = NULL;

          parent = gimp_viewable_get_parent (GIMP_VIEWABLE (layer));

          if (parent)
            new_parent = g_object_get_data (G_OBJECT (parent),
                                            "gimp-image-duplicate-item");

          gimp_image_add_layer (new_image, new_layer,
                                new_parent, count++, FALSE);
        }
    }

  for (list = all_layers; list; list = g_list_next (list))
    g_object_set_data (list->data, "gimp-image-duplicate-item", NULL);

  g_list_free (all_layers);

  return active_layer;
}

static GimpChannel *
gimp_image_duplicate_channels (GimpImage     *image,
                               GimpImage     *new_image,
                               GimpDrawable  *floating_sel_drawable,
                               GimpDrawable **new_floating_sel_drawable)
{
  GimpChannel *active_channel = NULL;
  GList       *all_channels;
  GList       *list;
  gint         count;

  all_channels = gimp_image_get_channel_list (image);

  for (list = all_channels, count = 0;
       list;
       list = g_list_next (list))
    {
      GimpChannel  *channel = list->data;
      GimpChannel  *new_channel;
      GimpViewable *parent;
      GimpChannel  *new_parent = NULL;

      new_channel = GIMP_CHANNEL (gimp_item_convert (GIMP_ITEM (channel),
                                                     new_image,
                                                     G_TYPE_FROM_INSTANCE (channel)));

      g_object_set_data (G_OBJECT (channel), "gimp-image-duplicate-item",
                         new_channel);

      /*  Make sure the copied channel doesn't say: "<old channel> copy"  */
      gimp_object_set_name (GIMP_OBJECT (new_channel),
                            gimp_object_get_name (GIMP_OBJECT (channel)));

      if (gimp_image_get_active_channel (image) == channel)
        active_channel = new_channel;

      if (floating_sel_drawable == GIMP_DRAWABLE (channel))
        *new_floating_sel_drawable = GIMP_DRAWABLE (new_channel);

      parent = gimp_viewable_get_parent (GIMP_VIEWABLE (channel));

      if (parent)
        new_parent = g_object_get_data (G_OBJECT (parent),
                                        "gimp-image-duplicate-item");

      gimp_image_add_channel (new_image, new_channel,
                              new_parent, count++, FALSE);
    }

  for (list = all_channels; list; list = g_list_next (list))
    g_object_set_data (list->data, "gimp-image-duplicate-item", NULL);

  g_list_free (all_channels);

  return active_channel;
}

static GimpVectors *
gimp_image_duplicate_vectors (GimpImage *image,
                              GimpImage *new_image)
{
  GimpVectors *active_vectors = NULL;
  GList       *all_vectors;
  GList       *list;
  gint         count;

  all_vectors = gimp_image_get_vectors_list (image);

  for (list = all_vectors, count = 0;
       list;
       list = g_list_next (list))
    {
      GimpVectors  *vectors = list->data;
      GimpVectors  *new_vectors;
      GimpViewable *parent;
      GimpVectors  *new_parent = NULL;

      new_vectors = GIMP_VECTORS (gimp_item_convert (GIMP_ITEM (vectors),
                                                     new_image,
                                                     G_TYPE_FROM_INSTANCE (vectors)));

      g_object_set_data (G_OBJECT (vectors), "gimp-image-duplicate-item",
                         new_vectors);

      /*  Make sure the copied vectors doesn't say: "<old vectors> copy"  */
      gimp_object_set_name (GIMP_OBJECT (new_vectors),
                            gimp_object_get_name (GIMP_OBJECT (vectors)));

      if (gimp_image_get_active_vectors (image) == vectors)
        active_vectors = new_vectors;

      parent = gimp_viewable_get_parent (GIMP_VIEWABLE (vectors));

      if (parent)
        new_parent = g_object_get_data (G_OBJECT (parent),
                                        "gimp-image-duplicate-item");

      gimp_image_add_vectors (new_image, new_vectors,
                              new_parent, count++, FALSE);
    }

  for (list = all_vectors; list; list = g_list_next (list))
    g_object_set_data (list->data, "gimp-image-duplicate-item", NULL);

  g_list_free (all_vectors);

  return active_vectors;
}

static void
gimp_image_duplicate_mask (GimpImage *image,
                           GimpImage *new_image)
{
  TileManager *src_tiles;
  TileManager *dest_tiles;
  PixelRegion  srcPR, destPR;

  src_tiles  =
    gimp_drawable_get_tiles (GIMP_DRAWABLE (gimp_image_get_mask (image)));
  dest_tiles =
    gimp_drawable_get_tiles (GIMP_DRAWABLE (gimp_image_get_mask (new_image)));

  pixel_region_init (&srcPR, src_tiles,
                     0, 0,
                     gimp_image_get_width  (image),
                     gimp_image_get_height (image),
                     FALSE);
  pixel_region_init (&destPR, dest_tiles,
                     0, 0,
                     gimp_image_get_width  (image),
                     gimp_image_get_height (image),
                     TRUE);

  copy_region (&srcPR, &destPR);

  new_image->selection_mask->bounds_known   = FALSE;
  new_image->selection_mask->boundary_known = FALSE;
}

static void
gimp_image_duplicate_components (GimpImage *image,
                                 GimpImage *new_image)
{
  gint count;

  for (count = 0; count < MAX_CHANNELS; count++)
    {
      new_image->visible[count] = image->visible[count];
      new_image->active[count]  = image->active[count];
    }
}

static void
gimp_image_duplicate_guides (GimpImage *image,
                             GimpImage *new_image)
{
  GList *list;

  for (list = gimp_image_get_guides (image);
       list;
       list = g_list_next (list))
    {
      GimpGuide *guide    = list->data;
      gint       position = gimp_guide_get_position (guide);

      switch (gimp_guide_get_orientation (guide))
        {
        case GIMP_ORIENTATION_HORIZONTAL:
          gimp_image_add_hguide (new_image, position, FALSE);
          break;

        case GIMP_ORIENTATION_VERTICAL:
          gimp_image_add_vguide (new_image, position, FALSE);
          break;

        default:
          g_error ("Unknown guide orientation.\n");
        }
    }
}

static void
gimp_image_duplicate_sample_points (GimpImage *image,
                                    GimpImage *new_image)
{
  GList *list;

  for (list = gimp_image_get_sample_points (image);
       list;
       list = g_list_next (list))
    {
      GimpSamplePoint *sample_point = list->data;

      gimp_image_add_sample_point_at_pos (new_image,
                                          sample_point->x,
                                          sample_point->y,
                                          FALSE);
    }
}

static void
gimp_image_duplicate_grid (GimpImage *image,
                           GimpImage *new_image)
{
  if (image->grid)
    gimp_image_set_grid (new_image, image->grid, FALSE);
}

static void
gimp_image_duplicate_quick_mask (GimpImage *image,
                                 GimpImage *new_image)
{
  new_image->quick_mask_state    = image->quick_mask_state;
  new_image->quick_mask_inverted = image->quick_mask_inverted;
  new_image->quick_mask_color    = image->quick_mask_color;
}

static void
gimp_image_duplicate_parasites (GimpImage *image,
                                GimpImage *new_image)
{
  if (image->parasites)
    {
      g_object_unref (new_image->parasites);
      new_image->parasites = gimp_parasite_list_copy (image->parasites);
    }
}
