/*
 * Copyright (C) 1997-1998 Janne Lf <jlof@student.oulu.fi>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <math.h>
#include <gtk/gtk.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include "gtkglarea.h"
#include "trackball.h"


extern void draw_alien(void);	/* from alien.c */
extern void draw_penguin(void);	/* from penguin.c */
void (*draw_func)(void) = draw_penguin;

/* trackball */
float beginx,beginy;	/* last recorded mouse position */
float	curquat[4];	/* current quaternion */
float	lastquat[4];	/* last quaternion */

gint button_press_event(GtkWidget *widget, GdkEventButton *event)
{
	if (event->button == 1) {
		beginx = event->x;
		beginy = event->y;
		trackball(lastquat, 0.0, 0.0, 0.0, 0.0);
	}
	return TRUE;
}
gint motion_notify_event(GtkWidget *widget, GdkEventMotion *event)
{
	int x, y;
	GdkModifierType state;
	if (event->is_hint) {
		gdk_window_get_pointer(event->window, &x, &y, &state);
	} else {
		x = event->x;
		y = event->y;
		state = event->state;
	}
    	if (state & GDK_BUTTON1_MASK) {
		int width  = GTK_WIDGET(widget)->allocation.width;
		int height = GTK_WIDGET(widget)->allocation.height;
		/* recalculate spin speed */
		trackball(lastquat,
			(2.0*beginx - width) / width,
		        (height- 2.0*beginy) / height,
			(2.0*x - width) / width,
			(height- 2.0*y) / height);
		beginx = x;
		beginy = y;
	}
  	return TRUE;
}



void initgl(void)
{
	/* initialize view, no need to worry about vieport size (glViewport),
	   it is always automatically set to widget size */
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(45, 1.0, 1,100);
	glMatrixMode(GL_MODELVIEW);


	/* remove back faces */
	glEnable(GL_CULL_FACE);
	glEnable(GL_DEPTH_TEST);

	/* speedups */
	glEnable(GL_DITHER);
	glShadeModel(GL_SMOOTH);
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
	glHint(GL_POLYGON_SMOOTH_HINT, GL_FASTEST);

	/* light */
	if (1) {
		GLfloat light0_pos[4]   = { -50.0, 50.0, 0.0, 1.0 };
		GLfloat light0_color[4] = { .6, .6, .6, 1.0 };		/* white light */

		GLfloat light1_pos[4]   = {  50.0, 50.0, 0.0, 1.0 };
		GLfloat light1_color[4] = { .4, .4, 1, 1.0 };		/* cold blue light */

		glLightfv(GL_LIGHT0, GL_POSITION, light0_pos);
		glLightfv(GL_LIGHT0, GL_DIFFUSE,  light0_color);

		glLightfv(GL_LIGHT1, GL_POSITION, light1_pos);
		glLightfv(GL_LIGHT1, GL_DIFFUSE,  light1_color);

		glEnable(GL_LIGHT0);
		glEnable(GL_LIGHT1);
		glEnable(GL_LIGHTING);

		glColorMaterial(GL_FRONT_AND_BACK,GL_AMBIENT_AND_DIFFUSE);
		glEnable(GL_COLOR_MATERIAL);

	}



}

gint resize(GtkWidget *widget, GdkEventConfigure *event)
{
	/* resize viewport */
	gtk_gl_area_begingl(GTK_GL_AREA(widget));
	glViewport(0,0, widget->allocation.width, widget->allocation.height);
	gtk_gl_area_endgl(GTK_GL_AREA(widget));
	return TRUE;
}

gint draw(gpointer gp)
{
	GtkGLArea *glarea = GTK_GL_AREA(gp);
	GLfloat m[4][4];

	static int init = 0;

	/* start opengl rendering, you must put all opengl calls
	 * inside gtk_gl_area_begingl/gtk_gl_area_endgl pairs
	*/
	gtk_gl_area_begingl(glarea);

	if (!init) {
		trackball(curquat , 0.0, 0.0, 0.0, 0.0);
		trackball(lastquat, 0.0, 0.0, 0.0, 0.0);
		initgl();
		init = 1;
	}
		
	glClearColor(.3,.4,.6,1);
	glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

	glLoadIdentity();
	glTranslatef(0,0,-30);		/* object position */
	build_rotmatrix(m,curquat);	/* rotation */
	glMultMatrixf(&m[0][0]);

	add_quats(lastquat, curquat, curquat);	/* spin */

	draw_func();

	/* opengl rendering done for now */
	gtk_gl_area_endgl(glarea);

	/* swap buffers, note that all GLX contexts share the same notion
	 * of which are front and which are back buffers.
	 */
	gtk_gl_area_swapbuffers(glarea);

	return TRUE;
}



gint toggle(GtkWidget *a, GtkWidget *b)
{
	if (GTK_TOGGLE_BUTTON(a)->active) {
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(b), FALSE);
		if  (draw_func == draw_penguin)
			draw_func = draw_alien;
		else
			draw_func = draw_penguin;
	} else {
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(b), TRUE);
	}
	return TRUE;
}


int
main( int argc, char **argv )
{
	GtkWidget *window,*vbox;

	gtk_init( &argc, &argv );

	/* toplevel window */
	window = gtk_window_new( GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(window), "GtkGLArea Demo");
	gtk_container_border_width(GTK_CONTAINER(window),10);
	gtk_signal_connect (GTK_OBJECT (window), "destroy",
	                    GTK_SIGNAL_FUNC (gtk_exit), NULL);
	gtk_signal_connect (GTK_OBJECT (window), "delete_event",
	                    GTK_SIGNAL_FUNC (gtk_exit), NULL);

	/* vertical box */
	vbox = gtk_vbox_new(FALSE,10);
	gtk_container_add(GTK_CONTAINER(window),vbox);
	gtk_widget_show(vbox);


	/* text label */
	{
		GtkWidget *label = gtk_label_new("Left button drag over the image\nto spin the object");
		gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
		gtk_widget_show(label);
	}

	/* make opengl widget, attrList has the same format as
	   glXChooseVisual()'s attribute list. glarea widget is created
	   with requested type of visual, each widget has it's own GLX
	   context. When widget is resized it's opengl viewport is
	   automatically resized (glViewport is called automatically).
	*/
	{
		int attrList[] = { GLX_RGBA, GLX_DOUBLEBUFFER, GLX_DEPTH_SIZE,1, None };
		GtkWidget *glarea = gtk_gl_area_new(attrList);

		if (glarea == NULL) {
			g_print("Can't create GtkGLArea widget\n");
			gtk_exit(1);
		}

	        gtk_widget_set_usize(glarea,200,200);   /* minumum size */
		gtk_widget_set_events(glarea,	/* events */
			GDK_BUTTON_PRESS_MASK|
			GDK_BUTTON_RELEASE_MASK|
			GDK_POINTER_MOTION_MASK|
			GDK_POINTER_MOTION_HINT_MASK);

		gtk_signal_connect (GTK_OBJECT (glarea), "configure_event",
		                    GTK_SIGNAL_FUNC(resize), NULL);
		gtk_signal_connect (GTK_OBJECT (glarea), "motion_notify_event",
		                    GTK_SIGNAL_FUNC(motion_notify_event), NULL);
	        gtk_signal_connect (GTK_OBJECT (glarea), "button_press_event",
	 	                    GTK_SIGNAL_FUNC(button_press_event), NULL);

	

		gtk_box_pack_start(GTK_BOX(vbox), glarea, TRUE, TRUE, 0);
		gtk_widget_show(glarea);

		/* do drawing in idle function */
		gtk_idle_add(draw, glarea);
	}

	/* penguin/alien buttons */
	{
		GtkWidget *hbox = gtk_hbox_new(FALSE,0);
		GtkWidget *button1 = gtk_toggle_button_new_with_label("penguin");
		GtkWidget *button2 = gtk_toggle_button_new_with_label("alien");

		gtk_box_pack_start(GTK_BOX(hbox), button1, TRUE, TRUE, 0);
		gtk_widget_show(button1);

		gtk_box_pack_start(GTK_BOX(hbox), button2, TRUE, TRUE, 0);
		gtk_widget_show(button2);

		gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
		gtk_widget_show(hbox);

		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button1), TRUE);
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button2), FALSE);

		gtk_signal_connect(GTK_OBJECT(button1),"toggled", GTK_SIGNAL_FUNC(toggle), (gpointer)button2);
		gtk_signal_connect(GTK_OBJECT(button2),"toggled", GTK_SIGNAL_FUNC(toggle), (gpointer)button1);
	}


	gtk_widget_show(window);

	gtk_main();
	
	return 0;
}
