# signal handling for Python GTK

%{

/* signal marshallers -- Ideas Stolen from Perl Gtk*/
static PyObject *GtkArg_to_Tuple(int nparams, GtkArg *args) {
    PyObject *params;
    int i;

    params = PyTuple_New(nparams);
    for (i=0; i < nparams; i++)
        switch (GTK_FUNDAMENTAL_TYPE(args[i].type)) {
            case GTK_TYPE_INVALID:
                fprintf(stderr, "invalid argument type?\n");
                Py_INCREF(Py_None);
                PyTuple_SetItem(params, i, Py_None);
                break;
            case GTK_TYPE_NONE:
                Py_INCREF(Py_None);
                PyTuple_SetItem(params, i, Py_None);
                break;
            case GTK_TYPE_CHAR:
                PyTuple_SetItem(params, i, PyString_FromStringAndSize(
                      &(GTK_VALUE_CHAR(args[i])), 1));
                break;
            case GTK_TYPE_BOOL:
                PyTuple_SetItem(params, i, PyInt_FromLong(GTK_VALUE_BOOL(
                      args[i])));
                break;
            case GTK_TYPE_INT:
                PyTuple_SetItem(params, i, PyInt_FromLong(GTK_VALUE_INT(
                      args[i])));
                break;
            case GTK_TYPE_UINT:
                PyTuple_SetItem(params, i, PyInt_FromLong(GTK_VALUE_UINT(
                      args[i])));
                break;
            case GTK_TYPE_LONG:
                PyTuple_SetItem(params, i, PyInt_FromLong(GTK_VALUE_LONG(
                      args[i])));
                break;
            case GTK_TYPE_ULONG:
                PyTuple_SetItem(params, i, PyInt_FromLong(GTK_VALUE_ULONG(
                      args[i])));
                break;
            case GTK_TYPE_FLOAT:
                PyTuple_SetItem(params, i, PyFloat_FromDouble(
                      GTK_VALUE_FLOAT(args[i])));
                break;
            case GTK_TYPE_DOUBLE:
                PyTuple_SetItem(params, i, PyFloat_FromDouble(
                      GTK_VALUE_DOUBLE(args[i])));
                break;
            case GTK_TYPE_STRING:
                PyTuple_SetItem(params, i, PyString_FromString(
                      GTK_VALUE_STRING(args[i])));
                break;
            case GTK_TYPE_ENUM:
                PyTuple_SetItem(params, i, PyInt_FromLong(GTK_VALUE_ENUM(
                      args[i])));
                break;
            case GTK_TYPE_FLAGS:
                PyTuple_SetItem(params, i, PyInt_FromLong(GTK_VALUE_FLAGS(
                      args[i])));
                break;
            case GTK_TYPE_ARGS:
                PyTuple_SetItem(params, i, GtkArg_to_Tuple(GTK_VALUE_ARGS(
                           args[i]).n_args, GTK_VALUE_ARGS(args[i]).args));
                break;
            case GTK_TYPE_OBJECT:
                PyTuple_SetItem(params, i, PyGtk_New(GTK_VALUE_OBJECT(
                    args[i])));
                break;
            case GTK_TYPE_POINTER:
                PyTuple_SetItem(params, i,PyInt_FromLong((long)
                    GTK_VALUE_POINTER(args[i])));
                break;
            case GTK_TYPE_BOXED:
                /* conversion is specific to the type which is boxed
                 * builtin boxed types: GtkAcceleratorTable, GtkStyle,
                 *   GdkColormap, GdkVisual, GdkFont, GdkWindow, GdkEvent */
                if (args[i].type == GTK_TYPE_ACCELERATOR_TABLE)
                    PyTuple_SetItem(params, i, PyGtkAccelerator_New(
                        GTK_VALUE_BOXED(args[i])));
                else if (args[i].type == GTK_TYPE_STYLE)
                     PyTuple_SetItem(params, i, PyGtkStyle_New(
                         GTK_VALUE_BOXED(args[i])));
                else if (args[i].type == GTK_TYPE_GDK_EVENT)
                     PyTuple_SetItem(params, i, PyGdkEvent_New(
                         GTK_VALUE_BOXED(args[i])));
                else if (args[i].type == GTK_TYPE_GDK_FONT)
                     PyTuple_SetItem(params, i, PyGdkFont_New(
                         GTK_VALUE_BOXED(args[i])));
                else if (args[i].type ==  GTK_TYPE_GDK_COLOR)
                     PyTuple_SetItem(params, i, PyGdkColor_New(
                         GTK_VALUE_BOXED(args[i])));
                break;
            case GTK_TYPE_FOREIGN:
                Py_INCREF((PyObject *)(GTK_VALUE_FOREIGN(args[i]).data));
                PyTuple_SetItem(params, i, GTK_VALUE_FOREIGN(args[i]).data);
                break;
            case GTK_TYPE_CALLBACK:
                Py_INCREF((PyObject *)(GTK_VALUE_CALLBACK(args[i]).data));
                PyTuple_SetItem(params, i, GTK_VALUE_CALLBACK(args[i]).data);
            case GTK_TYPE_SIGNAL:
                Py_INCREF((PyObject *)(GTK_VALUE_SIGNAL(args[i]).d));
                PyTuple_SetItem(params, i, GTK_VALUE_SIGNAL(args[i]).d);
            default:
	        fprintf(stderr, "Can't decode parameter (type %s)\n",
			gtk_type_name(args[i].type));
	        Py_INCREF(Py_None);
		PyTuple_SetItem(params, i, Py_None);
                break;
        }
    return params;
}

/* set the signal/callback return type from the python return type */
static void PyGtk_SetRetType(GtkArg *ret, GtkType ret_type, PyObject *py_ret) {
    switch (GTK_FUNDAMENTAL_TYPE(ret_type)) {
    case GTK_TYPE_NONE:
    case GTK_TYPE_INVALID:
      break;
    case GTK_TYPE_BOOL:
      if (py_ret == Py_None || PyInt_Check(py_ret) && PyInt_AsLong(py_ret)==0)
	*GTK_RETLOC_BOOL(*ret) = FALSE;
      else
	*GTK_RETLOC_BOOL(*ret) = TRUE;
      break;
    case GTK_TYPE_CHAR:
      if (!PyString_Check(py_ret) || PyString_Size(py_ret) != 1)
	*GTK_RETLOC_CHAR(*ret) = '\0';
      else
	*GTK_RETLOC_CHAR(*ret) = PyString_AsString(py_ret)[0];
      break;
    case GTK_TYPE_ENUM:
    case GTK_TYPE_FLAGS:
    case GTK_TYPE_INT:
      if (!PyInt_Check(py_ret))
	*GTK_RETLOC_INT(*ret) = 0;
      else
	*GTK_RETLOC_INT(*ret) = PyInt_AsLong(py_ret);
      break;
    case GTK_TYPE_UINT:
      if (!PyInt_Check(py_ret))
	*GTK_RETLOC_UINT(*ret) = 0;
      else
	*GTK_RETLOC_UINT(*ret) = PyInt_AsLong(py_ret);
      break;
    case GTK_TYPE_LONG:
      if (!PyInt_Check(py_ret))
	*GTK_RETLOC_LONG(*ret) = 0;
      else
	*GTK_RETLOC_LONG(*ret) = PyInt_AsLong(py_ret);
      break;
    case GTK_TYPE_ULONG:
      if (!PyInt_Check(py_ret))
	*GTK_RETLOC_ULONG(*ret) = 0;
      else
	*GTK_RETLOC_ULONG(*ret) = PyInt_AsLong(py_ret);
      break;
    case GTK_TYPE_FLOAT:
      if (!PyFloat_Check(py_ret))
	*GTK_RETLOC_FLOAT(*ret) = 0;
      else
	*GTK_RETLOC_FLOAT(*ret) = PyFloat_AsDouble(py_ret);
      break;
    case GTK_TYPE_DOUBLE:
      if (!PyFloat_Check(py_ret))
	*GTK_RETLOC_DOUBLE(*ret) = 0;
      else
	*GTK_RETLOC_DOUBLE(*ret) = PyFloat_AsDouble(py_ret);
      break;
    case GTK_TYPE_STRING:
      if (!PyString_Check(py_ret))
	*GTK_RETLOC_STRING(*ret) = NULL;
      else
	*GTK_RETLOC_STRING(*ret) = g_strdup(PyString_AsString(py_ret));
      break;
    case GTK_TYPE_OBJECT:
      if (!PyGtk_Check(py_ret))
	*GTK_RETLOC_OBJECT(*ret) = NULL;
      else
	*GTK_RETLOC_OBJECT(*ret) = PyGtk_Get(py_ret);
      break;
    case GTK_TYPE_BOXED:
      if (ret_type == GTK_TYPE_ACCELERATOR_TABLE) {
	if (!PyGtkAccelerator_Check(py_ret))
	  *GTK_RETLOC_BOXED(*ret) = NULL;
	else
	  *GTK_RETLOC_BOXED(*ret) = PyGtkAccelerator_Get(py_ret);
      } else if (ret_type == GTK_TYPE_STYLE) {
	if (!PyGtkStyle_Check(py_ret))
	  *GTK_RETLOC_BOXED(*ret) = NULL;
	else
	  *GTK_RETLOC_BOXED(*ret) = PyGtkStyle_Get(py_ret);
      } else if (ret_type == GTK_TYPE_GDK_EVENT) {
	if (!PyGdkEvent_Check(py_ret))
	  *GTK_RETLOC_BOXED(*ret) = NULL;
	else
	  *GTK_RETLOC_BOXED(*ret) = PyGdkEvent_Get(py_ret);
      } else if (ret_type == GTK_TYPE_GDK_FONT) {
	if (!PyGdkFont_Check(py_ret))
	  *GTK_RETLOC_BOXED(*ret) = NULL;
	else
	  *GTK_RETLOC_BOXED(*ret) = PyGdkFont_Get(py_ret);
      } else if (ret_type == GTK_TYPE_GDK_COLOR) {
	if (!PyGdkColor_Check(py_ret))
	  *GTK_RETLOC_BOXED(*ret) = NULL;
	else
	  *GTK_RETLOC_BOXED(*ret) = PyGdkColor_Get(py_ret);
      }
      break;
    case GTK_TYPE_POINTER:
    default:
      break;
    }
}

static
void PyGtk_SignalMarshal(GtkObject *object, /*gpointer*/ PyObject *func,
                       int nparams, GtkArg *args, GtkType *arg_types,
                                                 GtkType return_type) {
    PyObject *arg_list, *params, *ret;
    int i;

    ret = PyTuple_New(1);
    PyTuple_SetItem(ret, 0, PyGtk_New(object));

    for (i=0; i < nparams; i++) {
        if (arg_types[i] != GTK_TYPE_NONE) {
            args[i].type = arg_types[i];
        }
    }
    arg_list = GtkArg_to_Tuple(nparams, args);
    params = PySequence_Concat(ret, arg_list);
    Py_DECREF(ret);
    Py_DECREF(arg_list);
    ret = PyObject_CallObject(func, params);
    Py_DECREF(params);
    if (ret == NULL) {
        PyErr_Print();
	PyErr_Clear();
	return;
    }
    PyGtk_SetRetType(&args[nparams], return_type, ret);
    Py_DECREF(ret);
}

static
void PyGtk_SignalDestroy(/*gpointer*/ PyObject *func) {
    Py_DECREF(func);
}

static PyObject *gtk__signal_connect(PyObject *self, PyObject *args) {
    PyGtk_Object *obj;
    char *name;
    PyObject *func;
    int signum;

    if (!PyArg_ParseTuple(args, "O!sO:gtk_signal_connect", &PyGtk_Type, &obj,
                    &name, &func))
        return NULL;
    if (!PyCallable_Check(func)) {
        PyErr_SetString(PyExc_TypeError, "third argument must be callable");
        return NULL;
    }
    Py_INCREF(func);
    signum = gtk_signal_connect(PyGtk_Get(obj), name, NULL, func);
    return PyInt_FromLong(signum);
}

static PyObject *gtk__signal_connect_after(PyObject *self, PyObject *args) {
    PyGtk_Object *obj;
    char *name;
    PyObject *func;
    int signum;

    if (!PyArg_ParseTuple(args, "O!sO:gtk_signal_connect_after",
                    &PyGtk_Type, &obj, &name, &func))
        return NULL;
    if (!PyCallable_Check(func)) {
        PyErr_SetString(PyExc_TypeError, "third argument must be callable");
        return NULL;
    }
    Py_INCREF(func);
    signum = gtk_signal_connect_after(PyGtk_Get(obj), name, NULL, func);
    return PyInt_FromLong(signum);
}

%}

// I haven't included gtk_signal_connect_object, since it is difficult
// to implement in C.  Here is a sample implementation in Python:
//
//  def gtk_signal_connect_object(obj, name, func, cobj):
//    class sigfunc:
//      def __init__(self, obj, func): self.obj=obj ; self.func = func
//      def __call__(self, obj, *args): apply(self.func, (self.obj,)+args)
//    return gtk_signal_connect(obj, name, sigfunc(cobj, func))
//
%native(gtk_signal_connect) PyObject *gtk__signal_connect(
        PyObject *self, PyObject *args);
%native(gtk_signal_connect_after) PyObject *gtk__signal_connect_after(
        PyObject *self, PyObject *args);

void gtk_signal_disconnect(GtkObject *o, int signum);
void gtk_signal_disconnect_by_data(GtkObject *o, PyObject *func);
void gtk_signal_handler_block(GtkObject *o, int signum);
void gtk_signal_handler_block_by_data(GtkObject *o, PyObject *func);
void gtk_signal_handler_unblock(GtkObject *o, int signum);
void gtk_signal_handler_unblock_by_data(GtkObject *o, PyObject *func);
void gtk_signal_handlers_destroy(GtkObject *o);

void gtk_signal_emit_stop_by_name(GtkObject *o, char *name);
