/* Rapicorn
 * Copyright (C) 2005 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include "image.hh"
#include "itemimpl.hh"
#include "painter.hh"
#include "factory.hh"
#include <ext/hash_map>

namespace Rapicorn {

struct HashName {
  size_t
  operator() (const char *name) const
  {
    /* 31 bit hash function */
    const char *p = name;
    size_t h = *p;
    if (h)
      for (p += 1; *p != '\0'; p++)
        h = (h << 5) - h + *p;
    return h;
  }
};

struct EqualNames {
  bool
  operator() (const char *name1, const char *name2) const
  {
    return strcmp (name1, name2) == 0;
  }
};

static __gnu_cxx::hash_map <const char*, const uint8*, HashName, EqualNames> builtin_pixstreams; // FIXME: needs locking

const uint8*
Image::lookup_builtin_pixstream (const char *builtin_name)
{
  return builtin_pixstreams[builtin_name];
}

void
Image::register_builtin_pixstream (const char  * const builtin_name,
                                   const uint8 * const builtin_pixstream)
{
  builtin_pixstreams[builtin_name] = builtin_pixstream;
}

static const uint8* get_broken16_pixdata (void);

class ImageImpl : public virtual ItemImpl, public virtual Image {
  const PixelImage *m_pimage;
  int               m_xoffset, m_yoffset;
public:
  explicit ImageImpl() :
    m_pimage (NULL),
    m_xoffset (0),
    m_yoffset (0)
  {}
  void
  reset ()
  {
    m_xoffset = m_yoffset = 0;
    if (m_pimage)
      {
        const PixelImage *mpi = m_pimage;
        m_pimage = NULL;
        mpi->unref();
      }
    invalidate();
  }
  ~ImageImpl()
  {}
  virtual void          image_file        (const String &filename)      { load_image_file (filename); }
  virtual void
  builtin_pixstream (const String &builtin_name)
  {
    const uint8 *pixstream = lookup_builtin_pixstream (builtin_name.c_str());
    if (!pixstream)
      pixstream = get_broken16_pixdata();
    load_pixstream (pixstream);
  }
  virtual ErrorType
  load_image_file (const String &filename)
  {
    reset();
    ErrorType error = UNKNOWN_FORMAT;
    if (error)
      load_pixstream (get_broken16_pixdata());
    return error;
  }
  virtual ErrorType
  load_pixstream (const uint8 *gdkp_pixstream)
  {
    reset();
    ErrorType error = NONE;
    const PixelImage *pimage = pixel_image_from_pixstream (gdkp_pixstream, &error);
    if (pimage)
      {
        pimage->ref_sink();
        error = load_pixel_image (pimage);
        pimage->unref();
      }
    else
      load_pixstream (get_broken16_pixdata());
    return error;
  }
  virtual ErrorType
  load_pixel_image (const PixelImage *pimage)
  {
    reset();
    if (!pimage)
      {
        /* no setting a "broken" image here, to allow
         * load_pixel_image(NULL) for setting up an empty image
         */
        return EXCESS_DIMENSIONS;
      }
    const PixelImage *oi = m_pimage;
    m_pimage = pimage;
    m_pimage->ref_sink();
    if (oi)
      oi->unref();
    return NONE;
  }
protected:
  virtual void
  size_request (Requisition &requisition)
  {
    if (m_pimage)
      {
        requisition.width += m_pimage->width();
        requisition.height += m_pimage->height();
      }
  }
  virtual void
  size_allocate (Allocation area)
  {
    allocation (area);
    if (m_pimage)
      {
        m_xoffset = (area.width - m_pimage->width()) / 2;
        m_yoffset = (area.height - m_pimage->height()) / 2;
      }
    else
      m_xoffset = m_yoffset = 0;
  }
public:
  void
  render (Display &display)
  {
    if (m_pimage)
      {
        int ix = allocation().x + m_xoffset, iy = allocation().y + m_yoffset;
        int ih = m_pimage->height();
        display.push_clip_rect (ix, iy, m_pimage->width(), ih);
        Plane &plane = display.create_plane();
        for (int y = plane.ystart(); y < plane.ybound(); y++)
          {
            uint nth = y - iy;
            const uint32 *row = m_pimage->row (ih - (1 + nth));
            row += plane.xstart() - ix;
            uint32 *span = plane.poke_span (plane.xstart(), y, plane.width());
            uint32 *limit = span + plane.width();
            while (span < limit)
              *span++ = Color (*row++).premultiplied();
          }
        display.pop_clip_rect();
      }
  }
  virtual const PropertyList&
  list_properties()
  {
    static Property *properties[] = {
      MakeProperty (Image, image_file, _("Image Filename"), _("Load an image from a file, only PNG images can be loaded."), "", "rw"),
      MakeProperty (Image, builtin_pixstream, _("Builtin Pixstream"), _("Load an image from a builtin pixel stream."), "", "rw"),
    };
    static const PropertyList property_list (properties, Item::list_properties());
    return property_list;
  }
};
static const ItemFactory<ImageImpl> image_factory ("Rapicorn::Image");

class PixelImageImpl : virtual public PixelImage {
  uint m_width, m_height;
  uint32 *m_pixels;
public:
  PixelImageImpl() :
    m_width (0),
    m_height (0),
    m_pixels (NULL)
  {}
  ~PixelImageImpl()
  {
    if (m_pixels)
      delete[] m_pixels;
    m_pixels = NULL;
  }
  void
  resize (uint iwidth,
          uint iheight)
  {
    if (m_pixels)
      delete[] m_pixels;
    m_width = iwidth;
    m_height = iheight;
    m_pixels = new uint32[iwidth * iheight];
  }
  uint32*
  pixels ()
  {
    return m_pixels;
  }
  virtual int           width  () const       { return m_width; }
  virtual int           height () const       { return m_height; }
  virtual const uint32* row    (uint y) const { return m_pixels + y * width(); }
};

static Image::ErrorType
fill_pixel_image_from_pixstream (bool                has_alpha,
                                 bool                rle_encoded,
                                 uint                pixdata_width,
                                 uint                pixdata_height,
                                 const uint8        *encoded_pixdata,
                                 PixelImageImpl     &pimage)
{
  if (pixdata_width < 1 || pixdata_width > 16777216 ||
      pixdata_height < 1 || pixdata_height > 16777216)
    return Image::EXCESS_DIMENSIONS;
  uint bpp = has_alpha ? 4 : 3;
  if (!encoded_pixdata)
    return Image::DATA_CORRUPT;

  pimage.resize (pixdata_width, pixdata_height);

  if (rle_encoded)
    {
      const uint8 *rle_buffer = encoded_pixdata;
      uint32 *image_buffer = pimage.pixels();
      uint32 *image_limit = image_buffer + pixdata_width * pixdata_height;

      while (image_buffer < image_limit)
        {
          uint length = *(rle_buffer++);
          if ((length & ~128) == 0)
            return Image::DATA_CORRUPT;
          bool check_overrun = false;
          if (length & 128)
            {
              length = length - 128;
              check_overrun = image_buffer + length > image_limit;
              if (check_overrun)
                length = image_limit - image_buffer;
              if (bpp < 4)
                while (length--)
                  *image_buffer++ = (0xff << 24) | (rle_buffer[0] << 16) | (rle_buffer[1] << 8) | rle_buffer[2];
              else
                while (length--)
                  *image_buffer++ = (rle_buffer[3] << 24) | (rle_buffer[0] << 16) | (rle_buffer[1] << 8) | rle_buffer[2];
              rle_buffer += bpp;
            }
          else
            {
              check_overrun = image_buffer + length > image_limit;
              if (check_overrun)
                length = image_limit - image_buffer;
              if (bpp < 4)
                while (length--)
                  {
                    *image_buffer++ = (0xff << 24) | (rle_buffer[0] << 16) | (rle_buffer[1] << 8) | rle_buffer[2];
                    rle_buffer += 3;
                  }
              else
                while (length--)
                  {
                    *image_buffer++ = (rle_buffer[3] << 24) | (rle_buffer[0] << 16) | (rle_buffer[1] << 8) | rle_buffer[2];
                    rle_buffer += 4;
                  }
            }
          if (check_overrun)
            return Image::DATA_CORRUPT;
        }
    }
  else
    for (uint y = 0; y < pixdata_height; y++)
      {
        uint32 *row = pimage.pixels() + y * pimage.width();
        uint length = pixdata_width;
        if (bpp < 4)
          while (length--)
            {
              *row++ = (0xff << 24) | (encoded_pixdata[0] << 16) | (encoded_pixdata[1] << 8) | encoded_pixdata[2];
              encoded_pixdata += 3;
            }
        else
          while (length--)
            {
              *row++ = (encoded_pixdata[3] << 24) | (encoded_pixdata[0] << 16) | (encoded_pixdata[1] << 8) | encoded_pixdata[2];
              encoded_pixdata += 4;
            }
      }

  return Image::NONE;
}

static inline const uint8*
get_uint32 (const uint8 *stream, uint *result)
{
  *result = (stream[0] << 24) + (stream[1] << 16) + (stream[2] << 8) + stream[3];
  return stream + 4;
}

const PixelImage*
Image::pixel_image_from_pixstream (const uint8 *gdkp_pixstream,
                                   ErrorType   *error_type)
{
  ErrorType dummy = DATA_CORRUPT;
  error_type = error_type ? error_type : &dummy;
  return_val_if_fail (gdkp_pixstream != NULL, NULL);

  *error_type = UNKNOWN_FORMAT;
  const uint8 *s = gdkp_pixstream;
  if (strncmp ((char*) s, "GdkP", 4) != 0)
    return NULL;
  s += 4;

  *error_type = UNKNOWN_FORMAT;
  uint len;
  s = get_uint32 (s, &len);
  if (len < 24)
    return NULL;

  *error_type = UNKNOWN_FORMAT;
  uint type;
  s = get_uint32 (s, &type);
  if (type != 0x02010001 &&     /* RLE/8bit/RGB */
      type != 0x01010001 &&     /* RAW/8bit/RGB */
      type != 0x02010002 &&     /* RLE/8bit/RGBA */
      type != 0x01010002)       /* RAW/8bit/RGBA */
    return NULL;

  *error_type = EXCESS_DIMENSIONS;
  uint rowstride, width, height;
  s = get_uint32 (s, &rowstride);
  s = get_uint32 (s, &width);
  s = get_uint32 (s, &height);
  if (width < 1 || height < 1)
    return NULL;

  PixelImageImpl *pimage = new PixelImageImpl();
  *error_type = fill_pixel_image_from_pixstream ((type & 0xff) == 0x02,
                                                 (type >> 24) == 0x02,
                                                 width, height, s,
                                                 *pimage);
  if (!*error_type)
    return pimage;
  pimage->ref_sink();
  pimage->unref();
  return NULL;
}

static const uint8*
get_broken16_pixdata (void)
{
  /* GdkPixbuf RGBA C-Source image dump 1-byte-run-length-encoded */
#ifdef __SUNPRO_C
#pragma align 4 (broken16_pixdata)
#endif
#ifdef __GNUC__
  static const uint8 broken16_pixdata[] __attribute__ ((__aligned__ (4))) = 
#else
    static const uint8 broken16_pixdata[] = 
#endif
    { ""
      /* Pixbuf magic (0x47646b50) */
      "GdkP"
      /* length: header (24) + pixel_data (485) */
      "\0\0\1\375"
      /* pixdata_type (0x2010002) */
      "\2\1\0\2"
      /* rowstride (64) */
      "\0\0\0@"
      /* width (16) */
      "\0\0\0\20"
      /* height (16) */
      "\0\0\0\20"
      /* pixel_data: */
      "\221\0\0\0\0\203\0\0\0\377\2>\0\10\377\0\0\0\377\202\0\0\0\0\1\253\0"
      "\30\\\203\0\0\0\377\2\0\0\0\177\0\0\0\77\203\0\0\0\0\1\0\0\0\377\202"
      "\377\377\377\377\2\377r\205\377\377Jc\262\202\0\0\0\0\7\377D_\301\377"
      "\201\222\377\377\377\377\377\0\0\0\377\300\300\300\377\0\0\0\177\0\0"
      "\0\77\202\0\0\0\0\1\0\0\0\377\202\377\377\377\377\2\377\"A\377\377Kd"
      "D\202\0\0\0\0\7\377\32;\367\377\317\325\377\377\377\377\377\0\0\0\377"
      "\377\377\377\377\300\300\300\377\0\0\0\177\202\0\0\0\0\4\0\0\0\377\377"
      "\377\377\377\377\324\332\377\377\25""6\371\202\0\0\0\0\2\377Kd=\377\33"
      ";\377\202\377\377\377\377\204\0\0\0\377\202\0\0\0\0\4\0\0\0\377\377\377"
      "\377\377\377\206\227\377\377A\\\307\202\0\0\0\0\2\377.K\224\377k\177"
      "\377\205\377\377\377\377\1\0\0\0\377\202\0\0\0\0\11\0\0\0\377\377\377"
      "\377\377\377\257\272\377\377\5(\363\377\0$8\0\0\0\0\377\0$P\377\5(\363"
      "\377\305\315\370\204\377\377\377\377\1\0\0\0\377\202\0\0\0\0\1\0\0\0"
      "\377\202\377\377\377\377\7\377\273\305\366\377\4'\366\377\0$R\0\0\0\0"
      "\377\0$A\377\2%\364\377\247\264\360\203\377\377\377\377\1\0\0\0\377\202"
      "\0\0\0\0\1\0\0\0\377\203\377\377\377\377\7\377\325\333\376\377\12,\365"
      "\377\0$w\0\0\0\0\377\0$)\377\2%\355\377|\216\350\202\377\377\377\377"
      "\1\0\0\0\377\202\0\0\0\0\1\0\0\0\377\204\377\377\377\377\11\377\366\367"
      "\377\377(F\352\377\1%\251\377\0$\10\377\0$\11\377\3'\310\377F`\350\377"
      "\367\370\377\0\0\0\377\202\0\0\0\0\1\0\0\0\377\204\377\377\377\377\11"
      "\377\365\366\377\3777S\361\377\4'\256\377\0$\6\377\0$\12\377\5(\301\377"
      "F`\354\377\371\371\377\0\0\0\377\202\0\0\0\0\1\0\0\0\377\203\377\377"
      "\377\377\12\377\356\360\377\377+H\355\377\1%\252\377\0$\2\377\0$\21\377"
      "\6)\312\377Jd\357\377\375\375\377\377\377\377\377\0\0\0\377\202\0\0\0"
      "\0\1\0\0\0\377\202\377\377\377\377\10\377\356\360\377\377\35=\360\377"
      "\1%\216\377\0$\1\377\0$\21\377\3'\327\377h}\357\377\376\376\377\202\377"
      "\377\377\377\1\0\0\0\377\202\0\0\0\0\11\0\0\0\377\377\377\377\377\377"
      "\340\344\377\377\15/\365\377\2%z\0\0\0\0\377\0$\37\377\3&\353\377\200"
      "\222\364\204\377\377\377\377\1\0\0\0\377\202\0\0\0\0\7\0\0\0\377&\0\5"
      "\377\0\0\0\377\326\0\36p\0\0\0\0\377\0$&\370\0#\351\207\0\0\0\377\221"
      "\0\0\0\0"
    };
  return broken16_pixdata;
}

} // Rapicorn
