/* GLIB - Library of useful routines for C programming
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * gmain.c: Main loop abstraction, timeouts, and idle functions
 * Copyright 1998 Owen Taylor
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <glib.h>
#include "gmain.h"
#include <sys/time.h>
#include <unistd.h>

/* Types */

typedef struct _GIdleData GIdleData;
typedef struct _GTimeoutData GTimeoutData;
typedef struct _GSource GSource;

typedef enum {
  G_SOURCE_NEED_DESTROY,
  G_SOURCE_READY,
  G_SOURCE_IN_CALL
} GSourceFlags;

struct _GSource {
  gint priority;
  guint tag;
  GSourceFlags flags;
  GSourceFuncs *funcs;
  gpointer source_data;
  gpointer user_data;
  GDestroyNotify notify;
};

struct _GMainLoop {
  gboolean flag;
};

struct _GIdleData {
  GSourceFunc callback;
};

struct _GTimeoutData {
  GTimeVal    expiration;
  gint        interval;
  GSourceFunc callback;
};

/* Forward declarations */

static guint    g_main_find_poll_index (gint      priority);
static gboolean g_timeout_prepare      (gpointer  source_data, 
					GTimeVal *current_time,
					gint     *timeout);
static gboolean g_timeout_check        (gpointer  source_data,
					GTimeVal *current_time);
static gboolean g_timeout_dispatch     (gpointer  source_data,
					GTimeVal *current_time,
					gpointer  user_data);
static gboolean g_idle_prepare         (gpointer  source_data, 
					GTimeVal *current_time,
					gint     *timeout);
static gboolean g_idle_check           (gpointer  source_data,
					GTimeVal *current_time);
static gboolean g_idle_dispatch        (gpointer  source_data,
					GTimeVal *current_time,
					gpointer  user_data);

/* Data */

static GSList *pending_dispatches = NULL;
static GList *source_list = NULL;

static GSourceFuncs timeout_funcs = {
  g_timeout_prepare,
  g_timeout_check,
  g_timeout_dispatch,
  (GDestroyNotify)g_free
};

static GSourceFuncs idle_funcs = {
  g_idle_prepare,
  g_idle_check,
  g_idle_dispatch,
  (GDestroyNotify)g_free
};

static GArray *fd_array = NULL;
static GArray *priority_array = NULL;

static GPollFunc poll_func = (GPollFunc)poll;

/* Hooks for adding to the main loop */

/* Use knowledge of insert_sorted algorithm here to make
 * sure we insert at the end of equal priority items
 */
static gint
g_source_compare (gconstpointer a, gconstpointer b)
{
  const GSource *source_a = a;
  const GSource *source_b = a;

  return (source_a->priority < source_b->priority) ? -1 : 1;
}

guint 
g_source_add (gint           priority, 
	      GSourceFuncs  *funcs,
	      gpointer       source_data, 
	      gpointer       user_data,
	      GDestroyNotify notify)
{
  static guint next_tag = 0;
  GSource *source = g_new (GSource, 1);

  source->priority = priority;
  source->tag = ++next_tag;
  source->flags = 0;
  source->funcs = funcs;
  source->source_data = source_data;
  source->user_data = user_data;
  source->notify = notify;
  
  source_list = g_list_insert_sorted (source_list, source, g_source_compare);

  return source->tag;
}

static void 
g_source_remove_node (GList *node)
{
  GSource *source = node->data;

  if (source->flags & G_SOURCE_IN_CALL)
    source->flags |= G_SOURCE_NEED_DESTROY;
  else
    {
      source_list = g_list_remove_link (source_list, node);
      if (pending_dispatches)
	pending_dispatches = g_slist_remove (pending_dispatches, node->data);

      if (source->notify)
	source->notify (source->user_data);
      source->funcs->destroy (source->source_data);
      
      g_free (source);
      g_list_free_1 (node);
    }
}

void 
g_source_remove (guint tag)
{
  GList *tmp_list = source_list;

  while (tmp_list)
    {
      GSource *source = tmp_list->data;
      GList *node = tmp_list;
      tmp_list = tmp_list->next;

      if (source->tag == tag)
	{
	  g_source_remove_node (node);
	  break;
	}
    }
}

void 
g_source_remove_by_user_data (gpointer user_data)
{
  GList *tmp_list = source_list;

  while (tmp_list)
    {
      GSource *source = tmp_list->data;
      GList *node = tmp_list;
      tmp_list = tmp_list->next;

      if (source->user_data == user_data)
	{
	  g_source_remove_node (node);
	  break;
	}
    }
}

void 
g_source_remove_by_source_data (gpointer source_data)
{
  GList *tmp_list = source_list;

  while (tmp_list)
    {
      GSource *source = tmp_list->data;
      GList *node = tmp_list;
      tmp_list = tmp_list->next;

      if (source->source_data == source_data)
	{
	  g_source_remove_node (node);
	  break;
	}
    }
}

void g_get_current_time (GTimeVal *result)
{
  gettimeofday ((struct timeval *)result, NULL);
}

/* Running the main loop */

static void
g_main_dispatch (GTimeVal *current_time)
{
  while (pending_dispatches != NULL)
    {
      gboolean need_destroy;
      
      GSource *source = pending_dispatches->data;
      pending_dispatches = g_slist_remove_link (pending_dispatches, pending_dispatches);
      
      source->flags |= G_SOURCE_IN_CALL;
      need_destroy = !source->funcs->dispatch (source->source_data, 
					       current_time,
					       source->user_data);
      source->flags &= ~G_SOURCE_IN_CALL;

      if (need_destroy || source->flags & G_SOURCE_NEED_DESTROY)
	{
	  GList *node = g_list_find (source_list, source);
	  g_source_remove_node (node);
	}
    }
}

/* Run a single iteration of the mainloop, or, if !dispatch
 * check to see if any events need dispatching, but don't
 * run the loop.
 */
static gboolean
g_main_iterate (gboolean block, gboolean dispatch)
{
  GTimeVal current_time;
  gint nready = 0;
  gint current_priority = 0;
  GList *tmp_list;
  gint timeout;
  gint npoll;

  g_return_val_if_fail (!block || dispatch, FALSE);

  if (!fd_array)
    {
      fd_array = g_array_new (FALSE, FALSE, sizeof(GPollFD));
      priority_array = g_array_new (FALSE, FALSE, sizeof(gint));
    }

  g_get_current_time (&current_time);
  
  /* If recursing, finish up current dispatch, before starting over */
  if (pending_dispatches)
    {
      if (dispatch)
	g_main_dispatch (&current_time);
      
      return TRUE;
    }

  /* Prepare all sources */

  timeout = block ? -1 : 0;
  
  tmp_list = source_list;
  while (tmp_list)
    {
      GSource *source = tmp_list->data;
      gint source_timeout;
      
      tmp_list = tmp_list->next;

      if ((nready > 0) && (source->priority > current_priority))
	break;

      if (source->flags & G_SOURCE_IN_CALL)
	continue;

      if (source->funcs->prepare (source->source_data,
				  &current_time,
				  &source_timeout))
	{
	  if (!dispatch)
	    return TRUE;
	  else
	    {
	      source->flags |= G_SOURCE_READY;
	      nready++;
	      current_priority = source->priority;
	      timeout = 0;
	    }
	}

      if (source_timeout >= 0)
	{
	  if (timeout < 0)
	    timeout = source_timeout;
	  else
	    timeout = MIN (timeout, source_timeout);
	}
    }

  /* poll(), if necessary */

  if (nready > 0)
    npoll = g_main_find_poll_index (current_priority);
  else
    npoll = fd_array->len;

  if ((npoll > 0) || (timeout > 0))
    {
      gint i;

      for (i=0; i<npoll; i++)
	g_array_index (fd_array, GPollFD, i).revents = 0;
      (*poll_func) ((GPollFD *)fd_array->data, npoll, timeout);
      if (timeout != 0)
	g_get_current_time (&current_time);
    }

  /* Check to see what sources need to be dispatched */

  tmp_list = source_list;
  nready = 0;
  
  while (tmp_list)
    {
      GSource *source = tmp_list->data;
      tmp_list = tmp_list->next;

      if ((nready > 0) && (source->priority > current_priority))
	break;

      if (source->flags & G_SOURCE_IN_CALL)
	continue;

      if ((source->flags & G_SOURCE_READY) ||
	  source->funcs->check (source->source_data,
				 &current_time))
	{
	  if (dispatch)
	    {
	      source->flags &= ~G_SOURCE_READY;
	      pending_dispatches = g_slist_prepend (pending_dispatches, source);
	      current_priority = source->priority;
	      nready++;
	    }
	  else
	    return TRUE;
	}
    }

  /* Now invoke the callbacks */

  if (pending_dispatches)
    {
      pending_dispatches = g_slist_reverse (pending_dispatches);
      g_main_dispatch (&current_time);
      return TRUE;
    }
  else
    return FALSE;
}

/* See if any events are pending
 */
gboolean 
g_main_pending ()
{
  return g_main_iterate (FALSE, FALSE);
}

/* Run a single iteration of the mainloop. If block is FALSE,
 * will never block
 */
void g_main_iteration (gboolean block)
{
  g_main_iterate (block, TRUE);
}

GMainLoop *
g_main_new ()
{
  GMainLoop *result = g_new (GMainLoop, 1);
  result->flag = FALSE;

  return result;
}

void 
g_main_run (GMainLoop *loop)
{
  loop->flag = FALSE;
  while (!loop->flag)
    g_main_iterate (TRUE, TRUE);
}

void 
g_main_quit (GMainLoop *loop)
{
  loop->flag = TRUE;
}

void 
g_main_destroy (GMainLoop *loop)
{
  g_free (loop);
}

/* Find the first index where priority < priority_array[i] */
static guint
g_main_find_poll_index (int priority)
{
  guint first = 0;
  guint last;
  guint mid;

  if ((priority_array->len == 0) ||
      (priority < g_array_index (priority_array, int, 1)))
    return 0;

  last = priority_array->len - 1;
  if (priority >= g_array_index (priority_array, int, last))
    return last + 1;
  
  /* invariant - priority_array[first] <= priority < priority_array[last] */

  while (1)
    {
      mid = (first + last)/2;
      if (mid == first)
	break;
      else if (g_array_index (priority_array, int, mid) <= priority)
	first = mid;
      else
	last = mid;
    }

  return last;
}

GPollFD *
g_main_poll_add (gint priority, gint fd, gshort events)
{
  gint i;
  GPollFD newfd;

  newfd.fd = fd;
  newfd.events = events;

  if (!fd_array)
    {
      fd_array = g_array_new (FALSE, FALSE, sizeof(GPollFD));
      priority_array = g_array_new (FALSE, FALSE, sizeof(gint));
    }

  i = g_main_find_poll_index (priority);

  g_array_insert_vals (priority_array, i, &priority, 1);
  g_array_insert_vals (fd_array, i, &newfd, 1);

  return &g_array_index (fd_array, GPollFD, i);
}

void 
g_main_poll_remove (GPollFD *fd)
{
  g_return_if_fail (fd_array != NULL);
  
  g_array_remove_index (priority_array, fd - (GPollFD *)fd_array->data);
  g_array_remove_index (fd_array, fd - (GPollFD *)fd_array->data);
}

void 
g_main_set_poll_func (GPollFunc func)
{
  if (func)
    poll_func = func;
  else
    poll_func = (GPollFunc)poll;
}

/* Timeouts */

static gboolean 
g_timeout_prepare  (gpointer source_data, 
		    GTimeVal *current_time,
		    gint    *timeout)
{
  glong msec;
  GTimeoutData *data = source_data;

  msec = (data->expiration.tv_sec  - current_time->tv_sec) * 1000 +
         (data->expiration.tv_usec - current_time->tv_usec) / 1000;

  *timeout = (msec <= 0) ? 0 : msec;

  return (msec <= 0);
}

static gboolean 
g_timeout_check    (gpointer source_data,
		    GTimeVal *current_time)
{
  GTimeoutData *data = source_data;

  return (data->expiration.tv_sec < current_time->tv_sec) ||
         ((data->expiration.tv_sec == current_time->tv_sec) &&
	  (data->expiration.tv_usec <= current_time->tv_usec));
}

static gboolean
g_timeout_dispatch (gpointer source_data, 
		    GTimeVal *current_time,
		    gpointer user_data)
{
  GTimeoutData *data = source_data;

  if (data->callback(user_data))
    {
      data->expiration.tv_sec = current_time->tv_sec;
      data->expiration.tv_usec = current_time->tv_usec + data->interval * 1000;
      if (data->expiration.tv_usec >= 1000000)
	{
	  data->expiration.tv_usec -= 1000000;
	  data->expiration.tv_sec++;
	}
      return TRUE;
    }
  else
    return FALSE;
}

guint 
g_timeout_add_full (gint           priority,
		    guint          interval, 
		    GSourceFunc    function,
		    gpointer       data,
		    GDestroyNotify notify)
{
  GTimeoutData *timeout_data = g_new (GTimeoutData, 1);

  timeout_data->interval = interval;
  timeout_data->callback = function;
  g_get_current_time (&timeout_data->expiration);

  timeout_data->expiration.tv_usec += timeout_data->interval * 1000;
  if (timeout_data->expiration.tv_usec >= 1000000)
    {
      timeout_data->expiration.tv_usec -= 1000000;
      timeout_data->expiration.tv_sec--;
    }

  return g_source_add (priority, &timeout_funcs, timeout_data, data, notify);
}

guint 
g_timeout_add (guint32        interval,
	       GSourceFunc    function,
	       gpointer       data)
{
  return g_timeout_add_full (0, interval, function, data, NULL);
}

/* Idle functions */

static gboolean 
g_idle_prepare  (gpointer source_data, 
		 GTimeVal *current_time,
		 gint     *timeout)
{
  timeout = 0;
  return TRUE;
}

static gboolean 
g_idle_check    (gpointer  source_data,
		 GTimeVal *current_time)
{
  return TRUE;
}

static gboolean
g_idle_dispatch (gpointer source_data, 
		 GTimeVal *current_time,
		 gpointer user_data)
{
  GIdleData *data = source_data;

  return (*data->callback)(user_data);
}

guint 
g_idle_add_full (gint          priority,
		 GSourceFunc    function,
		 gpointer       data,
		 GDestroyNotify notify)
{
  GIdleData *idle_data = g_new (GIdleData, 1);

  idle_data->callback = function;

  return g_source_add (priority, &idle_funcs, idle_data, data, notify);
}

guint 
g_idle_add (GSourceFunc    function,
	    gpointer       data)
{
  return g_idle_add_full (0, function, data, NULL);
}
