/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include "buffer_flat.h"


static void
gimp_flat_buffer_delete (GimpBuffer * buffer)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  /* free any image memory we hold */
  gimp_memptr_uninit (&flat_buffer->data);

  /* clean up the base class */
  gimp_buffer_uninit (GIMP_BUFFER (flat_buffer));
  
  /* delete the object */
  g_free (flat_buffer);
}


static gboolean
gimp_flat_buffer_focus (GimpBuffer  *buffer,
                        GimpPortion *portion,
                        gint         x,
                        gint         y)
{
  if ((x >= 0) && (x < buffer->width) &&
      (y >= 0) && (y < buffer->height))
    {
      /* ok, the point is within the buffer */
      portion->x = x;
      portion->y = y;
      portion->x0 = 0;
      portion->y0 = 0;
      portion->x1 = buffer->width;
      portion->y1 = buffer->height;
    }
  else
    {
      /* bad, the point is not in the buffer */
      portion->x = 0;
      portion->y = 0;
      portion->x0 = 0;
      portion->y0 = 0;
      portion->x1 = 0;
      portion->y1 = 0;
    }

  return TRUE;
}

#define X (&flat_buffer->data)

static gboolean
gimp_flat_buffer_alloc (GimpBuffer   *buffer,
                        GimpPortion  *portion,
                        Alloc         how)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);
  gint size = buffer->width * buffer->height * buffer->depth;

  switch (how)
    {
    case ALLOC_ALLOC:
      if (gimp_memptr_alloc (X, size) == TRUE)
        {
          return TRUE;
        }
      g_warning ("alloc of flatbuffer failed");
      break;

    case ALLOC_UNALLOC:
      if (gimp_memptr_unalloc (X) == TRUE)
        {
          flat_buffer->valid = FALSE;
          return TRUE;
        }
      g_warning ("unalloc of flatbuffer failed");
      break;

    case ALLOC_NONE:
      g_warning ("bad value");
      break;
    }

  return FALSE;
}


static gboolean
gimp_flat_buffer_validate (GimpBuffer  *buffer,
                           GimpPortion *portion, 
                           Validate     how)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  switch (how)
    {
    case VALIDATE_VALIDATE:
      if (flat_buffer->valid == TRUE)
        return TRUE;

      if (! gmp_is_alloced (X))
        {
          if (buffer->autoalloc == FALSE)
            return FALSE;
          
          if (gimp_flat_buffer_alloc (buffer, portion,
                                      ALLOC_ALLOC) != TRUE)
            {
              g_warning ("flatbuffer autoalloc failed");
              return FALSE;
            }
        }

      if (gimp_memptr_use (X, TRUE) != TRUE)
        {
          g_warning ("flatbuffer validate use failed");
          return FALSE;
        }

      /* HOOK: PERFORM VALIDATION HERE */
      memset (X->data, 0,
              (buffer->width *
               buffer->height *
               buffer->depth));
      
      if (gimp_memptr_unuse (X) != TRUE)
        {
          g_warning ("flatbuffer validate unuse failed");
          return FALSE;
        }

      flat_buffer->valid = TRUE;
      return TRUE;


    case VALIDATE_INVALIDATE:
      if (gmp_usecount (X) != 0)
        {
          g_warning ("tried to invalidate an in-use flatbuffer");
          return FALSE;
        }

      flat_buffer->valid = FALSE;
      return TRUE;


    case VALIDATE_NONE:
      g_warning ("bad value");
      break;
    }

  return FALSE;
}



static gboolean
gimp_flat_buffer_use (GimpBuffer  *buffer,
                      GimpPortion *portion, 
                      Use          how)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  switch (how)
    {
    case USE_NONE:
    case USE_RELEASE:
      break;

    case USE_READ:
    case USE_UPDATE:
    case USE_WRITE:
      if (flat_buffer->valid != TRUE)
        {
          if (buffer->autovalidate == FALSE)
            return FALSE;
          
          if (gimp_flat_buffer_validate (buffer, portion,
                                         VALIDATE_VALIDATE) != TRUE)
            {
              g_warning ("flatbuffer autovalidate failed");
              return FALSE;
            }
        }
      break;
    }

  switch (how)
    {
    case USE_READ:
    case USE_UPDATE:
      if (gimp_memptr_use (X, FALSE) == TRUE)
        return TRUE;
      g_warning ("flatbuffer clean use failed");
      break;

    case USE_WRITE:
      if (gimp_memptr_use (X, TRUE) == TRUE)
        return TRUE;
      g_warning ("flatbuffer dirty use failed");
      break;

    case USE_RELEASE:
      if (gimp_memptr_unuse (X) == TRUE)
        return TRUE;
      g_warning ("flatbuffer unuse failed");
      break;

    case USE_NONE:
      g_warning ("bad value");
      break;
    }

  return FALSE;
}


static gboolean
gimp_flat_buffer_query (GimpBuffer    *buffer,
                        GimpPortion   *portion,
                        GimpMemStatus *status)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  status->alloced   = (gmp_is_alloced (X) ? TRUE : FALSE);
  status->valid     = flat_buffer->valid;
  status->usecount  = (gmp_usecount (X));

  return TRUE;
}


static gboolean
gimp_flat_buffer_data (GimpBuffer      *buffer,
                       GimpPortion     *portion,
                       GimpPixelArray  *array)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  /* describe the memory layout */
  array->tag       = buffer->tag;
  array->width     = buffer->width - portion->x;
  array->height    = buffer->height - portion->y;
  array->pixstride = buffer->depth;
  array->rowstride = buffer->depth * buffer->width;

  /* get a pointer to the memory */
  array->data = X;

  /* and save an offset to the first pixel */
  array->offset = (portion->y * array->rowstride) +
                  (portion->x * array->pixstride);

  return TRUE;
}

#undef X


static GimpFlatBufferClass my_class =
{
  {
    STORAGE_FLAT,
    gimp_flat_buffer_delete,
    gimp_flat_buffer_focus,
    gimp_flat_buffer_alloc,
    gimp_flat_buffer_validate,
    gimp_flat_buffer_use,
    gimp_flat_buffer_query,
    gimp_flat_buffer_data
  }
};






GimpFlatBuffer *
gimp_flat_buffer_new (Tag   tag,
                      gint  width,
                      gint  height)
{
  GimpFlatBuffer * flat_buffer;

  /* alloc a new object */
  flat_buffer = g_new (GimpFlatBuffer, 1);

  /* init the base class */
  gimp_buffer_init (GIMP_BUFFER (flat_buffer),
                    tag, width, height);

  /* init the memory pointer */
  gimp_memptr_init (&flat_buffer->data);

  /* it has no valid data yet */
  flat_buffer->valid = FALSE;

  /* remember what sort of object we are */
  GIMP_BUFFER (flat_buffer)->klass = (void*) &my_class;

  /* return the new object */
  return flat_buffer;
}


