/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "memory.h"


/* GimpMemMgr tracks, caches, and swaps all GimpMem objects in the
   system.  Each is assigned a unique GimpMemId */
struct _GimpMemMgr
{
  /* fixme: put a caching swapping memory manager here */
  int foo;
};




/* given a GimpMemId, find the GimpMem */
#define gimp_memmgr_lookup(x) ((GimpMem*)(x))

/* create a new GimpMem */
GimpMemId
gimp_memmgr_new (gint size)
{
  GimpMem * mem = g_new (GimpMem, 1);
  mem->data = g_malloc (size);
  mem->size = size;
  mem->ptrcount = 0;
  mem->usecount = 0;
  mem->sharable = TRUE;
  return (GimpMemId) mem;
}


/* get the current value of the real data pointer */
void *
gimp_memmgr_ptr (GimpMemId id)
{
  GimpMem * mem = gimp_memmgr_lookup (id);
  if (mem == NULL)
    return NULL;
  if (mem->usecount == 0)
    return NULL;
  return mem->data;
}


/* a ptr has just gone from 0->1 uses */
gboolean
gimp_memmgr_use (GimpMemId id,
                 gboolean  dirty)
{
  GimpMem * mem = gimp_memmgr_lookup (id);

  if (mem == NULL)
    return FALSE;

  mem->usecount++;

  if (dirty == TRUE)
    mem->sharable = FALSE;

  return TRUE;
}


/* a ptr has just gone from 1->0 uses */
gboolean
gimp_memmgr_unuse (GimpMemId id)
{
  GimpMem * mem = gimp_memmgr_lookup (id);

  if (mem == NULL)
    return FALSE;

  mem->usecount--;

  if (mem->usecount == 0)
    mem->sharable = TRUE;

  return TRUE;
}


/* make a copy of the mem, return new id */
GimpMemId
gimp_memmgr_dup (GimpMemId id)
{
  GimpMem * mem;
  GimpMem * newmem;
  GimpMemId newid;

  /* get the old mem */
  mem = gimp_memmgr_lookup (id);
  if (mem == NULL)
    return InvalidGimpMemId;

  /* make a new mem */
  newid = gimp_memmgr_new (mem->size);
  if (newid == InvalidGimpMemId)
    return InvalidGimpMemId;

  /* get the new mem */
  newmem = gimp_memmgr_lookup (newid);
  if (newmem == NULL)
    return InvalidGimpMemId;
  
  /* copy the data */
  memcpy (newmem->data, mem->data, mem->size);
  
  /* return the new id */
  return newid;
}


/* add a ptr to mem.  if mem doesn;t want one, make a copy and add the
   ptr to that.  if 'use' is true, ptr is carrying outstanding uses.
   return the id of whoever got the ptr */
GimpMemId
gimp_memmgr_attach (GimpMemId id,
                    gboolean  use)
{
  GimpMem * mem;

  /* get the memory */
  mem = gimp_memmgr_lookup (id);
  if (mem == NULL)
    return InvalidGimpMemId;

  /* check if we need to make a new copy */
  if (mem->sharable != TRUE)
    {
      /* make a copy of the unsharable memory */
      id = gimp_memmgr_dup (id);

      /* get the new memory */
      mem = gimp_memmgr_lookup (id);
      if (mem == NULL)
        return InvalidGimpMemId;
    }

  /* add ptr to mem */
  mem->ptrcount++;
  
  /* add uses to mem */
  if (use == TRUE)
    mem->usecount++;
  
  return id;
}


GimpMemId
gimp_memmgr_detach (GimpMemId id,
                    gboolean  use)
{
  /* get the memory */
  GimpMem * mem = gimp_memmgr_lookup (id);
  if (mem == NULL)
    return InvalidGimpMemId;

  /* drop the use */
  if (use == TRUE)
    {
      mem->usecount--;
      if (mem->usecount == 0)
        mem->sharable = TRUE;
    }

  /* drop the ptr */
  mem->ptrcount--;

  /* check if we dropped last ptr */
  if (mem->ptrcount == 0)
    {
      g_free (mem->data);
      g_free (mem);
    }

  return id;
}


GimpMemId
gimp_memmgr_split (GimpMemId id,
                   gboolean  use)
{
  /* get the memory */
  GimpMem * mem = gimp_memmgr_lookup (id);
  if (mem == NULL)
    return InvalidGimpMemId;

  /* is our memory private already? */
  if (mem->ptrcount == 1)
    return id;

  /* drop the old body */
  id = gimp_memmgr_detach (id, use);
  if (id == InvalidGimpMemId)
    return InvalidGimpMemId;

  /* make a copy */
  id = gimp_memmgr_dup (id);
  if (id == InvalidGimpMemId)
    return InvalidGimpMemId;

  /* attach to new body */
  id = gimp_memmgr_attach (id, use);
  if (id == InvalidGimpMemId)
    return InvalidGimpMemId;

  return id;
}











static void
gimp_memptr_unlink (GimpMemPtr  *ptr)
{
  g_return_if_fail (ptr != NULL);

  /* if the ptr is actually attached to something */
  if (d_is_alloced (ptr))
    {
      /* drop any remaining uses */
      if ((ptr->usecount != 0) &&
          (gimp_memmgr_unuse (ptr->id) != TRUE))
        {
          g_warning ("gimp_memptr_unlink unuse failed");
        }

      /* detach the pointer */
      if (gimp_memmgr_detach (ptr->id, FALSE) == InvalidGimpMemId)
        {
          g_warning ("gimp_memptr_unlink detach failed");
        }
    }

  /* reinit the pointer to null */
  gimp_memptr_init (ptr);
}


void
gimp_memptr_init (GimpMemPtr *ptr)
{
  g_return_if_fail (ptr != NULL);
  
  ptr->id = InvalidGimpMemId;
  ptr->usecount = 0;
  ptr->joined = FALSE;
  ptr->size = 0;
  ptr->data = NULL;
}


void
gimp_memptr_uninit (GimpMemPtr *ptr)
{
  g_return_if_fail (ptr != NULL);
  
  if (d_usecount (ptr) != 0)
    g_warning ("uniniting an in-use gmp");
  
  gimp_memptr_unlink (ptr);  
}


gboolean
gimp_memptr_alloc (GimpMemPtr  *ptr,
                   gint         size)
{
  GimpMemId id;

  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (size > 0, FALSE);
  g_return_val_if_fail (! d_is_alloced (ptr), FALSE);

  /* reserve the memory */
  id = gimp_memmgr_new (size);
  if (id == InvalidGimpMemId)
    {
      g_warning ("gimp_memptr_alloc new failed");
      return FALSE;
    }

  /* attach it */
  id = gimp_memmgr_attach (id, FALSE);
  if (id == InvalidGimpMemId)
    {
      g_warning ("gimp_memptr_alloc attach failed");
      return FALSE;
    }

  ptr->id = id;
  ptr->size = size;
  
  return TRUE;
}


gboolean 
gimp_memptr_unalloc (GimpMemPtr *ptr)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (d_usecount(ptr) == 0, FALSE);

  /* detach our pointer from the memory block */
  gimp_memptr_unlink (ptr);  

  return TRUE;
}


gboolean
gimp_memptr_use (GimpMemPtr *ptr,
                 gboolean    dirty)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (d_is_alloced (ptr), FALSE);

  /* COW if we're writing to possibly shared memory */
  if ((dirty == TRUE) && (d_is_joined (ptr)))
    {
      if (gimp_memptr_split (ptr) != TRUE)
        {
          g_warning ("gimp_memptr_use split failed");
          return FALSE;
        }
    }

  /* if this is the first use for ptr, let the GimpMem know */
  if (ptr->usecount == 0)
    {
      if (gimp_memmgr_use (ptr->id, dirty) != TRUE)
        {
          g_warning ("gimp_memptr_use use failed");
          return FALSE;
        }
    }

  /* increment our local count and cache the data ptr */
  ptr->usecount++;
  ptr->data = gimp_memmgr_ptr (ptr->id);

  return TRUE;
}


gboolean
gimp_memptr_unuse (GimpMemPtr *ptr)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (d_usecount (ptr) != 0, FALSE);

  /* if this is the last use for ptr, let the GimpMem know */
  if (ptr->usecount == 1)
    {
      if (gimp_memmgr_unuse (ptr->id) != TRUE)
        {
          g_warning ("gimp_memptr_unuse unuse failed");
          return FALSE;
        }
      
      /* drop cached data ptr */
      ptr->data = NULL;
    }
  
  /* decrement our local count */
  ptr->usecount--;
  
  return TRUE;
}


gboolean
gimp_memptr_join (GimpMemPtr *ptr,
                  GimpMemPtr *newptr)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (newptr != NULL, FALSE);
  g_return_val_if_fail (d_is_alloced (ptr), FALSE);
  g_return_val_if_fail (! d_is_alloced (newptr), FALSE);

  /* attach newptr to ptr */
  newptr->id = gimp_memmgr_attach (ptr->id, FALSE);

  /* see if it worked */
  if (newptr->id == InvalidGimpMemId)
    {
      g_warning ("gimp_memptr_join attach failed");
      return FALSE;
    }

  /* if we joined, remember to split on COW */
  if (newptr->id == ptr->id)
    {
      ptr->joined = TRUE;
      newptr->joined = TRUE;
    }

  /* cache our size */
  newptr->size = d_size (ptr);
  
  return TRUE;
}


gboolean
gimp_memptr_split (GimpMemPtr *ptr)
{
  g_return_val_if_fail (ptr != NULL, FALSE);

  /* see if we need to split */
  if (d_is_alloced (ptr) && d_is_joined (ptr))
    {
      /* do the split */
      GimpMemId newid = gimp_memmgr_split (ptr->id, (ptr->usecount
                                                     ? TRUE
                                                     : FALSE));
  
      /* see if it worked */
      if (newid == InvalidGimpMemId)
        {
          g_warning ("gimp_memptr_split failed");
          return FALSE;
        }

      /* save new id and remember that we're not joined to anyone
         anymore */
      ptr->id = newid;
      ptr->joined = FALSE;
      if (ptr->usecount)
        ptr->data = gimp_memmgr_ptr (newid);
    }

  return TRUE;
}


