
#include <stdlib.h>
#include <libguile.h>
#include "memory.h"


static long mem_tag;

static SCM
make_mem (void)
{
  GimpMemPtr *mem;
  SCM mem_smob;

  mem = (GimpMemPtr *) scm_must_malloc (sizeof (GimpMemPtr), "mem");
  gimp_memptr_init (mem);

  SCM_NEWCELL (mem_smob);
  SCM_SETCDR (mem_smob, mem);
  SCM_SETCAR (mem_smob, mem_tag);

  return mem_smob;
}

static SCM
mem_p (SCM mem_smob)
{
  if SCM_IMP(mem_smob) return SCM_BOOL_F;
  return SCM_CAR(mem_smob)==mem_tag ? SCM_BOOL_T : SCM_BOOL_F;
}

static SCM
alloc_mem (SCM mem_smob, SCM s_size)
{
  GimpMemPtr * mem;
  int size;
  
  SCM_ASSERT ((SCM_NIMP (mem_smob) && SCM_CAR (mem_smob) == mem_tag),
	      mem_smob, SCM_ARG1, "alloc-mem");
  SCM_ASSERT (SCM_INUMP (s_size),
              s_size,  SCM_ARG2, "alloc-mem");

  mem = (GimpMemPtr *) SCM_CDR (mem_smob);
  size = SCM_INUM (s_size);

  if (gimp_memptr_alloc (mem, size) != TRUE)
    return SCM_BOOL_F;
  
  return SCM_BOOL_T;
}

static SCM
unalloc_mem (SCM mem_smob)
{
  GimpMemPtr * mem;
  
  SCM_ASSERT ((SCM_NIMP (mem_smob) && SCM_CAR (mem_smob) == mem_tag),
	      mem_smob, SCM_ARG1, "unalloc-mem");

  mem = (GimpMemPtr *) SCM_CDR (mem_smob);

  if (gimp_memptr_unalloc (mem) != TRUE)
    return SCM_BOOL_F;
  
  return SCM_BOOL_T;
}

static SCM
use_mem (SCM mem_smob, SCM s_dirty)
{
  GimpMemPtr * mem;
  gboolean dirty;
  
  SCM_ASSERT ((SCM_NIMP (mem_smob) && SCM_CAR (mem_smob) == mem_tag),
	      mem_smob, SCM_ARG1, "use-mem");
  SCM_ASSERT (scm_boolean_p (s_dirty),
              s_dirty,  SCM_ARG2, "use-mem");

  mem = (GimpMemPtr *) SCM_CDR (mem_smob);
  dirty = SCM_FALSEP (s_dirty) ? FALSE : TRUE;
  
  if (gimp_memptr_use (mem, dirty) != TRUE)
    return SCM_BOOL_F;
  
  return SCM_BOOL_T;
}

static SCM
unuse_mem (SCM mem_smob)
{
  GimpMemPtr * mem;
  
  SCM_ASSERT ((SCM_NIMP (mem_smob) && SCM_CAR (mem_smob) == mem_tag),
	      mem_smob, SCM_ARG1, "unuse-mem");

  mem = (GimpMemPtr *) SCM_CDR (mem_smob);

  if (gimp_memptr_unuse (mem) != TRUE)
    return SCM_BOOL_F;
  
  return SCM_BOOL_T;
}

static SCM
join_mem (SCM mem_smob, SCM mem2_smob)
{
  GimpMemPtr * mem, * mem2;
  
  SCM_ASSERT ((SCM_NIMP (mem_smob) && SCM_CAR (mem_smob) == mem_tag),
	      mem_smob, SCM_ARG1, "join-mem");
  SCM_ASSERT ((SCM_NIMP (mem2_smob) && SCM_CAR (mem2_smob) == mem_tag),
	      mem2_smob, SCM_ARG2, "join-mem");

  mem = (GimpMemPtr *) SCM_CDR (mem_smob);
  mem2 = (GimpMemPtr *) SCM_CDR (mem2_smob);
  
  if (gimp_memptr_join (mem, mem2) != TRUE)
    return SCM_BOOL_F;
  
  return SCM_BOOL_T;
}

static SCM
split_mem (SCM mem_smob)
{
  GimpMemPtr * mem;
  
  SCM_ASSERT ((SCM_NIMP (mem_smob) && SCM_CAR (mem_smob) == mem_tag),
	      mem_smob, SCM_ARG1, "split-mem");

  mem = (GimpMemPtr *) SCM_CDR (mem_smob);
  
  if (gimp_memptr_split (mem) != TRUE)
    return SCM_BOOL_F;
  
  return SCM_BOOL_T;
}

static SCM
info_mem (SCM mem_smob)
{
  GimpMemPtr * mem;
  SCM head, id, h_usecount, joined, h_data;
  SCM body, data, size, ptrcount, usecount, sharable;
  
  SCM_ASSERT ((SCM_NIMP (mem_smob) && SCM_CAR (mem_smob) == mem_tag),
	      mem_smob, SCM_ARG1, "info-mem");

  mem = (GimpMemPtr *) SCM_CDR (mem_smob);

  if (mem->mem == NULL)
    {
      id         = scm_makfrom0str("");
      h_usecount = SCM_MAKINUM(0);
      joined     = SCM_BOOL_F;
      h_data     = scm_makfrom0str("");
      
      data       = scm_makfrom0str("");
      size       = SCM_MAKINUM(0);
      ptrcount   = SCM_MAKINUM(0);
      usecount   = SCM_MAKINUM(0);
      sharable   = SCM_BOOL_F;
    }
  else
    {
      char buf[32];
      
      sprintf (buf, "%p", mem->mem);
      id         = scm_makfrom0str (buf);
      h_usecount = SCM_MAKINUM(mem->usecount);
      joined     = (mem->joined) ? SCM_BOOL_T : SCM_BOOL_F;
      sprintf (buf, "%p", mem->data);
      h_data     = scm_makfrom0str (buf);

      sprintf (buf, "%p", mem->mem->data);
      data       = scm_makfrom0str (buf);
      size       = SCM_MAKINUM(mem->mem->size);
      ptrcount   = SCM_MAKINUM(mem->mem->ptrcount);
      usecount   = SCM_MAKINUM(mem->mem->usecount);
      sharable   = ((mem->mem->sharable) ? SCM_BOOL_T : SCM_BOOL_F);
    }
  
  head = scm_listify (id, h_usecount, joined, h_data,
                      SCM_UNDEFINED);
  body = scm_listify (data, size, ptrcount, usecount, sharable,
                      SCM_UNDEFINED);
  return scm_listify (head, body, SCM_UNDEFINED);
}


static SCM
mark_mem (SCM mem_smob)
{
  return SCM_BOOL_F;
}

static scm_sizet
free_mem (SCM mem_smob)
{
  GimpMemPtr *mem = (GimpMemPtr *) SCM_CDR (mem_smob);
  scm_sizet size = sizeof (GimpMemPtr);

  gimp_memptr_uninit (mem);
  free (mem);

  return size;
}

static int
print_mem (SCM mem_smob, SCM port, scm_print_state *pstate)
{
  GimpMemPtr *mem = (GimpMemPtr *) SCM_CDR (mem_smob);
  char buffer[256];
  
  sprintf (buffer, "#<mem %p %p", mem, mem->mem);
  scm_puts (buffer, port);
  scm_puts (">", port);

  /* non-zero means success */
  return 1;
}

static scm_smobfuns mem_funs = {
  mark_mem, free_mem, print_mem, 0
};

void
init_mem_type ()
{
  mem_tag = scm_newsmob (&mem_funs);

  scm_make_gsubr ("make-gimpmem",    0, 0, 0, make_mem);

  scm_make_gsubr ("gimpmem?",   1, 0, 0, mem_p);
  scm_make_gsubr ("gimpmem-info",    1, 0, 0, info_mem);

  scm_make_gsubr ("gimpmem-alloc",   2, 0, 0, alloc_mem);
  scm_make_gsubr ("gimpmem-unalloc", 1, 0, 0, unalloc_mem);
  scm_make_gsubr ("gimpmem-use",     2, 0, 0, use_mem);
  scm_make_gsubr ("gimpmem-unuse",   1, 0, 0, unuse_mem);
  scm_make_gsubr ("gimpmem-join",    2, 0, 0, join_mem);
  scm_make_gsubr ("gimpmem-split",   1, 0, 0, split_mem);
}

