// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <cc++/process.h>
#include <cc++/slog.h>
#include "server.h"

namespace server {
using namespace ost;
using namespace std;

bool Methods::scrDial(void)
{
	unsigned len = 0;
	const char *cp;
	
	state.join.answer_timer = state.timeout = getTimeoutKeyword("timeout");
	
	while(NULL != (cp = getValue(NULL)))
	{
		snprintf(state.join.digits + len, sizeof(state.join.digits) - len, "%s", cp);
		len = strlen(state.join.digits);
	}

	state.join.dial = state.join.digits;
	setState(STATE_DIAL);
	return false;
}

bool Methods::scrSignal(void)
{
	Line *line = getLine();
	Event event;
	BayonneSession *session;

	if(!var_pid)
	{
		error("not-connected");
		return true;
	}

	session = getSid(var_pid);
	
	if(!session)
	{
		error("session-invalid");
		return true;
	}

	memset(&event, 0, sizeof(event));
	if(!strnicmp(line->cmd, "busy", 4))
		event.id = CHILD_BUSY;
	else
		event.id = CHILD_FAILED;
	session->queEvent(&event);
	strcpy(var_pid, "none");
	advance();
	return true;
}

bool Methods::scrWait(void)
{
	const char *cp = getKeyword("id");
	state.timeout = getTimeoutKeyword("timeout");
	
	if(!cp)
		cp = getValue(NULL);

	if(cp)
	{
		state.join.peer = getSid(cp);
		if(!state.join.peer)
		{
			error("session-invalid");
			return true;
		}
	}
	else
		state.join.peer = NULL;

	setState(STATE_WAIT);
	return false;
}
	
bool Methods::scrJoin(void)
{
	Event event;

	state.timeout = getTimeoutKeyword("timeout");
	state.join.peer = getSid(getValue(NULL));

	if(!state.join.peer)
	{
		error("session-invalid");
		return true;
	}

	memset(&event, 0, sizeof(event));
	event.id = JOIN_PEER;
	release();
	if(!state.join.peer->postEvent(&event))
	{
		error("peer-invalid");
		return true;
	}
	setState(STATE_JOIN);
	return false;
}

bool Methods::scrTransfer(void)
{
	char nbuf[256];
	unsigned len = 0;
	const char *cp;

	while(NULL != (cp = getValue(NULL)))
	{
		snprintf(nbuf + len, sizeof(nbuf) - len, "%s", cp);
		len = strlen(nbuf);
	}

	switch(iface)
	{
	case IF_PSTN:	
		state.tone.exiting = true;
		if(audio.tone)
			delete audio.tone;
		state.tone.digits[0] = '!';
		setString(state.tone.digits + 1, sizeof(state.tone.digits) - 1, nbuf);
		audio.tone = new DTMFTones(state.tone.digits, 20000, getToneFraming(), driver->getInterdigit());
		setState(STATE_TONE);
		return false;
	}
	error("not-supported");
	return true;
}

bool Methods::scrStart(void)
{
	char nbuf[256];
	const char *name = getValue(NULL);
	char *number = NULL;
	unsigned len = 0;
	const char *cp = getMember();
	Event event;
	ScriptImage *img = useImage();
	Name *scr;
	Line *sel;
	BayonneSession *child;
	BayonneSpan *span;
	const char *caller = getKeyword("caller");
	const char *display = getKeyword("display");
	Symbol *sym = getKeysymbol("session", 16);
	unsigned idx = 0;
	timeslot_t pos;
	unsigned count;
	Audio::Level level = 26000;
	TelTone::tonekey_t *key = NULL;
	BayonneDriver *drv = NULL;

	if(cp)
	{
		drv = BayonneDriver::get(cp);
		if(!drv)
		{
			error("start-driver-missing");
			return true;
		}
	}

	while(NULL != (cp = getValue(NULL)))
	{
		number = nbuf;
		snprintf(nbuf + len, sizeof(nbuf) - len, "%s", cp);
		len = strlen(nbuf);
	}

	scr = img->getScript(name);
	if(!scr)
	{
		endImage(img);
		if(!scriptEvent("start:missing"))
			goto failed;
		return true;
	}

	if(scr->access != ScriptInterp::scrPUBLIC)
	{
		endImage(img);
		if(!scriptEvent("start:invalid"))
			goto failed;
		return true;
	}

	release();	// release any global locks...

	if(drv || iface == IF_INET)
		goto any;

	sel = scr->select;
	if(!sel)
		goto any;

	while(sel)
	{
		idx = 0;
		cp = strchr(cp, '.');
		if(cp && !stricmp(cp, ".span"))
		     while(NULL != (cp = sel->args[idx++]))
		{
			span = BayonneSpan::get(atoi(cp));
			if(!span)
				continue;

			pos = span->getFirst();		
			count = span->getCount();
			while(count--)
			{
				child = getSession(pos++);
				if(!child)
					continue;

				child->enter();
				if(child->isIdle())
					goto start;
				child->leave();
			}
		}
	
		else while(NULL != (cp = sel->args[idx++]))
		{
			child = getSid(cp);
			if(!child)
				continue;

			child->enter();
			if(child->isIdle())
				goto start;
			child->leave();
		}
		sel = sel->next;
	}
	goto busy;

any:
	if(!drv)
		drv = driver;

	child = drv->getIdle();
	if(!child)
	{
busy:
		endImage(img);
		if(!scriptEvent("start:busy"))
			goto failed;
		return true;
	}

	child->enter();

start:
	if(caller)
		child->setConst("session.caller", caller);

	if(display)
		child->setConst("session.display", display);

	memset(&event, 0, sizeof(event));	
	event.id = START_OUTGOING;
	event.start.img = img;
	event.start.scr = scr;
	event.start.parent = this;
	event.start.dialing = number;

	if(child->postEvent(&event))
		goto success;

	child->leave();
	endImage(img);	

failed:
	if(!scriptEvent("start:failed"))
	{
		ScriptInterp::commit(sym, "none");
		error("start-failed");
	}
	return true;

success:
	if(sym)
		ScriptInterp::commit(sym, child->getExternal("session.id"));
	strcpy(state.tone.sessionid, child->getExternal("session.id"));
	child->leave();
	
	state.timeout = getTimeoutKeyword("timeout");
	state.tone.duration = getTimeoutKeyword("duration");
	if(state.timeout && state.timeout != TIMEOUT_INF)
	{
		if(audio.tone)
		{
			delete audio.tone;
			audio.tone = NULL;
		}
		cp = getKeyword("tone");
		if(cp)
			key = TelTone::find(cp, runtime.getLast("location"));
		if(key)
			audio.tone = new TelTone(key, level, getToneFraming());
		if(number)
			setState(STATE_CONNECT);
		else if(state.tone.duration && state.tone.duration != TIMEOUT_INF)
			setState(STATE_CONNECT);
		else
			setState(STATE_START);
		return false;
	}

	advance();
	return true;
}

bool Methods::scrDetach(void)
{
	Event event;
	BayonneSession *s;
	const char *cp;
	const char *errmsg = NULL;

	while(NULL != (cp = getValue(NULL)))
	{
		s = getSid(cp);
		if(!s)
		{
			errmsg = "invalid-timeslot";
			continue;
		}
		memset(&event, 0, sizeof(event));
		event.id = DETACH_CHILD;
		queEvent(&event);
	}

	if(errmsg)
		error(errmsg);
	else
		advance();
	return true;
}	

bool Methods::scrCancel(void)
{
	Event event;
	BayonneSession *s;
	const char *cp;
	const char *errmsg = NULL;

	while(NULL != (cp = getValue(NULL)))
	{
		s = getSid(cp);
		if(!s)
		{
			errmsg = "invalid-timeslot";
			continue;
		}
		memset(&event, 0, sizeof(event));
		event.id = CANCEL_CHILD;
		queEvent(&event);
	}

	if(errmsg)
		error(errmsg);
	else
		advance();
	return true;
}	

bool Methods::scrDTMF(void)
{
	const char *cp = getKeyword("level");
	Audio::Level level = 26000;
	Audio::timeout_t interdigit;
	unsigned len = 0;

	if(cp)
		level = atoi(cp);

	// lets not detect ourself...
	disableDTMF();

	interdigit = getTimeoutKeyword("interdigit");
	state.timeout = TIMEOUT_INF;
	if(interdigit == TIMEOUT_INF || !interdigit)
		interdigit = driver->getInterdigit();
	if(interdigit > 1000)
		interdigit /= 1000;
	
	if(audio.tone)
		delete audio.tone;

	state.tone.exiting = false;
	state.tone.digits[0] = 0;
	while(len < 64 && NULL != (cp = getValue(NULL)))
	{
		snprintf(state.tone.digits + len, sizeof(state.tone.digits) - len, "%s", cp);		
		len = strlen(state.tone.digits);
	}

	audio.tone = new DTMFTones(state.tone.digits, level, getToneFraming(), interdigit);	
	setState(STATE_TONE);
	return false;
}

bool Methods::scrMF(void)
{
	const char *cp = getKeyword("level");
	Audio::Level level = 26000;
	Audio::timeout_t interdigit;
	unsigned len = 0;

	if(cp)
		level = atoi(cp);

	disableDTMF();

	interdigit = getTimeoutKeyword("interdigit");
	state.timeout = TIMEOUT_INF;
	if(interdigit == TIMEOUT_INF || !interdigit)
		interdigit = driver->getInterdigit();
	if(interdigit > 1000)
		interdigit /= 1000;
	
	if(audio.tone)
		delete audio.tone;

	state.tone.exiting = false;
	state.tone.digits[0] = 0;
	while(len < 64 && NULL != (cp = getValue(NULL)))
	{
		snprintf(state.tone.digits + len, sizeof(state.tone.digits) - len, "%s", cp);		
		len = strlen(state.tone.digits);
	}

	audio.tone = new MFTones(state.tone.digits, level, getToneFraming(), interdigit);	
	setState(STATE_TONE);
	return false;
}


bool Methods::scrTonegen(void)
{
	Audio::Level level = 26000;
	const char *cp = getKeyword("level");
	const char *f1, *f2;

	state.timeout = getTimeoutKeyword("duration");
	state.tone.exiting = false;

	if(cp)
		level = atoi(cp);

	if(audio.tone)
	{
		delete audio.tone;
		audio.tone = NULL;
	}

	cp = getKeyword("frequency");
	f1 = getKeyword("freq1");
	f2 = getKeyword("freq2");

	if(!f1)
		f1 = "0";

	if(!f2)
		f2 = "0";

	if(cp)
		audio.tone = new AudioTone(atoi(cp), level, getToneFraming());
	else
		audio.tone = new AudioTone(atoi(f1), atoi(f2), level, level, getToneFraming());

	setState(STATE_TONE);	
	return false;
}

bool Methods::scrTone(void)
{
	Line *line = getLine();
	const char *name = line->cmd;
	const char *loc = getKeyword("location");
	const char *cp = getKeyword("level");
	TelTone::tonekey_t *key;
	Audio::Level level = 26000;

	if(cp)
		level = atoi(cp);

	state.timeout = getTimeoutKeyword("timeout");
	state.tone.exiting = false;

	if(!loc)
		loc = runtime.getLast("location");

	if(audio.tone)
	{
		delete audio.tone;
		audio.tone = NULL;
	}

	if(!stricmp(name, "dialtone"))
		name = "dial";
	else if(!stricmp(name, "ringback"))
		name = "ring";
	else if(!stricmp(name, "ringtone"))
		name = "ring";
	else if(!stricmp(name, "busytone"))
		name = "busy";
	else if(!stricmp(name, "beep"))
		name = "record";
	else if(!stricmp(name, "callwait"))
		name = "waiting";
	else if(!stricmp(name, "callback"))
		name = "recall";

	key = TelTone::find(name, loc);
	if(!key)
	{
		error("invalid-tone");
		return true;
	}

	audio.tone = new TelTone(key, level, getToneFraming());
	setState(STATE_TONE);
	return false;
}

bool Methods::scrKey(void)
{
	const char *cp;
	const char *val = getKeyword("value");
	const char *ind = getKeyword("index");
	Symbol *sym;

	while(NULL != (cp = getOption(NULL)))
	{
		sym = mapSymbol(cp, PersistProperty::getSize());
		if(!sym)
			continue;
		PersistProperty::refresh(sym, ind, val);
	}
	advance();
	return true;
}

bool Methods::scrTimeslot(void)
{
	const char *cp;
	ScriptProperty *p = &typeTimeslot;
	Symbol *sym;

	while(NULL != (cp = getOption(NULL)))
	{
		sym = mapSymbol(cp, 16 + sizeof(p));
		if(!sym)
			continue;

		if(sym->type != ScriptSymbols::symINITIAL)
			continue;

		sym->type = ScriptSymbols::symPROPERTY;
		memcpy(sym->data, &p, sizeof(p));
		sym->data[sizeof(p)] = 0;
	}

	advance();
	return true;
}

bool Methods::scrPosition(void)
{
	const char *cp;
	ScriptProperty *p = &typePosition;
	Symbol *sym;

	while(NULL != (cp = getOption(NULL)))
	{
		sym = mapSymbol(cp, 13 + sizeof(p));
		if(!sym)
			continue;

		if(sym->type != ScriptSymbols::symINITIAL)
			continue;

		sym->type = ScriptSymbols::symPROPERTY;
		memcpy(sym->data, &p, sizeof(p));
		strcpy(sym->data + sizeof(p), "00:00:00.000");
	}

	advance();
	return true;
}

#ifdef	HAVE_LIBEXEC

bool Methods::scrLibexec(void)
{
	const char *cp = getMember();

	if(!Libexec::create(this))
	{
		error("libexec-failed");
		return true;
	}
	state.lib = getLine();
	setState(STATE_LIBEXEC);
	return false;
}

#else

bool Methods::scrLibexec(void)
{
	error("libexec-unsupported");
	return true;
}

#endif

bool Methods::scrEndmenu(void)
{
	if(!state.menu)
	{
		error("menu-undefined");
		return true;
	}

	if(stack > state.stack)
	{
		error("menu-overflow");
		return true;
	}
		
	state.menu = NULL;

	frame[stack].tranflag = false;
	return scrCleardigits();
}

bool Methods::scrMenu(void)
{
	if(state.menu)
	{
		error("menu-defined");
		return true;
	}

	state.menu = getName();
	state.stack = stack;

	frame[stack].tranflag = true;

	return scrCleardigits();
}

bool Methods::scrCleardigits(void)
{
	state.timeout = getTimeoutValue("0");

	if(state.timeout && !requiresDTMF())
	{
		*dtmf_digits = 0;
		return true;
	}

	if(!state.timeout)
	{
		*dtmf_digits = 0;
		advance();
		return true;
	}

	setState(STATE_CLEAR);
	return false;
}

bool Methods::scrVoicelib(void)
{
	const char *lib = getValue(NULL);
	char bts[8];
	char *p;
	BayonneTranslator *t;

	snprintf(bts, sizeof(bts), "%s", lib);
	p = strchr(bts, '/');
	if(p)
		*p = 0;
	t = BayonneTranslator::get(bts);
	if(!t)
	{
		slog.error("language %s not loaded", bts);
		error("voicelib-invalid");
		return true;
	}

	lib = audio.getVoicelib(lib);
	if(lib == NULL)
	{
		slog.error("voice library %s invalid", lib);
		error("voicelib-invalid");
	}
	else
	{
		translator = t;
		voicelib = lib;
		advance();
	}
	return true;
}

bool Methods::scrRoute(void)
{
	const char *cp = getMember();
	const char *opt;
	char evt[65];
	
	if(!cp)
		cp = "digits";

	while(NULL != (opt = getValue(NULL)))
	{
		snprintf(evt, sizeof(evt), "%s:%s", cp, opt);
		if(digitEvent(evt))
			return false;
	}
	
	error("no-digit-match");
	return true;
}
	
bool Methods::scrCollect(void)
{
        const char *cp = getMember();     

        if(!requiresDTMF())    
                return true;

	if(!cp)
		cp = "digits";

	state.input.route = cp;
	state.input.var = getOption(NULL);
        state.timeout = getTimeoutKeyword("timeout");
        state.input.interdigit = getTimeoutKeyword("interdigit");
        state.input.format = getKeyword("format");
	state.input.ignore = getKeyword("ignore");

        if(state.timeout == TIMEOUT_INF)
                state.timeout = 6000;

        if(state.input.interdigit == TIMEOUT_INF)
                state.input.interdigit = state.timeout;

        state.input.exit = getExitKeyword("#");

        cp = getKeyword("count");
        if(!cp)
                cp = "0";

      	state.input.size = state.input.count = atoi(cp);
        if(state.input.format)
                state.input.size = strlen(state.input.format);

        if(state.input.format && !state.input.count)
        {
                cp = state.input.format;
                while(*cp)
                {
                        if(*(cp++) == '?')
                                ++state.input.count;
                }
        }

	if(!state.input.count)
	{
		error("input-no-count");
		return true;
	}

        if(state.input.count > MAX_DTMF)
                state.input.count = MAX_DTMF;

	setState(STATE_COLLECT);
	return false;
}

bool Methods::scrInput(void)
{
	const char *cp;
	Symbol *sym;

	if(!requiresDTMF())
		return true;

	state.input.var = getOption(NULL);
	state.timeout = getTimeoutKeyword("timeout");
	state.input.interdigit = getTimeoutKeyword("interdigit");
	state.input.format = getKeyword("format");

	if(state.timeout == TIMEOUT_INF)
		state.timeout = 6000;

	if(state.input.interdigit == TIMEOUT_INF)
		state.input.interdigit = state.timeout;

	state.input.exit = getExitKeyword("#");

	cp = getKeyword("count");
	if(!cp)
		cp = "0";

	state.input.size = state.input.count = atoi(cp);

	if(state.input.format)
		state.input.size = strlen(state.input.format);

	state.input.required = 0;
	if(state.input.format)
	{
		cp = state.input.format;
		while(*cp)
		{
			if(*(cp++) == '?')
				++state.input.required;
		}
		if(!state.input.count)
			state.input.count = state.input.required;
	}
		
	sym = mapSymbol(state.input.var, state.input.size);
	if(!sym)
	{
		error("input-no-symbol");
		return true;
	}
	if(!state.input.size)
		state.input.count = ScriptSymbols::storage(sym);

	if(!state.input.count)
		state.input.count = state.input.size;

	if(!state.input.count)
	{
		error("cannot-save-input");
		return true;
	}
	if(state.input.count > MAX_DTMF)
		state.input.count = MAX_DTMF;

	if(state.input.required > MAX_DTMF)
		state.input.required = MAX_DTMF;

	if(!state.input.required)
		state.input.required = state.input.count;

	setState(STATE_INPUT);
	return false;	
}

bool Methods::scrRead(void)
{
	const char *cp;
	Symbol *sym;

	if(!requiresDTMF())
		return true;

	state.input.var = getOption(NULL);
	state.timeout = getTimeoutKeyword("timeout");
	state.input.interdigit = getTimeoutKeyword("interdigit");
	state.input.format = getKeyword("format");

	if(state.timeout == TIMEOUT_INF)
		state.timeout = 6000;

	if(state.input.interdigit == TIMEOUT_INF)
		state.input.interdigit = state.timeout;

	state.input.exit = getExitKeyword(NULL);

	cp = getKeyword("count");
	if(!cp)
		cp = "0";

	state.input.size = state.input.count = atoi(cp);

	if(state.input.format)
		state.input.size = strlen(state.input.format);

	state.input.required = 0;
	if(state.input.format)
	{
		cp = state.input.format;
		while(*cp)
		{
			if(*(cp++) == '?')
				++state.input.required;
		}
		if(!state.input.count)
			state.input.count = state.input.required;
	}
		
	sym = mapSymbol(state.input.var, state.input.size);
	if(!sym)
	{
		error("input-no-symbol");
		return true;
	}
	if(!state.input.size)
		state.input.count = ScriptSymbols::storage(sym);

	if(!state.input.count)
		state.input.count = state.input.size;

	if(!state.input.count)
	{
		error("cannot-save-input");
		return true;
	}
	if(state.input.count > MAX_DTMF)
		state.input.count = MAX_DTMF;

	if(state.input.required > MAX_DTMF)
		state.input.required = MAX_DTMF;

	if(!state.input.required)
		state.input.required = state.input.count;

	setState(STATE_READ);
	return false;	
}

bool Methods::scrKeyinput(void)
{

	if(!requiresDTMF())
		return true;

	state.inkey.menu = getKeyword("menu");
	state.inkey.var = getOption(NULL);
	state.timeout = getTimeoutKeyword("timeout");

	if(state.timeout == TIMEOUT_INF)
		state.timeout = 0;
		
	setState(STATE_INKEY);
	return false;
}

bool Methods::scrPickup(void)
{
	if(!offhook)
	{
		setState(STATE_PICKUP);
		return false;
	}
	advance();
	return true;
}

bool Methods::scrSync(void)
{
	time_t now;

	state.timeout = getTimeoutValue();

	time(&now);
	now -= starttime;
	if(now >= state.timeout / 1000)
		state.timeout = 0;
	else
		state.timeout -= (now * 1000l);

	if(!state.timeout)
	{
		advance();
		return true;
	}
	setState(STATE_SLEEP);
	return false;
}	

bool Methods::scrSleep(void)
{
	state.timeout = getTimeoutValue();

	if(!state.timeout)
	{
		advance();
		return true;
	}

	setState(STATE_SLEEP);
	return false;
}

bool Methods::scrHangup(void)
{
	setState(STATE_HANGUP);
	return false;
}	

bool Methods::scrErase(void)
{
	const char *cp = getAudio(false);
	
	if(cp)
	{
		slog.error("%s: %s", logname, cp);
		error("invalid-erase");
		return true;
	}

	cp = audio.getFilename(getValue(NULL), true);
	if(!cp)
	{
		error("erase-invalid-file");
		return true;
	}
	remove(cp);
	advance();
	return true;
}	

bool Methods::scrMove(void)
{
	const char *cp = getAudio(false);
	const char *dp;
	const char *se, *de;
	
	if(cp)
	{
		slog.error("%s: %s", logname, cp);
		error("invalid-move");
		return true;
	}

	cp = audio.getFilename(getValue(NULL), true);
	if(!cp)
	{
		error("invalid-source");
		return true;
	}
	
	dp = audio.getFilename(getValue(NULL), true);
	if(!cp)
	{
		error("invalid-target");
		return true;
	}

	se = strrchr(cp, '.');
	de = strrchr(dp, '.');

	if(!se || !de)
	{
		error("invalid-names");
		return true;
	}

	if(stricmp(se, de))
	{
		error("filename-mismatch");
		return true;
	}

	if(rename(cp, dp))
	{
		if(!signalScript(SIGNAL_FAIL))
			error("move-failed");
		return true;
	}

	advance();
	return true;
}	

bool Methods::scrCopy(void)
{
	const char *cp = getAudio(false);
	const char *dp;
	const char *se, *de;
	
	if(cp)
	{
		slog.error("%s: %s", logname, cp);
		error("invalid-move");
		return true;
	}

	cp = audio.getFilename(getValue(NULL), true);
	if(!cp)
	{
		error("invalid-source");
		return true;
	}
	
	dp = audio.getFilename(getValue(NULL), true);
	if(!cp)
	{
		error("invalid-target");
		return true;
	}

	se = strrchr(cp, '.');
	de = strrchr(dp, '.');

	if(!se || !de)
	{
		error("invalid-names");
		return true;
	}

	if(stricmp(se, de))
	{
		error("filename-mismatch");
		return true;
	}
	new CopyThread(dynamic_cast<ScriptInterp*>(this), cp, dp);
	return false;
}	

bool Methods::scrBuild(void)
{
	unsigned count = 0;
	const char *cp;
	Audio::Info info;

	cp = getAudio();
        if(cp)
        {
                slog.error("%s: %s", logname, cp);
                if(!signalScript(SIGNAL_FAIL))
                        error("invalid-audio");      
                return true;
        }

        cp = audio.translator->speak(this);
        if(cp)
        {
                slog.error("%s: %s", logname, cp);
                error("invalid-translation");
                return true;
        }

	info.framing = audio.framing;
	info.encoding = audio.encoding;
	info.annotation = (char *)getKeyword("note");
	info.rate = Audio::getRate(audio.encoding);

	if(info.rate == Audio::rateUnknown)
		info.rate = Audio::rate8khz;

	new BuildThread(dynamic_cast<ScriptInterp *>(this), &audio, &info, state.audio.list);
	return false;
}

bool Methods::scrPrompt(void)
{
	unsigned count = 0;
	const char *cp = getMember();

	if(cp && stricmp(translator->getId(), cp))
	{
		advance();
		return true;
	}

	if(state.menu && *dtmf_digits)
	{
		advance();
		return true;
	}

        memset(&state.audio, 0, sizeof(state.audio));
	audio.offset = NULL;

	if(state.menu && !requiresDTMF())
		return true;

	if(state.menu)
		state.audio.exitkey = true;
	else
		state.audio.exitkey = false;

	state.audio.mode = Audio::modeReadAny;

	cp = getAudio();
        if(cp)
        {
                slog.error("%s: %s", logname, cp);
                if(!signalScript(SIGNAL_FAIL))
                        error("invalid-audio");
                return true;
        }

        cp = audio.translator->speak(this);
        if(cp)
        {
                slog.error("%s: %s", logname, cp);
                error("invalid-translation");
                return true;
        }

	setState(STATE_PLAY);
	return false;
}

bool Methods::scrReplay(void)
{
	const char *cp;
        memset(&state.audio, 0, sizeof(state.audio));

        state.audio.exitkey = false;
        if(state.menu)
                state.audio.exitkey = true;

        state.audio.exit = getExitKeyword(NULL);
	state.audio.menu = getMenuKeyword(NULL);

        cp = getAudio();
        if(cp)
        {
                slog.error("%s: %s", logname, cp);
                if(!signalScript(SIGNAL_FAIL))
                        error("invalid-audio");
                return true;
        }

        state.audio.list[0] = getValue(NULL);
	state.audio.list[1] = NULL;
	state.audio.mode = Audio::modeRead;
	setState(STATE_PLAY);
	return false;
}

bool Methods::scrRecord(void)
{
	const char *cp = getMember();

	memset(&state.audio, 0, sizeof(state.audio));

	state.audio.exitkey = false;
	if(state.menu)
		state.audio.exitkey = true;

	if(cp)
		state.audio.compress = true;

	state.audio.exit = getExitKeyword(NULL);
	state.audio.menu = getMenuKeyword(NULL);
	state.audio.total = getTimeoutKeyword("timeout");
	state.audio.silence = getTimeoutKeyword("silence");
	state.audio.intersilence = getTimeoutKeyword("intersilence");
	state.audio.note = getKeyword("note");

	if(state.audio.total == TIMEOUT_INF)
		state.audio.total = 60000;

	if(state.audio.silence == TIMEOUT_INF)
		state.audio.silence = 0;

	if(state.audio.intersilence == TIMEOUT_INF)
		state.audio.intersilence = state.audio.silence;

	cp = getAudio();
	if(cp)
	{
                slog.error("%s: %s", logname, cp);
                if(!signalScript(SIGNAL_FAIL))
                        error("invalid-audio");
                return true;
        }

	state.audio.list[0] = getValue(NULL);

	state.audio.mode = Audio::modeCreate;
	if(audio.offset)
		state.audio.mode = Audio::modeWrite;

	setState(STATE_RECORD);
	return false;
}

bool Methods::scrAppend(void)
{
	const char *cp = getMember();

	memset(&state.audio, 0, sizeof(state.audio));

	state.audio.exitkey = false;
	if(state.menu)
		state.audio.exitkey = true;

	if(cp)
		state.audio.compress = true;

	audio.offset = NULL;
	state.audio.exit = getExitKeyword(NULL);
	state.audio.menu = getMenuKeyword(NULL);
	state.audio.total = getTimeoutKeyword("timeout");
	state.audio.silence = getTimeoutKeyword("silence");
	state.audio.intersilence = getTimeoutKeyword("intersilence");

	if(state.audio.total == TIMEOUT_INF)
		state.audio.total = 60000;

	if(state.audio.silence == TIMEOUT_INF)
		state.audio.silence = 0;

	if(state.audio.intersilence == TIMEOUT_INF)
		state.audio.intersilence = state.audio.silence;

	cp = getAudio();
	if(cp)
	{
                slog.error("%s: %s", logname, cp);
                if(!signalScript(SIGNAL_FAIL))
                        error("invalid-audio");
                return true;
        }

	state.audio.list[0] = getValue(NULL);
	state.audio.mode = Audio::modeAppend;

	setState(STATE_RECORD);
	return false;
}

bool Methods::scrPlay(void)
{
	unsigned count = 0;
	const char *cp = getMember();

        if(cp && stricmp(translator->getId(), cp))
        {
                advance();
                return true;
        } 

        memset(&state.audio, 0, sizeof(state.audio));
	audio.offset = NULL;
	state.audio.exitkey = false;
	state.audio.mode = Audio::modeReadAny;

	cp = getAudio();
        if(cp)
        {
                slog.error("%s: %s", logname, cp);
                if(!signalScript(SIGNAL_FAIL))
                        error("invalid-audio");
                return true;
        }

        cp = audio.translator->speak(this);
        if(cp)
        {
                slog.error("%s: %s", logname, cp);
                error("invalid-translation");
                return true;
        }

	setState(STATE_PLAY);
	return false;
}

bool Methods::scrPathname(void)
{
	unsigned count = 0;
	const char *cp = getAudio();
	const char *path;
	FILE *fp;

	if(cp)
	{
		slog.error("%s: %s", logname, cp);
		if(!signalScript(SIGNAL_FAIL))
			error("invalid-audio");
		return true;
	}

	cp = audio.translator->speak(this);
	if(cp)
	{
		slog.error("%s: %s", logname, cp);
		error("invalid-translation");
		return true;
	}	

	fp = fopen(server->getLast("output"), "a");
	serialize.enter();
	while(state.audio.list[count])
	{
		cp = state.audio.list[count++];
		path = audio.getFilename(cp, false);
		if(path)
		{
			printf("%s = %s\n", cp, path);
			fprintf(fp, "%s = %s\n", cp, path);
		}
		else
		{
			printf("%s = invalid\n", cp);
			fprintf(fp, "%s = invalid\n", cp);
		}
	}
	serialize.leave();
	fclose(fp);
	advance();
	return true;
}
		
bool Methods::scrEcho(void)
{
	const char *val;
	FILE *fp = fopen(server->getLast("output"), "a");
	serialize.enter();

	while(NULL != (val = getValue(NULL)))
	{
		fputs(val, fp);
		cout << val;
	}
	
	fputc('\n', fp);
	cout << endl;
	serialize.leave();
	fclose(fp);
	advance();
	return true;
}

} // end namespace
