// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception, you may use this file as part of a free software
// library without restriction.  Specifically, if other files instantiate
// templates or use macros or inline functions from this file, or you compile
// this file and link it with other files to produce an executable, this
// file does not by itself cause the resulting executable to be covered by
// the GNU General Public License.  This exception does not however
// invalidate any other reasons why the executable file might be covered by
// the GNU General Public License.
//
// This exception applies only to the code released under the name GNU
// ccScript.  If you copy code from other releases into a copy of GNU
// ccScript, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own for GNU ccScript, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.
//

#ifndef	CCXX_LIBEXEC_H_
#define	CCXX_LIBEXEC_H_

#ifndef	CCXX_BAYONNE_H_
#include <cc++/bayonne.h>
#endif

#ifndef	CCXX_SLOG_H_
#include <cc++/slog.h>
#endif

#ifndef	CCXX_PROCESS_H_
#include <cc++/process.h>
#endif

namespace ost {

/**
 * Container class for applications implimenting the libexec process
 * method of Bayonne interfacing.  This is intended for writing external
 * apps and is neatly tucked away into libbayonne as well.
 *
 * @author David Sugar <dyfet@gnutelephony.org>
 * @short Libexec process interface class.
 */
class __EXPORT Libexec
{
private:
	static Keydata *head;
	static Keydata *args;
	static const char *tsid;
	static bool eof;
	static int result;
	static char position[16];
	static char digits[64];
	static unsigned exitcode;

	static int getResult(const char *id = NULL, char *buffer = NULL, unsigned size = 0);

public:
	/**
	 * Initialize libexec.
	 */
	static void init(void);

	/**
	 * Get a header record item.
	 *
	 * @param id of header or sys env item.
	 * @return string value of requested item or NULL.
	 */
	static const char *getEnv(const char *id);

	/**
	 * Get a named libexec command line argument.
	 *
	 * @param id of libexec argument.
	 * @return string value of requested argument or NULL.
	 */
	static const char *getArg(const char *id);

	/**
	 * Get a symbol value from the server.
	 *
	 * @param id of server symbol.
	 * @param buffer to save value into.
	 * @param size of buffer.
	 * @return true if success.
	 */
	static bool getVar(const char *id, char *buffer, unsigned size);

	/**
	 * Clear a server symbol.
	 *
	 * @param id of server symbol.
	 * @return true if success.
	 */
	static bool clrVar(const char *id);

	/**
 	 * Create a new server symbol of specified size.
	 *
	 * @param id of server symbol.
	 * @param size of server symbol.
	 * @return true if succesful.
	 */
	static bool newVar(const char *id, unsigned size);
	
	/**
	 * Set a symbol value to the server.
	 *
	 * @param id of server symbol.
	 * @param value of server symbol.
	 * @return true if successful.
	 */
	static bool setVar(const char *id, const char *value);
	 
	/**
	 * Add/cat a symbol value to a server symbol.
	 *
	 * @param id of server symbol.
	 * @param value of server symbol.
	 * @return true if successful.
	 */
	static bool addVar(const char *id, const char *value);

	/**
	 * Send results back to the server, generic.
	 *
	 * @param value of result.
	 * @return true if successful.
	 */
	static bool setResult(const char *value);

	/**
	 * Get last dtmf digit buffer response.
	 *
	 * @param buffer to save results into.
	 * @param size of buffer.
	 * @return dtmf digits.
	 */
	static const char *getDigits(char *buffer, unsigned size);

	/**
	 * Get last audio position marker.
	 *
	 * @return position marker.
	 */
	static inline const char *getPosition(void)
		{return position;};

	/**
	 * Hangup session...
	 */
	static void hangup(void);

	/**
	 * Resume server, pass it exit code...
	 */
	static void resume(unsigned code = 0);

	/**
	 * Read a single key of input...
	 *
	 * @param timeout to wait for key.
	 * @return char code read.
	 */
	static char inkey(timeout_t timeout);

	/**
	 * Read a input line.
	 *
	 * @param buffer to save input into.
	 * @param size of buffer (# of chars + 1) to read.
	 * @param timeout to wait for digit.
	 * @return true if successful.
	 */
	static bool input(char *buffer, unsigned size, timeout_t timeout);

	/**
	 * Prompt user with text.
	 *
	 * @param text to send through phrasebook.
	 * @param voice to optionally use.
	 * @return true if success.
	 */
	static bool prompt(const char *text, const char *voice = NULL);

	/**
	 * Record file.
	 *
	 * @param file to record.
	 * @param duration of file.
	 * @param optional silence detect.
	 * @param optional file offset to use.
	 */
	static bool record(const char *name, timeout_t total, timeout_t silence = 0, const char *offset = NULL);

	/**
	 * Replay (a recorded) audio file.
	 *
	 * @param file to replay.
	 * @param optional offset to use.
	 */
	static bool replay(const char *name, const char *offset = NULL);

	/**
	 * Clear user input.
	 *
	 * @return true if successful.
	 */
	static bool clear(void);

	/**
	 * Wait for user input.
	 *
	 * @param timeout to wait.
	 * @return true if input pending.
	 */
	static bool wait(timeout_t timeout);

	/**
	 * is live check.
	 */
	static bool isLive(void);

	/**
	 * Get exit reason if server triggered.
	 *
	 * @return exit reason.
	 */
	static inline unsigned getExit(void)
		{return exitcode;};
	
};	 

class __EXPORT BayonneTSession : public BayonneSession
{
protected:
	friend class __EXPORT BayonneSysexec;

        void sysVar(const char *tsid, char *id, const char *value, int size);
        void sysHeader(const char *tsid);
        void sysArgs(const char *tsid);
        void sysStatus(const char *tsid);  
	void sysRecord(const char *tsid, char *token);
	void sysReplay(const char *tsid, char *token);
	void sysFlush(const char *tsid);
	void sysWait(const char *tsid, char *token);
	void sysPrompt(const char *tsid, const char *voice, const char *text);
	void sysInput(const char *tsid, char *token);
	void sysHangup(const char *tsid);
	void sysExit(const char *tsid, char *token);
	void sysReturn(const char *tsid, const char *text);
};

/**
 * Core class for any server which impliments libexec functionality.
 *
 * @author David Sugar
 * @short Server system execution interface
 */
class __EXPORT BayonneSysexec : protected Thread, protected Bayonne
{
private:
	static bool exiting;
#ifndef	WIN32
	static int iopair[2];
#endif
	static BayonneSysexec *libexec;

	static void execute(unsigned slot, const char *cmd);
	static void readline(char *buf, unsigned max);

	void run(void);

	BayonneSysexec();
	~BayonneSysexec();

public:
	static bool create(BayonneSession *s);
	static void allocate(const char *path, size_t bs = 0, int pri = 0, const char *modpath = NULL);
	static void cleanup(void);
	static void startup(void);
};

}; // namespace

#endif
