// Copyright (C) 2005 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "module.h"

namespace module {
using namespace ost;
using namespace std;

Session::Session(TCPSocket &server) :
TCPSession(server)
{
	monmap = new char[ts_used];
	memset(monmap, 0, ts_used);
}

void Session::startSelect(const char *name, const char *dial, const char *caller, const char *display)
{
	BayonneSpan *span;
	BayonneDriver *trunk = BayonneDriver::getTrunking();
	BayonneSession *session;
	ScriptImage *img = useImage();
	Name *scr = img->getScript(name);
	Event event;
	Line *sel = scr->select;
	unsigned idx = 0, count;
	const char *cp;
	timeslot_t pos;

        if(!scr)
                *tcp() << "*** unknown script specified" << endl;
        else if(scr && scr->access != scrPUBLIC)
                *tcp() << "*** invalid script specified" << end;

	if(!scr || !scr->select || scr->access != scrPUBLIC)
	{
		endImage(img);
		if(trunk && scr)
		{
			startDriver(trunk, name, dial, caller, display);
			return;
		}
		if(scr && !scr->select && scr->access == scrPUBLIC)
			*tcp() << "*** missing select records" << endl;
		return;
	}

	while(sel)
	{
                idx = 0;
                cp = strchr(sel->cmd, '.');
                if(cp && !stricmp(cp, ".span"))
                     while(NULL != (cp = sel->args[idx++]))
                {
                        span = BayonneSpan::get(atoi(cp));
                        if(!span)   
                                continue;

                        pos = span->getFirst();
                        count = span->getCount();     
                        while(count--)
                        {
                                session = getSession(pos++);
                                if(!session)
                                        continue;

                                session->enter();
                                if(session->isIdle())
                                        goto start;
                                session->leave();
                        }
		}
                else while(NULL != (cp = sel->args[idx++]))
                {
                        session = getSid(cp);
                        if(!session)
                                continue;

                        session->enter();
                        if(session->isIdle())
                                goto start;
                        session->leave();
                }
		sel = sel->next;

	}
	endImage(img);
	*tcp() << "*** no selected ports available" << endl;
	return;

start:
        memset(&event, 0, sizeof(event));
        event.id = START_OUTGOING;
        event.start.img = img; 
        event.start.scr = scr;
        event.start.dialing = dial;

        session->setConst("session.caller", caller);
        session->setConst("session.display", display);

        if(!session->postEvent(&event))
        {
                *tcp() << "*** session unable to start" << endl;
                session->leave();
                endImage(img);
                return;
        }

	*tcp() << "starting call " << session->getExternal("session.id") << endl; 
	session->leave();
	return;
}

void Session::startDriver(BayonneDriver *d, const char *name, const char *dial, const char *caller, const char *display)
{
        BayonneSession *session;
        ScriptImage *img = useImage();
        Name *scr = img->getScript(name);
        Event event;

	if(!d)
		*tcp() << "*** invalid driver specified" << endl;
	else if(!scr)
		*tcp() << "*** unknown script specified" << endl;
	else if(scr && scr->access != scrPUBLIC)
		*tcp() << "*** invalid script specified" << end;

	if(!d || !scr || scr->access != scrPUBLIC)
	{
		endImage(img);
		return;
	}

        if(!caller)      
        {
                display = "bayonne";
                caller = "none";
        }

        if(!display)
                display = caller;


        session = d->getIdle();
        if(!session)
        {
		*tcp() << "*** no idle ports available" << endl;
                endImage(img);  
                return;
        }

        session->enter();
        memset(&event, 0, sizeof(event));
        event.id = START_OUTGOING;
        event.start.img = img;
        event.start.scr = scr;
        event.start.dialing = dial;

        session->setConst("session.caller", caller);
        session->setConst("session.display", display);

        if(!session->postEvent(&event))
        {
		*tcp() << "*** session unable to start" << endl;
                session->leave();
                endImage(img);  
                return;
        }

	*tcp() << "starting call " << session->getExternal("session.id") << endl;
	session->leave();
	return;
}

void Session::help(void)
{
	*tcp() << "bye - disconnect session" << endl;
	*tcp() << "calls [sid..] - display active calls" << endl;
	*tcp() << "drivers [names..] - display loaded drivers" << endl;
	*tcp() << "mon state|all timeslots.. - enable state monitoring" << endl;
	*tcp() << "nomon timeslots.. - disable state monitoring" << endl;
	*tcp() << "reload - reload script configuration" << endl;
	*tcp() << "shutdown - shutdown server" << endl;
	*tcp() << "spans - list active spans in server" << endl;
	*tcp() << "start script [num [cid]] - start a script running" << endl;
	*tcp() << "stat - show status line record" << endl;
	*tcp() << "stop sid - stop a running session" << endl;
	*tcp() << "uptime - show current server uptime" << endl;
}

void Session::uptime(void)
{
	unsigned long upt = Bayonne::uptime();
	char dur[12];

	if(upt < 100 * 3600)
		snprintf(dur, sizeof(dur), "%02d:%02d:%02d",
			upt / 3600, (upt / 60) % 60, upt % 60);
	else
		snprintf(dur, sizeof(dur), "%d days", upt / (3600 * 24));	
	
	*tcp() << "uptime: " << dur << endl;
}

void Session::monitor(char **list)
{
	BayonneSession *session;
	Event event;
	char *evt = *(list++);
	char *ts;

	while(*list)
	{
		ts = *(list++);
		session = getSid(ts);
		if(!session)
		{
			*tcp() << "*** " << ts << " invalid" << endl;
			continue;
		}

		memset(&event, 0, sizeof(event));
		event.id = ENABLE_LOGGING;
		event.debug.output = tcp();
		event.debug.logstate = evt;
		if(session->postEvent(&event))
			monmap[session->getSlot()] = 'x';
		else
			*tcp() << "*** " << ts << " unavailable" << endl;
	}
}

void Session::nomon(char **list)
{
	BayonneSession *session;
	Event event;
	char *ts;
	timeslot_t pos = 0;

	if(!*list)
	{
		while(pos < ts_used)
		{
			session = getSession(pos);
			if(!monmap[pos] || !session)
			{
				++pos;
				continue;
			}

			monmap[pos++] = 0;
			memset(&event, 0, sizeof(event));
			event.id = DISABLE_LOGGING;
			event.debug.output = tcp();
			session->postEvent(&event);
			*tcp() << (pos - 1) << " monitoring disabled" << endl;
		}			
		return;
	}
	while(*list)
	{
		ts = *(list++);
		session = getSid(ts);
		if(!session)
		{
			*tcp() << "*** " << ts << " invalid" << endl;
			continue;
		}

		memset(&event, 0, sizeof(event));
		event.id = DISABLE_LOGGING;
		event.debug.output = tcp();
		session->postEvent(&event);
		monmap[session->getSlot()] = 0;
		*tcp() << ts << " monitoring disabled" << endl;
	}
}

void Session::calls(char **list)
{
	BayonneSession *session;
	char lr[128];
	const char *caller, *dialed, *display;
	unsigned count = 0;

	while(*list)
	{
		session = getSid(*(list++));
		if(!session)
			continue;
	
		session->enter();
		if(session->isIdle())
		{
			session->leave();
			continue;
		}
		caller = session->getSymbol("session.caller");
		dialed = session->getSymbol("session.dialed");
		display = session->getSymbol("session.display");
		if(!caller)
			caller = "unknown caller";
		if(!dialed)
			dialed = "unknown dialed";
		if(!display)
			display = "";
		snprintf(lr, sizeof(lr), "%-14s %-16s %-16s %-12s %s",
                        session->getExternal("session.deviceid"),
                        session->getExternal("session.id"),
                        session->getExternal("session.pid"),
                        session->getExternal("session.duration"),
                        session->getExternal("session.type")); 
		*tcp() << lr << endl;
		snprintf(lr, sizeof(lr), " %-20s %-20s %s",
			caller, dialed, display);
		*tcp() << lr << endl;
		session->leave();
		++count;
	}
	if(!count)
		*tcp() << "*** no active calls found" << endl;
}

void Session::allcalls(void)
{
	BayonneSession *session;
	char lr[128];
	timeslot_t pos = 0;
	const char *caller, *dialed, *display;
	unsigned count = 0;

	while(pos < ts_used)
	{
		session = getSession(pos++);
		if(!session)
			continue;
	
		session->enter();
		if(session->isIdle())
		{
			session->leave();
			continue;
		}
		caller = session->getSymbol("session.caller");
		dialed = session->getSymbol("session.dialed");
		display = session->getSymbol("session.display");
		if(!caller)
			caller = "unknown caller";
		if(!dialed)
			dialed = "unknown dialed";
		if(!display)
			display = "";
		snprintf(lr, sizeof(lr), "%-14s %-16s %-16s %-12s %s",
                        session->getExternal("session.deviceid"),
                        session->getExternal("session.id"),
                        session->getExternal("session.pid"),
                        session->getExternal("session.duration"),
                        session->getExternal("session.type")); 
		*tcp() << lr << endl;
		snprintf(lr, sizeof(lr), " %-20s %-20s %s",
			caller, dialed, display);
		*tcp() << lr << endl;
		session->leave();
		++count;
	}
	if(!count)
		*tcp() << "*** no active calls found" << endl;
}



void Session::spans(void)
{
	BayonneSpan *span;
	char lr[80];

	unsigned spid = 0;
	if(!BayonneSpan::getSpans())
	{
		*tcp() << "*** no spans active" << endl;
		return;
	}

	while(spid < BayonneSpan::getSpans())
	{
		span = BayonneSpan::get(spid);
		snprintf(lr, sizeof(lr), "%03d %04d %04d", spid,
			span->getFirst(), span->getCount());
		*tcp() << lr << endl;
	}
}

void Session::drivers(char **list)
{
	BayonneDriver *driver;
	char lr[80];
	while(*list)
	{
		driver = BayonneDriver::get(*list);
		snprintf(lr, sizeof(lr), "%-16s %04d %04d %03d %03d", 
			*list, driver->getFirst(), driver->getCount(), 
			driver->getSpanFirst(), driver->getSpansUsed());
		*tcp() << lr << endl;
		++list;
	}
}

void Session::run(void)
{
	Event event;
	BayonneSession *session;
	bool login = false;
	int retries = 0;
	char buffer[256];
	char *argv[128];
	char *tok;
	char *bp;
	int argc;
	tpport_t port;
	IPV4Address addr = getLocal(&port);
	const char *name = server->getLast("servername");
	const char *ver = server->getLast("serverversion");
	*tcp() << "welcome to " << name << " " << ver << " monitor at " << addr.getHostname() << endl;
	*tcp() << ts_used << " timeslots active; " << ts_count << " timeslots allocated" << endl;
        *tcp() << "secret: ";
        flush();

	while(Service::active && !login && retries++ < 3)
	{
		buffer[0] = 0;
		if(isPending(pendingInput, 10000))
		{
			getline(buffer, sizeof(buffer));
			if(!buffer[0])
				Thread::exit();
			
			bp = strtok_r(buffer, "\r\n", &tok);
			if(bp)
				Thread::sleep(1000);
			if(bp && !strcmp(bp, Service::tcp.getLast("secret")))
				login = true;
			else
			{
				*tcp() << "secret: ";
				flush();
			}
		}
	}

	if(login)
	{
		*tcp() << "-> ";
		flush();
	}

	while(Service::active && login)
	{
		if(isPending(pendingInput, 100))
		{
			buffer[0] = 0;
			getline(buffer, sizeof(buffer));
			if(!buffer[0])
				Thread::exit();

			argc = 0;
			bp = strtok_r(buffer, " \t\r\n", &tok);
			while(bp && argc < 127)
			{
				argv[argc++] = bp;
				bp = strtok_r(NULL, " \t\r\n", &tok);
			}
			argv[argc] = NULL;

			if(!*argv)
			{
			}
			else if(!stricmp(*argv, "quit") || !stricmp(*argv, "bye") || !stricmp(*argv, "goodbye"))
			{
				login = false;
				break;
			}
			else if(!strnicmp(*argv, "nomon", 5))
			{
				nomon(&argv[1]);
			}
			else if(!strnicmp(*argv, "mon", 3))
			{
				if(!argv[1])
				{
					*tcp() << "*** events missing to monitor" << endl;
					goto skip;
				}
				if(!argv[2])
				{
					*tcp() << "*** no timeslots listed to monitor" << endl;
					goto skip;
				}
				monitor(&argv[1]);
			}
			else if(!stricmp(*argv, "calls"))
			{
				if(argv[1])
					calls(&argv[1]);
				else
					allcalls();
			}
			else if(!stricmp(*argv, "spans"))
				spans();
			else if(!stricmp(*argv, "drivers"))
			{
				if(!argv[1])
					BayonneDriver::list(&argv[1], 120);
				drivers(&argv[1]);
			}
			else if(!stricmp(*argv, "start"))
			{
				if(!argv[1])
				{
					*tcp() << "*** missing script name" << endl;
					goto skip;
				}
				if(argv[2])
					startSelect(argv[1], argv[2], argv[3], argv[4]);
				else
					startSelect(argv[1], NULL, NULL, NULL);
			}
			else if(!stricmp(*argv, "stop"))
			{
				if(!argv[1])
				{
					*tcp() << "*** missing session id" << endl;
					goto skip;
				}
				session = getSid(argv[1]);
				if(!session)
				{
					*tcp() << "*** invalid or unknown call session" << endl;
					goto skip;
				}
				memset(&event, 0, sizeof(event));
				event.id = STOP_SCRIPT;
				if(session->postEvent(&event))
					*tcp() << "script stopped" << endl;
				else
					*tcp() << "*** stop ignored" << endl;
			}
			else if(!strnicmp(*argv, "stat", 4))
				*tcp() << status << endl;
			else if(!stricmp(*argv, "uptime"))
				Session::uptime();
			else if(!stricmp(*argv, "reload"))
				Bayonne::reload();
			else if(!stricmp(*argv, "help"))
				help();
			else if(!stricmp(*argv, "shutdown"))
			{
				Service::active = false;
				login = false;
				Bayonne::down();
				break;
			}
			else
				*tcp() << "*** unknown command" << endl;

skip:			
			*tcp() << "-> ";
			flush();
		}
		Thread::yield();
	}
	*tcp() << "goodbye" << endl;
	Thread::exit();
}

void Session::final(void)
{
	Event event;
	timeslot_t pos = 0;
	BayonneSession *session;

	while(pos < ts_used)
	{
		if(monmap[pos])
		{
			session = getSession(pos);
			if(session)
			{
				memset(&event, 0, sizeof(event));
				event.id = DISABLE_LOGGING;
				event.debug.output = tcp();
				session->postEvent(&event);
			}
		}
		++pos;
	}

	delete[] monmap;
}

} // namespace 
