/*
 * Electric(tm) VLSI Design System
 *
 * File: usrarc.c
 * User interface tool: simple arc routing
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * support@staticfreesoft.com
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "efunction.h"
#include "tecschem.h"
#include "tecgen.h"

/* prototypes for local routines */
static TECHNOLOGY *us_gettech(NODEINST*, PORTPROTO*);
static ARCINST *us_directarcinst(NODEINST*, PORTPROTO*, ARCPROTO*, NODEINST*, PORTPROTO*, ARCPROTO*,
	NODEPROTO*, INTBIG, INTBIG, INTBIG, INTBIG, INTSML, INTBIG*);
static ARCINST *us_onebend(NODEINST*, PORTPROTO*, ARCPROTO*, NODEINST*, PORTPROTO*, ARCPROTO*,
	NODEPROTO*, INTBIG, INTBIG, INTBIG, INTBIG, ARCINST**, INTBIG*);
static ARCINST *us_twobend(NODEINST*, PORTPROTO*, ARCPROTO*, NODEINST*, PORTPROTO*, ARCPROTO*,
	NODEPROTO*, INTBIG, INTBIG, INTBIG, INTBIG, ARCINST**, ARCINST**, INTBIG*);
static INTBIG us_portdistance(NODEINST*, PORTPROTO*, INTBIG, INTBIG, INTBIG*, INTBIG*, INTBIG, INTSML);
static INTSML us_fitportangle(NODEINST*, PORTPROTO*, INTSML, INTSML);
static INTBIG us_figurearcwidth(ARCPROTO *typ, INTBIG wid, NODEINST *ni1, PORTPROTO *pp1,
	INTBIG ox1, INTBIG oy1, NODEINST *ni2, PORTPROTO *pp2, INTBIG ox2, INTBIG oy2);
static NODEPROTO *us_findconnectingpin(ARCPROTO *ap, GEOM *geom, PORTPROTO *pp);

/*
 * routine to run an arcinst from "fromgeom" to "togeom".  The prefered
 * ports to use on these objects is "frompp" and "topp" (presuming that
 * "fromgeom" and "togeom" are nodeinsts and "frompp" and "topp" are
 * not NOPORTPROTO).  The prefered layer for the arc is "typ" but
 * this may be overridden if other layers are possible and the prefered
 * layer isn't.  The prefered location for the arcinst is closest to
 * (prefx, prefy) if there is a choice.  If "nozigzag" is nonzero, do not
 * make 3-arc connections (only 1 or 2).  If the arcinst is run, the
 * routine returns its address (if two or three arcs are created, their
 * addresses are returned in "alt1" and "alt2").  Otherwise, it issues an
 * error and returns NOARCINST.  The required angle increment for the
 * arcs is "ang" tenth-degrees (900 for manhattan geometry).
 */
ARCINST *aconnect(GEOM *fromgeom, PORTPROTO *frompp, GEOM *togeom, PORTPROTO *topp,
	ARCPROTO *typ, INTBIG prefx, INTBIG prefy, ARCINST **alt1, ARCINST **alt2, INTSML ang,
	INTSML nozigzag)
{
	REGISTER NODEINST *fromnodeinst, *tonodeinst, *ni;
	REGISTER NODEPROTO *con;
	REGISTER ARCINST *ai, *newai;
	REGISTER PORTPROTO *fpt, *tpt, *pp;
	REGISTER ARCPROTO *ap, *fap, *tap;
	REGISTER TECHNOLOGY *tech, *tech1, *tech2;
	REGISTER INTSML j, t, ans, gent;
	REGISTER INTBIG i, bestdist, bestx, besty, usecontact;
	INTBIG x, y;
	char *result[2];
	extern COMCOMP us_noyesp;

	/* error check */
	if (fromgeom == togeom && fromgeom->entrytype != OBJNODEINST)
	{
		us_abortcommand(_("Cannot run from arc to itself"));
		return(NOARCINST);
	}

	/* handle special case of an arcinst connecting to a facet */
	if (fromgeom->entrytype == OBJARCINST || togeom->entrytype == OBJARCINST)
	{
		if (fromgeom->entrytype == OBJARCINST) ai = fromgeom->entryaddr.ai; else
			ai = togeom->entryaddr.ai;
		ni = NONODEINST;
		if (fromgeom->entrytype == OBJNODEINST) ni = fromgeom->entryaddr.ni;
		if (togeom->entrytype == OBJNODEINST) ni = togeom->entryaddr.ni;

		if (ni != NONODEINST && ni->proto->primindex == 0)
		{
			/* found facet connected to arcinst: search for closest portinst */
			bestdist = MAXINTBIG;
			for(fpt = ni->proto->firstportproto; fpt != NOPORTPROTO; fpt = fpt->nextportproto)
			{
				/* see if the arcinst can connect at this portinst */
				for(i=0; fpt->connects[i] != NOARCPROTO; i++)
					if (fpt->connects[i] == ai->proto) break;
				if (fpt->connects[i] == NOARCPROTO) continue;

				/* compute position for closest applicable portinst */
				i = us_portdistance(ni, fpt, prefx, prefy, &x, &y,
					defaultarcwidth(typ) - arcprotowidthoffset(typ), 0);
				if (i > bestdist) continue;
				bestdist = i;   bestx = x;   besty = y;
			}

			/* adjust prefered position to closest port (manhattan only!!!) */
			if (bestdist < MAXINTBIG)
			{
				if (ai->end[0].xpos == ai->end[1].xpos) prefy = besty;
				if (ai->end[0].ypos == ai->end[1].ypos) prefx = bestx;
			}
		}
	}

	/* get actual nodes for each end of connection */
	fromnodeinst = us_getnodeonarcinst(&fromgeom, &frompp, togeom, topp, prefx, prefy, 0);
	if (fromnodeinst == NONODEINST)
	{
		us_abortcommand(_("Cannot create splitting pin"));
		return(NOARCINST);
	}

	tonodeinst = us_getnodeonarcinst(&togeom, &topp, fromgeom, frompp, prefx, prefy, 0);
	if (tonodeinst == NONODEINST)
	{
		us_abortcommand(_("Cannot create splitting pin"));
		return(NOARCINST);
	}

	/* default to single port on one-port nodeinsts */
	if (frompp == NOPORTPROTO)
		if ((fpt = fromnodeinst->proto->firstportproto) != NOPORTPROTO)
			if (fpt->nextportproto == NOPORTPROTO) frompp = fpt;
	if (topp == NOPORTPROTO)
		if ((tpt = tonodeinst->proto->firstportproto) != NOPORTPROTO)
			if (tpt->nextportproto == NOPORTPROTO) topp = tpt;

	/* sillyness check */
	if (fromnodeinst == tonodeinst && frompp == topp)
	{
		(void)initinfstr();
		(void)formatinfstr(_("Are you sure you want to run an arc from node %s to itself?"),
			describenodeinst(fromnodeinst));
		ans = ttygetparam(returninfstr(), &us_noyesp, 1, result);
		if (ans <= 0 || (result[0][0] != 'y' && result[0][0] != 'Y')) return(NOARCINST);
	}

	/* reset all arcproto connection bits */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
		for(ap = tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
			ap->userbits &= ~CANCONNECT;

	/* set bits in arc prototypes that can make the connection */
	t = gent = 0;
	for(fpt = fromnodeinst->proto->firstportproto; fpt !=NOPORTPROTO; fpt = fpt->nextportproto)
	{
		if (frompp != NOPORTPROTO && frompp != fpt) continue;
		for (tpt = tonodeinst->proto->firstportproto; tpt != NOPORTPROTO; tpt = tpt->nextportproto)
		{
			if (topp != NOPORTPROTO && topp != tpt) continue;

			/* set those bits common to both "fpt" and "tpt" */
			for(i=0; fpt->connects[i] != NOARCPROTO; i++)
			{
				for(j=0; tpt->connects[j] != NOARCPROTO; j++)
				{
					if (tpt->connects[j] != fpt->connects[i]) continue;
					fpt->connects[i]->userbits |= CANCONNECT;
					if (fpt->connects[i]->tech == gen_tech) gent++; else t++;
					break;
				}
			}
		}
	}

	/* if no common ports, look for a contact */
	con = NONODEPROTO;
	usecontact = 0;
	if (t == 0)
	{
		tech = fromnodeinst->proto->tech;
		if (tech != gen_tech)
		{
			for(con = tech->firstnodeproto; con != NONODEPROTO; con = con->nextnodeproto)
			{
				if (((con->userbits&NFUNCTION) >> NFUNCTIONSH) != NPCONTACT) continue;
				pp = con->firstportproto;
				for(fpt = fromnodeinst->proto->firstportproto; fpt != NOPORTPROTO; fpt = fpt->nextportproto)
				{
					if (frompp != NOPORTPROTO && frompp != fpt) continue;
					for(i=0; pp->connects[i] != NOARCPROTO; i++)
					{
						if (pp->connects[i]->tech == gen_tech) continue;
						for(j=0; fpt->connects[j] != NOARCPROTO; j++)
							if (fpt->connects[j] == pp->connects[i]) break;
						if (fpt->connects[j] != NOARCPROTO) break;
					}
					if (pp->connects[i] != NOARCPROTO) break;
				}
				if (fpt == NOPORTPROTO) continue;
				fap = fpt->connects[j];

				for (tpt = tonodeinst->proto->firstportproto; tpt != NOPORTPROTO; tpt = tpt->nextportproto)
				{
					if (topp != NOPORTPROTO && topp != tpt) continue;
					for(i=0; pp->connects[i] != NOARCPROTO; i++)
					{
						if (pp->connects[i]->tech == gen_tech) continue;
						for(j=0; tpt->connects[j] != NOARCPROTO; j++)
							if (tpt->connects[j] == pp->connects[i]) break;
						if (tpt->connects[j] != NOARCPROTO) break;
					}
					if (pp->connects[i] != NOARCPROTO) break;
				}
				if (tpt == NOPORTPROTO) continue;
				tap = tpt->connects[j];

				/* this contact connects */
				t = 1;
				usecontact = 1;
				break;
			}
		}
	}

	/* if no common ports, don't run the arcinst */
	if (t+gent == 0)
	{
		us_abortcommand(_("Cannot find arc that connects %s to %s"), geomname(fromgeom),
			geomname(togeom));
		return(NOARCINST);
	}

	/* if a contact must be used, try it */
	if (usecontact != 0)
	{
		newai = us_makeconnection(fromnodeinst, frompp, fap, tonodeinst, topp, tap,
			con, prefx, prefy, alt1, alt2, ang, nozigzag, 0);
		if (newai != NOARCINST) return(newai);
	}

	/* see if the default arc prototype can be used */
	if ((typ->userbits&CANCONNECT) != 0)
	{
		newai = us_makeconnection(fromnodeinst, frompp, typ, tonodeinst, topp, typ,
			getpinproto(typ), prefx, prefy, alt1, alt2, ang, nozigzag, 0);
		if (newai != NOARCINST) return(newai);
	}

	/* default arc prototype cannot be used: try others in this technology */
	for(ap = el_curtech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
	{
		if ((ap->userbits&CANCONNECT) == 0) continue;
		if (ap == typ) continue;
		newai = us_makeconnection(fromnodeinst, frompp, ap, tonodeinst, topp, ap,
			getpinproto(ap), prefx, prefy, alt1, alt2, ang, nozigzag, 0);
		if (newai != NOARCINST) return(newai);
	}

	/* current technology bad: see if this is a cross-technology connect */
	tech1 = us_gettech(fromnodeinst, frompp);
	tech2 = us_gettech(tonodeinst, topp);
	if (tech1 == tech2)
	{
		/* if current technology not that of the two nodes, check it */
		if (tech1 != el_curtech)
			for(ap = tech1->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
		{
			if ((ap->userbits&CANCONNECT) == 0) continue;
			if (ap == typ) continue;
			newai = us_makeconnection(fromnodeinst, frompp, ap, tonodeinst, topp, ap,
				getpinproto(ap), prefx, prefy, alt1, alt2, ang, nozigzag, 0);
			if (newai != NOARCINST) return(newai);
		}
	} else
	{
		/* current technology bad: try other technologies */
		for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
			if (tech != el_curtech)
				for(ap = tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
		{
			if ((ap->userbits&CANCONNECT) == 0) continue;
			newai = us_makeconnection(fromnodeinst, frompp, ap, tonodeinst, topp, ap,
				getpinproto(ap), prefx, prefy, alt1, alt2, ang, nozigzag, 0);
			if (newai != NOARCINST) return(newai);
		}
	}
	us_abortcommand(_("There is no way to connect these objects"));
	return(NOARCINST);
}

/*
 * routine to return the technology associated with node "ni", port "pp".
 */
TECHNOLOGY *us_gettech(NODEINST *ni, PORTPROTO *pp)
{
	for(;;)
	{
		if (ni->proto->primindex != 0) break;
		if (pp == NOPORTPROTO) break;
		ni = pp->subnodeinst;
		pp = pp->subportproto;
	}
	return(ni->proto->tech);
}

/*
 * routine to try to run an arcinst from nodeinst "fromnodeinst" to nodeinst
 * "tonodeinst".  The default port prototypes to use are in "fromportproto" and
 * "toportproto".  The arcs to use are "fromarcproto" and "toarcproto".  The
 * connecting pin (if necessary) is "con".  If it can be done, the arcinst is
 * created and the routine returns its address, otherwise it returns
 * NOARCINST.  If two or three arcs are created, their addresses are returned
 * in "alt1" and "alt2".  The preferred arcinst runs closest to (prefx, prefy).
 * The most direct route will be traveled, but an additional nodeinst or two
 * may have to be created.  The angle increment for the arcs is "ang".
 * If "nozigzag" is nonzero, disallow 3-arc connections (only 1 or 2).
 */
ARCINST *us_makeconnection(NODEINST *fromnodeinst, PORTPROTO *fromportproto, ARCPROTO *fromarcproto,
	NODEINST *tonodeinst, PORTPROTO *toportproto, ARCPROTO *toarcproto, NODEPROTO *con, INTBIG prefx,
	INTBIG prefy, ARCINST **alt1, ARCINST **alt2, INTSML ang, INTSML nozigzag, INTBIG *fakecoords)
{
	REGISTER PORTPROTO *fpt, *tpt;
	REGISTER INTBIG bestdist, dist, fwid, twid, w;
	REGISTER INTSML i;
	REGISTER ARCINST *newai;
	INTBIG x, y, hx, hy, lx, ly;

	/* see if the cursor is near a port on a facet */
	bestdist = MAXINTBIG;
	if (fromportproto == NOPORTPROTO && fromnodeinst->proto->primindex == 0)
		for(fpt = fromnodeinst->proto->firstportproto; fpt != NOPORTPROTO; fpt = fpt->nextportproto)
	{
		for(i=0; fpt->connects[i] != NOARCPROTO; i++)
			if (fromarcproto == fpt->connects[i]) break;
		if (fpt->connects[i] == NOARCPROTO) continue;

		/* portproto may be valid: compute distance to it */
		portposition(fromnodeinst, fpt, &x, &y);
		dist = abs(x-prefx) + abs(y-prefy);
		if (dist > bestdist) continue;
		bestdist = dist;   fromportproto = fpt;
	}
	if (toportproto == NOPORTPROTO && tonodeinst->proto->primindex == 0)
		for(tpt = tonodeinst->proto->firstportproto; tpt != NOPORTPROTO; tpt = tpt->nextportproto)
	{
		for(i=0; tpt->connects[i] != NOARCPROTO; i++)
			if (toarcproto == tpt->connects[i]) break;
		if (tpt->connects[i] == NOARCPROTO) continue;

		/* portproto may be valid: compute distance to it */
		portposition(tonodeinst, tpt, &x, &y);
		dist = abs(x-prefx) + abs(y-prefy);
		if (dist > bestdist) continue;
		bestdist = dist;   toportproto = tpt;
	}
	if (fromportproto == NOPORTPROTO || toportproto == NOPORTPROTO) return(NOARCINST);

	/* determine the width of this arc from others on the nodes */
	fwid = defaultarcwidth(fromarcproto);
	w = us_widestarcinst(fromarcproto, fromnodeinst, fromportproto);
	if (w > fwid) fwid = w;
	twid = defaultarcwidth(toarcproto);
	w = us_widestarcinst(toarcproto, tonodeinst, toportproto);
	if (w > twid) twid = w;

	/* now first check for direct connection */
	newai = us_directarcinst(fromnodeinst, fromportproto, fromarcproto,
		tonodeinst, toportproto, toarcproto, con,
			prefx, prefy, fwid, twid, ang, fakecoords);
	if (newai != NOARCINST)
	{
		*alt1 = *alt2 = NOARCINST;
		return(newai);
	}

	/* now try a zig-zag if Manhattan or diagonal */
	if ((ang == 450 || ang == 900) && nozigzag == 0)
	{
		/*
		 * check location of cursor with respect to fromnode and tonode.
		 * There are nine possibilities.  Each implies a specific routing
		 * request (cases where cursor is lined up horizontally or vertically
		 * with either fromnode or tonode are not included as they default to
		 * L-connections)
		 *       1   2   3
		 *         *             [fromnode]
		 *       4   5   6
		 *             *         [tonode]
		 *       7   8   9
		 */
		portposition(fromnodeinst, fromportproto, &lx, &ly);
		portposition(tonodeinst, toportproto, &hx, &hy);
		if (hx < lx)
		{
			x = lx;   lx = hx;   hx = x;
		}
		if (hy < ly)
		{
			y = ly;   ly = hy;   hy = y;
		}

		/* if cursor location is in 2, 4, 5, 6, or 8, try two-bend connection */
		if ((prefx >= lx && prefx <= hx) || (prefy >= ly && prefy <= hy))
		{
			newai = us_twobend(fromnodeinst, fromportproto, fromarcproto, tonodeinst,
				toportproto, toarcproto, con, prefx, prefy, fwid, twid, alt1, alt2, fakecoords);
			if (newai != NOARCINST) return(newai);
		}
	}

	/* next check for a one-bend connection */
	newai = us_onebend(fromnodeinst, fromportproto, fromarcproto, tonodeinst, toportproto,
		toarcproto, con, prefx, prefy, fwid, twid, alt1, fakecoords);
	if (newai != NOARCINST)
	{
		*alt2 = NOARCINST;
		return(newai);
	}

	/* finally check for any zig-zag connection */
	newai = us_twobend(fromnodeinst, fromportproto, fromarcproto, tonodeinst, toportproto,
		toarcproto, con, prefx, prefy, fwid, twid, alt1, alt2, fakecoords);
	if (newai != NOARCINST) return(newai);

	/* give up */
	return(NOARCINST);
}

/*
 * routine to check for direct connection from nodeinst "fromnodeinst", port "fromppt",
 * arc "fromarcproto" to nodeinst "tonodeinst",  port "toppt", arc "toarcproto" (with
 * a possible via of type "con"). and create an arc that is "wid" wide if possible.
 * It is prefered that the arcinst be close to (prefx, prefy).  If "fakecoords" is
 * zero, create the arc if possible.  If "fakecoords" is nonzero, do not create an arc,
 * but just store the coordinates of the two endpoints in the four integers there (and
 * return any non-NOARCINST value).  If the arcinst is created, the routine
 * returns its address, otherwise it returns NOARCINST.  The angle increment
 * for the arc is "ang" tenth-degrees (900 for manhattan geometry).
 */
ARCINST *us_directarcinst(NODEINST *fromnodeinst, PORTPROTO *fromppt, ARCPROTO *fromarcproto,
	NODEINST *tonodeinst, PORTPROTO *toppt, ARCPROTO *toarcproto, NODEPROTO *con,
	INTBIG prefx, INTBIG prefy, INTBIG fwid, INTBIG twid, INTSML ang, INTBIG *fakecoords)
{
	REGISTER PORTPROTO *fpt, *tpt, *fstart, *tstart;
	REGISTER ARCINST *ai;
	REGISTER NODEINST *ni;
	PORTPROTO *fromportproto, *toportproto;
	REGISTER INTSML i, j, trange, frange, bad, gotpath;
	INTBIG bestdist, dist, bestpdist, pdist, bestfx,bestfy, besttx,bestty, otheralign, fpx, fpy,
		tpx, tpy, flx, fly, fhx, fhy, tlx, tly, thx, thy, wid, frwid, trwid, x, y, bestxi, bestyi,
		pxs, pys, lx, hx, ly, hy;
	static POLYGON *fpoly = NOPOLYGON, *tpoly = NOPOLYGON;

	/* get polygons */
	if (fpoly == NOPOLYGON) fpoly = allocstaticpolygon(4, us_tool->cluster);
	if (tpoly == NOPOLYGON) tpoly = allocstaticpolygon(4, us_tool->cluster);

	/* determine true width */
	portposition(tonodeinst, toppt, &tpx, &tpy);
	portposition(fromnodeinst, fromppt, &fpx, &fpy);

	fwid = us_figurearcwidth(fromarcproto, fwid, fromnodeinst, fromppt, prefx, prefy, tonodeinst, toppt, prefx, prefy);
	frwid = fwid - arcprotowidthoffset(fromarcproto);
	twid = us_figurearcwidth(toarcproto, twid, fromnodeinst, fromppt, prefx, prefy, tonodeinst, toppt, prefx, prefy);
	trwid = twid - arcprotowidthoffset(toarcproto);
	wid = maxi(fwid, twid);

	/* assume default prefered positions for port locations */
	tpx = (tonodeinst->highx + tonodeinst->lowx) / 2;
	tpy = (tonodeinst->highy + tonodeinst->lowy) / 2;
	fpx = (fromnodeinst->highx + fromnodeinst->lowx) / 2;
	fpy = (fromnodeinst->highy + fromnodeinst->lowy) / 2;

	/* precompute better positions if ports were specified */
	if (toppt != NOPORTPROTO)
	{
		portposition(tonodeinst, toppt, &tpx, &tpy);
		tpoly->xv[0] = prefx;   tpoly->yv[0] = prefy;   tpoly->count = 1;
		shapeportpoly(tonodeinst, toppt, tpoly, 1);
		reduceportpoly(tpoly, tonodeinst, toppt, trwid);
		if ((toppt->userbits&PORTISOLATED) != 0)
		{
			/* use prefered location on isolated ports */
			tpx = prefx;   tpy = prefy;
			closestpoint(tpoly, &tpx, &tpy);
		}
		tstart = toppt;
	} else tstart = tonodeinst->proto->firstportproto;
	if (fromppt != NOPORTPROTO)
	{
		portposition(fromnodeinst, fromppt, &fpx, &fpy);
		fpoly->xv[0] = prefx;   fpoly->yv[0] = prefy;   fpoly->count = 1;
		shapeportpoly(fromnodeinst, fromppt, fpoly, 1);
		reduceportpoly(fpoly, fromnodeinst, fromppt, frwid);
		if ((fromppt->userbits&PORTISOLATED) != 0)
		{
			/* use prefered location on isolated ports */
			fpx = prefx;   fpy = prefy;
			closestpoint(fpoly, &fpx, &fpy);
		}
		fstart = fromppt;
	} else fstart = fromnodeinst->proto->firstportproto;

	/* check again to make sure the ports align */
	if (toppt != NOPORTPROTO)
	{
		tpx = fpx;   tpy = fpy;
		closestpoint(tpoly, &tpx, &tpy);
	}
	if (fromppt != NOPORTPROTO)
	{
		fpx = tpx;   fpy = tpy;
		closestpoint(fpoly, &fpx, &fpy);
	}
	if (toppt != NOPORTPROTO)
	{
		tpx = fpx;   tpy = fpy;
		closestpoint(tpoly, &tpx, &tpy);
	}

	/* search every possible port on the "from" node */
	bestdist = bestpdist = MAXINTBIG;
	for(fpt = fstart; fpt != NOPORTPROTO; fpt = fpt->nextportproto)
	{
		if (fromppt != NOPORTPROTO && fromppt != fpt) break;

		/* see if the port has an opening for this type of arcinst */
		for(i=0; fpt->connects[i] != NOARCPROTO; i++)
			if (fromarcproto == fpt->connects[i]) break;
		if (fpt->connects[i] == NOARCPROTO) continue;

		/* potential port: get information about its position if not already known */
		if (fromppt == NOPORTPROTO)
		{
			fpoly->xv[0] = prefx;   fpoly->yv[0] = prefy;   fpoly->count = 1;
			shapeportpoly(fromnodeinst, fpt, fpoly, 1);

			/* handle arc width offset from node edge */
			reduceportpoly(fpoly, fromnodeinst, fpt, frwid);

			/* find single closest point */
			fpx = tpx;   fpy = tpy;
			closestpoint(fpoly, &fpx, &fpy);
		}

		/* correlate with every possible port on the "to" node */
		for(tpt = tstart; tpt != NOPORTPROTO; tpt = tpt->nextportproto)
		{
			if (toppt != NOPORTPROTO && toppt != tpt) break;

			/* silly to run from port to itself when ports are unspecified */
			if (fromnodeinst == tonodeinst && fpt == tpt &&
				(fromppt == NOPORTPROTO || toppt == NOPORTPROTO)) continue;

			/* see if the port has an opening for this type of arcinst */
			for(i=0; tpt->connects[i] != NOARCPROTO; i++)
				if (toarcproto == tpt->connects[i]) break;
			if (tpt->connects[i] == NOARCPROTO) continue;

			/* get the shape of the "to" port if not already done */
			if (toppt == NOPORTPROTO)
			{
				tpoly->xv[0] = prefx;   tpoly->yv[0] = prefy;   tpoly->count = 1;
				shapeportpoly(tonodeinst, tpt, tpoly, 1);

				/* handle arc width offset from node edge */
				reduceportpoly(tpoly, tonodeinst, tpt, trwid);

				/* find single closest point */
				tpx = fpx;   tpy = fpy;
				closestpoint(tpoly, &tpx, &tpy);
			}

			/* check directionality of ports */
			trange = (INTSML)(((tpt->userbits&PORTARANGE) >> PORTARANGESH) * 10);
			frange = (INTSML)(((fpt->userbits&PORTARANGE) >> PORTARANGESH) * 10);

			/* ignore range calculations for serpentine transistors */
			if ((tonodeinst->proto->userbits&HOLDSTRACE) != 0 &&
				getvalkey((INTBIG)tonodeinst, VNODEINST, VINTEGER|VISARRAY, el_trace) != NOVARIABLE)
					trange = 1800;
			if ((fromnodeinst->proto->userbits&HOLDSTRACE) != 0 &&
				getvalkey((INTBIG)fromnodeinst, VNODEINST, VINTEGER|VISARRAY, el_trace) != NOVARIABLE)
					frange = 1800;

			/* make sure ranges are acceptable */
			if (trange != 1800 || frange != 1800)
			{
				/* determine angle between port centers */
				bad = 0;
				if (fpx != tpx || fpy != tpy)
				{
					/* see if the angle is permitted */
					i = figureangle(fpx, fpy, tpx, tpy);
					if (us_fitportangle(fromnodeinst, fpt, i, frange) != 0) bad = 1; else
						if (us_fitportangle(tonodeinst, tpt, (INTSML)(i+1800), trange) != 0)
							bad = 1;
				}

				/* special case for ports that overlap */
				if (bad != 0)
				{
					flx = (fromnodeinst->lowx+fromnodeinst->highx)/2;
					fly = (fromnodeinst->lowy+fromnodeinst->highy)/2;
					fhx = (tonodeinst->lowx+tonodeinst->highx)/2;
					fhy = (tonodeinst->lowy+tonodeinst->highy)/2;
					if (flx != fhx || fly != fhy)
					{
						j = figureangle(flx, fly, fhx, fhy);
						if ((j+1800)%3600 == i &&
							us_fitportangle(fromnodeinst, fpt, j, frange) == 0 &&
								us_fitportangle(tonodeinst, tpt, (INTSML)(j+1800), trange) == 0)
									bad = 0;
					}
				}
				if (bad != 0) continue;
			}

			/* see if an arc can connect at this angle */
			if (ang == 0)
			{
				/* no angle restrictions: simply use the chosen locations */
				gotpath = 1;
			} else
			{
				/* if manhattan is possible, test it first */
				gotpath = 0;
				if (900 / ang * ang == 900)
				{
					/* manhattan angle restriction: check directly */
					if (tpx == fpx || tpy == fpy) gotpath = 1;
				}
				if (gotpath == 0 && ang != 900)
				{
					flx = fhx = fpx;   fly = fhy = fpy;
					tlx = thx = tpx;   tly = thy = tpy;

					/* arbitrary angle restrictions: try all angle possibilities */
					for(i=0; i<3600; i += ang)
						if (arcconnects(i, flx,fhx,fly,fhy, tlx,thx,tly,thy, &fpx,&fpy, &tpx,&tpy) != 0)
					{
						gotpath = 1;
						break;
					}
				}
			}
			if (gotpath == 0) continue;

			/* for manhattan arcs, adjust if edge alignment requested */
			if (fpx == tpx)
			{
				/* arcinst runs vertically */
				getbbox(fpoly, &flx, &fhx, &fly, &fhy);
				getbbox(tpoly, &tlx, &thx, &tly, &thy);
				if (us_edgealignment != 0 && flx != fhx && tlx != thx)
				{
					/* make the arc edge align */
					x = us_alignvalue(fpx - frwid/2, us_edgealignment, &otheralign) + frwid/2;
					otheralign += frwid/2;
					if (x <= mini(fhx,thx) && x >= maxi(flx,tlx)) tpx = fpx = x; else
						if (otheralign <= mini(fhx,thx) && otheralign >= maxi(flx,tlx))
							fpx = tpx = otheralign;

					/* try to align the ends */
					y = us_alignvalue(tpy+trwid/2, us_edgealignment, &otheralign) - trwid/2;
					otheralign -= trwid/2;
					if (isinside(tpx, y, tpoly) != 0) tpy = y; else
						if (isinside(tpx, otheralign, tpoly) != 0) tpy = otheralign;
					y = us_alignvalue(fpy+frwid/2, us_edgealignment, &otheralign) - frwid/2;
					otheralign -= frwid/2;
					if (isinside(fpx, y, fpoly) != 0) fpy = y; else
						if (isinside(fpx, otheralign, fpoly) != 0) fpy = otheralign;
				}
			} else if (fpy == tpy)
			{
				/* arcinst runs horizontally */
				getbbox(fpoly, &flx, &fhx, &fly, &fhy);
				getbbox(tpoly, &tlx, &thx, &tly, &thy);
				if (us_edgealignment != 0 && fly != fhy && tly != thy)
				{
					/* make the arc edge align */
					y = us_alignvalue(tpy - trwid/2, us_edgealignment, &otheralign) + trwid/2;
					otheralign += trwid/2;
					if (y <= mini(fhy,thy) && y >= maxi(fly,tly)) tpy = fpy = y; else
						if (otheralign <= mini(fhy,thy) && otheralign >= maxi(fly,tly))
							tpy = fpy = otheralign;

					/* try to align the ends */
					x = us_alignvalue(tpx+trwid/2, us_edgealignment, &otheralign) - trwid/2;
					otheralign -= trwid/2;
					if (isinside(x, tpy, tpoly) != 0) tpx = x; else
						if (isinside(otheralign, tpy, tpoly) != 0) tpx = otheralign;
					x = us_alignvalue(fpx+frwid/2, us_edgealignment, &otheralign) - frwid/2;
					otheralign -= frwid/2;
					if (isinside(fpx, x, fpoly) != 0) fpx = x; else
						if (isinside(otheralign, fpy, fpoly) != 0) fpx = otheralign;
				}
			}

			/* if this path is longer than another, forget it */
			dist = abs(fpx-tpx) + abs(fpy-tpy);
			if (dist > bestdist) continue;

			/* if this path is same as another check prefered position */
			pdist = (fpx==tpx) ? abs(prefx-fpx) : abs(prefy-fpy);
			if (dist == bestdist && pdist > bestpdist) continue;

			/* this is best: remember it */
			bestdist = dist;       bestpdist = pdist;
			fromportproto = fpt;   toportproto = tpt;
			bestfx = fpx;          bestfy = fpy;
			besttx = tpx;          bestty = tpy;
		}
	}

	if (bestdist < MAXINTBIG)
	{
		if (fakecoords == 0)
		{
			if (fromarcproto != toarcproto)
			{
				/* create the intermediate nodeinst */
				if (bestfx == besttx)
				{
					bestxi = bestfx;
					bestyi = prefy;
				} else if (bestfy == bestty)
				{
					bestxi = prefx;
					bestyi = bestfy;
				} else
				{
					bestxi = prefx;
					bestyi = prefy;
				}
				defaultnodesize(con, &pxs, &pys);
				lx = bestxi - pxs/2;   hx = lx + pxs;
				ly = bestyi - pys/2;   hy = ly + pys;
				ni = newnodeinst(con, lx, hx, ly, hy, 0, 0, fromnodeinst->parent);
				if (ni == NONODEINST)
				{
					us_abortcommand(_("Cannot create connecting contact"));
					return(NOARCINST);
				}
				endobjectchange((INTBIG)ni, VNODEINST);

				ai = us_runarcinst(fromnodeinst, fromportproto, bestfx, bestfy, 
					ni, con->firstportproto, bestxi, bestyi, fromarcproto, wid);
				if (ai == NOARCINST) return(NOARCINST);

				ai = us_runarcinst(ni, con->firstportproto, bestxi, bestyi, 
					tonodeinst, toportproto, besttx, bestty, toarcproto, wid);
				if (ai == NOARCINST) return(NOARCINST);
				ttyputmsg(_("Created 1 %s and 1 %s arc"), fromarcproto->protoname, toarcproto->protoname);
			} else
			{
				ai = us_runarcinst(fromnodeinst, fromportproto, bestfx, bestfy, 
					tonodeinst, toportproto, besttx, bestty, fromarcproto, wid);
				if (ai == NOARCINST) return(NOARCINST);
				ttyputmsg(_("Created 1 %s arc"), fromarcproto->protoname);
			}
			return(ai);
		} else
		{
			fakecoords[0] = bestfx;
			fakecoords[1] = bestfy;
			fakecoords[2] = besttx;
			fakecoords[3] = bestty;
			return((ARCINST *)1);
		}
	}

	/* give up */
	return(NOARCINST);
}

/*
 * routine to check for a one-bend connection running from nodeinst
 * "fromnodeinst" to nodeinst "tonodeinst" on arcproto "typ" and create two
 * arcs that are "wid" wide if possible.  It is prefered that the bend pass
 * close to (prefx, prefy) and is on portproto "fromppt" (if "fromppt" is not
 * NOPORTPROTO) on the FROM nodeinst and on portproto "toppt" (if "toppt" is
 * not NOPORTPROTO) on the TO nodeinst.  If "fakecoords" is zero, create the arc
 * if possible.  If "fakecoords" is nonzero, do not create an arc, but just store
 * the coordinates of the two endpoints in the four integers there (and return any
 * non-NOARCINST value).  If the arcinst is created, the
 * routine returns its address and the second arc's address is returned in
 * "alt".  If no arc can be created, the routine returns NOARCINST.
 */
ARCINST *us_onebend(NODEINST *fromnodeinst, PORTPROTO *fromppt, ARCPROTO *fromarcproto,
	NODEINST *tonodeinst, PORTPROTO *toppt, ARCPROTO *toarcproto, NODEPROTO *con,
	INTBIG prefx, INTBIG prefy, INTBIG fwid, INTBIG twid, ARCINST **alt, INTBIG *fakecoords)
{
	REGISTER NODEINST *ni;
	REGISTER PORTPROTO *fpt, *tpt;
	PORTPROTO *fromportproto, *toportproto;
	REGISTER INTSML i, bad, frange, trange;
	REGISTER INTBIG lx, hx, ly, hy, frwid, trwid, wid;
	REGISTER ARCINST *ai1, *ai2;
	static POLYGON *poly = NOPOLYGON;
	INTBIG fx, fy, tx, ty, xi, yi, bestxi, bestyi, bestdist, dist,
		bestfx, bestfy, besttx, bestty, altxi, altyi, otheralign, pxs, pys;
	INTSML found;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, us_tool->cluster);

	/* determine true width */
	fwid = us_figurearcwidth(fromarcproto, fwid, fromnodeinst, fromppt, prefx, prefy, tonodeinst, toppt, prefx, prefy);
	frwid = fwid - arcprotowidthoffset(fromarcproto);
	twid = us_figurearcwidth(toarcproto, twid, fromnodeinst, fromppt, prefx, prefy, tonodeinst, toppt, prefx, prefy);
	trwid = twid - arcprotowidthoffset(toarcproto);
	wid = maxi(fwid, twid);

	found = 0;    bestdist = MAXINTBIG;
	for(fpt = fromnodeinst->proto->firstportproto; fpt != NOPORTPROTO; fpt = fpt->nextportproto)
	{
		if (fromppt != NOPORTPROTO && fromppt != fpt) continue;

		/* see if the port has an opening for this type of arcinst */
		for(i=0; fpt->connects[i] != NOARCPROTO; i++)
			if (fromarcproto == fpt->connects[i]) break;
		if (fpt->connects[i] == NOARCPROTO) continue;

		/* potential port: get information about its position */
		if ((fpt->userbits&PORTISOLATED) != 0)
		{
			(void)us_portdistance(fromnodeinst, fpt, prefx, prefy, &fx, &fy, frwid, 1);
		} else
		{
			portposition(fromnodeinst, fpt, &fx, &fy);
		}

		for(tpt = tonodeinst->proto->firstportproto; tpt != NOPORTPROTO; tpt = tpt->nextportproto)
		{
			if (toppt != NOPORTPROTO && toppt != tpt) continue;

			/* should not run arcinst from port to itself */
			if (fromnodeinst == tonodeinst && fpt == tpt) continue;

			/* see if the port has an opening for this type of arcinst */
			for(i=0; tpt->connects[i] != NOARCPROTO; i++)
				if (toarcproto == tpt->connects[i]) break;
			if (tpt->connects[i] == NOARCPROTO) continue;

			/* potential portinst: get information about its position */
			if ((tpt->userbits&PORTISOLATED) != 0)
			{
				(void)us_portdistance(tonodeinst, tpt, prefx, prefy, &tx, &ty, trwid, 1);
			} else
			{
				portposition(tonodeinst, tpt, &tx, &ty);
			}

			if (abs(tx-prefx) + abs(fy-prefy) < abs(fx-prefx) + abs(ty-prefy))
			{
				xi = tx;  yi = fy;   altxi = fx;   altyi = ty;
			} else
			{
				xi = fx;  yi = ty;   altxi = tx;   altyi = fy;
			}

			/* see if port angles are correct */
			bad = 0;
			trange = (INTSML)(((tpt->userbits&PORTARANGE) >> PORTARANGESH) * 10);
			if (trange != 1800)
			{
				if (tx == xi && ty == yi) bad++; else
				{
					i = figureangle(tx, ty, xi, yi);
					if (us_fitportangle(tonodeinst, tpt, i, trange) != 0) bad++;
				}
			}
			frange = (INTSML)(((fpt->userbits&PORTARANGE) >> PORTARANGESH) * 10);
			if (frange != 1800)
			{
				if (fx == xi && fy == yi) bad++; else
				{
					i = figureangle(fx, fy, xi, yi);
					if (us_fitportangle(fromnodeinst, fpt, i, frange) != 0) bad++;
				}
			}

			/* if port angles are wrong, try the other inflexion point */
			if (bad != 0)
			{
				if (tx == altxi && ty == altyi) continue;
				i = figureangle(tx, ty, altxi, altyi);
				if (us_fitportangle(tonodeinst, tpt, i, trange) != 0) continue;
				if (fx == altxi && fy == altyi) continue;
				i = figureangle(fx, fy, altxi, altyi);
				if (us_fitportangle(fromnodeinst, fpt, i, frange) != 0) continue;
				xi = altxi;   yi = altyi;
			}

			/* see if this path is better than any previous ones */
			dist = abs(fx-tx) + abs(fy-ty);
			if (dist > bestdist) continue;

			/* select this path */
			found++;               bestdist = dist;
			fromportproto = fpt;   toportproto = tpt;
			bestxi = xi;           bestyi = yi;
			bestfx = fx;           bestfy = fy;
			besttx = tx;           bestty = ty;
		}
	}

	/* make one-bend arcinst */
	if (found != 0)
	{
		/* handle edge alignment */
		if (us_edgealignment != 0)
		{
			if (bestfx == bestxi)
			{
				/* see if "bestxi" and "bestfx" can be aligned */
				i = us_alignvalue(bestfx - fwid/2, us_edgealignment, &otheralign) + fwid/2;
				otheralign += fwid/2;
				shapeportpoly(fromnodeinst, fromportproto, poly, 0);
				if (isinside(i, bestfy, poly) != 0) bestfx = bestxi = i; else
					if (isinside(otheralign, bestfy, poly) != 0) bestfx = bestxi = otheralign;

				/* see if "bestyi" and "bestty" can be aligned */
				i = us_alignvalue(bestty - twid/2, us_edgealignment, &otheralign) + twid/2;
				otheralign += twid/2;
				shapeportpoly(tonodeinst, toportproto, poly, 0);
				if (isinside(besttx, i, poly) != 0) bestty = bestyi = i; else
					if (isinside(besttx, otheralign, poly) != 0) bestty = bestyi = otheralign;
			} else if (bestfy == bestyi)
			{
				/* see if "bestyi" and "bestfy" can be aligned */
				i = us_alignvalue(bestfy - fwid/2, us_edgealignment, &otheralign) + fwid/2;
				otheralign += fwid/2;
				shapeportpoly(fromnodeinst, fromportproto, poly, 0);
				if (isinside(bestfx, i, poly) != 0) bestfy = bestyi = i; else
					if (isinside(bestfx, otheralign, poly) != 0) bestfy = bestyi = otheralign;

				/* see if "bestxi" and "besttx" can be aligned */
				i = us_alignvalue(besttx - twid/2, us_edgealignment, &otheralign) + twid/2;
				otheralign += twid/2;
				shapeportpoly(tonodeinst, toportproto, poly, 0);
				if (isinside(i, bestty, poly) != 0) besttx = bestxi = i; else
					if (isinside(otheralign, bestty, poly) != 0) besttx = bestxi = otheralign;
			}
		}

		/* run the connecting arcs */
		if (fakecoords == 0)
		{
			/* create the intermediate nodeinst */
			defaultnodesize(con, &pxs, &pys);
			lx = bestxi - pxs/2;   hx = lx + pxs;
			ly = bestyi - pys/2;   hy = ly + pys;
			ni = newnodeinst(con, lx, hx, ly, hy, 0, 0, fromnodeinst->parent);
			if (ni == NONODEINST)
			{
				us_abortcommand(_("Cannot create connecting pin"));
				return(NOARCINST);
			}
			endobjectchange((INTBIG)ni, VNODEINST);

			fpt = con->firstportproto;
			ai1 = us_runarcinst(fromnodeinst, fromportproto, bestfx, bestfy, ni, fpt,
				bestxi, bestyi, fromarcproto, wid);
			if (ai1 == NOARCINST) return(NOARCINST);
			ai2 = us_runarcinst(ni, fpt, bestxi, bestyi, tonodeinst, toportproto,
				besttx, bestty, toarcproto, wid);
			if (ai2 == NOARCINST) return(NOARCINST);
			if (fromarcproto == toarcproto) ttyputmsg(_("Created 2 %s arcs"), fromarcproto->protoname); else
				ttyputmsg(_("Created 1 %s and 1 %s arc"), fromarcproto->protoname, toarcproto->protoname);
			if (abs(bestfx-bestxi) + abs(bestfy-bestyi) >
				abs(bestxi-besttx) + abs(bestyi-bestty))
			{
				*alt = ai2;
				return(ai1);
			} else
			{
				*alt = ai1;
				return(ai2);
			}
		} else
		{
			fakecoords[0] = bestfx;
			fakecoords[1] = bestfy;
			fakecoords[2] = bestxi;
			fakecoords[3] = bestyi;
			fakecoords[4] = besttx;
			fakecoords[5] = bestty;
			*alt = (ARCINST *)1;
			return((ARCINST *)1);
		}
	}

	/* give up */
	return(NOARCINST);
}

/*
 * routine to check for a two-bend connection running from nodeinst
 * "fromnodeinst" to nodeinst "tonodeinst" on arcproto "typ" and create two
 * arcs that are "wid" wide if possible.  It is prefered that the jog pass
 * close to (prefx, prefy) and is on portproto "fromppt" (if "fromppt" is not
 * NOPORTPROTO) on the FROM nodeinst and on portproto "toppt" (if "toppt" is
 * not NOPORTPROTO) on the TO nodeinst.  If "fakecoords" is zero, create the arc
 * if possible.  If "fakecoords" is nonzero, do not create an arc, but just store
 * the coordinates of the two endpoints in the four integers there (and return any
 * non-NOARCINST value).  If the arcinst is created, the
 * routine returns its address and the other two arcs are returned in "alt1"
 * and "alt2".  If no arc can be created, the routine returns NOARCINST.
 */
ARCINST *us_twobend(NODEINST *fromnodeinst, PORTPROTO *fromppt, ARCPROTO *fromarcproto,
	NODEINST *tonodeinst, PORTPROTO *toppt, ARCPROTO *toarcproto, NODEPROTO *con,
	INTBIG prefx, INTBIG prefy, INTBIG fwid, INTBIG twid, ARCINST **alt1, ARCINST **alt2,
	INTBIG *fakecoords)
{
	REGISTER NODEINST *ni1, *ni2;
	REGISTER NODEPROTO *fpin;
	REGISTER PORTPROTO *fpt, *tpt;
	PORTPROTO *fromportproto, *toportproto;
	REGISTER INTSML i;
	REGISTER INTBIG lx, hx, ly, hy, frwid, trwid, wid;
	REGISTER ARCINST *ai;
	INTBIG fx, fy, tx, ty, xi1, yi1, xi2, yi2, bestdist, dist,
		bestfx, bestfy, besttx, bestty, fpxs, fpys, tpxs, tpys;

	/* determine true width */
	fwid = us_figurearcwidth(fromarcproto, fwid, fromnodeinst, fromppt, prefx, prefy, tonodeinst, toppt, prefx, prefy);
	frwid = fwid - arcprotowidthoffset(fromarcproto);
	twid = us_figurearcwidth(toarcproto, twid, fromnodeinst, fromppt, prefx, prefy, tonodeinst, toppt, prefx, prefy);
	trwid = twid - arcprotowidthoffset(toarcproto);
	wid = maxi(fwid, twid);

	/* find the "from" port */
	bestdist = MAXINTBIG;   fromportproto = NOPORTPROTO;
	for(fpt = fromnodeinst->proto->firstportproto; fpt != NOPORTPROTO; fpt = fpt->nextportproto)
	{
		if (fromppt != NOPORTPROTO && fromppt != fpt) continue;

		/* see if the port has an opening for this type of arcinst */
		for(i=0; fpt->connects[i] != NOARCPROTO; i++)
			if (fromarcproto == fpt->connects[i]) break;
		if (fpt->connects[i] == NOARCPROTO) continue;

		/* potential portinst: get information about its position */
		(void)us_portdistance(fromnodeinst, fpt, prefx, prefy, &fx, &fy, frwid, 1);

		dist = abs(fx-prefx) + abs(fy-prefy);
		if (dist > bestdist) continue;
		fromportproto = fpt;   bestdist = dist;
		bestfx = fx;           bestfy = fy;
	}
	if (fromportproto == NOPORTPROTO) return(NOARCINST);

	/* find the "to" port */
	bestdist = MAXINTBIG;   toportproto = NOPORTPROTO;
	for(tpt = tonodeinst->proto->firstportproto; tpt != NOPORTPROTO; tpt = tpt->nextportproto)
	{
		if (toppt != NOPORTPROTO && toppt != tpt) continue;

		/* see if the port has an opening for this type of arcinst */
		for(i=0; tpt->connects[i] != NOARCPROTO; i++)
			if (toarcproto == tpt->connects[i]) break;
		if (tpt->connects[i] == NOARCPROTO) continue;

		/* potential portinst: get information about its position */
		(void)us_portdistance(tonodeinst, tpt, prefx, prefy, &tx, &ty, trwid, 1);

		dist = abs(tx-prefx) + abs(ty-prefy);
		if (dist > bestdist) continue;
		toportproto = tpt;   bestdist = dist;
		besttx = tx;         bestty = ty;
	}
	if (toportproto == NOPORTPROTO) return(NOARCINST);

	/*
	 * figure out whether the jog will run horizontally or vertically.
	 * Use directionality constraints if they exist
	 */
	if (((fromportproto->userbits&PORTARANGE) >> PORTARANGESH) != 180)
	{
		i = us_bottomrecurse(fromnodeinst, fromportproto);
	} else if (((toportproto->userbits&PORTARANGE) >> PORTARANGESH) != 180)
	{
		i = us_bottomrecurse(tonodeinst, toportproto);
	} else if ((prefy > bestfy && prefy > bestty) || (prefy < bestfy && prefy < bestty))
	{
		/* jog is horizontal if prefy is above or below both ports */
		i = 900;
	} else if ((prefx > bestfx && prefx > besttx) || (prefx < bestfx && prefx < besttx))
	{
		/* jog is vertical if prefx is to right or left of both ports */
		i = 0;
	} else
	{
		/* if area between nodes is wider than tall, jog is vertical */
		if (abs(bestfx-besttx) > abs(bestfy-bestty)) i = 0; else i = 900;
	}
	i = (i+450) % 1800;   if (i < 0) i += 1800;
	gridalign(&prefx, &prefx, us_alignment);
	if (i < 900)
	{
		xi1 = xi2 = prefx;
		yi1 = bestfy;   yi2 = bestty;
	} else
	{
		xi1 = bestfx;   xi2 = besttx;
		yi1 = yi2 = prefy;
	}

	/* figure out what primitive nodeproto connects these arcs */
	fpin = getpinproto(fromarcproto);
	if (fpin == NONODEPROTO)
	{
		us_abortcommand(_("No pin for %s arcs"), describearcproto(fromarcproto));
		return(NOARCINST);
	}
	defaultnodesize(fpin, &fpxs, &fpys);
	defaultnodesize(con, &tpxs, &tpys);

	/* run the arcs */
	if (fakecoords == 0)
	{
		/* create the intermediate nodeinsts */
		lx = xi1 - fpxs/2;   hx = lx + fpxs;
		ly = yi1 - fpys/2;   hy = ly + fpys;
		ni1 = newnodeinst(fpin, lx,hx, ly,hy, 0, 0, fromnodeinst->parent);
		if (ni1 == NONODEINST)
		{
			us_abortcommand(_("Cannot create first connecting pin"));
			return(NOARCINST);
		}
		endobjectchange((INTBIG)ni1, VNODEINST);
		lx = xi2 - tpxs/2;   hx = lx + tpxs;
		ly = yi2 - tpys/2;   hy = ly + tpys;
		ni2 = newnodeinst(con, lx,hx, ly,hy, 0, 0, fromnodeinst->parent);
		if (ni2 == NONODEINST)
		{
			us_abortcommand(_("Cannot create second connecting pin"));
			return(NOARCINST);
		}
		endobjectchange((INTBIG)ni2, VNODEINST);

		*alt1 = us_runarcinst(fromnodeinst, fromportproto, bestfx, bestfy, ni1,
			fpin->firstportproto, xi1,yi1, fromarcproto, wid);
		if (*alt1 == NOARCINST) return(NOARCINST);
		ai = us_runarcinst(ni1,fpin->firstportproto, xi1,yi1, ni2,con->firstportproto,
			xi2,yi2, fromarcproto, wid);
		if (ai == NOARCINST) return(NOARCINST);
		*alt2 = us_runarcinst(ni2,con->firstportproto, xi2,yi2, tonodeinst,
			toportproto, besttx, bestty, toarcproto, wid);
		if (*alt2 == NOARCINST) return(NOARCINST);
		if (fromarcproto == toarcproto)
		{
			ttyputmsg(_("Created 3 %s arcs"), fromarcproto->protoname);
		} else
		{
			ttyputmsg(_("Created 2 %s arcs and 1 %s arc"), fromarcproto->protoname,
				toarcproto->protoname);
		}
		return(ai);
	}

	/* record the fake */
	fakecoords[0] = bestfx;
	fakecoords[1] = bestfy;
	fakecoords[2] = xi1;
	fakecoords[3] = yi1;
	fakecoords[4] = xi2;
	fakecoords[5] = yi2;
	fakecoords[6] = besttx;
	fakecoords[7] = bestty;
	*alt1 = (ARCINST *)1;
	*alt2 = (ARCINST *)1;
	return((ARCINST *)1);
}

/*
 * run an arcinst from portproto "fromportproto" of nodeinst "fromnodeinst" at
 * (fromx,fromy) to portproto "toportproto" of nodeinst "tonodeinst" at
 * (tox,toy).  The type of the arcinst is "ap" and the width is "wid".  The
 * routine returns the address of the newly created arcinst (NOARCINST on
 * error).
 */
ARCINST *us_runarcinst(NODEINST *fromnodeinst, PORTPROTO *fromportproto, INTBIG fromx, INTBIG fromy,
	NODEINST *tonodeinst, PORTPROTO *toportproto, INTBIG tox, INTBIG toy, ARCPROTO *ap, INTBIG wid)
{
	REGISTER ARCINST *ai;
	REGISTER INTBIG bits;

	/* see if nodes need to be undrawn to account for "Steiner Point" changes */
	if ((fromnodeinst->proto->userbits&WIPEON1OR2) != 0)
		startobjectchange((INTBIG)fromnodeinst, VNODEINST);
	if ((tonodeinst->proto->userbits&WIPEON1OR2) != 0)
		startobjectchange((INTBIG)tonodeinst, VNODEINST);

	/* create the arcinst */
	bits = us_makearcuserbits(ap);
	ai = newarcinst(ap, wid, bits, fromnodeinst, fromportproto, fromx,fromy,
		tonodeinst, toportproto, tox, toy, fromnodeinst->parent);
	if (ai == NOARCINST)
	{
		us_abortcommand(_("Problem creating the arc"));
		return(NOARCINST);
	}
	ai->changed = 0;
	endobjectchange((INTBIG)ai, VARCINST);
	us_setarcproto(ap, 0);

	/* see if nodes need to be redrawn to account for "Steiner Point" changes */
	if ((fromnodeinst->proto->userbits&WIPEON1OR2) != 0)
		endobjectchange((INTBIG)fromnodeinst, VNODEINST);
	if ((tonodeinst->proto->userbits&WIPEON1OR2) != 0)
		endobjectchange((INTBIG)tonodeinst, VNODEINST);
	return(ai);
}

/*
 * routine to find the width of the widest arcinst of type "ap" connected
 * to any port of nodeinst "ni" (if "ni" is primitive) or to port "por" of
 * nodeinst "ni" (if "ni" is complex).
 */
INTBIG us_widestarcinst(ARCPROTO *ap, NODEINST *ni, PORTPROTO *por)
{
	REGISTER INTBIG wid;
	REGISTER INTSML pindex;
	REGISTER PORTARCINST *pi;

	/* look at all arcs on the nodeinst */
	wid = 0;
	pindex = ni->proto->primindex;
	if (por == NOPORTPROTO) pindex = 1;
	for(;;)
	{
		for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
		{
			if (pindex == 0 && pi->proto != por) continue;
			if (pi->conarcinst->proto != ap) continue;
			if (pi->conarcinst->width > wid) wid = pi->conarcinst->width;
		}

		/* descend to the next level in the hierarchy */
		if (pindex != 0) break;
		ni = por->subnodeinst;
		pindex = ni->proto->primindex;
		por = por->subportproto;
	}
	return(wid);
}

/*
 * Routine to determine the proper width of arc of type "typ" with width "wid" and running from node
 * "ni1", port "pp1" to node "ni2", port "pp2".  Oversize nodes are considered when sizing the arc.
 */
INTBIG us_stretchtonodes(ARCPROTO *typ, INTBIG wid, NODEINST *ni1, PORTPROTO *pp1, INTBIG otherx, INTBIG othery)
{
#if 0		/* this turns out to be a bad idea.  So it is turned off. */
	REGISTER INTBIG xstretch, ystretch, i, rot, cx, cy;
	INTBIG ni1x, ni1y;

	/* see if node 1 is stretched */
	xstretch = (ni1->highx - ni1->lowx) - (ni1->proto->highx - ni1->proto->lowx);
	ystretch = (ni1->highy - ni1->lowy) - (ni1->proto->highy - ni1->proto->lowy);
	if (xstretch > 0 || ystretch > 0)
	{
		rot = (ni1->rotation + 900 * ni1->transpose) % 3600;
		switch (rot)
		{
			case 0:
			case 1800:
				break;
			case 900:
			case 2700:
				i = xstretch;   xstretch = ystretch;   ystretch = i;
				break;
			default:
				return(wid);
		}
		portposition(ni1, pp1, &ni1x, &ni1y);
		cx = (ni1->lowx + ni1->highx) / 2;
		cy = (ni1->lowy + ni1->highy) / 2;
		if (ni1x == cx && ni1y == cy)
		{
			if (abs(ni1x-otherx) > abs(ni1y-othery))
			{
				/* horizontal wire: see if Y stretch allows growth */
				if (ystretch > wid - typ->nominalwidth) wid = ystretch + typ->nominalwidth;
			} else
			{
				/* vertical wire: see if X stretch allows growth */
				if (xstretch > wid - typ->nominalwidth) wid = xstretch + typ->nominalwidth;
			}
		} else
		{
			if (abs(ni1x-cx) > abs(ni1y-cy))
			{
				/* horizontal wire: see if Y stretch allows growth */
				if (ystretch > wid - typ->nominalwidth) wid = ystretch + typ->nominalwidth;
			} else
			{
				/* vertical wire: see if X stretch allows growth */
				if (xstretch > wid - typ->nominalwidth) wid = xstretch + typ->nominalwidth;
			}
		}
	}
#endif
	return(wid);
}

/*
 * Routine to determine the width to use for arc "typ", given a default width of "wid", that it
 * runs from "ni1/pp1" towards (ox1,oy1) and to "ni2/pp2" towards (ox2,oy2).  If either end is
 * a pin, the width calculation is not performed for it.
 */
INTBIG us_figurearcwidth(ARCPROTO *typ, INTBIG wid, NODEINST *ni1, PORTPROTO *pp1, INTBIG ox1, INTBIG oy1,
	NODEINST *ni2, PORTPROTO *pp2, INTBIG ox2, INTBIG oy2)
{
	INTBIG wid1, wid2, stretchwid, stretch1, stretch2, fun;
	REGISTER NODEINST *rni;
	REGISTER PORTPROTO *rpp;

	stretch1 = stretch2 = 1;

	/* see if node 1 is a pin */
	rni = ni1;   rpp = pp1;
	while (rni->proto->primindex == 0)
	{
		rni = rpp->subnodeinst;
		rpp = rpp->subportproto;
	}
	fun = nodefunction(rni);
	if (fun == NPPIN) stretch1 = 0;

	/* see if node 2 is a pin */
	rni = ni2;   rpp = pp2;
	while (rni->proto->primindex == 0)
	{
		rni = rpp->subnodeinst;
		rpp = rpp->subportproto;
	}
	fun = nodefunction(rni);
	if (fun == NPPIN) stretch2 = 0;

	if (stretch1 == 0 && stretch2 == 0) return(wid);

	wid1 = us_stretchtonodes(typ, wid, ni1, pp1, ox1, oy1);
	wid2 = us_stretchtonodes(typ, wid, ni2, pp2, ox2, oy2);
	if (stretch1 == 0) wid1 = wid2;
	if (stretch2 == 0) wid2 = wid1;
	stretchwid = mini(wid1, wid2);
	if (stretchwid > wid) wid = stretchwid;
	return(wid);
}

/*
 * routine to determine the nodeinst to be used when connecting the geometry
 * module pointed to by "ipos" and the geometry module in "othergeom".
 * The port prototype on the other object ("othergeom") is in "otherport" and
 * the port prototype on the object in "ipos" is in "ipp".  The prefered site
 * of connection is in (prefx, prefy).  If the first module (ipos) is an
 * arcinst, it may be split into two arcs and a nodeinst in which case that
 * nodeinst will be returned and the address of the geometry module will be
 * changed to point to that nodeinst.  If "fake" is nonzero, the newly created
 * node will be a fake one (not really created) for the purposes of determining
 * an intended connection only.
 */
NODEINST *us_getnodeonarcinst(GEOM **ipos, PORTPROTO **ipp, GEOM *othergeom,
	PORTPROTO *otherport, INTBIG prefx, INTBIG prefy, INTBIG fake)
{
	REGISTER ARCINST *ai, *oar, *ar1,*ar2;
	REGISTER ARCPROTO *ap;
	REGISTER NODEINST *ni, *fno, *tno;
	REGISTER PORTPROTO *fpt, *tpt, *pt1;
	REGISTER NODEPROTO *np, *pnt;
	REGISTER GEOM *geom;
	REGISTER INTBIG wid, bits1, bits2, lx, hx, ly, hy;
	INTBIG fendx, fendy, tendx, tendy, pxs, pys;
	static POLYGON *poly = NOPOLYGON;

	/* get the actual geometry modules */
	geom = *ipos;

	/* if the module is a nodeinst, return it */
	if (geom->entrytype == OBJNODEINST) return(geom->entryaddr.ni);

	/* if the other module is a primitive node, use center as break point */
	if (othergeom->entrytype == OBJNODEINST)
	{
		ni = othergeom->entryaddr.ni;
		if (otherport != NOPORTPROTO)
		{
			/* make sure there is a polygon */
			if (poly == NOPOLYGON) poly = allocstaticpolygon(1, us_tool->cluster);

			/* get the polygon describing the port */
			poly->xv[0] = prefx;   poly->yv[0] = prefy;   poly->count = 1;
			shapeportpoly(ni, otherport, poly, 1);

			/* determine the center of the polygon */
			getcenter(poly, &prefx, &prefy);
		} else if (ni->proto->primindex != 0)
		{
			prefx = (ni->lowx + ni->highx) / 2;
			prefy = (ni->lowy + ni->highy) / 2;
		}
	}

	/* find point on this arcinst closest to break point */
	ai = geom->entryaddr.ai;
	if (ai->end[0].xpos == ai->end[1].xpos)
	{
		/* vertical arcinst */
		if (othergeom->entrytype == OBJARCINST)
		{
			/* if two arcs are perpendicular, find point of intersection */
			oar = othergeom->entryaddr.ai;
			if (oar->end[0].ypos == oar->end[1].ypos) prefy=oar->end[0].ypos;
		}
	} else if (ai->end[0].ypos == ai->end[1].ypos)
	{
		/* horizontal arcinst */
		if (othergeom->entrytype == OBJARCINST)
		{
			/* if two arcs are perpendicular, find point of intersection */
			oar = othergeom->entryaddr.ai;
			if (oar->end[0].xpos == oar->end[1].xpos) prefx=oar->end[0].xpos;
		}
	}

	/* adjust to closest point */
	(void)closestpointtosegment(ai->end[0].xpos,ai->end[0].ypos,
		ai->end[1].xpos,ai->end[1].ypos, &prefx, &prefy);
	if (prefx == ai->end[0].xpos && prefy == ai->end[0].ypos)
	{
		*ipp = ai->end[0].portarcinst->proto;
		*ipos = ai->end[0].nodeinst->geom;
		return(ai->end[0].nodeinst);
	}
	if (prefx == ai->end[1].xpos && prefy == ai->end[1].ypos)
	{
		*ipp = ai->end[1].portarcinst->proto;
		*ipos = ai->end[1].nodeinst->geom;
		return(ai->end[1].nodeinst);
	}

	/* break is at (prefx, prefy): save information about the arcinst */
	fno = ai->end[0].nodeinst;    fpt = ai->end[0].portarcinst->proto;
	tno = ai->end[1].nodeinst;    tpt = ai->end[1].portarcinst->proto;
	fendx = ai->end[0].xpos;      fendy = ai->end[0].ypos;
	tendx = ai->end[1].xpos;      tendy = ai->end[1].ypos;
	ap = ai->proto;   wid = ai->width;  pnt = ai->parent;
	bits1 = bits2 = ai->userbits;
	if ((bits1&ISNEGATED) != 0)
	{
		if ((bits1&REVERSEEND) == 0) bits2 &= ~ISNEGATED; else
			bits1 &= ~ISNEGATED;
	}
	if (figureangle(fendx,fendy, prefx,prefy) != figureangle(prefx,prefy, tendx,tendy))
	{
		bits1 &= ~FIXANG;
		bits2 &= ~FIXANG;
	}

	/* create the splitting pin */
	np = us_findconnectingpin(ap, othergeom, otherport);
	if (np == NONODEPROTO) return(NONODEINST);
	defaultnodesize(np, &pxs, &pys);
	pt1 = np->firstportproto;
	lx = prefx - pxs/2;   hx = lx + pxs;
	ly = prefy - pys/2;   hy = ly + pys;
	if (fake != 0)
	{
		ni = dummynode();
		ni->lowx = lx;   ni->highx = hx;
		ni->lowy = ly;   ni->highy = hy;
		ni->proto = np;
	} else
	{
		ni = newnodeinst(np, lx,hx, ly,hy, 0, 0, pnt);
		if (ni == NONODEINST)
		{
			ttyputerr(_("Cannot create splitting pin"));
			return(NONODEINST);
		}
		endobjectchange((INTBIG)ni, VNODEINST);

		/* delete the old arcinst */
		startobjectchange((INTBIG)ai, VARCINST);
		if (killarcinst(ai)) ttyputerr(_("Error deleting original arc"));

		/* create the two new arcinsts */
		ar1 = newarcinst(ap, wid, bits1, fno, fpt, fendx, fendy, ni, pt1, prefx, prefy, pnt);
		ar2 = newarcinst(ap, wid, bits2, ni, pt1, prefx, prefy, tno, tpt, tendx, tendy, pnt);
		if (ar1 == NOARCINST || ar2 == NOARCINST)
		{
			ttyputerr(_("Error creating the split arc parts"));
			return(ni);
		}
		(void)copyvars((INTBIG)ai, VARCINST, (INTBIG)ar1, VARCINST);
		endobjectchange((INTBIG)ar1, VARCINST);
		endobjectchange((INTBIG)ar2, VARCINST);
	}

	/* return pointers to the splitting pin */
	*ipos = ni->geom;
	*ipp = pt1;
	return(ni);
}

/*
 * Routine to find the connecting node that can join arcs of type "ap" with
 * an object "geom" (which may be a node with port "pp").  Returns NONODEPROTO
 * if no connecting node can be found.
 */
NODEPROTO *us_findconnectingpin(ARCPROTO *ap, GEOM *geom, PORTPROTO *pp)
{
	REGISTER NODEPROTO *np;
	REGISTER ARCINST *ai;
	REGISTER PORTPROTO *firstpp;
	REGISTER TECHNOLOGY *tech;
	REGISTER INTBIG fun, i, j;

	/* first presume the pin that connects this type of arc */
	np = getpinproto(ap);

	/* if there is no other object, use this pin */
	if (geom == NOGEOM) return(np);

	/* ensure that it connects to this */
	if (geom->entrytype == OBJNODEINST)
	{
		if (pp == NOPORTPROTO) return(np);
		for(i=0; pp->connects[i] != NOARCPROTO; i++)
		{
			if (pp->connects[i] == ap) break;

			/* special case: bus arc can connect to a node with a wire-pin port */
			if (ap == sch_busarc && pp->connects[i] == sch_wirearc) break;
		}
		if (pp->connects[i] == NOARCPROTO) np = NONODEPROTO;
	} else
	{
		ai = geom->entryaddr.ai;
		if (ai->proto != ap) np = NONODEPROTO;
	}
	if (np == NONODEPROTO)
	{
		/* doesn't connect: look for a contact */
		tech = ap->tech;
		for(np = tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			fun = (np->userbits&NFUNCTION) >> NFUNCTIONSH;
			if (fun != NPCONTACT) continue;
			firstpp = np->firstportproto;

			/* make sure the original arc connects to this contact */
			for(i=0; firstpp->connects[i] != NOARCPROTO; i++)
				if (firstpp->connects[i] == ap) break;
			if (firstpp->connects[i] == NOARCPROTO) continue;

			/* make sure the other object connects to this contact */
			for(i=0; firstpp->connects[i] != NOARCPROTO; i++)
			{
				if (firstpp->connects[i]->tech != tech) continue;
				if (geom->entrytype == OBJNODEINST)
				{
					for(j=0; pp->connects[j] != NOARCPROTO; j++)
						if (pp->connects[j] == firstpp->connects[i]) break;
					if (pp->connects[j] != NOARCPROTO) break;
				} else
				{
					ai = geom->entryaddr.ai;
					if (ai->proto == firstpp->connects[i]) break;
				}
			}
			if (firstpp->connects[i] != NOARCPROTO) break;
		}
	}
	return(np);
}

/*
 * routine to report the distance of point (prefx,prefy) to port "pt" of node
 * instance "ni".  The closest point on the polygon is returned in (x,y),
 * given that the port will connect to an arc with width "wid".  If "purpose"
 * is nonzero, a new sub-port location within the port is desired from
 * the "shapeportpoly" routine.  Euclidean distance is not always used, but
 * at least the metric is consistent with itself.
 */
INTBIG us_portdistance(NODEINST *ni, PORTPROTO *pt, INTBIG prefx, INTBIG prefy, INTBIG *x,
	INTBIG *y, INTBIG wid, INTSML purpose)
{
	static POLYGON *poly = NOPOLYGON;
	REGISTER INTSML i, j;
	REGISTER INTBIG bestdist, px, py, nx, ny;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, us_tool->cluster);

	/* specify prefered location of new port */
	poly->xv[0] = prefx;   poly->yv[0] = prefy;   poly->count = 1;
	shapeportpoly(ni, pt, poly, purpose);
	switch (poly->style)
	{
		case FILLED:
			/* reduce the port area to the proper amount for this arc */
			reduceportpoly(poly, ni, pt, wid);

			/* for filled polygon, see if point is inside */
			if (isinside(prefx, prefy, poly) != 0)
			{
				*x = prefx;   *y = prefy;
				return(0);
			}

			*x = prefx;   *y = prefy;
			closestpoint(poly, x, y);
			return(0);

		case OPENED:
		case CLOSED:
			/* for OPENED/CLOSED polygons look for proximity to a vertex */
			bestdist = abs(poly->xv[0] - prefx) + abs(poly->yv[0] - prefy);
			*x = poly->xv[0];   *y = poly->yv[0];
			for(j=1; j<poly->count; j++)
			{
				i = abs(poly->xv[j] - prefx) + abs(poly->yv[j] - prefy);
				if (i < bestdist)
				{
					bestdist = i;
					*x = poly->xv[j];   *y = poly->yv[j];
				}
			}

			/* additionally, look for proximity to an edge */
			for(j=0; j<poly->count; j++)
			{
				if (j == 0)
				{
					if (poly->style == OPENED) continue;
					px = poly->xv[poly->count-1];
					py = poly->yv[poly->count-1];
				} else
				{
					px = poly->xv[j-1];
					py = poly->yv[j-1];
				}
				nx = poly->xv[j];
				ny = poly->yv[j];

				/* handle vertical line that is perpendicular to point */
				if (px == nx && maxi(py, ny) >= prefy && mini(py, ny) <= prefy &&
					abs(px - prefx) < bestdist)
				{
					bestdist = i;
					*x = px;
					*y = prefy;
				}

				/* handle horizontal line that is perpendicular to point */
				if (py == ny && maxi(px, nx) >= prefx && mini(px, nx) <= prefx &&
					abs(py - prefy) < bestdist)
				{
					bestdist = i;
					*x = prefx;
					*y = py;
				}
			}
			return(bestdist);
	}

	/* bogus answer for unusual shapes!!! */
	return(0);
}

/*
 * routine to determine whether port "pp" of node "ni" can connect to an
 * arc at angle "angle" within range "range".  Returns nonzero if the
 * connection cannot be made.
 */
INTSML us_fitportangle(NODEINST *ni, PORTPROTO *pp, INTSML angle, INTSML range)
{
	REGISTER INTSML j;

	j = us_bottomrecurse(ni, pp);
	j = (j - angle) % 3600;   if (j < 0) j += 3600;
	if (j > 1800) j = 3600 - j;
	if (j > range) return(1);
	return(0);
}

/*
 * routine to recurse to the bottom (most primitive node) of a port and
 * compute the port orientation from the bottom up.
 */
INTSML us_bottomrecurse(NODEINST *ni, PORTPROTO *pp)
{
	REGISTER INTSML k;

	if (ni->proto->primindex == 0)
		k = us_bottomrecurse(pp->subnodeinst, pp->subportproto); else
			k = (INTSML)(((pp->userbits&PORTANGLE) >> PORTANGLESH) * 10);
	k += ni->rotation;
	if (ni->transpose != 0) k = 2700 - k;
	return(k);
}

/*
 * Routine to figure out the proper "cursor location" given that arcs are to be
 * drawn from "fromgeom/fromport" to "togeom/toport" and that the cursor is at
 * (xcur,ycur).  Changes (xcur,ycur) to be the proper location.
 */
INTSML us_figuredrawpath(GEOM *fromgeom, PORTPROTO *fromport, GEOM *togeom, PORTPROTO *toport,
	INTBIG *xcur, INTBIG *ycur)
{
	static POLYGON *poly = NOPOLYGON, *poly2 = NOPOLYGON;
	REGISTER NODEINST *tonode, *fromnode;
	INTBIG flx, fhx, fly, fhy, tlx, tly, thx, thy, lx, hx, ly, hy, overx[2], overy[2],
		cornerx[4], cornery[4];
	REGISTER INTBIG k, overcount, fx, fy, tx, ty, c1x, c1y, c2x, c2y,
		c1dist, c2dist, pxs, pys, dist, ox, oy, i, bestdist, j, x, y;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, us_tool->cluster);
	if (poly2 == NOPOLYGON) poly2 = allocstaticpolygon(4, us_tool->cluster);

	if (fromgeom->entrytype != OBJNODEINST || fromport == NOPORTPROTO)
	{
		us_abortcommand(_("Cannot connect to %s: it has no ports"),
			geomname(fromgeom));
		return(1);
	}
	fromnode = fromgeom->entryaddr.ni;

	ox = *xcur;   oy = *ycur;
	gridalign(xcur, ycur, us_alignment);

	/* user dragged over another object: connect them */
	tonode = us_getnodeonarcinst(&togeom, &toport, fromgeom,
		fromport, *xcur, *ycur, 1);
	if (tonode == NONODEINST)
	{
		us_abortcommand(_("Cannot find a way to connect these objects"));
		return(1);
	}
	if (toport == NOPORTPROTO)
	{
		us_abortcommand(_("Cannot connect to %s: it has no ports"),
			geomname(togeom));
		return(1);
	}

	/* change cursor according to port angle */
	shapeportpoly(fromnode, fromport, poly2, 0);
	getbbox(poly2, &flx, &fhx, &fly, &fhy);
	shapeportpoly(tonode, toport, poly2, 0);
	getbbox(poly2, &tlx, &thx, &tly, &thy);
	lx = mini(flx, tlx);
	hx = maxi(fhx, thx);
	ly = mini(fly, tly);
	hy = maxi(fhy, thy);
	dist = maxi(hx-lx, hy-ly);
	overcount = 0;
	if (((fromport->userbits&PORTARANGE) >> PORTARANGESH) != 180)
	{
		k = us_bottomrecurse(fromnode, fromport) % 3600;
		if (k < 0) k += 3600;
		overx[overcount] = (flx+fhx)/2 + mult(cosine((INTSML)k), dist);
		overy[overcount] = (fly+fhy)/2 + mult(sine((INTSML)k), dist);
		overcount++;
	}
	if (((toport->userbits&PORTARANGE) >> PORTARANGESH) != 180)
	{
		k = us_bottomrecurse(tonode, toport) % 3600;
		if (k < 0) k += 3600;
		overx[overcount] = (tlx+thx)/2 + mult(cosine((INTSML)k), dist);
		overy[overcount] = (tly+thy)/2 + mult(sine((INTSML)k), dist);
		overcount++;
	}
	if (overcount == 2)
	{
		pxs = (overx[0] + overx[1]) / 2;
		pys = (overy[0] + overy[1]) / 2;
	} else
	{
		pxs = (flx + fhx) / 2;
		pys = (tly + thy) / 2;
		if (overcount > 0)
		{
			pxs = overx[0];   pys = overy[0];
		} else
		{
			/* use slight cursor differences to pick a corner */
			fx = (flx+fhx)/2;
			fy = (fly+fhy)/2;
			tx = (tlx+thx)/2;
			ty = (tly+thy)/2;
			c1x = fx;   c1y = ty;
			c2x = tx;   c2y = fy;
			c1dist = computedistance(ox, oy, c1x, c1y) +
				computedistance(c1x, c1y, fx, fy);
			c2dist = computedistance(ox, oy, c2x, c2y) +
				computedistance(c2x, c2y, fx, fy);
			if (c1dist < c2dist)
			{
				pxs = c1x;   pys = c1y;
			} else
			{
				pxs = c2x;   pys = c2y;
			}
		}
		cornerx[0] = lx-1;   cornery[0] = ly-1;
		cornerx[1] = lx-1;   cornery[1] = hy+1;
		cornerx[2] = hx+1;   cornery[2] = hy+1;
		cornerx[3] = hx+1;   cornery[3] = ly-1;
		for(i=0; i<4; i++)
		{
			dist = abs(cornerx[i]-pxs) + abs(cornery[i]-pys);

			/* LINTED "bestdist" used in proper order */
			if (i == 0 || dist < bestdist)
			{
				bestdist = dist;
				j = i;
			}
		}
		pxs = cornerx[j]*2 - (lx+hx)/2;
		pys = cornery[j]*2 - (ly+hy)/2;

		/* set coordinate in the center if the ports overlap in either axis */
		if (fromport != NOPORTPROTO)
		{
			shapeportpoly(fromnode, fromport, poly, 0);
			getbbox(poly, &flx, &fhx, &fly, &fhy);
			shapeportpoly(tonode, toport, poly, 0);
			getbbox(poly, &tlx, &thx, &tly, &thy);
			lx = maxi(flx, tlx);   hx = mini(fhx, thx);
			ly = maxi(fly, tly);   hy = mini(fhy, thy);
			if (lx <= hx || ly <= hy)
			{
				pxs = (lx + hx) / 2;
				pys = (ly + hy) / 2;
			}
		}
	}

	/* isolated ports simply use cursor location */
	if ((fromport != NOPORTPROTO &&
		(fromport->userbits&PORTISOLATED) != 0) ||
			(toport->userbits&PORTISOLATED) != 0)
	{
		portposition(fromnode, fromport, &lx, &ly);
		portposition(tonode, toport, &hx, &hy);
		if (hx < lx)
		{
			x = lx;   lx = hx;   hx = x;
		}
		if (hy < ly)
		{
			y = ly;   ly = hy;   hy = y;
		}

		/* if cursor location is within range of ports, use it */
		if (*xcur >= lx && *xcur <= hx) pxs = *xcur;
		if (*ycur >= ly && *ycur <= hy) pys = *ycur;
	}
	*xcur = pxs;
	*ycur = pys;
	return(0);
}
